/**
 * copyright 2004, Jean-Christophe Hoelt <jeko@ios-software.com>
 *
 * This program is released under the terms of the GNU Lesser General Public Licence.
 */
%{
    #include <stdio.h>
    #include <string.h>
    #include "goom_script_scanner.h"
    
    int yylex(void);
    void yyerror(char *);
    extern GoomScriptScanner *currentScanner;

    static NodeType *nodeNew(const char *str, int type);
    static void nodeFreeInternals(NodeType *node);
    static void nodeFree(NodeType *node);

    static void commit_node(NodeType *node);
    static void precommit_node(NodeType *node);

    static NodeType *new_constInt(const char *str);
    static NodeType *new_constFloat(const char *str);
    static NodeType *new_var(const char *str);
    static NodeType *new_param(const char *str);
    static NodeType *new_nop(const char *str);
    static NodeType *new_op(const char *str, int type, int nbOp);

    static int allocateLabel();
    #define allocateTemp allocateLabel

    /* SETTER */
    static NodeType *new_set(NodeType *lvalue, NodeType *expression) {
        NodeType *set = new_op("set", OPR_SET, 2);
        set->opr.op[0] = lvalue;
        set->opr.op[1] = expression;
        return set;
    }
    static void commit_set(NodeType *set) {
        precommit_node(set->opr.op[1]);
        printf("set.f %s %s\n", set->opr.op[0]->str, set->opr.op[1]->str);
        currentScanner->instr = instr_init(currentScanner, "set.f", INSTR_SETF, 2);
        commit_node(set->opr.op[0]);
        commit_node(set->opr.op[1]);
    }

    /* FLOAT */
    static NodeType *new_float_decl(NodeType *name) {
        NodeType *fld = new_op("float", OPR_DECLARE_FLOAT, 1);
        fld->opr.op[0] = name;
        return fld;
    }
    static void commit_float(NodeType *var) {
        printf("float %s\n", var->opr.op[0]->str);
        currentScanner->instr = instr_init(currentScanner, "float", INSTR_INT, 1);
        commit_node(var->opr.op[0]);
    }
    
    /* INT */
    static NodeType *new_int_decl(NodeType *name) {
        NodeType *intd = new_op("int", OPR_DECLARE_INT, 1);
        intd->opr.op[0] = name;
        return intd;
    }
    static void commit_int(NodeType *var) {
        printf("int %s\n", var->opr.op[0]->str);
        currentScanner->instr = instr_init(currentScanner, "int", INSTR_INT, 1);
        commit_node(var->opr.op[0]);
    }

    /* commodity method for add, mult, ... */

    static int is_tmp_expr(NodeType *node) {
        return node->str && !strncmp(node->str,"__tmp",5);
    }

    static void precommit_expr(NodeType *expr, const char *type, int instr_id) {

        char stmp[256];
        NodeType *tmp;
        int toAdd;

        /* compute "left" and "right" */
        precommit_node(expr->opr.op[0]);
        precommit_node(expr->opr.op[1]);

        if (is_tmp_expr(expr->opr.op[0])) {
            strcpy(stmp, expr->opr.op[0]->str);
            toAdd = 1;
        }
        else if (is_tmp_expr(expr->opr.op[1])) {
            strcpy(stmp,expr->opr.op[1]->str);
            toAdd = 0;
        }
        else {
            /* declare a float to store the result */
            sprintf(stmp,"__tmp%i",allocateTemp());
            commit_node(new_float_decl(new_var(stmp)));
            /* set the float to the value of "op1" */
            commit_node(new_set(new_var(stmp),expr->opr.op[0]));
            toAdd = 1;
        }

        /* add op2 to tmp */
        printf("%s %s %s\n", type, stmp, expr->opr.op[toAdd]->str);
        currentScanner->instr = instr_init(currentScanner, type, instr_id, 2);
        commit_node(new_var(stmp));
        commit_node(expr->opr.op[toAdd]);
    
        /* redefine the ADD node now as the computed variable */
        nodeFreeInternals(expr);
        tmp = new_var(stmp);
        *expr = *tmp;
        free(tmp);
    }

    static NodeType *new_expr2(const char *name, int id, NodeType *expr1, NodeType *expr2) {
        NodeType *add = new_op(name, id, 2);
        add->opr.op[0] = expr1;
        add->opr.op[1] = expr2;
        return add;
    }

    /* ADD */
    static NodeType *new_add(NodeType *expr1, NodeType *expr2) {
        return new_expr2("add.f", OPR_ADD, expr1, expr2);
    }
    static void precommit_add(NodeType *add) {
        precommit_expr(add,"add.f",INSTR_ADDF);
    }

    /* MUL */
    static NodeType *new_mul(NodeType *expr1, NodeType *expr2) {
        return new_expr2("mul.f", OPR_MUL, expr1, expr2);
    }
    static void precommit_mul(NodeType *mul) {
        precommit_expr(mul,"mul.f",INSTR_MULF);
    }
    
    /* EQU */
    static NodeType *new_equ(NodeType *expr1, NodeType *expr2) {
        return new_expr2("isequal.f", OPR_EQU, expr1, expr2);
    }
    static void precommit_equ(NodeType *mul) {
        precommit_expr(mul,"isequal.f",INSTR_ISEQUALF);
    }
    
    /* INF */
    static NodeType *new_low(NodeType *expr1, NodeType *expr2) {
        return new_expr2("islower.f", OPR_LOW, expr1, expr2);
    }
    static void precommit_low(NodeType *mul) {
        precommit_expr(mul,"islower.f",INSTR_ISLOWERF);
    }
    
    /* IF */
    static NodeType *new_if(NodeType *expression, NodeType *instr) {
        NodeType *node = new_op("if", OPR_IF, 2);
        node->opr.op[0] = expression;
        node->opr.op[1] = instr;
        return node;
    }
    static void commit_if(NodeType *node) {

        char slab[1024];
        precommit_node(node->opr.op[0]);

        /* jzero.i <expression> <endif> */
        sprintf(slab, "|eif%d|", allocateLabel());
        printf("jzero.i %s %s\n", node->opr.op[0]->str, slab);
        currentScanner->instr = instr_init(currentScanner, "jzero.i", INSTR_JZERO, 2);
        commit_node(node->opr.op[0]);
        instr_add_param(currentScanner->instr, slab, TYPE_LABEL);

        /* [... instrs of the if ...] */
        commit_node(node->opr.op[1]);
        /* label <endif> */
        printf("label %s\n", slab);
        currentScanner->instr = instr_init(currentScanner, "label", INSTR_LABEL, 1); 
        instr_add_param(currentScanner->instr, slab, TYPE_LABEL);
    }

    /* BLOCK */
    static NodeType *new_block(NodeType *lastNode) {
        NodeType *blk = new_op("block", OPR_BLOCK, 2);
        blk->opr.op[0] = new_nop("start_of_block");
        blk->opr.op[1] = lastNode;        
        return blk;
    }
    static void commit_block(NodeType *node) {
        commit_node(node->opr.op[0]->opr.next);
    }

    /** **/

    static NodeType *rootNode = 0; // TODO: reinitialiser a chaque compilation.
    static NodeType *lastNode = 0;
    static NodeType *gsl_append(NodeType *curNode) {
        if (lastNode)
            lastNode->opr.next = curNode;
        lastNode = curNode;
        while(lastNode->opr.next) lastNode = lastNode->opr.next;
        if (rootNode == 0)
            rootNode = curNode;
        return curNode;
    }

    static int lastLabel = 0;
    int allocateLabel() {
        return ++lastLabel;
    }
    void releaseLabel(int n) {
        if (n == lastLabel)
            lastLabel--;
    }

    void gsl_commit_compilation() {
        commit_node(rootNode);
        rootNode = 0;
        lastNode = 0;
    }
    
    void precommit_node(NodeType *node) {
        /* do here stuff for expression.. for exemple */
        switch(node->type) {
            case OPR_NODE:
                switch(node->opr.type) {
                    case OPR_ADD: precommit_add(node); break;
                    case OPR_MUL: precommit_mul(node); break;
                    case OPR_EQU: precommit_equ(node); break;
                    case OPR_LOW: precommit_low(node); break;
                }
        }
    }
    
    void commit_node(NodeType *node) {

        if (node == 0) return;
        
        switch(node->type) {
            case OPR_NODE:
                switch(node->opr.type) {
                    case OPR_DECLARE_FLOAT: commit_float(node); break;
                    case OPR_DECLARE_INT:   commit_int(node);   break;
                    case OPR_SET:           commit_set(node); break;
                    case OPR_IF:            commit_if(node); break;
                    case OPR_BLOCK:         commit_block(node); break;
                    case EMPTY_NODE:        printf("NOP\n"); break;
                }

                commit_node(node->opr.next); /* recursive for the moment, maybe better to do something iterative? */
                break;

            case PARAM_NODE:       instr_add_param(currentScanner->instr, node->str, TYPE_PARAM); break;
            case VAR_NODE:         instr_add_param(currentScanner->instr, node->str, TYPE_VAR); break;
            case CONST_INT_NODE:   instr_add_param(currentScanner->instr, node->str, TYPE_INTEGER); break;
            case CONST_FLOAT_NODE: instr_add_param(currentScanner->instr, node->str, TYPE_FLOAT); break;
        }
        nodeFree(node);
    }
%}

%union {
    int intValue;
    float floatValue;
    char charValue;
    char strValue[2048];
    NodeType *nPtr;
  };
  
%token <strValue>   TYPE_INTEGER
%token <strValue>   TYPE_FLOAT
%token <strValue>   TYPE_VAR TYPE_PARAM

%type <nPtr> expression variable constValue instruction start_block
%left '=' '<'
%left '+'
%left '*'

%token INT_TK FLOAT_TK

%%

gsl: gsl instruction { gsl_append($2); }
   |
   ;

instruction: variable '=' expression ';'            { $$ = new_set($1,$3); }
           | FLOAT_TK TYPE_VAR '=' expression ';'   { $$ = new_float_decl(new_var($2));
                                                      $$->opr.next = new_set(new_var($2), $4); }
           | FLOAT_TK TYPE_VAR ';'                  { $$ = new_float_decl(new_var($2)); }
           | INT_TK TYPE_VAR  ';'                   { $$ = new_int_decl(new_var($2)); }
           | '(' expression ')' '?' instruction     { $$ = new_if($2,$5); }
           | '{' start_block gsl '}'                { lastNode = $2->opr.op[1]; $$=$2; }
           | ';'                                    { $$ = new_nop("nop"); }
           ;

start_block: { $$ = new_block(lastNode); lastNode = $$->opr.op[0]; }
           ;

expression: variable   { $$ = $1; }
          | constValue { $$ = $1; }
          | expression '*' expression { $$ = new_mul($1,$3); } 
          | expression '+' expression { $$ = new_add($1,$3); } 
          | expression '=' expression { $$ = new_equ($1,$3); } 
          | expression '<' expression { $$ = new_low($1,$3); } 
          | '(' expression ')'        { $$ = $2; }
          ;

variable: TYPE_VAR    { $$ = new_var($1);   }
        | TYPE_PARAM  { $$ = new_param($1); }
        ;

constValue: TYPE_FLOAT   { $$ = new_constFloat($1); }
          | TYPE_INTEGER { $$ = new_constInt($1); } 
          ;

%%

    NodeType *nodeNew(const char *str, int type) {
        NodeType *node = (NodeType*)malloc(sizeof(NodeType));
        node->type = type;
        node->str  = (char*)malloc(strlen(str)+1);
        strcpy(node->str, str);
        return node;
    }

    void nodeFreeInternals(NodeType *node) {
        free(node->str);
    }
    
    void nodeFree(NodeType *node) {
        nodeFreeInternals(node);
        free(node);
    }

    NodeType *new_constInt(const char *str) {
        NodeType *node = nodeNew(str, CONST_INT_NODE);
        node->constInt.val = atoi(str);
        return node;
    }

    NodeType *new_constFloat(const char *str) {
        NodeType *node = nodeNew(str, CONST_FLOAT_NODE);
        node->constFloat.val = atof(str);
        return node;
    }

    NodeType *new_var(const char *str) {
        NodeType *node = nodeNew(str, VAR_NODE);
        return node;
    }
    
    NodeType *new_param(const char *str) {
        NodeType *node = nodeNew(str, PARAM_NODE);
        return node;
    }
    
    NodeType *new_nop(const char *str) {
        NodeType *node = new_op(str, EMPTY_NODE, 0);
        return node;
    }
    
    NodeType *new_op(const char *str, int type, int nbOp) {
        int i;
        NodeType *node = nodeNew(str, OPR_NODE);
        node->opr.next = 0;
        node->opr.type = type;
        node->opr.nbOp = nbOp;
        for (i=0;i<nbOp;++i) node->opr.op[i] = 0;
        return node;
    }

void yyerror(char *str) {
    fprintf(stderr, "GSL ERROR: %s\n", str);
}

