/* pkchar.cpp							-*- C++ -*-
   Time-stamp: "97/01/03 17:55:36 mik"

   Copyright (C) 1991, 92, 93, 96, 97
	Christian Schenk  <cschenk@berlin.snafu.de>

   This file is part of MiKTeX.

   MiKTeX is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   MiKTeX is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MiKTeX; if not, write to the Free Software Foundation,
   Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include "pkchar.h"
#include "common.h"

#define twopwr(n) (((dvi_integer) 1) << (n))
const size_t bits_per_rword = 16;
const dvi_byte *_pkchar::pras;
_pkchar::rword_t _pkchar::rword;
dvi_integer _pkchar::rword_weight;
dvi_integer _pkchar::repeat_count;
dvi_integer _pkchar::cur_x;

const dvi_integer
_pkchar::power[32] =
{
  twopwr(0), twopwr(1), twopwr(2), twopwr(3),
  twopwr(4), twopwr(5), twopwr(6), twopwr(7),
  twopwr(8), twopwr(9), twopwr(10), twopwr(11),
  twopwr(12), twopwr(13), twopwr(14), twopwr(15),
  twopwr(16), twopwr(17), twopwr(18), twopwr(19),
  twopwr(20), twopwr(21), twopwr(22), twopwr(23),
  twopwr(24), twopwr(25), twopwr(26), twopwr(27),
  twopwr(28), twopwr(29), twopwr(30), twopwr(31)
};

const dvi_integer
_pkchar::gpower[33] =
{
  twopwr(0)-1, twopwr(1)-1, twopwr(2)-1, twopwr(3)-1,
  twopwr(4)-1, twopwr(5)-1, twopwr(6)-1, twopwr(7)-1,
  twopwr(8)-1, twopwr(9)-1, twopwr(10)-1, twopwr(11)-1,
  twopwr(12)-1, twopwr(13)-1, twopwr(14)-1, twopwr(15)-1,
  twopwr(16)-1, twopwr(17)-1, twopwr(18)-1, twopwr(19)-1,
  twopwr(20)-1, twopwr(21)-1, twopwr(22)-1, twopwr(23)-1,
  twopwr(24)-1, twopwr(25)-1, twopwr(26)-1, twopwr(27)-1,
  twopwr(28)-1, twopwr(29)-1, twopwr(30)-1, twopwr(31)-1,
  -1
};

_pkchar::_pkchar ()
  
  : raster (0),
    praster (0),
    cc (0),
    tfm (0),
    dx (0),
    w (0),
    h (0),
    hoff (0),
    voff (0)
     
{
  dprintf1 ("constructing _pkchar object %p\n", this);
}

_pkchar::~_pkchar ()
     
{
  dprintf1 ("deleting _pkchar object %p\n", this);
  if (praster)
    {
      delete [] praster;
      praster = 0;
    }
  if (raster)
    {
      delete [] raster;
      raster = 0;
    }
}

dvi_bool
_pkchar::isshort () const

{
  return (l3() < 4 ? dvi_true : dvi_false);
}

dvi_bool
_pkchar::isxshort () const

{
  return (l3() < 7 && ! isshort() ? dvi_true : dvi_false);
}

dvi_bool
_pkchar::islong () const

{
  return (l3() == 7 ? dvi_true : dvi_false);
}

dvi_bool
_pkchar::read (_ifwrdstream &file,
	       int flag,
	       dvi_integer z)

{
  _pkchar::flag = flag;
  dvi_integer alpha, beta;	// quantities used in the scaling computation
  
  // replace z by z' and compute alpha/beta
  alpha = 16;
  while (z >= 040000000)
    {
      z /= 2;
      alpha += alpha;
    }
  beta = 256 / alpha;
  alpha *= z;

  if (isshort ())
    {
      // Read character preamble (short form)
      pl = (((dvi_integer) l3() % 4) << 8 | file.readbyte ()) - 8;
      cc = file.readbyte ();
      tfm = file.readtrio ();
      dx = file.readbyte ();
      w = file.readbyte ();
      h = file.readbyte ();
      hoff = file.readsignedbyte ();
      voff = file.readsignedbyte ();
    }
  else if (isxshort ())
    {
      // Read character preamble (extended short form)
      pl = (((dvi_integer) l3() % 4) << 16 | file.readpair ()) - 13;
      cc = file.readbyte ();
      tfm = file.readtrio ();
      dx = file.readpair ();
      w = file.readpair (); 
      h = file.readpair (); 
      hoff = file.readsignedpair ();
      voff = file.readsignedpair ();
    }
  else
    {
      // Read character preamble (long form)
      pl = file.readsignedquad () - 24;
      cc = file.readsignedquad ();
      tfm = file.readsignedquad ();
      dx = file.readsignedquad ();
      dx = (dx + 0x10000) >> 16;
      w = file.readsignedquad ();
      h = file.readsignedquad ();
      hoff = file.readsignedquad ();
      voff = file.readsignedquad ();
    }
  
  // Convert the width value
  dvi_integer b0 = (tfm >> 24) & 0xff;
  dvi_integer b1 = (tfm >> 16) & 0xff;
  dvi_integer b2 = (tfm >> 8) & 0xff;
  dvi_integer b3 = tfm & 0xff;
  
  tfm = (((((b3 * z) / 0400) + (b2 * z)) / 0400) + (b1 * z)) / beta;
  if (b0 == 255)
    tfm -= alpha;
  
  dprintf2 ("Char: '%c' (0x%x)", (char) cc, cc);
  dprintf1 (", tfm=%ld (scaled)", (long) tfm);
  dprintf1 (", dx=%ld", (long) dx);
  dprintf1 (", w=%ld", (long) w);
  dprintf1 (", h=%ld", (long) h);
  dprintf1 (", hoff=%ld", (long) hoff);
  dprintf1 (", voff=%ld\n", (long) voff);

  // Read raster data
  dprintf1 ("reading %ld raster bytes\n", pl);
  praster = new dvi_byte[pl];
  file.read (praster, pl);

  return (dvi_true);
}

int
_pkchar::getbyte () const

{
  return (*pras++);
}

int
_pkchar::getnybble ()

{
  int temp;
  if (bit_weight == 0)
    {
      input_byte = getbyte ();
      bit_weight = 16;
    }
  temp = input_byte / bit_weight;
  input_byte -= temp * bit_weight;
  bit_weight /= 16;
  return (temp);
}

dvi_bool
_pkchar::getbit ()

{
  dvi_bool temp;
   
  bit_weight /= 2;
  if (bit_weight == 0)
    {
      input_byte = getbyte ();
      bit_weight = 128;
    }

  temp = (input_byte >= bit_weight ? dvi_true : dvi_false);
  if (temp)
    input_byte -= bit_weight;

  return (temp);
}

dvi_integer
_pkchar::getpackednumber ()

{
  dvi_integer i = getnybble ();
  dvi_integer j;
  if (i == 0)
    {
      do
	{
	  j = getnybble ();
	  i++;
	}
      while (j == 0);
      while (i > 0)
	{
	  j = j * 16 + getnybble ();
	  i--;
	}
      return (j - 15 + (13 - dyn_f) * 16 + dyn_f);
    }
  else if (i <= dyn_f)
    return (i);
  else if (i < 14)
    return ((i - dyn_f - 1 ) * 16 + getnybble () + dyn_f + 1);
  else
    {
      if (i == 14)
	repeat_count = getpackednumber ();
      else
	repeat_count = 1;
      return (getpackednumber ());
    }
}

dvi_integer _pkchar::rword_width;
int _pkchar::dyn_f;
dvi_bool _pkchar::turn_on;
int _pkchar::input_byte;
int _pkchar::bit_weight;

void
_pkchar::unpack ()

{
  pras = praster;
  rword_width = (w + bits_per_rword - 1) / bits_per_rword;
  dprintf1 ("rword_width=%ld\n", rword_width);
  if (w == 0 || h == 0)
    return;			// shouldn't happen
  raster = new rword_t[h * rword_width];
  rword_t *ras = raster;
  dyn_f = flag >> 4;
  cur_x = 0;
  turn_on = (flag & 8 ? dvi_true : dvi_false);
  bit_weight = 0;
  if (dyn_f == 14)
    {
      // Get raster by bits
      bit_weight = 0;
      for (dvi_integer i = 1; i <= h; i++)
	{
	  rword = 0;
	  rword_weight = bits_per_rword-1;
	  for (dvi_integer j = 1; j <= w; j++)
	    {
	      if (getbit ())
		rword += power[rword_weight];
	      rword_weight--;
	      if (rword_weight == -1)
		{
#if defined (_WIN32)
		  *ras++ = ~ rword;
#else
		  *ras++ = rword;
#endif
		  rword = 0;
		  rword_weight = bits_per_rword-1;
		}
	    }
	  if (rword_weight < bits_per_rword-1)
#if defined (_WIN32)
	    *ras++ = ~ rword;
#else
	    *ras++ = rword;
#endif
	}
    }
  else
    {
      // Create normally packed raster
      dvi_integer rows_left = h;// how many rows left?
      dvi_integer h_bit = w;	// what is our horizontal position?
      repeat_count = 0;
      rword_weight = bits_per_rword;
      rword = 0;
      while (rows_left > 0)
	{
	  dvi_integer count;
	  count = getpackednumber (); // how many bits of current color left?
	  while (count > 0)
	    {
	      if ((count < rword_weight) && (count < h_bit))
		{
		  if (turn_on)
		    rword += gpower[rword_weight] -
		      gpower[rword_weight - count];
		  h_bit -= count;
		  rword_weight -= count;
		  count = 0;
		}
	      else if ((count >= h_bit) && (h_bit <= rword_weight))
		{
		  if (turn_on)
		    rword += gpower[rword_weight] -
		      gpower[rword_weight - h_bit];
#if defined (_WIN32)
		  *ras++ = ~ rword;
#else
		  *ras++ = rword;
#endif
		  // Send row
		  for (dvi_integer i = 1; i <= repeat_count; i++)
		    {
		      for (dvi_integer j = 1; j <= rword_width; j++, ras++)
#if defined (_WIN32)
			*ras = ras[- rword_width];
#else
			*ras = ras[- rword_width];
#endif
		    }
		  rows_left = rows_left - repeat_count - 1;
		  repeat_count = 0;
		  rword = 0;
		  rword_weight = bits_per_rword;
		  count -= h_bit;
		  h_bit = w;
		}
	      else
		{
		  if (turn_on)
		    rword += gpower[rword_weight];
#if defined (_WIN32)
		  *ras++ = ~ rword;
#else
		  *ras++ = rword;
#endif
		  rword = 0;
		  count -= rword_weight;
		  h_bit -= rword_weight;
		  rword_weight = bits_per_rword;
		}
	    }
	  turn_on = (turn_on ? dvi_false : dvi_true);
	}
      if ((rows_left != 0) || (h_bit != w))
	; // FIXME: error ("Bad pk file---more bits than required!");
    }
}

#if defined (RASTER_DEBUG)

void
_pkchar::printraster () const

{
  const rword_t *p = raster;
  for (dvi_integer i = 0; i < h; i++, p++)
    {
      rword_t w1 = *p;
      size_t bit_count = 0;
      for (dvi_integer j = 0; j < w; j++)
	{
	  if (bit_count == bits_per_rword)
	    {
	      p++;
	      w1 = *p;
	      bit_count = 0;
	    }
	  if (w1 & 0x8000) // FIXME
	    dputchar (' ');
	  else
	    dputchar ('*');
	  w1 <<= 1;
	  bit_count++;
	}
      dputchar ('\n');
    }
}

#endif // RASTER_DEBUG

void
_pkchar::makebitmap ()

{
  if (raster == 0)
    {
      dprintf1 ("creating bitmap for character %ld\n", cc);
      unpack ();
#if defined (RASTER_DEBUG)
      printraster ();
#endif
      unsigned long rlen = (unsigned long) (rword_width * sizeof(rword_t) * h);
      dprintf3 ("%ld*%ld=%lu raster bytes\n", w, h, rlen);
      long len = rlen / sizeof (rword_t);
      for (dvi_byte *p = (dvi_byte *) raster; len; len--, p += 2)
	{
	  dvi_byte t = *p;
	  *p = p[1];
	  p[1] = t;
	}
    }
}
