/*
Copyright (C) 2016, 2017 Siep Kroonenberg

This file is part of TLaunch.

TLaunch is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

TLaunch is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with TLaunch.  If not, see <http://www.gnu.org/licenses/>.
*/

// various utilities for tlaunch

#include "tla_utils.h"

// shlobj.h not invoked by our own header files because inlined functions
// in it are not necessarily inlined by gcc: duplicate symbols.

#include <shlobj.h>

// logging
void wr_log( wchar_t * fmt, ... ) {
  if( logstream ) {
    va_list ap;
    va_start( ap, fmt );
    vfwprintf( logstream, fmt, ap );
    va_end( ap );
    fwprintf( logstream, L"\r\n" );
    fflush( logstream );
  }
} // wr_log

/////////////////////////////////
// calloc wrapper

// Our program does not use calloc for large allocations, so a failed
// calloc probably means that the OS is seriously short on memory. In
// this case getting out instantly is the only reasonable action.

void * calloc_fatal( size_t nitems, size_t it_size ) {
  void * p;
  p = calloc( nitems, it_size );
  if ( !p ) ExitProcess( 1 );
  return p;
}

/////////////// STRINGS //////////////////////////

// copy n wchars to new string and add null character, with allocation.
wchar_t * tl_wcsncpy( wchar_t * source, size_t n ) {
  wchar_t * p;
  if( !source ) return NULL;
  p = calloc_fatal( n + 1, sizeof( wchar_t ));
  wcsncpy( p, source, n );
  p[n] = L'\0';
  return p;
}

// make copy of string, with allocation
wchar_t * tl_wcscpy( wchar_t * source ) {
  if( source ) return tl_wcsncpy( source, wcslen( source ) );
  else return NULL;
}

// Concatenate several strings, with allocation. Caller frees memory.
wchar_t * tl_wcsnconcat( int n, ...) {
  int l = 1; // aggregate length; start with 1 for terminating zero
  int i;
  wchar_t * buf, * p;
  va_list argptr;
  va_start(argptr, n);
  for (i=0; i<n; ++i) {
    p = va_arg( argptr, wchar_t * );
    if ( p ) l += wcslen(p);
  }
  va_end(argptr);
  buf = calloc_fatal(l, sizeof(wchar_t));
  buf[0] = L'\0';
  va_start(argptr, n);
  for (i=0; i<n; ++i) {
    p = va_arg(argptr, wchar_t *);
    if ( p ) wcscat( buf, p );
  }
  va_end(argptr);
  buf[ l-1 ] = L'\0';
  return buf;
} // tl_wcsnconcat

// wide-char [v]asprintf with automatic sizing and allocation
// adapted from source code of asprintf and vasprintf
int tl_vwasprintf( wchar_t ** ret, const wchar_t * fmt, va_list ap ) {
  FILE * fp;
  wchar_t * buf;
  int len;

  fp = fopen( "NUL", "w" );
  if ( ! fp ) {
    *ret = NULL;
    return -1;
  }
  len = vfwprintf( fp, fmt, ap );
  fclose( fp );
  if( len < 0 ) return -1;
  buf = calloc_fatal( len + 1, sizeof( wchar_t ));
  if ( vswprintf( buf, len, fmt, ap ) != len ) {
    FREE0( buf );
    *ret = NULL;
    return -1;
  }
  *ret = buf;
  return len;
}

int tl_wasprintf( wchar_t ** ret, const wchar_t * fmt, ... ) {
  int len;
  va_list ap;
  va_start( ap, fmt );
  len = tl_vwasprintf( ret, fmt, ap );
  va_end( ap );
  return len;
}

// in-place removal of \n and \r at the end of a string
void chomp( wchar_t * str ) {
  int len;
  if( !str ) return;
  len = wcslen( str );
  while( len > 0 && ( str[ len - 1 ] == L'\n' || str[ len - 1 ] == L'\r' ||
    str[ len - 1 ] == L' ' )) str[ --len ] = L'\0';
}

// in-place character substitution
void wchar_subst( wchar_t * s, wchar_t oldchar, wchar_t newchar ) {
  int l;
  if( !s ) return;
  l = 0;
  for( l = 0; l < wcslen( s ); l++ ) {
    if( s[ l ] == oldchar ) s[ l ] = newchar;
  }
}

// for assembling a list of warnings
void add_to_mess( wchar_t ** mess, wchar_t * fmt, ... ) {
  wchar_t * new_mess, * to_add;
  va_list ap;
  if( fmt == NULL ) return;
  va_start( ap, fmt );
  tl_vwasprintf( &to_add, fmt, ap );
  va_end( ap );
  // toss out empty strings
  if ( *mess && !**mess ) FREE0( *mess ); // should not happen
  if ( to_add && !*to_add ) FREE0( to_add );
  if( *mess && to_add ) {
    new_mess = tl_wcsnconcat( 3, *mess, L"\r\n", to_add );
    FREE0( *mess );
    FREE0( to_add );
    *mess = new_mess;
  } else if( to_add ) {
    *mess = to_add;
  } // else *mess unchanged
} // add_to_mess

void set_err( wchar_t * fmt, ... ) {
  va_list ap;
  wchar_t * new_err, * old_err;
  old_err = errmess;
  new_err = NULL;
  va_start( ap, fmt );
  tl_vwasprintf( &new_err, fmt, ap );
  va_end( ap );
  if( !new_err ) tldie( NULL, L"set_err called with empty error message" );
  if( logstream )
    wr_log( L"%ls\r\n", new_err );
  if( old_err && errkeep ) { // add new_err to old_err
    errmess = tl_wcsnconcat( 3, old_err, L"\r\n", new_err );
    FREE0( new_err );
  } else {
    errmess = new_err;
  }
  FREE0( old_err ); // ok if old_err NULL
  errkeep = 0;
} // set_err

// Convert memory block of given size and unknown encoding to utf-16 without BOM
// This function always returns a newly-allocated string.
// It returns NULL on invalid input, because e.g. we do not want
// accidents with unintended path names.
wchar_t * to_wide_string( BYTE * s, int l ) {

  size_t nits, nitems, ll;
  wchar_t * dest;
  union { void * pv; BYTE * pb; char * pc; wchar_t * pw; } pts;
  UINT codepage = 0;
  int maybe_ansi = 0;

  if( !s ) return NULL;

  if ( l < 4 ) {
    set_err( L"to_wide_string: memory block too small" );
    return NULL;
  }

  // different pointer types for s
  pts.pv = s;
  if (pts.pb[0]=='\xFE' && pts.pb[1]=='\xFF') {
    // UTF-16 (BE) BOM
    set_err( L"String encoding not ASCII, UTF-8 or UTF-16 LE");
    return NULL;
  } else if (pts.pb[0]=='\xFF' && pts.pb[1]=='\xFE') {
    // UTF-16 (LE) BOM
    pts.pw += 1;
    // number of remaining wchars
    ll = (( l / sizeof( wchar_t )) - 1 );
    // tl_wcsncpy adds null wchar to copy
    return tl_wcsncpy( pts.pw, ll );
  } else if( pts.pb[0] != '\0' && pts.pb[1] == '\0' &&
       pts.pb[2] != '\0' && pts.pb[3] == '\0' ) {
      // UTF-16 (LE) without BOM most likely encoding
    ll = l / sizeof( wchar_t );
    return tl_wcsncpy( pts.pw, ll );
  } else if ( pts.pb[0] == '\0' || pts.pb[1] == '\0' || pts.pb[2] == '\0' ) {
    // No UTF-[8|16|32] BOM, cannot be  UTF-8
    set_err( L"String encoding undetermined or unsupported");
    return NULL;
  } else if (pts.pb[0]=='\xEF' && pts.pb[1]=='\xBB' && pts.pb[2]=='\xBF') {
    // UTF-8 BOM
    pts.pb += 3;
    ll = l - 3;
  } else {
    maybe_ansi = 1;
    // treat as UTF-8. Ignore exotic encodings.
    ll = l;
  }
  // non-ansi/utf8 cases have already returned

  /* standard C mbstowcs: did not work
    nits = mbstowcs( &dummy, pts.pb, 1 ); // get required length ex. \0
    if ( nits == -1 ) tldie( NULL, L"to_wide_string: invalid Unicode" );
    if ( !nits ) tldie( NULL, L"to_wide_string: conversion failed" );
    dest = calloc_fatal( nits + 1, sizeof( wchar_t ));
    nitems = mbstowcs( dest, pts.pb, nits );
    if ( nitems != nits )
      tldie( NULL, L"to_wide_string: utf8: something went wrong" );
    dest[nits] = L'\0';
  */
  // Windows API
  // parameters: codepage, flags (N/A), input buffer, input length in bytes,
  // output buffer, output buffer size in wchars incl. L'\0'
  // For the last two, use NULL and 0 to first get required size
  nits = MultiByteToWideChar( CP_UTF8, MB_ERR_INVALID_CHARS,
      pts.pc, ll, NULL, 0 );
  if ( !nits && maybe_ansi ) {
    if( GetLastError() == ERROR_NO_UNICODE_TRANSLATION ) {
      // try ANSI
      nits = MultiByteToWideChar( CP_ACP, MB_ERR_INVALID_CHARS,
          pts.pc, ll, NULL, 0 );
      if( !nits ) {
        set_err( L"to_wide_string: conversion failed" );
        return NULL;
      }
      codepage = CP_ACP; // system default ANSI code page
    } else if ( !nits ) {
      set_err( L"to_wide_string: conversion failed" );
      return NULL;
    } else {
      codepage = CP_UTF8;
    }
  }
  dest = calloc_fatal( nits+1, sizeof( wchar_t ));
  nitems = MultiByteToWideChar( codepage, MB_ERR_INVALID_CHARS,
      pts.pc, ll, dest, nits );
  if ( nitems != nits ) tldie( NULL, L"to_wide_string: something went wrong" );
  dest[nits] = L'\0';
  return dest;
} // to_wide_string

////// READ FILE AT ONE GO //////

BYTE * slurp_file( wchar_t * fname, DWORD max_size, DWORD * pread ) {

  __int64 ll;
  LARGE_INTEGER lll;
  BYTE * buf;
  HANDLE hf;
  BOOL result;

  hf = CreateFile( // 'Create' can also mean 'Open'
      fname,
      GENERIC_READ,
      FILE_SHARE_READ,
      NULL, // no inheriting
      OPEN_EXISTING,
      0, // attributes; ignored for opening existing files for reading
      NULL
  );
  if ( hf == INVALID_HANDLE_VALUE ) {
    set_err( L"Could not open %ls", fname );
    return NULL;
  }
  if ( !GetFileSizeEx( hf, &lll )) {
    CloseHandle( hf );
    set_err( L"Could not open %ls", fname );
    return NULL; // error condition, could call GetLastError to find out more
  }
  ll = lll.QuadPart;
  if ( ll < 0 ) {
    CloseHandle( hf );
    set_err( L"Failed to read size of file %ls", fname );
    return NULL;
  }
  if ( ll == 0 ) {
    CloseHandle( hf );
    set_err( L"Empty file %ls", fname );
    return NULL;
  }
  if ( ll > max_size ) {
    set_err( L"Oversize file %ls; read partially", fname );
    errkeep = 1;
    ll = max_size;
  }
  buf = calloc_fatal( ll, 1 );
  result = ReadFile( hf, buf, ll, pread, NULL );
  CloseHandle( hf );
  if ( !result ) {
    set_err( L"Failure reading %ls", fname );
    FREE0( buf );
  }
  return (void *)buf;
} // slurp_file


/// READ FILE AT ONE GO AND CONVERT TO WCHAR STRING

wchar_t * slurp_file_to_wcs( wchar_t * fname, DWORD max_size ) {

  DWORD lread;
  BYTE * buf;
  wchar_t * buf2;

  FREE0( errmess );
  buf = slurp_file( fname, max_size, &lread );
  // save slurp errors
  errkeep = 1;
  if ( !buf || !lread ) buf2 = NULL;
  else buf2 = to_wide_string( buf, lread );
  FREE0( buf );

  return buf2;
} // slurp_file_to_wcs

// capture output via a pipe; may fail when other programs are invoked
wchar_t * capture_output( wchar_t * cmd, BOOL allow_overflow ) {
  wchar_t * cmd_copy, * buf2 = NULL, * sysroot;
  DWORD last_err;
  PROCESS_INFORMATION pi;
  STARTUPINFO si;
  SECURITY_ATTRIBUTES sat;

  BYTE * buf, * buferr;
  WORD bufpos, buferrpos;
  BYTE * overflow; // bit bucket for excess output
  HANDLE piperd, pipewr, piperrr, piperrw; // read- and write end of pipes
  BOOL did_it;
  DWORD bytes_read;

  // set up pipe; requires console object,
  // to be provided by CreateProcess
  piperd = NULL;
  pipewr = NULL;
  piperrr = NULL;
  piperrw = NULL;
  ZeroMemory( &sat, sizeof( sat ));
  sat.nLength = sizeof( SECURITY_ATTRIBUTES );
  sat.bInheritHandle = TRUE; // to be applied only to write end of pipe
  sat.lpSecurityDescriptor = NULL;
  if( !CreatePipe( &( piperd ), &( pipewr ), &sat, 0)) {
    set_err( L"Cannot set up std pipe" );
    return NULL;
  }
  if( !CreatePipe( &( piperrr ), &( piperrw ), &sat, 0)) {
    set_err( L"Cannot set up err pipe" );
    return NULL;
  }
  // no inheriting of read end of pipes
  if( !SetHandleInformation( piperd, HANDLE_FLAG_INHERIT, 0 )) {
    set_err( L"Failed SetHandleInformation" );
    CloseHandle( pipewr );
    CloseHandle( piperd );
    return NULL;
  }
  if( !SetHandleInformation( piperrr, HANDLE_FLAG_INHERIT, 0 )) {
    set_err( L"Failed SetHandleInformation" );
    CloseHandle( piperrw );
    CloseHandle( piperrr );
    return NULL;
  }

  // create child process
  ZeroMemory( &pi, sizeof( pi ));
  ZeroMemory( &si, sizeof( si ));
  si.cb = sizeof(STARTUPINFO);
  si.dwFlags = STARTF_USESTDHANDLES | STARTF_USESHOWWINDOW;
  si.wShowWindow = SW_HIDE;
  si.hStdOutput = pipewr;
  si.hStdError = piperrw;
  // si.hStdError = GetStdHandle( STD_ERROR_HANDLE );
  si.hStdInput = GetStdHandle( STD_INPUT_HANDLE );
  cmd_copy = tl_wcscpy( cmd );
  sysroot = get_env( L"SystemRoot" );
  did_it = CreateProcess(
      NULL,
      cmd_copy, // should not be read-only!
      NULL,
      NULL,
      TRUE, // do inherit handles
      CREATE_NEW_CONSOLE | CREATE_NO_WINDOW, // creation flags
      NULL,
      sysroot, // working directory; NULL => directory of application
      &si,
      &pi );
  if( !did_it ) last_err = GetLastError( );
  FREE0( cmd_copy );
  FREE0( sysroot );
  CloseHandle( pipewr );
  CloseHandle( piperrw );
  if( !did_it )  {
    set_err( L"Failed CreateProcess \"%ls\": %lu", cmd, last_err );
    CloseHandle( piperd );
    CloseHandle( piperrr );
    return NULL;
  }
  CloseHandle( pi.hThread ); // this does NOT kill anything

  // read outputs as they become available
  buf = calloc_fatal( READSIZE, 1 );
  bufpos = 0;
  buferr = calloc_fatal( READSIZE, 1 );
  buferrpos = 0;
  overflow = NULL; // allocate only when needed

  for( ;; ) {
    DWORD std_done = FALSE, err_done = FALSE;
    Sleep( 10 );
    if( !std_done && bufpos < READSIZE ) {
      std_done = !ReadFile(
          piperd, buf + bufpos,
          ( bufpos + RDBUF <= READSIZE ) ? RDBUF : READSIZE - bufpos,
          &bytes_read, NULL );
      if( !std_done && bytes_read > 0 )
        bufpos += bytes_read;
    } else if( !std_done && allow_overflow ) {
      if( !overflow ) {
        overflow = calloc_fatal( RDBUF, 1 );
        set_err( L"Too many data; surplus truncated" );
        errkeep = 1;
      }
      std_done = !ReadFile( piperd, overflow, RDBUF, &bytes_read, NULL );
    } else { // too many data, overflow not allowed
      FREE0( buf );
      return NULL;
    }
    if( !err_done && buferrpos < READSIZE ) {
      err_done = !ReadFile(
          piperrr, buferr + buferrpos,
          ( buferrpos + RDBUF <= READSIZE ) ? RDBUF : READSIZE - buferrpos,
          &bytes_read, NULL );
      if( !err_done && bytes_read > 0 )
        buferrpos += bytes_read;
    } else if( !err_done ) { // here overflow always allowed
      if( !overflow ) {
        // use same overflow as stdout
        overflow = calloc_fatal( RDBUF, 1 );
        set_err( L"Too many error data; surplus truncated" );
        errkeep = 1;
      }
      err_done = !ReadFile( piperrr, overflow, RDBUF, &bytes_read, NULL );
    }
    if( std_done && err_done ) break;
  }
  if( overflow ) { FREE0( overflow ); }
  if( piperd != INVALID_HANDLE_VALUE ) CloseHandle( piperd );
  if( piperrr != INVALID_HANDLE_VALUE ) CloseHandle( piperrr );
  CloseHandle( pi.hProcess );
  if( !bufpos ) {
    FREE0( buf );
  }
  // log error output, if any
  if( buferrpos ) {
    buf2 = to_wide_string( buferr, buferrpos );
    chomp( buf2 ); // function checks for buf2 == NULL and buf2[0] == 0
    if( buf2 && buf2[0] ) wr_log( L"Error output from:\n%ls\n%ls", cmd, buf2 );
    FREE0( buf2 );
  }
  FREE0( buferr );
  if( !bufpos ) {
    FREE0( buf );
    return NULL;
  } else { // bufpos > 0
    buf2 = to_wide_string( buf, bufpos );
    chomp( buf2 );
    FREE0( buf );
    if( buf2 && !buf2[0] ) FREE0( buf2 );
    return buf2; // NULL or otherwise
  }
} // capture_output

// capture output via a temp file
wchar_t * capture_tmp( wchar_t * cmd, BOOL allow_overflow ) {
  wchar_t * cmd_full, * tempname, * tempname2, * sysroot;
  DWORD last_err;
  DWORD wait_ret;
  PROCESS_INFORMATION pi;
  STARTUPINFO si;
  wchar_t * buf;
  BOOL did_it;
  wr_log( L"Capturing output from %ls\n", cmd );

  // temporary files for output
  tempname = NULL;
  get_tempfilename( NULL, &tempname );
  if( !tempname ) {
    tldie( NULL, L"Cannot create tempfile" );
    return NULL;
  }
  tempname2 = NULL;
  get_tempfilename( NULL, &tempname2 );
  if( !tempname2 ) {
    tldie( NULL, L"Cannot create tempfile2" );
    return NULL;
  }
  // create child process
  ZeroMemory( &pi, sizeof( pi ));
  ZeroMemory( &si, sizeof( si ));
  si.cb = sizeof(STARTUPINFO);
  si.dwFlags = STARTF_USESHOWWINDOW;
  si.wShowWindow = SW_HIDE;
  // /s option: do not parse special characters inside the string,
  // not even enclosed double quotes.
  // it is up to the caller to put in the necessary quotes within cmd.
  // If redirecting to a file, '>' should be outside the quoted filename.
  // spaces right inside the surrounding quotes do not hurt
  // and may prevent interpretation of doubled double quotes at either end
  // but for now we leave them out.
  sysroot = get_env( L"SystemRoot" );
  tl_wasprintf( &cmd_full, L"cmd /s /c \"%ls >\"%ls\" 2>\"%ls\" \"",
      cmd, tempname, tempname2 );
  did_it = CreateProcess(
      NULL,
      cmd_full,
      NULL,
      NULL,
      FALSE, // do not inherit handles
      CREATE_NO_WINDOW, // creation flags
      NULL,
      sysroot, // working directory; NULL => directory of application
      &si,
      &pi );
  if( !did_it ) last_err = GetLastError( );
  FREE0( cmd_full );
  FREE0( sysroot );
  if( !did_it )  {
    set_err( L"Failed CreateProcess \"%ls\": %lu", cmd, last_err );
    return NULL;
  }
  CloseHandle( pi.hThread ); // this does NOT kill anything

  // wait for process to finish and handle output
  /* timeout
  wait_ret = WaitForSingleObject( pi.hProcess, 2000 * PATIENCE );
  did_it = ( wait_ret == WAIT_OBJECT_0 ) ? 1 : 0;
  if( wait_ret == WAIT_TIMEOUT ) {
    TerminateProcess( pi.hProcess, 1 );
    set_err( L"%ls did not finish in time", cmd );
  } */
  // no timeout
  wait_ret = WaitForSingleObject( pi.hProcess, INFINITE );
  did_it = ( wait_ret == WAIT_OBJECT_0 ) ? 1 : 0;
  if( wait_ret == WAIT_FAILED )
    set_err( L"%ls errorred out, code %lu", cmd, GetLastError( ));
  CloseHandle( pi.hProcess );

  // collect output whatever the outcome
  // first handle error output
  if( FILE_EXISTS( tempname2 )) {
    buf = slurp_file_to_wcs( tempname2, 10000000 );
    if( buf && !buf[0] ) FREE0( buf ); // do not keep single null char
    if( buf ) {
      wr_log( L"\nStderr from:\n%ls\n%ls\n", cmd, buf );
      FREE0( buf );
    }
    if( !keep_capts ) {
      delete_file( tempname2 ); // this function first tests existence
      if ( FILE_EXISTS( tempname2 )) {
        set_err( L"Failed to remove temp file %ls; error %lu", tempname2,
            GetLastError() );
      }
    }
  }
  // now return regular output to caller
  if( FILE_EXISTS( tempname )) {
    buf = slurp_file_to_wcs( tempname, 10000000 ); // may be NULL
    if( buf ) chomp( buf );
    if( buf && !buf[0] ) FREE0( buf );
    if( buf ) {
      wr_log( L"\nStdout from:\n%ls\n%ls\n", cmd, buf );
    }
    if( !keep_capts ) {
      delete_file( tempname ); // this function first tests existence
      if ( FILE_EXISTS( tempname ))
          set_err( L"Failed to remove temp file %ls; error %lu", tempname,
              GetLastError() );
    }
  }
  return buf; // NULL or otherwise
} // capture_tmp

// capture_tmp for a command consisting of single filename,
// possibly with spaces:
wchar_t * capture_tmp_quoted( wchar_t * cmd, BOOL allow_overflow ) {
  wchar_t * quoted, * result;
  quoted = tl_wcsnconcat( 3, L"\"", cmd, L"\"" );
  result = capture_tmp( quoted, allow_overflow );
  FREE0( quoted );
  return result;
}

////////// PROCESS ENVIRONMENT /////////////////////////////

// wrapper for GetEnvironmentVariable
wchar_t * get_env( wchar_t * ev ) {
  DWORD sz;
  wchar_t * val, dummy;
  sz = GetEnvironmentVariable( ev, &dummy, 1 );
  if ( sz<2 ) return NULL; // we do not want an empty string
  if ( sz >= TL_MAX_STR )
    tldie( NULL, L"Excessively long environment variable" );
  val = calloc_fatal( ++sz, sizeof( wchar_t ));
  sz = GetEnvironmentVariable( ev, val, sz );
  if ( !sz ) return NULL;
  val[sz] = L'\0';
  return val;
}

// wrapper for expanding environment variables inside a string
wchar_t * expand_env_vars( wchar_t * str ) {

  wchar_t dummy, * str_exp;
  DWORD sz;

  if( !wcschr( str, L'%' )) return tl_wcscpy( str );

  sz = ExpandEnvironmentStrings( str, &dummy, 1);
  if (!sz) {
    return NULL;
  }
  str_exp = calloc_fatal( sz, sizeof(wchar_t));
  sz = ExpandEnvironmentStrings( str, str_exp, sz);
  if (!sz) {
    FREE0( str_exp );
    return NULL;
  }
  return str_exp;
}

/////// REGISTRY ///////////////////////////////////////

// for diagnostic messages:
wchar_t * rootkey_name( HKEY rootkey ) {
  if( rootkey==HKEY_CLASSES_ROOT ) return L"HKCR";
  else if( rootkey==HKEY_CURRENT_USER ) return L"HKCU";
  else if( rootkey==HKEY_LOCAL_MACHINE ) return L"HKLM";
  else return L"HK_other";
}
wchar_t * regtype_name( DWORD type ) {
  if( type==REG_SZ ) return L"REG_SZ";
  else if( type==REG_EXPAND_SZ ) return L"REG_EXPAND_SZ";
  else if( type==REG_NONE ) return L"REG_NONE";
  else return L"other";
}

// wrappers for registry functions
// rootkey should be one of the predefined rootkeys HKEY_...
// for now, we only handle REG_SZ, REG_EXPAND_SZ and REG_NONE

// existence
int reg_key_exists( HKEY rootkey, wchar_t * regpath ) {
  HKEY hk_sub;

  RegOpenKeyEx( rootkey, regpath, 0, STANDARD_RIGHTS_READ, &hk_sub );
  if( hk_sub ) {
    RegCloseKey( hk_sub );
    return 1;
  } else {
    return 0;
  }
} // reg_key_exists

int reg_value_exists( HKEY rootkey, wchar_t * regpath, wchar_t * valname ) {
  HKEY hk_sub;
  LONG res; /*
  BYTE regbyte[100];
  DWORD reglen; */

  RegOpenKeyEx( rootkey, regpath, 0, STANDARD_RIGHTS_READ|KEY_QUERY_VALUE,
      &hk_sub);
  if( !hk_sub ) return 0;
  res = RegQueryValueEx( hk_sub, valname, NULL, NULL, NULL, NULL );
  RegCloseKey( hk_sub );
  return ( res != ERROR_FILE_NOT_FOUND ) ? 1 : 0;
  // inexplicable ERROR_ACCESS_DENIED is also a possible outcome
  // for an existing value so do not test for ERROR_SUCCESS
}

// getting //

// getting an unexpanded string value
// return sval=NULL, type=0 on failure
// no warnings if failure might be due to absent key or value
RegValueTyped
    reg_get_value_raw( HKEY rootkey, wchar_t * regpath, wchar_t * valname ) {

  LONG res;
  DWORD sz = 0, sz_w = 0;
  int stringdata;
  RegValueTyped rt = { NULL, REG_NONE };
  sz = 0;

  // first find out required size
  res = RegGetValue(
      rootkey,
      regpath,
      valname,
      RRF_RT_ANY | RRF_NOEXPAND,
      &(rt.type),
      NULL,
      &sz);
  if ( res != ERROR_SUCCESS ) {
    rt.type = REG_NONE;
    return rt;
  } else if( rt.type == REG_NONE ) {
    return rt;
  }
  // allocate string and then call RegGetValue for real;
  // may need to add null character if string data.
  stringdata =  rt.type==REG_SZ || rt.type==REG_EXPAND_SZ;
  if( !stringdata ) {
    set_err( L"Value type %ls for %ls\\%ls\\%ls unsupported\n",
        regtype_name( rt.type ), rootkey_name( rootkey ), regpath, valname );
    rt.type = REG_NONE;
    return rt;
  }
  if( sz<=1 ) { // only room for empty string
    rt.sval = NULL;
    return rt;
  }
  // n. of wchars plus room for null char added
  // in some cases this seems to be nessary, according to m$
  sz_w = (sz / sizeof(wchar_t)) + 1;
  // available buffer size to report to RegGetValue
  sz = sz_w * sizeof(wchar_t);
  rt.sval = calloc_fatal( sz_w, sizeof(wchar_t));
  res = RegGetValue(
      rootkey,
      regpath,
      valname,
      RRF_RT_ANY | RRF_NOEXPAND,
      &(rt.type),
      (BYTE *)(rt.sval),
      &sz);
  if (res != ERROR_SUCCESS) {
    set_err( L"Could not read value %ls\\%ls\\%ls\n",
        rootkey_name( rootkey ), regpath, valname );
    FREE0( rt.sval );
    rt.type = REG_NONE;
  }
  return rt;
}

// return a possibly expanded registry string value
// return NULL rather than an empty string
wchar_t * reg_get_value( HKEY rootkey, wchar_t * regpath, wchar_t * valname ) {
  RegValueTyped rt;
  wchar_t * retval;
  rt = reg_get_value_raw( rootkey, regpath, valname );
  if( ! rt.sval ) {
    return NULL;
  } else if( !rt.sval[0] ) { // empty string
    // postfix left-to-right associativity; higher precedence than prefix
    FREE0( rt.sval );
    return NULL;
  } else if( rt.type == REG_SZ ) {
    return rt.sval;
  } else if( rt.type == REG_EXPAND_SZ ) {
    retval = expand_env_vars( rt.sval );
    FREE0( rt.sval );
    return retval;
  } else {
    return NULL;
  }
}

// setting //

// special case: user environment variable; key exists, no subkeys needed,
// broadcast environment change
int reg_set_env_user( wchar_t * name, RegValueTyped value ) {
  LONG res;
  HKEY hk_sub;
  res = RegOpenKeyEx(
      HKEY_CURRENT_USER,
      L"environment",
      0,
      KEY_SET_VALUE,
      &hk_sub);
  if( res != ERROR_SUCCESS ) {
    tldie( NULL, L"Failed to open user environment in registry" );
    return 0;
  }
  if( value.sval && *(value.sval) ) {
    res = RegSetValueEx(
        hk_sub,
        name,
        0,
        value.type,
        (BYTE *)value.sval,
        ( wcslen( value.sval ) + 1 ) * sizeof( wchar_t )
    );
  } else {
    value.sval = L"\0"; // for error message in case of failure
    res = RegDeleteValue( hk_sub, name );
    if( res == ERROR_FILE_NOT_FOUND ) res = ERROR_SUCCESS;
  }
  RegCloseKey( hk_sub );
  if( res != ERROR_SUCCESS ) {
    set_err(
        L"Failed to set environment variable \"%ls\" to \"%ls\" in registry",
        name, value.sval );
    return 0;
  } else {
    SendMessageTimeout( HWND_BROADCAST, WM_SETTINGCHANGE,
        0, (LPARAM)L"Environment", SMTO_NORMAL, 200, NULL );
    return 1;
  }
} // reg_set_env_user

// general case
int reg_set_value( HKEY rootkey, wchar_t * regpath,
    wchar_t * name, RegValueTyped value ) {

  LONG res;
  HKEY hk_sub;

  res = RegCreateKeyEx( // can always be done in one go
      rootkey,
      regpath,
      0,
      NULL,
      0,
      KEY_WRITE,
      NULL,
      &hk_sub,
      NULL
  );
  if( res != ERROR_SUCCESS ) {
    set_err( L"Failed to open %ls\\%ls for writing\n",
        rootkey_name( rootkey ), regpath );
    return 0;
  }
  if( value.type != REG_NONE ) 
    res = RegSetValueEx(
        hk_sub,
        name,
        0,
        value.type,
        ( BYTE * )value.sval,
        ( wcslen( value.sval ) + 1 ) * sizeof( wchar_t )
    );
  else
    res = RegSetValueEx(
        hk_sub,
        name,
        0,
        REG_NONE,
        NULL,
        0
    );
  RegCloseKey( hk_sub );
  if( res != ERROR_SUCCESS ) {
    set_err( L"Failed to set %ls\\%ls\\%ls to %ls in registry",
        rootkey_name( rootkey ), regpath, name, value.sval );
    return 0;
  } else {
    return 1;
  }
}

// setting value, with automatically determined string type
int reg_set_stringvalue( HKEY rootkey, wchar_t * regpath,
    wchar_t * name, wchar_t * value ) {
  RegValueTyped vt;
  vt.sval = value;
  vt.type = wcschr( value, L'%' ) ? REG_EXPAND_SZ : REG_SZ;
  return reg_set_value( rootkey, regpath, name, vt );
}

// deleting
// this time around, we look at the return value of the API function
// rather than testing afterwards for existence

// typedef for RegDeleteTree prototype
// typedef LONG ( WINAPI * RegFu )( HKEY, wchar_t * );

// delete registry key recursively
int reg_del_key( HKEY rootkey, wchar_t * regpath ) {
  LONG res;

  // does the key exist at all?
  if( !reg_key_exists( rootkey, regpath )) return 1;

  res = RegDeleteTree( rootkey, regpath );
  if( res == ERROR_SUCCESS && !reg_key_exists( rootkey, regpath )) return 1;
  set_err( L"Failure to delete %ls\\%ls, error %d",
      rootkey_name( rootkey ), regpath, res );
  return 0;
}

// delete registry value
int reg_del_value( HKEY rootkey, wchar_t * regpath, wchar_t * name ) {
  LONG res;
  HKEY hk_sub;
  res = RegOpenKeyEx(
      HKEY_CURRENT_USER,
      regpath,
      0,
      KEY_SET_VALUE,
      &hk_sub);
  if( res == ERROR_FILE_NOT_FOUND ) {
    return 1;
  }
  if( res != ERROR_SUCCESS ) {
    set_err( L"Failed to open %ls\\%ls for writing\n",
        rootkey_name( rootkey ), regpath );
    return 0;
  }
  res = RegDeleteValue( hk_sub, name );
  RegCloseKey( hk_sub );
  if( res == ERROR_FILE_NOT_FOUND ) {
    return 1;
  } else if( res != ERROR_SUCCESS ) {
    set_err( L"Failed to delete %ls\\%ls\\%ls in registry, error %d\n",
        rootkey_name( rootkey ), regpath, name, res );
    return 0;
  } else {
    return 1;
  }
} // reg_del_value

////// PATH ///////////////////////////

int modify_user_path( wchar_t * dir, int add ) {
  wchar_t * path_exp, * p0, * p1, * newpath, * prevpath, * dir_exp;
  RegValueTyped userpath;
  int res, last, direxpand = 0;
  direxpand = wcschr( dir, L'%' ) ? 1 : 0;
  userpath = reg_get_value_raw( HKEY_CURRENT_USER, L"Environment", L"PATH" );
  // reg_get_value_raw only returns types REG_NONE, REG_SZ and REG_EXPAND_SZ
  if( !userpath.sval ) {
    if (add ) {
      userpath.sval = tl_wcscpy( dir );
      userpath.type = direxpand ? REG_EXPAND_SZ : REG_SZ;
      res = reg_set_env_user( L"PATH", userpath );
        // warns in case of failure
      FREE0( userpath.sval );
      return res;
    } else {
      return 1;
    }
  }
  // remaining case: non-empty old userpath.sval
  if( add )
    newpath = tl_wcscpy( dir );
  else
    newpath = NULL;
  p0 = userpath.sval;
  if( direxpand )
    dir_exp = expand_env_vars( dir );
  else dir_exp = dir;
  while( p0 && *p0 ) {
    p1 = wcschr( p0, L';' );
    if( p1 ) {
      *p1 = L'\0';
    } else {
      last = 1;
    }
    // tlwarn( L"Current component: %ls\n", p0 );
    path_exp = expand_env_vars( p0 );
    if( !same_file( dir_exp, path_exp )) { // keep path component
      if( newpath ) {
        prevpath = newpath;
        newpath = tl_wcsnconcat( 3, prevpath, L";", p0 );
        FREE0( prevpath );
      } else {
        newpath = tl_wcscpy( p0 );
      }
    }
    FREE0( path_exp );
    p0 = last ? NULL : ++p1;
  }
  userpath.sval = newpath;
  if( add && direxpand ) userpath.type = REG_EXPAND_SZ;
    // otherwise unchanged
  res = reg_set_env_user( L"PATH", userpath ); // warns in case of failure
  if( newpath && *newpath ) FREE0( newpath );
  if( direxpand ) FREE0( dir_exp );
  return res;
} // modify_user_path

////////////// SHELL FOLDERS //////////////////////////////

wchar_t * get_shell_folder( KNOWNFOLDERID rfid ) {
  wchar_t * ans, * folder_path = NULL;
  if( SHGetKnownFolderPath( &rfid, 0, NULL, &ans ) == S_OK ) {
    // copy to another string because the returned string
    // must be deallocated in a special way
    if( ans && ans[0] ) folder_path = tl_wcscpy( ans );
    CoTaskMemFree( ans );
  }
  return folder_path;
}

///////// SUPPORT FOR FILE TYPES; FINDING PROGRAMS ////////

// get the program part of a command; no existence check.
// use this function with care, e.g. may return perl.exe or rundll.exe.

wchar_t * prog_from_cmd( wchar_t * cmd ) {
  wchar_t * p1, * p2, * prog;
  if( !cmd || !cmd[0] ) return NULL;
  p1 = cmd;
  if( *cmd == L'"' ) {
    p2 = wcschr( ++p1, L'"' );
  } else {
    p2 = wcschr( p1, L' ' );
    if( !p2 ) p2 = wcschr( p1, L'\0' );
  }
  if( !p2 ) return NULL;
  prog = calloc_fatal( p2 - p1 + 1, sizeof( wchar_t ));
  wcsncpy( prog, p1, p2 - p1 );
  prog[p2 - p1] = L'\0';
  return prog;
}

void parse_iconspec( wchar_t * icf, wchar_t **iconfile, UINT * iconindex ) {
  wchar_t * ptr;
  *iconindex = 0;
  if( !icf ) { *iconfile = NULL; return; }
  ptr = wcsrchr( icf, L',' );
  if( ptr ) {
    *iconfile = tl_wcsncpy( icf, ptr - icf );
    if( !swscanf( ++ptr, L"%u", iconindex )) *iconindex = 0;
  } else {
    *iconfile = tl_wcscpy( icf );
    *iconindex = 0;
  }
  if( !FILE_EXISTS( *iconfile )) {
    FREE0( *iconfile );
    *iconindex = 0;
  }
}

/*
I. There are multiple ways in which extensions can be associated
with commands.  For finding existing associations, the w32 API
has functions which also take very generic ones, i.e. too many, into account.

II. In a normal run we shall only ever modify the HKCU registry hive.

III. However, for reading we also need to look at HKLM.

Alternative approaches to get at file associations:
AssocQueryString may return generic filetypes or secondary associations,
which makes it useless. Also, it uses COM.
For user customization, AssocQueryStringByKey, with key
HKCU\software\microsoft\windows\currentversion\explorer\fileexts,
returned nothing on W7_64.
Instead, we check HK[CU|LM]\software\classes separately.
Theoretically, just checking HKCR should do, but somehow that does
not always work.
*/

// find progid data associated with a progid
// Check HK[CU|LM]/software classes
// and collect info for a progid structure.
// A NULL p_ parameter signals that those data ares not desired.
// For clarity, there are now separate functions for ext and progid
void get_assoc_data_progid( wchar_t * progid, wchar_t ** p_cmd,
    wchar_t ** p_iconspec, wchar_t ** p_dflt_value ) {
  HKEY rk;
  wchar_t * regpath;
  int i;
  wchar_t * shell_cmd = NULL; // finding shell_cmd means success
  wchar_t * iconspec = NULL;
  wchar_t * dflt_value = NULL;

  if( !progid || !progid[0] )
    tldie( NULL, L"get_assoc_data_progid called with NULL progid" );

  for ( i=0;i<2;i++ ) {
    if( i == 0 ) rk = HKEY_CURRENT_USER; else rk = HKEY_LOCAL_MACHINE;
    regpath = tl_wcsnconcat( 3, L"software\\classes\\", progid,
        L"\\shell\\open\\command" );
    shell_cmd = reg_get_value( rk, regpath, L"" );
    // reg_get_value already converted empty string to NULL pointer
    FREE0( regpath );
    if( !shell_cmd ) {
      continue;
    } else {
      // collect icon data and default value, if desired and available.
      // do not derive from shell_cmd.
      if( p_iconspec ) {
        regpath = tl_wcsnconcat( 3, L"software\\classes\\", progid,
            L"\\DefaultIcon" );
        iconspec = reg_get_value( rk, regpath, L"" );
        FREE0( regpath );
      }
      if( p_dflt_value ) {
        regpath = tl_wcsnconcat( 2, L"software\\classes\\", progid );
        dflt_value = reg_get_value( rk, regpath, L"" );
        FREE0( regpath );
      }
      break;
    } // successfully acquired shell_cmd
  } // for i
  if( shell_cmd ) {
    if( p_cmd ) *p_cmd = shell_cmd;
    if( p_iconspec ) *p_iconspec = iconspec;
    if( p_dflt_value ) *p_dflt_value = dflt_value;
  } // otherwise leave at NULL
} // get_assoc_data_progid

void get_assoc_data_ext( wchar_t * ext, wchar_t ** p_progid, wchar_t ** p_cmd,
    wchar_t ** p_iconspec, wchar_t ** p_dflt_value ) {
  wchar_t * progid;
  // q_cmd arbiter of success, even if not otherwise needed
  wchar_t * q_cmd = NULL;
  HKEY rk;
  wchar_t * regpath;
  int i;
  // make sure all reference parameters are initialized
  if( p_progid ) *p_progid = NULL;
  if( p_cmd ) *p_cmd = NULL;
  if( p_iconspec ) *p_iconspec = NULL;
  if( p_dflt_value ) *p_dflt_value = NULL;
  for( i=0;i<2;i++ ) {
    if( i == 0 ) rk = HKEY_CURRENT_USER; else rk = HKEY_LOCAL_MACHINE;
    regpath = tl_wcsnconcat( 2, L"software\\classes\\", ext );
    progid = reg_get_value( rk, regpath, L"" );
    FREE0( regpath );
    if( !progid ) continue;
    get_assoc_data_progid( progid, &q_cmd, p_iconspec, p_dflt_value );
    if( !( q_cmd )) {
      FREE0( progid );
      // in this case, the other data should still be NULL
      continue;
    } else {
      if( p_cmd ) *p_cmd = q_cmd; else FREE0( q_cmd );
      break;
    }
  }
  if( progid && p_progid ) {
    *p_progid = progid;
  } else if( progid ) {
    FREE0( progid );
  }
} // get_assoc_data_ext

wchar_t * get_assoc_cmd_ext( wchar_t * ext ) {
  wchar_t * cmd;
  get_assoc_data_ext( ext, NULL, &cmd, NULL, NULL );
  return cmd;
} // get_assoc_cmd

// find full path of gui executable via application registration
// call only if the executable is not on the process searchpath

wchar_t * find_registered_app( wchar_t * prog ) {
  wchar_t * full_path, * new_path, * reg_path, * command;
  HKEY roots[2] = { HKEY_CURRENT_USER, HKEY_LOCAL_MACHINE };
  int h, l;
  for( h = 0; h<2; h++ ) {
    // first try app paths key
    l = tl_wasprintf( &reg_path,
      L"Software\\Microsoft\\Windows\\CurrentVersion\\App Paths\\%ls", prog );
    if( l < 0 ) tldie( NULL, L"Internal error in find_registered_app" );
    full_path = reg_get_value( roots[h], reg_path, NULL );
    FREE0( reg_path );
    if( full_path ) {
      if( full_path[0] != L'"' ) return full_path;
      // else remove quotes
      new_path = tl_wcsncpy( full_path + 1, wcslen( full_path ) - 2 );
      FREE0( full_path );
      return new_path;
    }
    // next try applications subkey
    l = tl_wasprintf( &reg_path,
      L"Software\\Classes\\Applications\\%ls\\shell\\open\\command", prog );
    if( l < 0 ) tldie( NULL, L"Internal error in find_registered_app" );
    command = reg_get_value( roots[h], reg_path, NULL );
    FREE0( reg_path );
    if( command ) {
      full_path = prog_from_cmd( command );
      FREE0( command );
    }
    if( full_path ) return full_path;
  } // else try HKLM
  return NULL;
} // find_registered_app

// creating file types and associations

// create progid registry key.
// tp refers to cmd and can be REG_SZ or REG_EXPAND_SZ.
// for other items, this choice is automatic.
int register_progid( wchar_t * progid, wchar_t * cmd, DWORD tp,
    wchar_t * iconspec, wchar_t * dflt_value ) {
  int res; // fail = 0, success = 1
  wchar_t * regpath;
  HKEY hk_sub;
  RegValueTyped val;
  if( !cmd ) { // just create the key
    regpath = tl_wcsnconcat( 2, L"software\\classes\\", progid );
    res = ( RegCreateKeyEx( HKEY_CURRENT_USER, regpath, 0, NULL, 0,
        KEY_WRITE, NULL, &hk_sub, NULL ) == ERROR_SUCCESS );
    if( hk_sub ) RegCloseKey( hk_sub );
  } else {
    regpath = tl_wcsnconcat( 3, L"software\\classes\\", progid,
        L"\\shell\\open\\command" );
    val.sval = cmd;
    val.type = tp;
    res = reg_set_value( HKEY_CURRENT_USER, regpath, L"", val );
    if( res ) {
      FREE0( regpath );
      regpath = tl_wcsnconcat( 3, L"software\\classes\\", progid,
          L"\\DefaultIcon" );
      reg_set_stringvalue( HKEY_CURRENT_USER, regpath, L"", iconspec );
    }
  }
  FREE0( regpath );
  if( res && dflt_value ) {
    FREE0( regpath );
    regpath = tl_wcsnconcat( 2, L"software\\classes\\", progid );
    reg_set_stringvalue( HKEY_CURRENT_USER, regpath, L"", dflt_value );
  }
  FREE0( regpath );
  SHChangeNotify( SHCNE_ASSOCCHANGED, SHCNF_IDLIST, NULL, NULL );
  return res;
}

// remove progid
int remove_progid( wchar_t * progid ) {
  int res;
  wchar_t * regpath;
  regpath = tl_wcsnconcat( 2, L"software\\classes\\", progid );
  res = reg_del_key( HKEY_CURRENT_USER, regpath );
  FREE0( regpath );
  return res;
}

int set_assoc_abs( wchar_t * ext, wchar_t * progid ) {
  int res;
  wchar_t * regpath;
  // don't bother deleting keys/values under
  // HKCU\\software\\microsoft\\windows\\currentversion\\explorer\\fileexts;
  // Their security settings frustrate changing them programmatically
  // Deleting returns success, and a subsequent attempt to open fails.
  // regpath = tl_wcsnconcat( 2,
  //   L"software\\Microsoft\\Windows\\CorrentVersion\\Explorer\\FileExts\\",
  //   ext );
  // reg_del_key( HKEY_CURRENT_USER, regpath );
  // // check logfile for failures
  // FREE0( regpath );
  regpath = tl_wcsnconcat( 2, L"software\\classes\\", ext );
  // 3rd parameter, value name, can be either NULL or empty string
  res = reg_set_stringvalue( HKEY_CURRENT_USER, regpath, L"", progid );
  FREE0( regpath );
  return res;
} // set_assoc_abs

// register progid as secondary association
void set_2nd_assoc( wchar_t * ext, wchar_t * progid ) {
  wchar_t * regpath;
  RegValueTyped rt;

  regpath = tl_wcsnconcat( 3, L"software\\classes\\", ext,
      L"\\OpenWithProgIds" );
  rt.type = REG_NONE;
  // member rt.sval `zero-length binary value' will not be used
  reg_set_value( HKEY_CURRENT_USER, regpath, progid, rt );
  FREE0( regpath );
} // set_2nd_assoc

// associate extension with progid
// mode 0: do nothing; 1: do not overwrite; 2: overwrite
int set_assoc( wchar_t * ext, wchar_t * progid, int mode ) {
  int res;
  wchar_t * old_pgid;
  if( !mode ) return 1;

  // existing valid association? preserve if to be overwritten
  get_assoc_data_ext( ext, &old_pgid, NULL, NULL, NULL );
  if ( old_pgid && mode > 1 ) set_2nd_assoc( ext, old_pgid );

  // register progid as secondary association
  set_2nd_assoc( ext, progid );

  // register progid as primary association if allowed
  if( mode > 1 || !old_pgid ) {
    res = set_assoc_abs( ext, progid );
    SHChangeNotify( SHCNE_ASSOCCHANGED, SHCNF_IDLIST, NULL, NULL );
  } else {
    res = 1;
  }
  if( old_pgid ) FREE0( old_pgid );
  // we report only on the success of the above registry setting
  return res;
} // set_assoc

// break association of extension with progid
int remove_assoc( wchar_t * ext, wchar_t * progid ) {
  int res;
  wchar_t * cur_assoc;
  wchar_t * regpath;
  regpath = tl_wcsnconcat( 2, L"Software\\classes\\", ext );
  cur_assoc = reg_get_value( HKEY_CURRENT_USER, regpath, L"" );
  // value name either NULL or empty string
  res = 1;
  if( cur_assoc && !_wcsicmp( cur_assoc, progid )) {
    res = reg_del_value( HKEY_CURRENT_USER, regpath, L"" );
    SHChangeNotify( SHCNE_ASSOCCHANGED, SHCNF_IDLIST, NULL, NULL );
  }
  FREE0( regpath );
  // also try to remove secondary file association, if any
  regpath = tl_wcsnconcat( 3, L"Software\\classes\\", ext, L"\\OpenWithProgIds" );
  reg_del_value( HKEY_CURRENT_USER, regpath, progid ); // forget return value
  FREE0( regpath );
  return res;
} // remove_assoc

// (un)register a program, optionally with searchpath prefix
int register_prog( wchar_t * path, wchar_t * prefix ) {
  int res; // fail = 0, success = 1
  wchar_t * regpath, * basename;
  if( !path || !path[0] ) return 0;
  basename = wcsrchr( path, L'\\' );
  if( !basename ) return 0;
  // leave backslash at the beginning
  regpath = tl_wcsnconcat( 2,
    L"Software\\Microsoft\\Windows\\CurrentVersion\\App Paths", basename );
  res = reg_set_stringvalue( HKEY_CURRENT_USER, regpath, L"", path );
  if( res && prefix )
    res = reg_set_stringvalue( HKEY_CURRENT_USER, regpath, L"Path", prefix );
  FREE0( regpath );
  return res;
} // register_prog

// should work with either full path or basename
int unregister_prog( wchar_t * path ){
  int res; // fail = 0, success = 1
  wchar_t * regpath, * basename = NULL;
  basename = wcsrchr( path, L'\\' );
  if( !basename ) basename = path;
  else basename++; // skip initial backslash
  regpath = tl_wcsnconcat( 2,
    L"Software\\Microsoft\\Windows\\CurrentVersion\\App Paths\\", basename );
  res = reg_del_key( HKEY_CURRENT_USER, regpath );
  FREE0( regpath );
  return res;
} // unregister_prog

// normalize path; parent should exist but path itself may not
wchar_t * normalize_path( wchar_t * fname ) {
  wchar_t * fn, dummy;
  DWORD sz, sz2;
  // according to MSDN docs, GetFullPathName only requires
  // that parent exists
  sz = GetFullPathName(
      fname,
      1,
      &dummy,
      NULL);
  if ( !sz ) return NULL; // here, sz includes the terminating 0
  fn = calloc_fatal( sz, sizeof( wchar_t ));
  sz2 = GetFullPathName(
      fname,
      sz,
      fn,
      NULL); // here, sz2 does not include the terminating 0!
  if ((( sz2 >= sz ) || !sz2 ) && fn ) { // hardly likely...
    FREE0( fn );
  }
  return fn;
}

// searches prog on process search path and returns full filename,
// or NULL if not found. prog should have an extension
// but no directory component.

wchar_t * find_on_path( wchar_t * prog ) {
  wchar_t * pg, * pspath, dummy;
  DWORD sz, sz2;
  pspath = get_env( L"path" );
  if ( !pspath ) return NULL;
  sz = SearchPath(
      pspath,
      prog,
      NULL,
      1,
      &dummy,
      NULL);
  if ( sz ) {
    pg = calloc_fatal( sz, sizeof( wchar_t ));
    sz2 = SearchPath(
        pspath,
        prog,
        L".exe",
        sz,
        pg,
        NULL);
    if ((( sz2 >= sz ) || !sz2 ) && pg ) {
      FREE0( pg );
    }
  } else {
    pg = NULL;
  }
  FREE0( pspath );
  return pg;
}

// find full path of program by any means
wchar_t * find_program( wchar_t * prog ) {
  wchar_t * basename, * pg_exe, * pg_full;
  basename = NULL;
  pg_exe = prog;
  if( !prog ) return NULL;

  // just basename or also path?
  basename = wcsrchr( prog, L'\\' );
  if( !basename ) basename = wcsrchr( prog, L'/' );
  if( basename )
    basename++;
  else
    basename = prog;
  // has extension?
  if( !wcschr( basename, L'.' ))
    pg_exe = tl_wcsnconcat( 2, prog, L".exe" );
  else
    pg_exe = prog;

  if( basename == prog ) {
    // no path; first try searchpath, then registered application
    pg_full = find_on_path( pg_exe );
    if( !pg_full ) pg_full = find_registered_app( pg_exe );
  } else if( FILE_EXISTS( pg_exe )) {
    pg_full = normalize_path( pg_exe );
  } else {
    pg_full = NULL;
  }
  if( pg_exe != prog ) FREE0( pg_exe );
  return pg_full;
}

// in a command, replace program with possibly quoted full path;
// returns NULL if the program cannot be found
wchar_t * normalize_command( wchar_t * command ) {
  wchar_t * prog, * prog_full, * remainder, * new_command;
  if( *command == L'"' ) {
    remainder = command + 1;
    remainder = wcschr( remainder, L'"' );
    if( !remainder ) return NULL;
    remainder++;
    if( *remainder != L' ' && *remainder != L'\0' ) return NULL;
  } else {
    remainder = wcschr( command, L' ' );
    if( !remainder ) remainder = wcschr( command, L'\0' );
    if( !remainder ) return NULL;
  }
  prog = prog_from_cmd( command );
  if( !prog ) return NULL;
  prog_full = find_program( prog );
  FREE0( prog );
  if( !prog_full ) return NULL;
  // remainder either empty string or starting with a space;
  // either way, safe to append
  if( wcschr( prog_full, ' ' ))
    tl_wasprintf( &new_command, L"\"%ls\"%ls", prog_full, remainder );
  else
    tl_wasprintf( &new_command, L"%ls%ls", prog_full, remainder );
  FREE0( prog_full );
  return new_command;
}

///////////// FILES /////////////////////////

// from stackoverflow question 8991192
__int64 win_filesize( wchar_t * name ) {
  HANDLE hFile = CreateFile( name, GENERIC_READ,
      FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING,
      FILE_ATTRIBUTE_NORMAL, NULL );
  LARGE_INTEGER size;

  if ( hFile==INVALID_HANDLE_VALUE )
      return -1; // error condition, could call GetLastError to find out more

  if ( !GetFileSizeEx( hFile, &size )) {
    CloseHandle( hFile );
    return -1; // error condition, could call GetLastError to find out more
  }

  CloseHandle( hFile );
  return size.QuadPart;
}

// returns TRUE if either the strings match case-insensitively or
// both files exist and are the same.
// rationale: we do not want to return FALSE just because a resource such as
// a removable drive or network share is temporarily unavailable.
int same_file( wchar_t * f1, wchar_t * f2 ) {

  HANDLE hf1, hf2;
  BY_HANDLE_FILE_INFORMATION fi1, fi2;
  int ans;

  if( !f1 || !f2 ) return 0;
  if( !_wcsicmp( f1, f2 )) return 1;

  /*
  http://stackoverflow.com/questions/562701/
  Open both files with CreateFile, call GetFileInformationByHandle for
  both, and compare dwVolumeSerialNumber, nFileIndexLow,
  nFileIndexHigh. If all three are equal they both point to the same
  file:
  */

  hf1 = CreateFile( f1, 0, // GENERIC_READ,
      FILE_SHARE_READ, NULL,
      OPEN_EXISTING, 0, NULL );
  if( hf1 == INVALID_HANDLE_VALUE ) return FALSE;
  if( !GetFileInformationByHandle( hf1, &fi1 )) {
    CloseHandle( hf1 );
    return 0;
  }
  hf2 = CreateFile( f2, 0, // GENERIC_READ,
      FILE_SHARE_READ, NULL,
      OPEN_EXISTING, 0, NULL );
  if( hf2 == INVALID_HANDLE_VALUE ) return 0;
  if( !GetFileInformationByHandle( hf2, &fi2 )) {
    CloseHandle( hf2 );
    return 0;
  }

  ans = 1;
  if( fi1.nFileIndexHigh != fi2.nFileIndexHigh ) ans = 0;
  else if( fi1.nFileIndexLow != fi2.nFileIndexLow ) ans = 0;
  else if( fi1.dwVolumeSerialNumber != fi2.dwVolumeSerialNumber ) ans = 0;
  CloseHandle( hf1 );
  CloseHandle( hf2 );
  return ans;
}

BOOL is_dir( wchar_t * fname ) {
  DWORD attrs;
  attrs = GetFileAttributes( fname );
  if( attrs == INVALID_FILE_ATTRIBUTES ) return FALSE;
  else if( attrs & FILE_ATTRIBUTE_DIRECTORY ) {
    return TRUE;
  } else return FALSE;
} // is_dir

void make_writable( wchar_t * fname ) {
  DWORD attrs = GetFileAttributes( fname );
  if( attrs == INVALID_FILE_ATTRIBUTES ) return;
  if( attrs & FILE_ATTRIBUTE_READONLY )
    SetFileAttributes( fname, attrs & ( ~FILE_ATTRIBUTE_READONLY ));
}

int delete_file( wchar_t * path ) {
  int i;
  for ( i=0; i<PATIENCE; i++ ) {
    if ( FILE_EXISTS( path )) {
      Sleep( 100 );
      DeleteFile( path );
    } else {
      return 1;
    }
  }
  return FILE_EXISTS( path ) ? 0 : 1;
}

// create directory and ancestors
int mkdir_recursive( wchar_t * dir ) {
  wchar_t * fullpath, * anc;
  DWORD attrs, len, l;
  attrs = GetFileAttributes( dir );
  if( attrs != INVALID_FILE_ATTRIBUTES ) { // path exists
    if( attrs & FILE_ATTRIBUTE_DIRECTORY ) return 1;
    return 0;
  }
  // dir does not exist
  fullpath = normalize_path( dir );
  len = wcslen( fullpath );
  while( 1 ) { // find existing ancestor
    anc = wcsrchr( fullpath, L'\\' );
    if( !anc ) {
      FREE0( fullpath );
      return 0;
    }
    *anc = L'\0';
    wprintf( L"Does %ls exist?\n", fullpath );
    attrs = GetFileAttributes( fullpath );
    if( attrs != INVALID_FILE_ATTRIBUTES ) { // path exists
      if( attrs & FILE_ATTRIBUTE_DIRECTORY ) break;
      else {
        FREE0( fullpath );
        return 0;
      }
    }
  }
  // all intermediate null chars were originally backslashes
  // so now revert them to backslashes one by one
  l = wcslen( fullpath );
  while( l < len ) {
    fullpath[l] = L'\\';
    l = wcslen( fullpath );
    wprintf( L"Creating %ls\n", fullpath );
    if( !CreateDirectory( fullpath, NULL )) {
      FREE0( fullpath );
      return 0;
    }
  }
  FREE0( fullpath );
  return 1;
} // mkdir_recursive

// 1: success; 0: failure
int rmdir_ifempty( wchar_t * dirname ) {
  DWORD result;
  if( RemoveDirectory( dirname )) return 1;
  result = GetLastError();
  // valid reason for failure? ok
  if( result == ERROR_FILE_NOT_FOUND || result == ERROR_PATH_NOT_FOUND ||
      result == ERROR_DIR_NOT_EMPTY ) return 1;
  return 0;
} // rmdir_ifempty

/*
remove directory recursively
we want this when clearing TL settings from launcher and also when
cleaning up after a failed install of the uninstaller

Alternatives:
- SHFileOperation: considered obsolete per Vista; cannot handle \\?\ prefix.
- IFileOperation interface: more complex than SHFileOperation,
  scanty documentation
- dirty trick involving comspec and ShellExec
  for a self-deleting uninstaller, we have to resort to this anyway,
  see forget_self in tlauch.c
- DIY recursion. Although Find[First|Next]File return filenames < MAX_PATH,
  this is just for the basename. The full path may be long.
  So this is what we use.
*/

static wchar_t * prefix_path( wchar_t * p ) {
  wchar_t * prefixed, * flipped;
  flipped = tl_wcscpy( p );
  wchar_subst( flipped, L'/', L'\\' );
  if( !wcsncmp( flipped, L"\\\\?\\", 4 )) return flipped;
  else if( !wcsncmp( flipped, L"\\\\?\\UNC\\", 8 )) return flipped;

  if( wcsncmp( flipped, L"\\\\", 2 ))
    prefixed = tl_wcsnconcat( 2, L"\\\\?\\", flipped );
  else
    prefixed = tl_wcsnconcat( 2, L"\\\\?\\UNC", flipped + 1 );
  FREE0( flipped );
  return prefixed;
} // prefix_path

// return values:
// -1: buffer too small
// 0: success
// >0: number of remaining direct children plus 1,
//     so 1 means dirname emptied but not removed
static int rmdir_recursive_nocheck( wchar_t * dirname ) {
  // Helper function for rmdir_recursive, which supplies dirname
  // as the beginning of a buffer of length TL_MAX_STR.
  // dirname should be a full path, with long-path prefix at the start.
  // The current function does the actual recursion, and needs only
  // to check against buffer overflow along the way.
  HANDLE hsearch;
  WIN32_FIND_DATA fd;
  DWORD len, undeleted = 0, last_err;
  BOOL backslashed;

  len = wcslen( dirname );
  // check whether dirname ends with backslash.
  // in that case, dirname indicates the root of a drive,
  // which cannot be removed, but we go ahead anyway.
  backslashed = dirname[ len - 1 ] == L'\\';
  if( backslashed ) dirname[ --len ] = L'\0';
  if( len + 2 < TL_MAX_STR )
    wcsncpy( dirname + len, L"\\*", 3 );
  else
    return -1;
  ZeroMemory( &fd, sizeof( fd ));
  hsearch  = FindFirstFile( dirname, &fd );
  if( hsearch == INVALID_HANDLE_VALUE ) last_err = GetLastError();
  if( hsearch == INVALID_HANDLE_VALUE && last_err != ERROR_FILE_NOT_FOUND ) {
    errkeep = 1;
    set_err( L"No search handle for %ls, error %lu", dirname, last_err );
    // restore dirname to what it was and return
    dirname[ len + ( backslashed ? 1 : 0 ) ] = L'\0';
    return -1;
  }
  // case ERROR_FILE_NOT_FOUND: skip recursive deletion,
  // but hang around for deletion of directory itself

  // recursive deletion: need original dirname, with backslash at end
  dirname[ len + 1 ] = L'\0';
  if( hsearch != INVALID_HANDLE_VALUE ) {
    do {
      // skip '.' and '..' entries
      if( !wcscmp( fd.cFileName, L"." ) || !wcscmp( fd.cFileName, L".." ))
        continue;
      // create full pathname of found file, including terminating 0
      wcsncpy( dirname + len + 1, fd.cFileName, wcslen( fd.cFileName ) + 1 );
      if( fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ) { // directory
        if( rmdir_recursive_nocheck( dirname )) {
          undeleted += 1;
        }
      } else { // regular file
        make_writable( dirname ); // does not apply to directories
        if( !DeleteFile( dirname )) {
          errkeep = 1;
          set_err( L"Failed to delete %ls: %lu\n", dirname, GetLastError( ));
          undeleted += 1;
        }
      }
    } while( FindNextFile( hsearch, &fd ));
    FindClose( hsearch );
  }

  // restore original dirname and remove directory if empty
  if( !backslashed ) dirname[ len ] = L'\0';
  if( undeleted ) return undeleted + 1;
  make_writable( dirname );
  if( !RemoveDirectory( dirname )) {
    errkeep = 1;
    set_err( L"Failed to delete %ls: %lu\n", dirname, GetLastError( ));
    return ++undeleted;
  } else {
    return 0;
  }
} //  rmdir_recursive_nocheck

// return values >= 0 from rmdir_recurse_nocheck; 0 is success, but
// we usually just check whether something was really deleted.
// return -2 for not a directory, or other problem
// return values of deeper levels get lost.
int rmdir_recursive( wchar_t * dirname ) {
  wchar_t * absdir0 = NULL, * absdir1 = NULL, * absdir = NULL;
  int res;

  if( !is_dir( dirname )) {
    set_err( L"%ls not a directory\n", dirname );
    return -1;
  }
  absdir0 = normalize_path( dirname );
  if( absdir0 ) {
    absdir1 = prefix_path( absdir0 );
    FREE0( absdir0 );
  }
  if( !absdir1 ) {
    set_err( L"Cannot get full path of %ls", dirname );
    return -1;
  }
  absdir = calloc_fatal( TL_MAX_STR, sizeof( wchar_t ));
  ZeroMemory( absdir, TL_MAX_STR * sizeof( wchar_t ));
  wcscpy( absdir, absdir1 );
  FREE0( absdir1 );
  // clean slate before rmdir_recursive_nocheck cumulates error messages
  FREE0( errmess );
  res = rmdir_recursive_nocheck( absdir );
  FREE0( absdir );
  return res;
} //  rmdir_recursive

// a crude test whether a string may be a url:
// a string containing "://" with at least one character before and after
int is_url( wchar_t * s ) {
  int i, is_ = 0;
  for ( i=1; i < wcslen( s ) - 3; i++ ) {
    if ( !wcsncmp( s + i, L"://", 3 )) {
      is_ = 1;
      break;
    }
  }
  return is_;
} // is_url

wchar_t tempdir[ MAX_TMP_PATH ]; // initialized to null wchars by compiler

void get_tempfilename( wchar_t * ext, wchar_t ** temp_fname ) {
  DWORD res, i;
  HANDLE hf;
  *temp_fname = NULL;
  if( ext && ext[0] != L'.' )
    tldie( NULL, L"Extension %ls does not start with '.'", ext );
  // ext NULL or valid extension
  if ( !tempdir[0] ) { // note: tempdir is an array, not a pointer
    res = GetTempPath( MAX_TMP_PATH, tempdir );
    if ( !res || res>=MAX_TMP_PATH ) tldie( NULL, L"no tempdir" );
  }
  i = 0; // keep count; do not keep trying forever
  while( i++<=50 && ( !( *temp_fname ))) {
    if( ext )
      res = tl_wasprintf( temp_fname, L"%lstl_%05x%ls", tempdir,
          rand() % 0x100000, ext );
    else
      res = tl_wasprintf( temp_fname, L"%lstl_%05x", tempdir,
          rand() % 0x100000 );
    if ( res <= 0 ) tldie( NULL, L"No tempfile name" );
    if ( res >= MAX_PATH ) tldie( NULL, L"Tempfile name too long" );
    if( FILE_EXISTS( *temp_fname )) continue;
    // make empty file
    hf = CreateFile(
      *temp_fname,
      GENERIC_WRITE, // desired access
      0, // applies to handle
      NULL, // optional security attributes
      CREATE_ALWAYS, // creation disposition: overwrite if file exists
      FILE_ATTRIBUTE_NORMAL, // flags and attributes
      NULL // optional handle for template file
    );
    if( hf == INVALID_HANDLE_VALUE ) {
      FREE0( *temp_fname );
    } else {
      CloseHandle( hf );
    }
  }
} // get_tempfilename

///////////////////// DIAGNOSTICS /////////////////////////

// full path of current executable.
// stupidly:
// - _wpgmptr is considered obsolete
// - _get_wpgmptr requires e.g. libmsvcr100.a or msvcr100.dll
// - GetModuleFileName won't tell you how big a buffer is really needed.

// so first use a really big buffer with GetModuleFileName, copy to a buffer of
// the right size, release the big buffer and return the correctly-sized one

wchar_t * get_own_path( void ) {
  wchar_t * buf, * retval;
  DWORD res, l;
  buf = calloc_fatal( TL_MAX_STR, sizeof( wchar_t ));
  res = GetModuleFileName( NULL, buf, TL_MAX_STR );
  if( res && res < TL_MAX_STR ) {
    buf[ TL_MAX_STR - 1 ] = L'\0'; // just in case
    l = wcslen( buf );
    retval = calloc_fatal( l + 1, sizeof( wchar_t ));
    wcsncpy( retval, buf, l + 1 );
  } else {
    retval = NULL;
  }
  FREE0( buf );
  return retval;
} // get_own_path

// check OS version
/*
BOOL os_geq( DWORD maver, DWORD miver ) {
  OSVERSIONINFOEX osVersionInfo;
  ULONGLONG maskCondition;
  ZeroMemory( &osVersionInfo, sizeof( OSVERSIONINFOEX ));
  osVersionInfo.dwOSVersionInfoSize = sizeof( OSVERSIONINFOEX );
  osVersionInfo.dwMajorVersion = maver;
  maskCondition = VerSetConditionMask( 0, VER_MAJORVERSION, VER_GREATER_EQUAL );
  // major < maver? => FALSE
  if( !VerifyVersionInfo( &osVersionInfo, VER_MAJORVERSION, maskCondition ))
    return FALSE;
  // major > maver => TRUE
  maskCondition = VerSetConditionMask( 0, VER_MAJORVERSION, VER_GREATER );
  if( VerifyVersionInfo( &osVersionInfo, VER_MAJORVERSION, maskCondition ))
    return TRUE;
  // equal; check minor version
  ZeroMemory( &osVersionInfo, sizeof( OSVERSIONINFOEX ));
  osVersionInfo.dwMinorVersion = miver;
  maskCondition = VerSetConditionMask( 0, VER_MINORVERSION, VER_GREATER_EQUAL );
  return VerifyVersionInfo( &osVersionInfo, VER_MINORVERSION, maskCondition );
} // os_geq
*/

BOOL is_elevated( void ) {
  // http://stackoverflow.com/questions/8046097/
  // assume >= vista, since tlaunch won't even start on XP

  BOOL ret = FALSE;
  HANDLE ProcessHandle, TokenHandle = NULL;
  TOKEN_ELEVATION Elevation;
  DWORD cbsize = sizeof( TOKEN_ELEVATION );
  ProcessHandle = GetCurrentProcess(); // no need to close afterwards
  if( OpenProcessToken( ProcessHandle, TOKEN_QUERY, &TokenHandle )) {
    if( GetTokenInformation(
        TokenHandle,
        TokenElevation, // NOT a variable!
        &Elevation,
        sizeof( Elevation ),
        &cbsize )) {
      ret = Elevation.TokenIsElevated;
      if( TokenHandle ) CloseHandle( TokenHandle );
    }
  }
  return ret;
}
