# coding: utf-8
# Name:        magnetic.py
# Author:      Mancuniancol
# Created on:  28.11.2016
# Licence:     GPL v.3: http://www.gnu.org/copyleft/gpl.html
"""
Magnetic module which manages the parsers and requests
"""
import urlparse
from threading import Thread
from urllib import quote_plus, unquote_plus

import filtering
from browser import read_torrent, get_cloudhole_key, get_cloudhole_clearance
from storage import *
from utils import *

parser_results = []
parser_name = []
available_parsers = 0
request_time = time.clock()


def process_torrent(self):
    """
    Process the torrent information from a webpage
    :param self: request
    :type self: parsersHandler
    """
    # request data
    parsed = urlparse.urlparse(self.path)
    info = urlparse.parse_qs(parsed.query)
    uri = info.get('uri', [''])[0]
    filename = str(uuid.uuid4()) + '.torrent'

    # headers
    self.send_response(200)
    self.send_header('Content-type', 'application/x-bittorrent')
    self.send_header('Content-Disposition', 'attachment; filename="%s"' % filename)
    self.send_header('Content-Transfer-Encoding', 'binary')
    self.send_header('Accept-Ranges', 'bytes')
    self.end_headers()

    # torrent
    self.wfile.write(read_torrent(unquote_plus(uri)))
    self.finish()


def process_parser(self):
    """
    parser call back with results
    :param self: request
    :type self: ParsersHandler
    """
    global parser_results
    global available_parsers
    global parser_name

    # parsing path
    parsed = urlparse.urlparse(self.path)
    addon_id = urlparse.parse_qs(parsed.query)['addonid'][0]
    content_length = int(self.headers['Content-Length'])
    payload = self.rfile.read(content_length)
    # headers
    self.send_response(200)
    self.send_header('Content-type', 'text/html')
    self.end_headers()
    # send ok
    self.wfile.write("OK")
    self.finish()
    # continue
    data = json.loads(payload)
    logger.info("parser " + addon_id + " returned " + str(len(data)) + " results in " + str(
        "%.1f" % round(time.clock() - request_time, 2)) + " seconds")
    parser_results.extend(data)
    available_parsers -= 1
    if addon_id in parser_name:
        parser_name.remove(addon_id)


def get_results(self):
    """
    Process to get the results
    :param self: request
    :type self: parsersHandler
    """
    global parser_results

    # init list
    parser_results = []

    # request data
    parsed = urlparse.urlparse(self.path)
    info = urlparse.parse_qs(parsed.query)
    operation = info.get('search', [''])[0]
    parser = info.get('parser', [''])[0]
    title = unquote_plus(str(info.get('title', [''])[0]).replace("'", ""))
    imdb_id = info.get('imdb_id', [''])[0]
    if len(imdb_id) < 5:
        imdb_id = find_imdb(title)
    if operation == 'general':
        method = 'search'
        general_item = {'imdb_id': str(imdb_id),
                        'title': title}
        payload = json.dumps(general_item)

    elif operation == "movie":
        method = "search_movie"
        year = info.get('year', [''])[0]
        movie_item = {'imdb_id': str(imdb_id),
                      'title': title,
                      'year': str(year)}
        payload = json.dumps(movie_item)

    elif operation == "episode":
        method = "search_episode"
        season = info.get('season', [''])[0]
        episode = info.get('episode', [''])[0]
        episode_item = {'imdb_id': str(imdb_id),
                        'title': title,
                        'season': int(season),
                        'episode': int(episode),
                        'absolute_number': int(0)}
        payload = json.dumps(episode_item)

    elif operation == "season":
        method = "search_season"
        season = info.get('season', [''])[0]
        season_item = {'imdb_id': str(imdb_id),
                       'title': title,
                       'season': int(season),
                       'absolute_number': int(0)}
        payload = json.dumps(season_item)

    else:
        self.send_response(200)
        self.send_header('Content-type', 'text/html')
        self.end_headers()
        # send the results
        self.wfile.write("OPERATION NOT FOUND")
        self.finish()
        return

    if len(title) == 0 or len(method) == 0:
        self.send_response(200)
        self.send_header('Content-type', 'text/html')
        self.end_headers()
        # send the results
        self.wfile.write("Payload Incomplete!!!      " + repr(payload))
        self.finish()
        return

    # check if the search is in cache
    database = Storage.open("parsers", 60 * 6, True)
    cache = database.get(payload, None)
    if cache is None or len(parser) > 0:
        # read the cloudhole key from the API
        if get_setting("use_cloudhole", bool):
            clearance, user_agent = get_cloudhole_clearance(get_cloudhole_key())
            set_setting('clearance', clearance)
            set_setting('user_agent', user_agent)

        # requests the results
        normalized_list = search(method, payload, parser)
        results = normalized_list
        # if there is results it will be saved in cache
        if len(normalized_list.get('magnets', [])) > 0:
            database[payload] = results
            database.sync()
    else:
        normalized_list = cache
        display_message_cache()

    logger.info("Filtering returned: " + str(len(normalized_list.get('magnets', []))) + " results")

    # headers
    self.send_response(200)
    self.send_header('Content-type', 'application/json')
    self.end_headers()
    # send the results
    self.wfile.write(json.dumps(normalized_list))
    self.finish()


def search(method, payload_json, parser=""):
    """
    Search for torrents - call parsers
    :param method: method of search
    :type method: str
    :param payload_json: parload in json format
    :type payload_json: str
    :param parser: name of parser, if the search is individually
    :type parser: str
    :return list of results
    """
    global parser_results
    global available_parsers
    global request_time
    global parser_name

    # reset global variables
    parser_results = []
    parser_name = []
    available_parsers = 0
    request_time = time.clock()
    # collect data
    if len(parser) == 0:
        addons = get_list_parsers_enabled()
    else:
        addons = [parser]

    if len(addons) == 0:
        # return empty list
        notify(string(32060), image=get_icon_path())
        logger.info("No parsers installed")
        return {'results': 0, 'duration': "0 seconds", 'magnets': []}

    p_dialog = xbmcgui.DialogProgressBG()
    p_dialog.create('Magnetic Manager', string(32061))
    for add_on in addons:
        available_parsers += 1
        parser_name.append(add_on)
        task = Thread(target=run_parser, args=(add_on, method, payload_json))
        task.start()

    parsers_time = time.clock()
    total = float(available_parsers)
    # while all parsers have not returned results or timeout not reached
    time_out = min(get_setting("timeout", int), 60)
    # if all parsers have returned results exit
    # check every 100ms
    while time.clock() - parsers_time < time_out and available_parsers > 0:
        xbmc.sleep(100)
        message = string(32062) % available_parsers if available_parsers > 1 else string(32063)
        p_dialog.update(int((total - available_parsers) / total * 100), message=message)

    # time-out parser
    if len(parser_name) > 0:
        message = ', '.join(parser_name)
        message = message.replace('script.magnetic.', '').title() + string(32064)
        logger.info(message)
        notify(message, ADDON_ICON)

    # filter magnets and append to results
    filtered_results = dict(magnets=filtering.apply_filters(parser_results))
    # append number and time on payload
    filtered_results['results'] = len(filtered_results['magnets'])
    filtered_results['duration'] = str("%.1f" % round(time.clock() - request_time, 2)) + " seconds"
    logger.info(
        "Magnetic search returned: %s results in %s" % (str(len(parser_results)), filtered_results['duration']))
    # destroy notification object
    p_dialog.close()
    del p_dialog

    return filtered_results


def run_parser(add_on, method, search_query):
    """
    Run external script
    :param add_on: add-on to run
    :type add_on: str
    :param method: method in the script
    :type method: str
    :param search_query: search or query
    :type search_query: str
    """
    logger.debug("Processing:" + add_on)
    xbmc.executebuiltin(
        "RunScript(" + add_on + "," + add_on + "," + method + "," + quote_plus(search_query.encode('utf-8')) + ")",
        True)
