/*******************************************************************************
 * Copyright (c) 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Zend Technologies
 *******************************************************************************/
package org.eclipse.php.internal.ui.folding;

import org.eclipse.jface.text.source.projection.IProjectionListener;
import org.eclipse.jface.text.source.projection.ProjectionViewer;
import org.eclipse.php.internal.ui.PHPUiPlugin;

/**
 * A proxy IProjectionListener and IStructuredTextFoldingProvider for the PHP
 * code folding extentions. This proxy checks the
 * PHPFoldingStructureProviderRegistry for the current IProjectionListener and
 * IStructuredTextFoldingProvider and propogate the request to them.
 * 
 * @author shalom
 */
public class PHPFoldingStructureProviderProxy implements IProjectionListener, IStructuredTextFoldingProvider {

	private IStructuredTextFoldingProvider currentFoldingProvider;

	public PHPFoldingStructureProviderProxy() {
		currentFoldingProvider = PHPUiPlugin.getDefault().getFoldingStructureProviderRegistry()
				.getCurrentFoldingProvider();
	}

	/**
	 * Tells this listener that projection has been enabled.
	 */
	@Override
	public void projectionEnabled() {
		IProjectionListener projectionListener = getProjectionListener();
		if (projectionListener != null) {
			projectionListener.projectionEnabled();
		}
	}

	/**
	 * Tells this listener that projection has been disabled.
	 */
	@Override
	public void projectionDisabled() {
		IProjectionListener projectionListener = getProjectionListener();
		if (projectionListener != null) {
			projectionListener.projectionDisabled();
		}
	}

	/**
	 * Installs this structure provider on the given viewer. Implementations
	 * should listen to the projection events generated by <code>viewer</code>
	 * and enable / disable generation of projection structure accordingly.
	 * 
	 * @param viewer
	 *            the projection viewer that displays the annotations created by
	 *            this structure provider
	 */
	@Override
	public void install(ProjectionViewer viewer) {
		IStructuredTextFoldingProvider foldingProvider = getFoldingProvider();
		if (foldingProvider != null) {
			foldingProvider.install(viewer);
		}
	}

	/**
	 * Uninstalls this structure provider. Any references to viewers should be
	 * cleared.
	 */
	@Override
	public void uninstall() {
		IStructuredTextFoldingProvider foldingProvider = getFoldingProvider();
		if (foldingProvider != null) {
			foldingProvider.uninstall();
		}
	}

	/**
	 * Initialize this provider with the correct document. Assumes projection is
	 * enabled. (otherwise, only install would have been called)
	 */
	@Override
	public void initialize() {
		IStructuredTextFoldingProvider foldingProvider = getFoldingProvider();
		if (foldingProvider != null) {
			foldingProvider.initialize();
		}
	}

	// Returns the current IStructuredTextFoldingProvider
	public IStructuredTextFoldingProvider getFoldingProvider() {
		return currentFoldingProvider;
	}

	// Returns the current IProjectionListener or null, if the current folding
	// provider is not a projection listener.
	private IProjectionListener getProjectionListener() {
		Object obj = getFoldingProvider();
		if (obj instanceof IProjectionListener) {
			return (IProjectionListener) obj;
		}
		return null;
	}

}
