/**
 * Copyright (c) 2017, 2019 itemis AG and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tamas Miklossy (itemis AG) - initial API and implementation
 *     Zoey Gerrit Prigge         - Generalized dependent attribute method
 *                                  to use with recordBased Node shapes (bug #454629)
 */
package org.eclipse.gef.dot.internal.language;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.dot.internal.DotAttributes;
import org.eclipse.gef.dot.internal.language.dot.AttrList;
import org.eclipse.gef.dot.internal.language.dot.AttrStmt;
import org.eclipse.gef.dot.internal.language.dot.Attribute;
import org.eclipse.gef.dot.internal.language.dot.AttributeType;
import org.eclipse.gef.dot.internal.language.dot.DotGraph;
import org.eclipse.gef.dot.internal.language.dot.EdgeRhs;
import org.eclipse.gef.dot.internal.language.dot.EdgeRhsNode;
import org.eclipse.gef.dot.internal.language.dot.EdgeStmtNode;
import org.eclipse.gef.dot.internal.language.dot.NodeId;
import org.eclipse.gef.dot.internal.language.dot.NodeStmt;
import org.eclipse.gef.dot.internal.language.dot.Stmt;
import org.eclipse.gef.dot.internal.language.dot.Subgraph;
import org.eclipse.gef.dot.internal.language.terminals.ID;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This class provides helper methods for walking the DOT abstract syntax tree.
 */
@SuppressWarnings("all")
public class DotAstHelper {
  public static NodeId getNodeId(final NodeId nodeId) {
    Object _xblockexpression = null;
    {
      final DotGraph dotGraph = EcoreUtil2.<DotGraph>getContainerOfType(nodeId, DotGraph.class);
      Iterable<NodeStmt> _allNodeStatements = DotAstHelper.getAllNodeStatements(dotGraph);
      for (final NodeStmt nodeStmt : _allNodeStatements) {
        boolean _and = false;
        NodeId _node = nodeStmt.getNode();
        boolean _tripleNotEquals = (_node != null);
        if (!_tripleNotEquals) {
          _and = false;
        } else {
          NodeId _node_1 = nodeStmt.getNode();
          ID _name = _node_1.getName();
          ID _name_1 = nodeId.getName();
          boolean _equals = Objects.equal(_name, _name_1);
          _and = _equals;
        }
        if (_and) {
          return nodeStmt.getNode();
        }
      }
      _xblockexpression = null;
    }
    return ((NodeId)_xblockexpression);
  }
  
  /**
   * Collects all nodeId EObjects having the same name as the baseNodeId
   */
  public static List<NodeId> getAllNodeIds(final NodeId baseNodeId) {
    LinkedList<NodeId> _xblockexpression = null;
    {
      final LinkedList<NodeId> result = CollectionLiterals.<NodeId>newLinkedList();
      final DotGraph dotGraph = EcoreUtil2.<DotGraph>getContainerOfType(baseNodeId, DotGraph.class);
      EList<Stmt> _stmts = dotGraph.getStmts();
      Iterable<NodeStmt> _filter = Iterables.<NodeStmt>filter(_stmts, NodeStmt.class);
      for (final NodeStmt nodeStmt : _filter) {
        {
          final NodeId nodeId = nodeStmt.getNode();
          boolean _and = false;
          boolean _and_1 = false;
          boolean _tripleNotEquals = (nodeId != null);
          if (!_tripleNotEquals) {
            _and_1 = false;
          } else {
            ID _name = nodeId.getName();
            ID _name_1 = baseNodeId.getName();
            boolean _equals = Objects.equal(_name, _name_1);
            _and_1 = _equals;
          }
          if (!_and_1) {
            _and = false;
          } else {
            boolean _notEquals = (!Objects.equal(nodeId, baseNodeId));
            _and = _notEquals;
          }
          if (_and) {
            result.add(nodeId);
          }
        }
      }
      EList<Stmt> _stmts_1 = dotGraph.getStmts();
      Iterable<EdgeStmtNode> _filter_1 = Iterables.<EdgeStmtNode>filter(_stmts_1, EdgeStmtNode.class);
      for (final EdgeStmtNode edgeStmtNode : _filter_1) {
        {
          NodeId nodeId = edgeStmtNode.getNode();
          boolean _and = false;
          boolean _and_1 = false;
          boolean _tripleNotEquals = (nodeId != null);
          if (!_tripleNotEquals) {
            _and_1 = false;
          } else {
            ID _name = nodeId.getName();
            ID _name_1 = baseNodeId.getName();
            boolean _equals = Objects.equal(_name, _name_1);
            _and_1 = _equals;
          }
          if (!_and_1) {
            _and = false;
          } else {
            boolean _tripleNotEquals_1 = (nodeId != baseNodeId);
            _and = _tripleNotEquals_1;
          }
          if (_and) {
            result.add(nodeId);
          }
          EList<EdgeRhs> _edgeRHS = edgeStmtNode.getEdgeRHS();
          final EdgeRhs edgeRHS = IterableExtensions.<EdgeRhs>head(_edgeRHS);
          if ((edgeRHS instanceof EdgeRhsNode)) {
            NodeId _node = ((EdgeRhsNode)edgeRHS).getNode();
            nodeId = _node;
            boolean _and_2 = false;
            boolean _and_3 = false;
            boolean _tripleNotEquals_2 = (nodeId != null);
            if (!_tripleNotEquals_2) {
              _and_3 = false;
            } else {
              ID _name_2 = nodeId.getName();
              ID _name_3 = baseNodeId.getName();
              boolean _equals_1 = Objects.equal(_name_2, _name_3);
              _and_3 = _equals_1;
            }
            if (!_and_3) {
              _and_2 = false;
            } else {
              boolean _tripleNotEquals_3 = (nodeId != baseNodeId);
              _and_2 = _tripleNotEquals_3;
            }
            if (_and_2) {
              result.add(nodeId);
            }
          }
        }
      }
      _xblockexpression = result;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the color scheme attribute value that is set for the given
   * attribute.
   * 
   * @param attribute
   *            The attribute to determine the color scheme attribute value
   *            for.
   * @return The color scheme value that is set for the given attribute, or
   *         null if it cannot be determined.
   */
  public static String getColorSchemeAttributeValue(final Attribute attribute) {
    return DotAstHelper.getDependedOnAttributeValue(attribute, DotAttributes.COLORSCHEME__GCNE);
  }
  
  /**
   * Returns an attribute value specified by attributeName that is set for given attribute
   * 
   * @param dependentAttribute
   * 			The attribute to determine a depending value for.
   * @param attributeName
   * 			The name of the attribute that the dependentAttribute depends on.
   * @return The attribute value set for the attribute specified by attributeName
   */
  public static String getDependedOnAttributeValue(final Attribute dependentAttribute, final String attributeName) {
    Object _xblockexpression = null;
    {
      final EdgeStmtNode edgeStmtNode = EcoreUtil2.<EdgeStmtNode>getContainerOfType(dependentAttribute, EdgeStmtNode.class);
      boolean _tripleNotEquals = (edgeStmtNode != null);
      if (_tripleNotEquals) {
        EList<AttrList> _attrLists = edgeStmtNode.getAttrLists();
        ID dependedOnValue = DotAstHelper.getAttributeValue(_attrLists, attributeName);
        boolean _tripleNotEquals_1 = (dependedOnValue != null);
        if (_tripleNotEquals_1) {
          return dependedOnValue.toValue();
        }
        ID _globalDependedOnValue = DotAstHelper.getGlobalDependedOnValue(edgeStmtNode, AttributeType.EDGE, attributeName);
        dependedOnValue = _globalDependedOnValue;
        boolean _tripleNotEquals_2 = (dependedOnValue != null);
        if (_tripleNotEquals_2) {
          return dependedOnValue.toValue();
        }
      }
      final NodeStmt nodeStmt = EcoreUtil2.<NodeStmt>getContainerOfType(dependentAttribute, NodeStmt.class);
      boolean _tripleNotEquals_3 = (nodeStmt != null);
      if (_tripleNotEquals_3) {
        EList<AttrList> _attrLists_1 = nodeStmt.getAttrLists();
        ID dependedOnValue_1 = DotAstHelper.getAttributeValue(_attrLists_1, attributeName);
        boolean _tripleNotEquals_4 = (dependedOnValue_1 != null);
        if (_tripleNotEquals_4) {
          return dependedOnValue_1.toValue();
        }
        ID _globalDependedOnValue_1 = DotAstHelper.getGlobalDependedOnValue(nodeStmt, AttributeType.NODE, attributeName);
        dependedOnValue_1 = _globalDependedOnValue_1;
        boolean _tripleNotEquals_5 = (dependedOnValue_1 != null);
        if (_tripleNotEquals_5) {
          return dependedOnValue_1.toValue();
        }
      }
      final AttrStmt attrStmt = EcoreUtil2.<AttrStmt>getContainerOfType(dependentAttribute, AttrStmt.class);
      boolean _tripleNotEquals_6 = (attrStmt != null);
      if (_tripleNotEquals_6) {
        EList<AttrList> _attrLists_2 = attrStmt.getAttrLists();
        final ID dependedOnValue_2 = DotAstHelper.getAttributeValue(_attrLists_2, attributeName);
        boolean _tripleNotEquals_7 = (dependedOnValue_2 != null);
        if (_tripleNotEquals_7) {
          return dependedOnValue_2.toValue();
        }
      }
      final Subgraph subgraph = EcoreUtil2.<Subgraph>getContainerOfType(dependentAttribute, Subgraph.class);
      boolean _tripleNotEquals_8 = (subgraph != null);
      if (_tripleNotEquals_8) {
        final ID dependedOnValue_3 = DotAstHelper.getAttributeValue(subgraph, attributeName);
        boolean _tripleNotEquals_9 = (dependedOnValue_3 != null);
        if (_tripleNotEquals_9) {
          return dependedOnValue_3.toValue();
        }
      }
      final DotGraph dotGraph = EcoreUtil2.<DotGraph>getContainerOfType(dependentAttribute, DotGraph.class);
      boolean _tripleNotEquals_10 = (dotGraph != null);
      if (_tripleNotEquals_10) {
        ID dependedOnValue_4 = DotAstHelper.getAttributeValueAll(dotGraph, attributeName);
        boolean _tripleNotEquals_11 = (dependedOnValue_4 != null);
        if (_tripleNotEquals_11) {
          return dependedOnValue_4.toValue();
        }
        ID _globalDependedOnValue_2 = DotAstHelper.getGlobalDependedOnValue(dotGraph, AttributeType.GRAPH, attributeName);
        dependedOnValue_4 = _globalDependedOnValue_2;
        boolean _tripleNotEquals_12 = (dependedOnValue_4 != null);
        if (_tripleNotEquals_12) {
          return dependedOnValue_4.toValue();
        }
      }
      _xblockexpression = null;
    }
    return ((String)_xblockexpression);
  }
  
  private static ID getGlobalDependedOnValue(final EObject eObject, final AttributeType attributeType, final String attributeName) {
    Object _xblockexpression = null;
    {
      final Subgraph subgraph = EcoreUtil2.<Subgraph>getContainerOfType(eObject, Subgraph.class);
      boolean _tripleNotEquals = (subgraph != null);
      if (_tripleNotEquals) {
        EList<Stmt> _stmts = subgraph.getStmts();
        final ID value = DotAstHelper.getAttributeValue(_stmts, attributeType, attributeName);
        boolean _tripleNotEquals_1 = (value != null);
        if (_tripleNotEquals_1) {
          return value;
        }
      }
      final DotGraph dotGraph = EcoreUtil2.<DotGraph>getContainerOfType(eObject, DotGraph.class);
      boolean _tripleNotEquals_2 = (dotGraph != null);
      if (_tripleNotEquals_2) {
        EList<Stmt> _stmts_1 = dotGraph.getStmts();
        final ID value_1 = DotAstHelper.getAttributeValue(_stmts_1, attributeType, attributeName);
        boolean _tripleNotEquals_3 = (value_1 != null);
        if (_tripleNotEquals_3) {
          return value_1;
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  private static ID getAttributeValue(final EList<Stmt> stmts, final AttributeType attributeType, final String attributeName) {
    Object _xblockexpression = null;
    {
      for (final Stmt stmt : stmts) {
        if ((stmt instanceof AttrStmt)) {
          AttributeType _type = ((AttrStmt)stmt).getType();
          boolean _equals = Objects.equal(_type, attributeType);
          if (_equals) {
            EList<AttrList> _attrLists = ((AttrStmt)stmt).getAttrLists();
            return DotAstHelper.getAttributeValue(_attrLists, attributeName);
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  public static ID getAttributeValue(final DotGraph graph, final String name) {
    Object _xblockexpression = null;
    {
      EList<Stmt> _stmts = graph.getStmts();
      for (final Stmt stmt : _stmts) {
        {
          ID _switchResult = null;
          boolean _matched = false;
          if (!_matched) {
            if (stmt instanceof Attribute) {
              _matched=true;
              _switchResult = DotAstHelper.getAttributeValue(((Attribute)stmt), name);
            }
          }
          final ID value = _switchResult;
          boolean _tripleNotEquals = (value != null);
          if (_tripleNotEquals) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  public static ID getAttributeValueAll(final DotGraph graph, final String name) {
    Object _xblockexpression = null;
    {
      EList<Stmt> _stmts = graph.getStmts();
      for (final Stmt stmt : _stmts) {
        {
          ID _switchResult = null;
          boolean _matched = false;
          if (!_matched) {
            if (stmt instanceof AttrStmt) {
              _matched=true;
              EList<AttrList> _attrLists = ((AttrStmt)stmt).getAttrLists();
              _switchResult = DotAstHelper.getAttributeValue(_attrLists, name);
            }
          }
          if (!_matched) {
            if (stmt instanceof Attribute) {
              _matched=true;
              _switchResult = DotAstHelper.getAttributeValue(((Attribute)stmt), name);
            }
          }
          final ID value = _switchResult;
          boolean _tripleNotEquals = (value != null);
          if (_tripleNotEquals) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  public static ID getAttributeValue(final Subgraph subgraph, final String name) {
    Object _xblockexpression = null;
    {
      EList<Stmt> _stmts = subgraph.getStmts();
      for (final Stmt stmt : _stmts) {
        {
          ID _switchResult = null;
          boolean _matched = false;
          if (!_matched) {
            if (stmt instanceof Attribute) {
              _matched=true;
              _switchResult = DotAstHelper.getAttributeValue(((Attribute)stmt), name);
            }
          }
          final ID value = _switchResult;
          boolean _tripleNotEquals = (value != null);
          if (_tripleNotEquals) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  /**
   * Returns the value of the first attribute with the give name or
   * <code>null</code> if no attribute could be found.
   * 
   * @param attrLists
   *            The {@link AttrList}s to search.
   * @param name
   *            The name of the attribute whose value is to be retrieved.
   * @return The attribute value or <code>null</code> in case the attribute
   *         could not be found.
   */
  public static ID getAttributeValue(final List<AttrList> attrLists, final String name) {
    Object _xblockexpression = null;
    {
      for (final AttrList attrList : attrLists) {
        {
          final ID value = DotAstHelper.getAttributeValue(attrList, name);
          boolean _tripleNotEquals = (value != null);
          if (_tripleNotEquals) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  private static ID getAttributeValue(final AttrList attrList, final String name) {
    EList<Attribute> _attributes = attrList.getAttributes();
    final Function1<Attribute, Boolean> _function = new Function1<Attribute, Boolean>() {
      public Boolean apply(final Attribute it) {
        ID _name = it.getName();
        String _value = _name.toValue();
        return Boolean.valueOf(Objects.equal(_value, name));
      }
    };
    Attribute _findFirst = IterableExtensions.<Attribute>findFirst(_attributes, _function);
    ID _value = null;
    if (_findFirst!=null) {
      _value=_findFirst.getValue();
    }
    return _value;
  }
  
  private static ID getAttributeValue(final Attribute attribute, final String name) {
    Object _xblockexpression = null;
    {
      ID _name = attribute.getName();
      String _value = _name.toValue();
      boolean _equals = _value.equals(name);
      if (_equals) {
        return attribute.getValue();
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  /**
   * Collects all node statements residing in the dot graph or in its subgraphs.
   */
  private static Iterable<NodeStmt> getAllNodeStatements(final DotGraph dotGraph) {
    Iterable<NodeStmt> _xblockexpression = null;
    {
      EList<Stmt> _stmts = dotGraph.getStmts();
      final Iterable<NodeStmt> nodeStamentsInDotGraph = Iterables.<NodeStmt>filter(_stmts, NodeStmt.class);
      final LinkedList<NodeStmt> nodeStatementsInSubgraphs = CollectionLiterals.<NodeStmt>newLinkedList();
      EList<Stmt> _stmts_1 = dotGraph.getStmts();
      Iterable<Subgraph> _filter = Iterables.<Subgraph>filter(_stmts_1, Subgraph.class);
      final Consumer<Subgraph> _function = new Consumer<Subgraph>() {
        public void accept(final Subgraph it) {
          List<NodeStmt> _allNodeStatementsInSubgraph = DotAstHelper.getAllNodeStatementsInSubgraph(it);
          Iterables.<NodeStmt>addAll(nodeStatementsInSubgraphs, _allNodeStatementsInSubgraph);
        }
      };
      _filter.forEach(_function);
      _xblockexpression = Iterables.<NodeStmt>concat(nodeStamentsInDotGraph, nodeStatementsInSubgraphs);
    }
    return _xblockexpression;
  }
  
  private static List<NodeStmt> getAllNodeStatementsInSubgraph(final Subgraph subgraph) {
    List<NodeStmt> _xblockexpression = null;
    {
      EList<Stmt> _stmts = subgraph.getStmts();
      Iterable<NodeStmt> _filter = Iterables.<NodeStmt>filter(_stmts, NodeStmt.class);
      final List<NodeStmt> nodeStatementInSubgraph = IterableExtensions.<NodeStmt>toList(_filter);
      EList<Stmt> _stmts_1 = subgraph.getStmts();
      Iterable<Subgraph> _filter_1 = Iterables.<Subgraph>filter(_stmts_1, Subgraph.class);
      final Consumer<Subgraph> _function = new Consumer<Subgraph>() {
        public void accept(final Subgraph it) {
          List<NodeStmt> _allNodeStatementsInSubgraph = DotAstHelper.getAllNodeStatementsInSubgraph(it);
          Iterables.<NodeStmt>addAll(nodeStatementInSubgraph, _allNodeStatementsInSubgraph);
        }
      };
      _filter_1.forEach(_function);
      _xblockexpression = nodeStatementInSubgraph;
    }
    return _xblockexpression;
  }
}
