/*********************************************************************
 * Copyright (c) 2019 Boeing
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Boeing - initial API and implementation
 **********************************************************************/

package org.eclipse.osee.define.rest.importing.parsers;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.osee.define.api.importing.RoughArtifact;
import org.eclipse.osee.define.api.importing.RoughArtifactCollector;
import org.eclipse.osee.define.api.importing.RoughArtifactKind;
import org.eclipse.osee.framework.core.enums.CoreArtifactTypes;
import org.eclipse.osee.framework.core.enums.CoreAttributeTypes;
import org.eclipse.osee.framework.jdk.core.result.XResultData;
import org.eclipse.osee.framework.jdk.core.util.Strings;
import org.eclipse.osee.orcs.OrcsApi;

/**
 * This is a special word outline extractor that handles documents generated by Doors. It is looking for tags wrapped in
 * braces {}. The tags define the scope and type of the artifact the import process creates.
 *
 * @author David W. Miller
 */

public class DoorsWordOutlineExtractor extends WordOutlineExtractorDelegate {
   private static final Pattern reqnamePattern = Pattern.compile("(\\{[^}]+\\})(.*?</w:p>)");
   private static final Pattern outlinePattern = Pattern.compile("(\\{[^}]+\\})");
   private static final Pattern doorsIdPattern = Pattern.compile("\\{[^-]+\\-([^}]+)\\}");
   private int lastReqNumber = 1;

   @Override
   protected void addChildRoughArtifact(OrcsApi orcsApi, XResultData results, String content, RoughArtifactCollector collector) {
      StringBuilder newName = new StringBuilder();
      StringBuilder newContent = new StringBuilder();
      if (needsNewArt(content, newName, newContent)) {
         setContent(); // finishes the previous rough artifact
         String number = String.format("%s.0.%d", lastHeaderNumber.toString(), lastReqNumber++);

         String name = newName.toString();
         if (name.contains("<w")) { // remove invalid tags from names
            name = name.replaceAll("<[^>]+>", "");
         }
         roughArtifact = new RoughArtifact(orcsApi, results, getRoughArtifactType(name));
         roughArtifact.setName(name);

         if (collector != null) {
            collector.addRoughArtifact(roughArtifact);
         }
         if (Strings.isValid(number)) {
            roughArtifact.setSectionNumber(number);
            roughArtifact.addAttribute(CoreAttributeTypes.ParagraphNumber, number);
         }
         Matcher match = doorsIdPattern.matcher(name);
         if (match.find()) {
            String toSet = match.group(1);
            roughArtifact.addAttribute(CoreAttributeTypes.StaticId, toSet);
         }
         roughArtifact.addAttribute(CoreAttributeTypes.PublishInline, "True");
         wordFormattedContent.append(newContent.toString());
         setArtifactType(roughArtifact, number);
         previousNamedArtifact = roughArtifact;
      } else {
         wordFormattedContent.append(content);
      }
   }

   private RoughArtifactKind getRoughArtifactType(String newName) {
      RoughArtifactKind toReturn = RoughArtifactKind.PRIMARY;
      if (newName.contains("Doc Dsc")) {
         toReturn = RoughArtifactKind.TERTIARY;
      } else if (newName.contains("Dsg Dsc")) {
         toReturn = RoughArtifactKind.QUATERNARY;
      }
      return toReturn;
   }

   private void setArtifactType(RoughArtifact art, String number) {
      RoughArtifactKind setType = art.getRoughArtifactKind();
      if (setType.equals(RoughArtifactKind.PRIMARY)) {
         if (number.startsWith("3.")) {
            art.setType(CoreArtifactTypes.SystemRequirementMsWord);
         }
      }
   }

   @Override
   public void processHeadingText(RoughArtifact roughArtifact, String headingText) {
      roughArtifact.setRoughArtifactKind(RoughArtifactKind.SECONDARY);
      Matcher match = outlinePattern.matcher(headingText);
      if (match.find()) {
         roughArtifact.setName(headingText.substring(0, match.start(1)).trim());
         Matcher matchId = doorsIdPattern.matcher(headingText);
         if (matchId.find()) {
            roughArtifact.addAttribute(CoreAttributeTypes.StaticId, matchId.group(1));
         }
         return;
      }
      roughArtifact.setName(headingText.trim());
   }

   @Override
   protected boolean checkSectionNumber(String sectionNumber) {
      if (sectionNumber.contains(".0.")) {
         return true; // special case of numbering requirements below a section number
      }
      return false;
   }

   @Override
   public void resetReqNumber() {
      lastReqNumber = 1;
   }

   @Override
   public String getName() {
      return "Doors Word Outline";
   }

   private boolean needsNewArt(String content, StringBuilder newName, StringBuilder newContent) {
      if (content.contains("####")) {
         return false; // special case for match in description
      }
      Matcher match = reqnamePattern.matcher(content);
      if (match.find()) {
         String begin = content.substring(0, match.start(1));
         newName.append(content.substring(match.start(1), match.start(2)));
         String endString = content.substring(match.start(2), match.end(2));
         newContent.append(begin + endString);
         return true; // only finds the first one
      }
      return false;
   }
}
