/*********************************************************************
 * Copyright (c) 2013 Boeing
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Boeing - initial API and implementation
 **********************************************************************/

package org.eclipse.osee.ats.ide.search;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import org.eclipse.osee.ats.api.data.AtsAttributeTypes;
import org.eclipse.osee.ats.ide.internal.AtsApiService;
import org.eclipse.osee.framework.core.data.ArtifactId;
import org.eclipse.osee.framework.core.enums.DeletionFlag;
import org.eclipse.osee.framework.core.exception.ArtifactDoesNotExist;
import org.eclipse.osee.framework.core.exception.MultipleArtifactsExist;
import org.eclipse.osee.framework.jdk.core.util.GUID;
import org.eclipse.osee.framework.skynet.core.artifact.Artifact;
import org.eclipse.osee.framework.skynet.core.artifact.search.ArtifactQuery;

/**
 * @author John Misinco
 */
public class AtsArtifactQuery {

   public static Artifact getArtifactFromId(String guidOrAtsId) {
      List<Artifact> artifacts = new LinkedList<>();

      if (GUID.isValid(guidOrAtsId)) {
         artifacts.add(ArtifactQuery.getArtifactFromId(guidOrAtsId, AtsApiService.get().getAtsBranch()));
      } else {
         artifacts.addAll(ArtifactQuery.getArtifactListFromAttributeValues(AtsAttributeTypes.AtsId,
            Collections.singleton(guidOrAtsId), AtsApiService.get().getAtsBranch(), 1));
      }

      if (artifacts.isEmpty()) {
         throw new ArtifactDoesNotExist("AtsArtifactQuery: No artifact found with id %s on ATS branch", guidOrAtsId);
      }
      if (artifacts.size() > 1) {
         throw new MultipleArtifactsExist("%d artifacts found with id %s", artifacts.size(), guidOrAtsId);
      }
      return artifacts.iterator().next();
   }

   public static List<Artifact> getArtifactListFromIds(Collection<String> guidsOrAtsIds) {
      List<Artifact> toReturn = new LinkedList<>();
      List<String> guids = new LinkedList<>();
      List<String> atsIds = new LinkedList<>();
      for (String guidOrAtsId : guidsOrAtsIds) {
         if (GUID.isValid(guidOrAtsId)) {
            guids.add(guidOrAtsId);
         } else {
            atsIds.add(guidOrAtsId.toUpperCase());
         }
      }

      if (!guids.isEmpty()) {
         List<Artifact> fromIds = ArtifactQuery.getArtifactListFromIds(guids, AtsApiService.get().getAtsBranch());
         toReturn.addAll(fromIds);
      }

      if (!atsIds.isEmpty()) {
         List<Artifact> fromIds = ArtifactQuery.getArtifactListFromAttributeValues(AtsAttributeTypes.AtsId, atsIds,
            AtsApiService.get().getAtsBranch(), atsIds.size());
         toReturn.addAll(fromIds);
      }

      return toReturn;
   }

   public static Artifact getArtifactFromId(long id) {
      return ArtifactQuery.getArtifactFromId(ArtifactId.valueOf(id), AtsApiService.get().getAtsBranch());
   }

   public static Artifact getArtifactFromIdOrNull(Long id) {
      return ArtifactQuery.getArtifactOrNull(ArtifactId.valueOf(id), AtsApiService.get().getAtsBranch(),
         DeletionFlag.EXCLUDE_DELETED);
   }
}