/**
 * <copyright>
 * 
 * Copyright (c) 2014 itemis and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     itemis - Initial API and implementation
 * 
 * </copyright>
 */
package org.eclipse.sphinx.emf.serialization.generators.xsd;

import com.google.common.base.Objects;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.ENamedElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.ExtendedMetaData;
import org.eclipse.emf.ecore.xml.type.XMLTypePackage;
import org.eclipse.sphinx.emf.metamodel.IMetaModelDescriptor;
import org.eclipse.sphinx.emf.metamodel.MetaModelDescriptorRegistry;
import org.eclipse.sphinx.emf.serialization.XMLPersistenceMappingExtendedMetaData;
import org.eclipse.sphinx.emf.serialization.generators.util.Ecore2XSDUtil;
import org.eclipse.sphinx.emf.serialization.generators.util.IGeneratorConstants;
import org.eclipse.sphinx.emf.serialization.generators.util.JavaXSDPrimitiveTypeMapping;
import org.eclipse.sphinx.emf.util.EcorePlatformUtil;
import org.eclipse.xsd.XSDAttributeDeclaration;
import org.eclipse.xsd.XSDAttributeGroupContent;
import org.eclipse.xsd.XSDAttributeUse;
import org.eclipse.xsd.XSDAttributeUseCategory;
import org.eclipse.xsd.XSDComplexTypeDefinition;
import org.eclipse.xsd.XSDCompositor;
import org.eclipse.xsd.XSDConstrainingFacet;
import org.eclipse.xsd.XSDDerivationMethod;
import org.eclipse.xsd.XSDElementDeclaration;
import org.eclipse.xsd.XSDEnumerationFacet;
import org.eclipse.xsd.XSDFactory;
import org.eclipse.xsd.XSDForm;
import org.eclipse.xsd.XSDImport;
import org.eclipse.xsd.XSDModelGroup;
import org.eclipse.xsd.XSDModelGroupDefinition;
import org.eclipse.xsd.XSDParticle;
import org.eclipse.xsd.XSDPatternFacet;
import org.eclipse.xsd.XSDProcessContents;
import org.eclipse.xsd.XSDSchema;
import org.eclipse.xsd.XSDSchemaContent;
import org.eclipse.xsd.XSDSimpleTypeDefinition;
import org.eclipse.xsd.XSDTypeDefinition;
import org.eclipse.xsd.XSDWildcard;
import org.eclipse.xsd.ecore.NameMangler;
import org.eclipse.xsd.util.XSDConstants;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

@SuppressWarnings("all")
public class Ecore2XSDFactory extends NameMangler {
  /**
   * The schema for schema namespace version to be used for new schemas.
   */
  protected String defaultXMLSchemaNamespace = XSDConstants.SCHEMA_FOR_SCHEMA_URI_2001;
  
  /**
   * The XML namespace version to be used for new schemas.
   */
  protected String defaultXMLNamespace = XSDConstants.XML_NAMESPACE_URI_1998;
  
  /**
   * The extended meta data used to determine the schema structure.
   */
  protected XMLPersistenceMappingExtendedMetaData xsdExtendedMetaData = XMLPersistenceMappingExtendedMetaData.INSTANCE;
  
  protected static XSDFactory xsdFactory = XSDFactory.eINSTANCE;
  
  public Set<String> patternCaseSet = new HashSet<String>();
  
  public Set<EPackage> referencedEcores = new HashSet<EPackage>();
  
  public Map<String, String> nsSchemaLocations = new HashMap<String, String>();
  
  /**
   * The schema for user schema namespace version to be used for new schemas.
   */
  protected String defaultUserSchemaNamespace;
  
  public EPackage ecoreModel;
  
  public Ecore2XSDFactory(final EPackage ecoreModel) {
    this.ecoreModel = ecoreModel;
  }
  
  public XSDSchema initSchema(final EPackage ePackage, final ResourceSet resourceSet, final IProgressMonitor monitor) {
    final SubMonitor progress = SubMonitor.convert(monitor, 100);
    boolean _isCanceled = progress.isCanceled();
    if (_isCanceled) {
      throw new OperationCanceledException();
    }
    progress.subTask("Initialize XSD schema");
    final XSDSchema xsdSchema = XSDFactory.eINSTANCE.createXSDSchema();
    final String targetNamespace = this.xsdExtendedMetaData.getNamespace(ePackage);
    xsdSchema.setTargetNamespace(targetNamespace);
    Map<String, String> namespaces = xsdSchema.getQNamePrefixToNamespaceMap();
    xsdSchema.setSchemaForSchemaQNamePrefix(IGeneratorConstants.DEFAULT_XML_SCHEMA_NAMESPACE_PREFIX);
    namespaces.put(IGeneratorConstants.DEFAULT_XML_SCHEMA_NAMESPACE_PREFIX, this.defaultXMLSchemaNamespace);
    final String nsPrefix = this.getRootNsPrefix(ePackage);
    Ecore2XSDUtil.handlePrefix(namespaces, nsPrefix, targetNamespace);
    xsdSchema.setElementFormDefault(XSDForm.QUALIFIED_LITERAL);
    xsdSchema.setAttributeFormDefault(XSDForm.UNQUALIFIED_LITERAL);
    xsdSchema.updateElement();
    this.initSchemaImports(ePackage, xsdSchema, monitor);
    this.defaultUserSchemaNamespace = targetNamespace;
    progress.done();
    return xsdSchema;
  }
  
  /**
   * To be override by custom
   */
  protected void initSchemaImports(final EPackage ePackage, final XSDSchema xsdSchema, final IProgressMonitor monitor) {
    final Map<String, String> externalSchemaLocations = this.xsdExtendedMetaData.getXMLExternalSchemaLocations(ePackage);
    final Set<EPackage> referencedEcores = this.getReferencedEcoreRootPackages(monitor);
    final Consumer<EPackage> _function = new Consumer<EPackage>() {
      public void accept(final EPackage it) {
        final String namespace = Ecore2XSDFactory.this.getGlobalXSDSchemaNamespace(it);
        final String schemaLocation = Ecore2XSDFactory.this.getSchemaLocation(it);
        Ecore2XSDFactory.this.nsSchemaLocations.put(namespace, schemaLocation);
      }
    };
    referencedEcores.forEach(_function);
    this.nsSchemaLocations.putAll(externalSchemaLocations);
    List<String> orderedNs = new ArrayList<String>();
    Set<String> _keySet = this.nsSchemaLocations.keySet();
    orderedNs.addAll(_keySet);
    Collections.<String>sort(orderedNs);
    final Consumer<String> _function_1 = new Consumer<String>() {
      public void accept(final String it) {
        EList<XSDSchemaContent> _contents = xsdSchema.getContents();
        String _get = Ecore2XSDFactory.this.nsSchemaLocations.get(it);
        XSDImport _createXSDImport = Ecore2XSDFactory.this.createXSDImport(it, _get);
        _contents.add(_createXSDImport);
      }
    };
    orderedNs.forEach(_function_1);
  }
  
  protected String getRootNsPrefix(final EPackage ePackage) {
    return ePackage.getNsPrefix();
  }
  
  public XSDImport createXSDImport(final String nameSpace, final String schemaLocation) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(nameSpace, schemaLocation);
    final XSDImport _result;
    synchronized (_createCache_createXSDImport) {
      if (_createCache_createXSDImport.containsKey(_cacheKey)) {
        return _createCache_createXSDImport.get(_cacheKey);
      }
      XSDImport _createXSDImport = Ecore2XSDFactory.xsdFactory.createXSDImport();
      _result = _createXSDImport;
      _createCache_createXSDImport.put(_cacheKey, _result);
    }
    _init_createXSDImport(_result, nameSpace, schemaLocation);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, XSDImport> _createCache_createXSDImport = CollectionLiterals.newHashMap();
  
  private void _init_createXSDImport(final XSDImport xsdImport, final String nameSpace, final String schemaLocation) {
    xsdImport.setNamespace(nameSpace);
    xsdImport.setSchemaLocation(schemaLocation);
  }
  
  /**
   * this method should be overriden to provide the actual referenced type xml name
   * in general case, it is the upper case of the fetched xml name
   * in EAST-ADL, it may also be +"-IREF" or +"-TREF"
   */
  protected String getTypeXMLName(final String xmlName, final ENamedElement eElement) {
    EObject _rootContainer = EcoreUtil.getRootContainer(eElement);
    final EPackage rootPackage = ((EPackage) _rootContainer);
    final String nsURI = rootPackage.getNsURI();
    String _namespace = this.xsdExtendedMetaData.getNamespace(this.ecoreModel);
    boolean _equals = nsURI.equals(_namespace);
    if (_equals) {
      return xmlName;
    } else {
      boolean _equals_1 = nsURI.equals(IGeneratorConstants.DEFAULT_ECORE_NAMESPACE);
      if (_equals_1) {
        return xmlName;
      } else {
        boolean _contains = this.referencedEcores.contains(rootPackage);
        if (_contains) {
          String result = Ecore2XSDUtil.getSingularName(eElement);
          return result;
        }
      }
    }
    return Ecore2XSDUtil.getSingularName(eElement);
  }
  
  /**
   * Get the global XSD namespace of the given root package, it may be different from the namespace of the root package.
   * For example, the namespace of AUTOSAR is "http://autosar.org/schema/r4.0", while the namespace of the root AUTOSAR package is "http://autosar.org/schema/r4.0/autosar40"
   */
  protected String getGlobalXSDSchemaNamespace(final EPackage rootPackage) {
    try {
      String namespace = rootPackage.getNsURI();
      final URI namespaceURI = new URI(namespace);
      final IMetaModelDescriptor mmDescriptor = MetaModelDescriptorRegistry.INSTANCE.getDescriptor(namespaceURI);
      boolean _notEquals = (!Objects.equal(mmDescriptor, null));
      if (_notEquals) {
        URI _namespaceURI = mmDescriptor.getNamespaceURI();
        String _string = _namespaceURI.toString();
        namespace = _string;
      }
      return namespace;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  protected String getSchemaLocation(final EPackage referencedRootPackage) {
    final String namespace = this.getGlobalXSDSchemaNamespace(referencedRootPackage);
    String schemaLocation = this.xsdExtendedMetaData.getXMLSchemaLocation(referencedRootPackage);
    boolean _equals = Objects.equal(null, schemaLocation);
    if (_equals) {
      boolean _equals_1 = namespace.equals(IGeneratorConstants.DEFAULT_ECORE_NAMESPACE);
      if (_equals_1) {
        schemaLocation = "platform:/plugin/org.eclipse.emf.ecore/model/Ecore.xsd";
      } else {
        final org.eclipse.emf.common.util.URI uri = this.getXSDSchemaFileURIFromNsURI(namespace);
        boolean _notEquals = (!Objects.equal(uri, null));
        if (_notEquals) {
          String _string = uri.toString();
          schemaLocation = _string;
        } else {
          String _name = referencedRootPackage.getName();
          String _plus = (_name + IGeneratorConstants.DEFAULT_XML_SCHEMA_NAMESPACE_PREFIX);
          schemaLocation = _plus;
        }
      }
    }
    return schemaLocation;
  }
  
  protected void loadImportReferencedXSD(final EPackage referencedRootPackage, final XSDSchema xsdSchema, final ResourceSet resourceSet) {
    final String namespace = this.getGlobalXSDSchemaNamespace(referencedRootPackage);
    String schemaLocation = this.xsdExtendedMetaData.getXMLSchemaLocation(referencedRootPackage);
    boolean _equals = Objects.equal(null, schemaLocation);
    if (_equals) {
      boolean _equals_1 = namespace.equals(IGeneratorConstants.DEFAULT_ECORE_NAMESPACE);
      if (_equals_1) {
        schemaLocation = "platform:/plugin/org.eclipse.emf.ecore/model/Ecore.xsd";
      } else {
        final org.eclipse.emf.common.util.URI uri = this.getXSDSchemaFileURIFromNsURI(namespace);
        boolean _notEquals = (!Objects.equal(uri, null));
        if (_notEquals) {
          String _string = uri.toString();
          schemaLocation = _string;
        } else {
          String _name = referencedRootPackage.getName();
          String _plus = (_name + IGeneratorConstants.DEFAULT_XML_SCHEMA_NAMESPACE_PREFIX);
          schemaLocation = _plus;
        }
      }
    }
    final XSDImport xsdImport = this.createXSDImport(namespace, schemaLocation);
    EList<XSDSchemaContent> _contents = xsdSchema.getContents();
    _contents.add(xsdImport);
  }
  
  protected org.eclipse.emf.common.util.URI getXSDSchemaFileURIFromNsURI(final String nsURI) {
    final IPath path = new Path("C:/Work/Eclipse-Kepler/runtime-EclipseApplication-Kepler/xsd.generator.test/model/referenced/hummingbird20.xsd");
    return EcorePlatformUtil.createURI(path);
  }
  
  /**
   * Get referenced ecore root packages.
   * If references to simple Ecore data types or XSD Types, such as EString, then the referenced Ecore package is not taken into account
   */
  protected Set<EPackage> getReferencedEcoreRootPackages(final IProgressMonitor monitor) {
    final SubMonitor progress = SubMonitor.convert(monitor, 100);
    boolean _isCanceled = progress.isCanceled();
    if (_isCanceled) {
      throw new OperationCanceledException();
    }
    final String targetNamespace = this.xsdExtendedMetaData.getNamespace(this.ecoreModel);
    TreeIterator<EObject> _eAllContents = this.ecoreModel.eAllContents();
    final Procedure1<EObject> _function = new Procedure1<EObject>() {
      public void apply(final EObject it) {
        if ((it instanceof EClass)) {
          EList<EClass> _eSuperTypes = ((EClass)it).getESuperTypes();
          final Consumer<EClass> _function = new Consumer<EClass>() {
            public void accept(final EClass it) {
              EObject _rootContainer = EcoreUtil.getRootContainer(it);
              final EPackage rootEPackage = ((EPackage) _rootContainer);
              final String uri = rootEPackage.getNsURI();
              boolean _notEquals = (!Objects.equal(uri, targetNamespace));
              if (_notEquals) {
                Ecore2XSDFactory.this.referencedEcores.add(rootEPackage);
              }
            }
          };
          _eSuperTypes.forEach(_function);
        } else {
          if ((it instanceof EStructuralFeature)) {
            final EClassifier eType = ((EStructuralFeature)it).getEType();
            final EObject container = EcoreUtil.getRootContainer(eType);
            if ((container instanceof EPackage)) {
              final String uri = ((EPackage)container).getNsURI();
              boolean _notEquals = (!Objects.equal(uri, targetNamespace));
              if (_notEquals) {
                boolean _or = false;
                boolean _equals = uri.equals(IGeneratorConstants.DEFAULT_ECORE_NAMESPACE);
                if (_equals) {
                  _or = true;
                } else {
                  boolean _equals_1 = XMLTypePackage.eNS_URI.equals(uri);
                  _or = _equals_1;
                }
                if (_or) {
                } else {
                  Ecore2XSDFactory.this.referencedEcores.add(((EPackage)container));
                }
              }
            }
          }
        }
        progress.worked(1);
        boolean _isCanceled = progress.isCanceled();
        if (_isCanceled) {
          throw new OperationCanceledException();
        }
      }
    };
    IteratorExtensions.<EObject>forEach(_eAllContents, _function);
    return this.referencedEcores;
  }
  
  protected String getElementXSDURI(final EClassifier eClassifier) {
    EObject _rootContainer = EcoreUtil.getRootContainer(eClassifier);
    final EPackage rootEPackage = ((EPackage) _rootContainer);
    final String namespace = this.xsdExtendedMetaData.getNamespace(rootEPackage);
    String result = this.xsdExtendedMetaData.getXMLName(eClassifier);
    String _typeXMLName = this.getTypeXMLName(result, eClassifier);
    result = _typeXMLName;
    boolean _notEquals = (!Objects.equal(namespace, null));
    if (_notEquals) {
      result = ((namespace + "#") + result);
    }
    return result;
  }
  
  protected String getQualifiedRootPackageName(final EPackage ePackage) {
    return this.ecoreModel.getName();
  }
  
  protected List<EStructuralFeature> getEAllRelevantStructuralFeatures(final EClass eClass) {
    return eClass.getEAllStructuralFeatures();
  }
  
  /**
   * Create a XSD-element for the class of which the "global_element" tag is true.
   * Add this XSD-element into the xsdSchema.
   */
  public void createGlobalElement3(final EPackage rootEPackageModel, final XSDSchema xsdSchema) {
    final List<EClassifier> globalEClassifiers = Ecore2XSDUtil.getGlobalElements(rootEPackageModel);
    final Consumer<EClassifier> _function = new Consumer<EClassifier>() {
      public void accept(final EClassifier it) {
        final XSDElementDeclaration xsdGlobalElement = Ecore2XSDFactory.xsdFactory.createXSDElementDeclaration();
        String _xMLName = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLName(it);
        xsdGlobalElement.setName(_xMLName);
        final String uri = Ecore2XSDFactory.this.getElementXSDURI(it);
        final XSDTypeDefinition xsdTypeDefinition = xsdSchema.resolveTypeDefinitionURI(uri);
        xsdGlobalElement.setTypeDefinition(xsdTypeDefinition);
        EList<XSDSchemaContent> _contents = xsdSchema.getContents();
        _contents.add(xsdGlobalElement);
      }
    };
    globalEClassifiers.forEach(_function);
  }
  
  /**
   * RMF ReqIF rule
   */
  public void createFixedRefTypes4(final XSDSchema xsdSchema) {
    final XSDSimpleTypeDefinition simpleTypeLocal = Ecore2XSDFactory.xsdFactory.createXSDSimpleTypeDefinition();
    simpleTypeLocal.setName("LOCAL-REF");
    final XSDTypeDefinition baseTypeLocal = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#IDREF"));
    simpleTypeLocal.setBaseTypeDefinition(((XSDSimpleTypeDefinition) baseTypeLocal));
    EList<XSDSchemaContent> _contents = xsdSchema.getContents();
    _contents.add(simpleTypeLocal);
    final XSDSimpleTypeDefinition simpleTypeGlobal = Ecore2XSDFactory.xsdFactory.createXSDSimpleTypeDefinition();
    simpleTypeGlobal.setName("GLOBAL-REF");
    final XSDTypeDefinition baseTypeGlobal = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    simpleTypeGlobal.setBaseTypeDefinition(((XSDSimpleTypeDefinition) baseTypeGlobal));
    EList<XSDSchemaContent> _contents_1 = xsdSchema.getContents();
    _contents_1.add(simpleTypeGlobal);
  }
  
  /**
   * Create xsd:complexType for a class
   */
  public XSDComplexTypeDefinition createClassTypeDefinition5(final EClass eClass, final XSDSchema xsdSchema, final ArrayList<EClass> referencedClass, final IProgressMonitor monitor) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(eClass, xsdSchema, referencedClass, monitor);
    final XSDComplexTypeDefinition _result;
    synchronized (_createCache_createClassTypeDefinition5) {
      if (_createCache_createClassTypeDefinition5.containsKey(_cacheKey)) {
        return _createCache_createClassTypeDefinition5.get(_cacheKey);
      }
      XSDComplexTypeDefinition _createXSDComplexTypeDefinition = Ecore2XSDFactory.xsdFactory.createXSDComplexTypeDefinition();
      _result = _createXSDComplexTypeDefinition;
      _createCache_createClassTypeDefinition5.put(_cacheKey, _result);
    }
    _init_createClassTypeDefinition5(_result, eClass, xsdSchema, referencedClass, monitor);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, XSDComplexTypeDefinition> _createCache_createClassTypeDefinition5 = CollectionLiterals.newHashMap();
  
  private void _init_createClassTypeDefinition5(final XSDComplexTypeDefinition xsdComplexTypeDefinition, final EClass eClass, final XSDSchema xsdSchema, final ArrayList<EClass> referencedClass, final IProgressMonitor monitor) {
    final SubMonitor progress = SubMonitor.convert(monitor, 100);
    boolean _isCanceled = progress.isCanceled();
    if (_isCanceled) {
      throw new OperationCanceledException();
    }
    String _name = eClass.getName();
    String _plus = ("create XSD Complex Type Definition for " + _name);
    progress.subTask(_plus);
    String _xMLName = this.xsdExtendedMetaData.getXMLName(eClass);
    xsdComplexTypeDefinition.setName(_xMLName);
    final XSDParticle xsdParticle = XSDFactory.eINSTANCE.createXSDParticle();
    xsdComplexTypeDefinition.setContent(xsdParticle);
    final XSDModelGroup xsdModelGroup = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup.setCompositor(XSDCompositor.SEQUENCE_LITERAL);
    xsdParticle.setContent(xsdModelGroup);
    final List<EStructuralFeature> relevantStructuralFeatures = this.getEAllRelevantStructuralFeatures(eClass);
    SubMonitor _newChild = progress.newChild(100);
    int _size = relevantStructuralFeatures.size();
    final SubMonitor subProgress = _newChild.setWorkRemaining(_size);
    final Consumer<EStructuralFeature> _function = new Consumer<EStructuralFeature>() {
      public void accept(final EStructuralFeature it) {
        XSDParticle xsdParticleForFeature = null;
        int _featureKind = ExtendedMetaData.INSTANCE.getFeatureKind(it);
        boolean _equals = Integer.valueOf(ExtendedMetaData.ELEMENT_FEATURE).equals(Integer.valueOf(_featureKind));
        if (_equals) {
          if ((it instanceof EAttribute)) {
            XSDParticle _switchResult = null;
            int _xMLPersistenceMappingStrategy = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLPersistenceMappingStrategy(it);
            switch (_xMLPersistenceMappingStrategy) {
              case XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__0100__FEATURE_ELEMENT:
                _switchResult = Ecore2XSDFactory.this.createClassElementAttribute5a(((EAttribute)it), xsdSchema);
                break;
              case XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__1100__FEATURE_WRAPPER_ELEMENT__FEATURE_ELEMENT:
                _switchResult = Ecore2XSDFactory.this.createEAttributeContained1100(((EAttribute)it), xsdSchema);
                break;
              default:
                Object _xblockexpression = null;
                {
                  int _xMLPersistenceMappingStrategy_1 = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLPersistenceMappingStrategy(it);
                  String _plus = ("Not supported: EAttribute " + Integer.valueOf(_xMLPersistenceMappingStrategy_1));
                  System.err.println(_plus);
                  _xblockexpression = null;
                }
                _switchResult = ((XSDParticle)_xblockexpression);
                break;
            }
            xsdParticleForFeature = _switchResult;
          } else {
            if ((it instanceof EReference)) {
              boolean _isContainment = ((EReference)it).isContainment();
              if (_isContainment) {
                XSDParticle _switchResult_1 = null;
                int _xMLPersistenceMappingStrategy_1 = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLPersistenceMappingStrategy(it);
                switch (_xMLPersistenceMappingStrategy_1) {
                  case XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__0100__FEATURE_ELEMENT:
                    _switchResult_1 = Ecore2XSDFactory.this.create_EReference_contained0100(it, eClass, xsdSchema);
                    break;
                  case XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__0101__FEATURE_ELEMENT__CLASSIFIER_ELEMENT:
                    _switchResult_1 = Ecore2XSDFactory.this.create_EReference_contained0101(it, eClass, xsdSchema);
                    break;
                  case XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__1001__FEATURE_WRAPPER_ELEMENT__CLASSIFIER_ELEMENT:
                    _switchResult_1 = Ecore2XSDFactory.this.create_EReference_contained1001(it, eClass, xsdSchema);
                    break;
                  case XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__1100__FEATURE_WRAPPER_ELEMENT__FEATURE_ELEMENT:
                    _switchResult_1 = Ecore2XSDFactory.this.create_EReference_contained1100(it, eClass, xsdSchema);
                    break;
                  default:
                    _switchResult_1 = null;
                    break;
                }
                xsdParticleForFeature = _switchResult_1;
              } else {
                XSDParticle _switchResult_2 = null;
                int _xMLPersistenceMappingStrategy_2 = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLPersistenceMappingStrategy(it);
                switch (_xMLPersistenceMappingStrategy_2) {
                  case XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__1001__FEATURE_WRAPPER_ELEMENT__CLASSIFIER_ELEMENT:
                    _switchResult_2 = Ecore2XSDFactory.this.createClassComposition5b(it, eClass);
                    break;
                  case XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__1100__FEATURE_WRAPPER_ELEMENT__FEATURE_ELEMENT:
                    _switchResult_2 = Ecore2XSDFactory.this.create_EReference_referenced1100Many_5l(it, eClass, xsdSchema, referencedClass);
                    break;
                  case XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__0100__FEATURE_ELEMENT:
                    _switchResult_2 = Ecore2XSDFactory.this.create_EReference_referenced0100Many_5m(it, eClass, xsdSchema, referencedClass);
                    break;
                  default:
                    _switchResult_2 = null;
                    break;
                }
                xsdParticleForFeature = _switchResult_2;
              }
            }
          }
          boolean _notEquals = (!Objects.equal(null, xsdParticleForFeature));
          if (_notEquals) {
            EList<XSDParticle> _contents = xsdModelGroup.getContents();
            _contents.add(xsdParticleForFeature);
          }
        } else {
          int _featureKind_1 = ExtendedMetaData.INSTANCE.getFeatureKind(it);
          boolean _equals_1 = Integer.valueOf(ExtendedMetaData.ATTRIBUTE_FEATURE).equals(Integer.valueOf(_featureKind_1));
          if (_equals_1) {
            if ((it instanceof EAttribute)) {
              final XSDAttributeUse xsdAttributeUse = Ecore2XSDFactory.this.createClassAttribute5d(((EAttribute)it), eClass, xsdSchema);
              EList<XSDAttributeGroupContent> _attributeContents = xsdComplexTypeDefinition.getAttributeContents();
              _attributeContents.add(xsdAttributeUse);
            }
          } else {
          }
        }
        subProgress.worked(1);
        boolean _isCanceled = subProgress.isCanceled();
        if (_isCanceled) {
          throw new OperationCanceledException();
        }
      }
    };
    relevantStructuralFeatures.forEach(_function);
    EList<XSDSchemaContent> _contents = xsdSchema.getContents();
    _contents.add(xsdComplexTypeDefinition);
  }
  
  /**
   * create xsd:element for a attribute with rule 5a of RMF
   * EAttribute & kind=element & FTFF (0100)
   * 
   * 5a. ClassElementAttribute ::=
   * 	<xsd:element
   * 		 name=" //Name of Attribute (single) // " minOccurs=" // Minimum // " maxOccurs=" // Maximum // "
   * 		( fixed=" // fixed value // " )?
   * 		type=" //Name of Attribute Type// "/>
   */
  public XSDParticle createClassElementAttribute5a(final EAttribute eAttribute, final XSDSchema xsdSchema) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    int _lowerBound = eAttribute.getLowerBound();
    xsdParticle.setMinOccurs(_lowerBound);
    int _upperBound = eAttribute.getUpperBound();
    xsdParticle.setMaxOccurs(_upperBound);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(eAttribute);
    xsdElement.setName(_xMLName);
    XSDTypeDefinition xsdTypeDefinition = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    final EClassifier baseType = eAttribute.getEAttributeType();
    boolean _and = false;
    if (!(baseType instanceof EDataType)) {
      _and = false;
    } else {
      Boolean _isXMLPrimitiveXsdType = this.isXMLPrimitiveXsdType(((EDataType) baseType));
      _and = (_isXMLPrimitiveXsdType).booleanValue();
    }
    if (_and) {
      final String xsdSimpleType = this.getXsdSimpleType(baseType);
      XSDTypeDefinition _resolveTypeDefinitionURI = xsdSchema.resolveTypeDefinitionURI(((this.defaultXMLSchemaNamespace + "#") + xsdSimpleType));
      xsdTypeDefinition = _resolveTypeDefinitionURI;
    } else {
      final String uri = this.getElementXSDURI(baseType);
      XSDTypeDefinition _resolveTypeDefinitionURI_1 = xsdSchema.resolveTypeDefinitionURI(uri);
      xsdTypeDefinition = _resolveTypeDefinitionURI_1;
    }
    xsdElement.setTypeDefinition(xsdTypeDefinition);
    xsdParticle.setContent(xsdElement);
    return xsdParticle;
  }
  
  /**
   * create xsd:element for a attribute with rule EAttributeContained1100
   * 
   * Example:
   * <pre>
   * <xsd:element name="PROPERTIES" minOccurs="0" maxOccurs="1" >
   *    <xsd:complexType>
   *      <xsd:choice minOccurs="0" maxOccurs="unbounded" >
   *        <xsd:element name="PROPERTY" type="node:STRING"/>
   *      </xsd:choice>
   *    </xsd:complexType>
   * </xsd:element>
   * </pre>
   */
  public XSDParticle createEAttributeContained1100(final EAttribute eAttribute, final XSDSchema xsdSchema) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    int _lowerBound = eAttribute.getLowerBound();
    boolean _greaterThan = (_lowerBound > 0);
    if (_greaterThan) {
      xsdParticle.setMinOccurs(1);
    } else {
      xsdParticle.setMinOccurs(0);
    }
    xsdParticle.setMaxOccurs(1);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLWrapperName = this.xsdExtendedMetaData.getXMLWrapperName(eAttribute);
    xsdElement.setName(_xMLWrapperName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    int _upperBound = eAttribute.getUpperBound();
    xsdParticle2.setMaxOccurs(_upperBound);
    int _lowerBound_1 = eAttribute.getLowerBound();
    xsdParticle2.setMinOccurs(_lowerBound_1);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final EDataType datatype = eAttribute.getEAttributeType();
    final XSDParticle xsdParticle3 = XSDFactory.eINSTANCE.createXSDParticle();
    final XSDElementDeclaration xsdElement2 = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(eAttribute);
    xsdElement2.setName(_xMLName);
    XSDTypeDefinition xsdTypeDefinition = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    Boolean _isXMLPrimitiveXsdType = this.isXMLPrimitiveXsdType(datatype);
    if ((_isXMLPrimitiveXsdType).booleanValue()) {
      final String xsdSimpleType = this.getXsdSimpleType(datatype);
      XSDTypeDefinition _resolveTypeDefinitionURI = xsdSchema.resolveTypeDefinitionURI(((this.defaultXMLSchemaNamespace + "#") + xsdSimpleType));
      xsdTypeDefinition = _resolveTypeDefinitionURI;
    } else {
      final String uri = this.getElementXSDURI(datatype);
      XSDTypeDefinition _resolveTypeDefinitionURI_1 = xsdSchema.resolveTypeDefinitionURI(uri);
      xsdTypeDefinition = _resolveTypeDefinitionURI_1;
    }
    xsdElement2.setTypeDefinition(xsdTypeDefinition);
    xsdParticle3.setContent(xsdElement2);
    EList<XSDParticle> _contents = xsdModelGroup2.getContents();
    _contents.add(xsdParticle3);
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
    return xsdParticle;
  }
  
  /**
   * create xsd:element for a feature that the type is no composite with rule 5b of RMF
   * EReference & !containment & TFFT (1001)
   */
  public XSDParticle createClassComposition5b(final EStructuralFeature feature, final EClass eClass) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    int _lowerBound = feature.getLowerBound();
    boolean _greaterThan = (_lowerBound > 0);
    if (_greaterThan) {
      xsdParticle.setMinOccurs(1);
    } else {
      xsdParticle.setMinOccurs(0);
    }
    xsdParticle.setMaxOccurs(1);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(feature);
    xsdElement.setName(_xMLName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    int _upperBound = feature.getUpperBound();
    xsdParticle2.setMaxOccurs(_upperBound);
    int _lowerBound_1 = feature.getLowerBound();
    xsdParticle2.setMinOccurs(_lowerBound_1);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final EClassifier typeeClassifier = feature.getEType();
    final XSDParticle xsdParticle3 = XSDFactory.eINSTANCE.createXSDParticle();
    final XSDElementDeclaration xsdElement2 = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    xsdElement2.setName("xxxxxxxx");
    final String typeeClassifierURI = this.getElementXSDURI(typeeClassifier);
    final XSDTypeDefinition xsdTypeDefinition = xsdElement2.resolveTypeDefinitionURI(typeeClassifierURI);
    xsdElement2.setTypeDefinition(xsdTypeDefinition);
    xsdParticle3.setContent(xsdElement2);
    EList<XSDParticle> _contents = xsdModelGroup2.getContents();
    _contents.add(xsdParticle3);
    if ((typeeClassifier instanceof EClass)) {
      List<EClass> _findESubTypesOf = Ecore2XSDUtil.findESubTypesOf(((EClass)typeeClassifier));
      final Consumer<EClass> _function = new Consumer<EClass>() {
        public void accept(final EClass it) {
          final XSDParticle xsdParticleSubType = XSDFactory.eINSTANCE.createXSDParticle();
          final XSDElementDeclaration xsdElementSubType = XSDFactory.eINSTANCE.createXSDElementDeclaration();
          String _xMLName = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLName(it);
          xsdElementSubType.setName(_xMLName);
          final String subTypeURI = Ecore2XSDFactory.this.getElementXSDURI(it);
          final XSDTypeDefinition xsdTypeDefinitionSubType = xsdElementSubType.resolveTypeDefinitionURI(subTypeURI);
          xsdElementSubType.setTypeDefinition(xsdTypeDefinitionSubType);
          xsdParticleSubType.setContent(xsdElementSubType);
          EList<XSDParticle> _contents = xsdModelGroup2.getContents();
          _contents.add(xsdParticleSubType);
        }
      };
      _findESubTypesOf.forEach(_function);
    }
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
    return xsdParticle;
  }
  
  /**
   * create xsd:element for a feature that the type is composite with rule 5c of RMF
   * EReference & containment & TFFT (1001) -- pattern 00010 in AUTOSAR
   * 
   * 5c. ClassCompositions ::=
   * 	<xsd:element name=" // Name of Target Property (plural) // " minOccurs="( "0" | "1" )" maxOccurs="1">
   * 		<xsd:complexType>
   * 			<xsd:choice minOccurs="//Minimum of Target Property//" maxOccurs="// Max of Target Property// ">
   * 				(<xsd:element
   * 					  name=" // XML Name of Target (Sub) Class // "
   * 					  type=" //namespace// ":" // XML Name of Target Class // " /> )+
   * 			</xsd:choice>
   * 		/xsd:complexType>
   * 	</xsd:element>
   */
  public XSDParticle create_EReference_contained1001(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    int _lowerBound = feature.getLowerBound();
    boolean _greaterThan = (_lowerBound > 0);
    if (_greaterThan) {
      xsdParticle.setMinOccurs(1);
    } else {
      xsdParticle.setMinOccurs(0);
    }
    xsdParticle.setMaxOccurs(1);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLWrapperName = this.xsdExtendedMetaData.getXMLWrapperName(feature);
    xsdElement.setName(_xMLWrapperName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    int _upperBound = feature.getUpperBound();
    xsdParticle2.setMaxOccurs(_upperBound);
    int _lowerBound_1 = feature.getLowerBound();
    xsdParticle2.setMinOccurs(_lowerBound_1);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final EClassifier typeEClassifier = feature.getEType();
    if ((typeEClassifier instanceof EClass)) {
      List<EClass> _findAllConcreteTypes = Ecore2XSDUtil.findAllConcreteTypes(((EClass)typeEClassifier), null);
      final Consumer<EClass> _function = new Consumer<EClass>() {
        public void accept(final EClass it) {
          final XSDParticle xsdParticleSubType = XSDFactory.eINSTANCE.createXSDParticle();
          final XSDElementDeclaration xsdElementSubType = XSDFactory.eINSTANCE.createXSDElementDeclaration();
          String _xMLName = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLName(it, feature);
          final String subClassXMLName = Ecore2XSDFactory.this.getTypeXMLName(_xMLName, it);
          xsdElementSubType.setName(subClassXMLName);
          final String subTypeURI = Ecore2XSDFactory.this.getElementXSDURI(it);
          final XSDTypeDefinition xsdTypeDefinitionSubType = xsdElementSubType.resolveTypeDefinitionURI(subTypeURI);
          xsdElementSubType.setTypeDefinition(xsdTypeDefinitionSubType);
          xsdParticleSubType.setContent(xsdElementSubType);
          EList<XSDParticle> _contents = xsdModelGroup2.getContents();
          _contents.add(xsdParticleSubType);
        }
      };
      _findAllConcreteTypes.forEach(_function);
    }
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
    return xsdParticle;
  }
  
  /**
   * create xsd:element for a feature that the type is composite with rule 5c of RMF
   * EReference & containment & FTFT(0101) - pattern 00012 in AUTOSAR
   */
  public XSDParticle createClassComposition5c0101(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    int _lowerBound = feature.getLowerBound();
    xsdParticle.setMinOccurs(_lowerBound);
    int _upperBound = feature.getUpperBound();
    xsdParticle.setMaxOccurs(_upperBound);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLWrapperName = this.xsdExtendedMetaData.getXMLWrapperName(feature);
    xsdElement.setName(_xMLWrapperName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    int _lowerBound_1 = feature.getLowerBound();
    xsdParticle2.setMinOccurs(_lowerBound_1);
    xsdParticle2.setMaxOccurs(1);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final EClassifier typeEClassifier = feature.getEType();
    final XSDParticle xsdParticle3 = XSDFactory.eINSTANCE.createXSDParticle();
    final XSDElementDeclaration xsdElement2 = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _name = this.xsdExtendedMetaData.getName(typeEClassifier);
    xsdElement2.setName(_name);
    XSDTypeDefinition xsdTypeDefinition = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    boolean _and = false;
    if (!(typeEClassifier instanceof EDataType)) {
      _and = false;
    } else {
      Boolean _isXMLPrimitiveXsdType = this.isXMLPrimitiveXsdType(((EDataType) typeEClassifier));
      _and = (_isXMLPrimitiveXsdType).booleanValue();
    }
    if (_and) {
      final String xsdSimpleType = this.getXsdSimpleType(typeEClassifier);
      XSDTypeDefinition _resolveTypeDefinitionURI = xsdSchema.resolveTypeDefinitionURI(((this.defaultXMLSchemaNamespace + "#") + xsdSimpleType));
      xsdTypeDefinition = _resolveTypeDefinitionURI;
    } else {
      final String uri = this.getElementXSDURI(typeEClassifier);
      XSDTypeDefinition _resolveTypeDefinitionURI_1 = xsdSchema.resolveTypeDefinitionURI(uri);
      xsdTypeDefinition = _resolveTypeDefinitionURI_1;
    }
    xsdElement2.setTypeDefinition(xsdTypeDefinition);
    xsdParticle3.setContent(xsdElement2);
    EList<XSDParticle> _contents = xsdModelGroup2.getContents();
    _contents.add(xsdParticle3);
    if ((typeEClassifier instanceof EClass)) {
      List<EClass> _findESubTypesOf = Ecore2XSDUtil.findESubTypesOf(((EClass)typeEClassifier));
      final Consumer<EClass> _function = new Consumer<EClass>() {
        public void accept(final EClass it) {
          final XSDParticle xsdParticleSubType = XSDFactory.eINSTANCE.createXSDParticle();
          final XSDElementDeclaration xsdElementSubType = XSDFactory.eINSTANCE.createXSDElementDeclaration();
          String _xMLName = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLName(it);
          xsdElementSubType.setName(_xMLName);
          final String subTypeURI = Ecore2XSDFactory.this.getElementXSDURI(it);
          final XSDTypeDefinition xsdTypeDefinitionSubType = xsdElementSubType.resolveTypeDefinitionURI(subTypeURI);
          xsdElementSubType.setTypeDefinition(xsdTypeDefinitionSubType);
          xsdParticleSubType.setContent(xsdElementSubType);
          EList<XSDParticle> _contents = xsdModelGroup2.getContents();
          _contents.add(xsdParticleSubType);
        }
      };
      _findESubTypesOf.forEach(_function);
    }
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
    return xsdParticle;
  }
  
  /**
   * create xsd:attribute for a attribute with rule 5d of RMF
   * 
   * 5d. ClassAttribute ::=
   * 	<xsd:attribute name="// XML name of Attribute //"
   * 				   type="//Type of Attribute // "
   * 				   use= "prohibited" | "optional" | "required"  />
   */
  public XSDAttributeUse createClassAttribute5d(final EAttribute attribute, final EClass eClass, final XSDSchema xsdSchema) {
    XSDAttributeUse xsdAttributeUse = Ecore2XSDFactory.xsdFactory.createXSDAttributeUse();
    final XSDAttributeDeclaration attributeDeclaration = XSDFactory.eINSTANCE.createXSDAttributeDeclaration();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(attribute);
    attributeDeclaration.setName(_xMLName);
    XSDTypeDefinition xsdTypeDefinition = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    final EClassifier baseType = attribute.getEType();
    boolean _and = false;
    if (!(baseType instanceof EDataType)) {
      _and = false;
    } else {
      Boolean _isXMLPrimitiveXsdType = this.isXMLPrimitiveXsdType(((EDataType) baseType));
      _and = (_isXMLPrimitiveXsdType).booleanValue();
    }
    if (_and) {
      final String xsdSimpleType = this.getXsdSimpleType(baseType);
      XSDTypeDefinition _resolveTypeDefinitionURI = xsdSchema.resolveTypeDefinitionURI(((this.defaultXMLSchemaNamespace + "#") + xsdSimpleType));
      xsdTypeDefinition = _resolveTypeDefinitionURI;
    } else {
      final String uri = this.getElementXSDURI(baseType);
      XSDTypeDefinition _resolveTypeDefinitionURI_1 = xsdSchema.resolveTypeDefinitionURI(uri);
      xsdTypeDefinition = _resolveTypeDefinitionURI_1;
    }
    attributeDeclaration.setTypeDefinition(((XSDSimpleTypeDefinition) xsdTypeDefinition));
    xsdAttributeUse.setContent(attributeDeclaration);
    int _upperBound = attribute.getUpperBound();
    boolean _equals = (_upperBound == 0);
    if (_equals) {
      xsdAttributeUse.setUse(XSDAttributeUseCategory.PROHIBITED_LITERAL);
    } else {
      int _lowerBound = attribute.getLowerBound();
      boolean _equals_1 = (_lowerBound == 0);
      if (_equals_1) {
        xsdAttributeUse.setUse(XSDAttributeUseCategory.OPTIONAL_LITERAL);
      } else {
        xsdAttributeUse.setUse(XSDAttributeUseCategory.REQUIRED_LITERAL);
      }
    }
    return xsdAttributeUse;
  }
  
  /**
   * create xsd:element for a EReference with rule 5i.EReference_contained1100Many of RMF
   * EReference & containment & upperBound>1 && TTFF (1100)
   * same as createClassComposition5c1001
   * 
   * 5i:EReference_contained1100Many ::=
   * 	<xsd:element name=" // Name of Target Property (plural) // " minOccurs="( "0" | "1" )" maxOccurs="1">
   * 		<xsd:complexType>
   * 			<xsd:choice minOccurs="//Minimum of Target Property//" maxOccurs="// Max of Target Property// ">
   * 				(<xsd:element
   * 					  name=" // feature XML Name // "
   * 					  type=" //namespace// ":" // XML Name of Target Class // " /> )+
   * 			</xsd:choice>
   * 		</xsd:complexType>
   * </xsd:element>
   */
  public XSDParticle create_EReference_contained1100(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    int _lowerBound = feature.getLowerBound();
    boolean _greaterThan = (_lowerBound > 0);
    if (_greaterThan) {
      xsdParticle.setMinOccurs(1);
    } else {
      xsdParticle.setMinOccurs(0);
    }
    xsdParticle.setMaxOccurs(1);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLWrapperName = this.xsdExtendedMetaData.getXMLWrapperName(feature);
    xsdElement.setName(_xMLWrapperName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    int _upperBound = feature.getUpperBound();
    xsdParticle2.setMaxOccurs(_upperBound);
    int _lowerBound_1 = feature.getLowerBound();
    xsdParticle2.setMinOccurs(_lowerBound_1);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final EClassifier typeEClassifier = feature.getEType();
    final XSDParticle xsdParticle3 = XSDFactory.eINSTANCE.createXSDParticle();
    final XSDElementDeclaration xsdElement2 = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(feature);
    xsdElement2.setName(_xMLName);
    XSDTypeDefinition xsdTypeDefinition = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    boolean _and = false;
    if (!(typeEClassifier instanceof EDataType)) {
      _and = false;
    } else {
      Boolean _isXMLPrimitiveXsdType = this.isXMLPrimitiveXsdType(((EDataType) typeEClassifier));
      _and = (_isXMLPrimitiveXsdType).booleanValue();
    }
    if (_and) {
      final String xsdSimpleType = this.getXsdSimpleType(typeEClassifier);
      XSDTypeDefinition _resolveTypeDefinitionURI = xsdSchema.resolveTypeDefinitionURI(((this.defaultXMLSchemaNamespace + "#") + xsdSimpleType));
      xsdTypeDefinition = _resolveTypeDefinitionURI;
    } else {
      final String uri = this.getElementXSDURI(typeEClassifier);
      XSDTypeDefinition _resolveTypeDefinitionURI_1 = xsdSchema.resolveTypeDefinitionURI(uri);
      xsdTypeDefinition = _resolveTypeDefinitionURI_1;
    }
    xsdElement2.setTypeDefinition(xsdTypeDefinition);
    xsdParticle3.setContent(xsdElement2);
    EList<XSDParticle> _contents = xsdModelGroup2.getContents();
    _contents.add(xsdParticle3);
    if ((typeEClassifier instanceof EClass)) {
      List<EClass> _findESubTypesOf = Ecore2XSDUtil.findESubTypesOf(((EClass)typeEClassifier));
      final Consumer<EClass> _function = new Consumer<EClass>() {
        public void accept(final EClass it) {
          final XSDParticle xsdParticleSubType = XSDFactory.eINSTANCE.createXSDParticle();
          final XSDElementDeclaration xsdElementSubType = XSDFactory.eINSTANCE.createXSDElementDeclaration();
          String _xMLName = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLName(feature);
          xsdElementSubType.setName(_xMLName);
          final String subTypeURI = Ecore2XSDFactory.this.getElementXSDURI(it);
          final XSDTypeDefinition xsdTypeDefinitionSubType = xsdElementSubType.resolveTypeDefinitionURI(subTypeURI);
          xsdElementSubType.setTypeDefinition(xsdTypeDefinitionSubType);
          xsdParticleSubType.setContent(xsdElementSubType);
          EList<XSDParticle> _contents = xsdModelGroup2.getContents();
          _contents.add(xsdParticleSubType);
        }
      };
      _findESubTypesOf.forEach(_function);
    }
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
    return xsdParticle;
  }
  
  /**
   * create xsd:element for a EReference with rule 5j.EReference_contained0101Single of RMF
   * EReference & containment & upperBound=1 && FTFT (0101)
   * if (sub types >1):
   * 
   * 5j_a:EReference_contained0101Single ::=
   * 
   * 	<xsd:element name="//Name of Target Property (single)//" minOccurs="0|1" maxOccurs="1">
   * 		<xsd:complexType>
   * 			<xsd:choice minOccurs="//Minimum of Target Property//" maxOccurs="1 ">
   * 				(<xsd:element
   * 					  name=" // XML Name of Target (Sub) Class // "
   * 					  type="//namespace// ":" // XML Name of Target Class // " /> )+
   * 			</xsd:choice>
   * 		</xsd:complexType>
   * 	</xsd:element>
   */
  public XSDParticle create_EReference_contained0101(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    int _lowerBound = feature.getLowerBound();
    boolean _greaterThan = (_lowerBound > 0);
    if (_greaterThan) {
      xsdParticle.setMinOccurs(1);
    } else {
      xsdParticle.setMinOccurs(0);
    }
    int _upperBound = feature.getUpperBound();
    xsdParticle.setMaxOccurs(_upperBound);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    xsdParticle.setContent(xsdElement);
    String _xMLName = this.xsdExtendedMetaData.getXMLName(feature);
    xsdElement.setName(_xMLName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    xsdParticle2.setMaxOccurs(1);
    int _lowerBound_1 = feature.getLowerBound();
    xsdParticle2.setMinOccurs(_lowerBound_1);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final EClassifier typeEClassifier = feature.getEType();
    if ((typeEClassifier instanceof EClass)) {
      List<EClass> _findAllConcreteTypes = Ecore2XSDUtil.findAllConcreteTypes(((EClass)typeEClassifier), null);
      final Consumer<EClass> _function = new Consumer<EClass>() {
        public void accept(final EClass it) {
          final XSDParticle xsdParticleSubType = XSDFactory.eINSTANCE.createXSDParticle();
          final XSDElementDeclaration xsdElementSubType = XSDFactory.eINSTANCE.createXSDElementDeclaration();
          String _xMLName = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLName(it, feature);
          final String subClassXMLName = Ecore2XSDFactory.this.getTypeXMLName(_xMLName, it);
          xsdElementSubType.setName(subClassXMLName);
          final String subTypeURI = Ecore2XSDFactory.this.getElementXSDURI(it);
          final XSDTypeDefinition xsdTypeDefinitionSubType = xsdElementSubType.resolveTypeDefinitionURI(subTypeURI);
          xsdElementSubType.setTypeDefinition(xsdTypeDefinitionSubType);
          xsdParticleSubType.setContent(xsdElementSubType);
          EList<XSDParticle> _contents = xsdModelGroup2.getContents();
          _contents.add(xsdParticleSubType);
        }
      };
      _findAllConcreteTypes.forEach(_function);
    }
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    return xsdParticle;
  }
  
  /**
   * create xsd:element for a EReference with rule 5j.EReference_contained0101Single of RMF
   * EReference & containment & upperBound=1 && FTFT (0101)
   * if not (sub types >1):
   * 
   * 5j_b:EReference_contained0101Single ::=
   * 
   * <xsd:element name='" // Name of Target Property (single) // "'
   * 				minOccurs='"//Minimum of Target Property//"'
   * 				maxOccurs='"// Max of Target Property// "'
   * 				type="// xml Name of Target Class //"  >
   * </xsd:element>
   */
  public XSDParticle create_EReference_contained0100(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    int _lowerBound = feature.getLowerBound();
    xsdParticle.setMinOccurs(_lowerBound);
    int _upperBound = feature.getUpperBound();
    xsdParticle.setMaxOccurs(_upperBound);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(feature);
    xsdElement.setName(_xMLName);
    final EClassifier typeeClassifier = feature.getEType();
    XSDTypeDefinition xsdTypeDefinition = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    boolean _and = false;
    if (!(typeeClassifier instanceof EDataType)) {
      _and = false;
    } else {
      Boolean _isXMLPrimitiveXsdType = this.isXMLPrimitiveXsdType(((EDataType) typeeClassifier));
      _and = (_isXMLPrimitiveXsdType).booleanValue();
    }
    if (_and) {
      final String xsdSimpleType = this.getXsdSimpleType(typeeClassifier);
      XSDTypeDefinition _resolveTypeDefinitionURI = xsdSchema.resolveTypeDefinitionURI(((this.defaultXMLSchemaNamespace + "#") + xsdSimpleType));
      xsdTypeDefinition = _resolveTypeDefinitionURI;
    } else {
      final String uri = this.getElementXSDURI(typeeClassifier);
      XSDTypeDefinition _resolveTypeDefinitionURI_1 = xsdSchema.resolveTypeDefinitionURI(uri);
      xsdTypeDefinition = _resolveTypeDefinitionURI_1;
    }
    xsdElement.setTypeDefinition(xsdTypeDefinition);
    xsdParticle.setContent(xsdElement);
    return xsdParticle;
  }
  
  /**
   * To be overriden by custom
   * 
   * create xsd:element for a EReference with rule 5l.EReference_referenced1100Many of RMF
   * EReference & !containment & upperBound>1 && TTFF (1100)
   * 
   * (1) in case typeAttributeName is not "xsi:type"
   * 5l:EReference_referenced1100Many ::=
   * 	<xsd:element name="//roleXmlNamePlural//" minOccurs="//lowerMultiplicity//" maxOccurs="1">
   * 	  <xsd:complexType>
   * 		<xsd:choice minOccurs="//lowerMultiplicity//" maxOccurs="//upperMultiplicity//">
   * 		   <xsd:element name="//roleXmlName//">
   * 			</xsd:complexType>
   * 			   <xsd:simpleContent>
   * 				  <xsd:extension base="xsd:string">
   * 				    <xsd:attribute name="typeAttributeName" type="//typeXmlNsPrefix : typeXmlName//--SUBTYPE-ENUM" use="optional"/>
   * 				  </xsd:extension>
   * 			   </xsd:simpleContent>
   * 			</xsd:complexType>
   * 		   </xsd:element>
   * 		</xsd:choice>
   *   </xsd:complexType>
   * 	</xsd:element>
   * 
   * (2) in case typeAttributeName is "xsi:type":
   * 		5l:EReference_referenced1100Many_XSITypeTrue ::=
   * 		<xsd:element name="//roleXmlNamePlural//" minOccurs="//lowerMultiplicity//" maxOccurs="1">
   * 	  		<xsd:complexType>
   * 				<xsd:choice minOccurs="//lowerMultiplicity//" maxOccurs="//upperMultiplicity//">
   * 		   			<xsd:element name="//roleXmlName//" type = "xsd:string"/>
   * 				</xsd:choice>
   *   		</xsd:complexType>
   * 		/xsd:element>
   */
  public XSDParticle create_EReference_referenced1100Many_5l(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema, final ArrayList<EClass> referencedClass) {
    String _xMLTypeAttributeName = this.xsdExtendedMetaData.getXMLTypeAttributeName(feature);
    boolean _equals = _xMLTypeAttributeName.equals("xsi:type");
    boolean _not = (!_equals);
    if (_not) {
      return this.create_EReference_referenced1100Many_5l_XSITypeFalse(feature, eClass, xsdSchema, referencedClass);
    }
    return this.create_EReference_referenced1100Many_5l_XSITypeTrue(feature, eClass, xsdSchema, referencedClass);
  }
  
  /**
   * create xsd:element for a EReference with rule 5l.EReference_referenced1100Many of RMF
   * EReference & !containment & upperBound>1 && TTFF (1100) && typeAttributeName != "xsi:type"
   * 
   * 5l:EReference_referenced1100Many_XSITypeFalse ::=
   * 	<xsd:element name="//roleXmlNamePlural//" minOccurs="//lowerMultiplicity//" maxOccurs="1">
   * 	  <xsd:complexType>
   * 		<xsd:choice minOccurs="//lowerMultiplicity//" maxOccurs="//upperMultiplicity//">
   * 		   <xsd:element name="//roleXmlName//">
   * 			</xsd:complexType>
   * 			   <xsd:simpleContent>
   * 				  <xsd:extension base="xsd:string">
   * 				    <xsd:attribute name="typeAttributeName" type="//typeXmlNsPrefix : typeXmlName//--SUBTYPE-ENUM" use="optional"/>
   * 				  </xsd:extension>
   * 			   </xsd:simpleContent>
   * 			</xsd:complexType>
   * 		   </xsd:element>
   * 		</xsd:choice>
   *   </xsd:complexType>
   * 	</xsd:element>
   */
  public XSDParticle create_EReference_referenced1100Many_5l_XSITypeFalse(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema, final ArrayList<EClass> referencedClass) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    final String roleName = this.xsdExtendedMetaData.getXMLWrapperName(feature);
    int lowerBound = feature.getLowerBound();
    if ((lowerBound > 1)) {
      lowerBound = 1;
    }
    xsdParticle.setMinOccurs(lowerBound);
    xsdParticle.setMaxOccurs(1);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    xsdElement.setName(roleName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    int _upperBound = feature.getUpperBound();
    xsdParticle2.setMaxOccurs(_upperBound);
    int _lowerBound = feature.getLowerBound();
    xsdParticle2.setMinOccurs(_lowerBound);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final XSDParticle xsdParticle3 = XSDFactory.eINSTANCE.createXSDParticle();
    final XSDElementDeclaration xsdElement2 = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(feature);
    xsdElement2.setName(_xMLName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition2 = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    xsdComplexTypeDefinition2.setDerivationMethod(XSDDerivationMethod.EXTENSION_LITERAL);
    final XSDSimpleTypeDefinition simpleTypeDefinition = Ecore2XSDFactory.xsdFactory.createXSDSimpleTypeDefinition();
    final XSDTypeDefinition baseTypeDef = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    xsdComplexTypeDefinition2.setBaseTypeDefinition(baseTypeDef);
    xsdComplexTypeDefinition2.setContent(simpleTypeDefinition);
    final XSDAttributeUse simpleAttributeUse = Ecore2XSDFactory.xsdFactory.createXSDAttributeUse();
    final XSDAttributeDeclaration xsdAttribute = Ecore2XSDFactory.xsdFactory.createXSDAttributeDeclaration();
    String _xMLTypeAttributeName = this.xsdExtendedMetaData.getXMLTypeAttributeName(feature);
    xsdAttribute.setName(_xMLTypeAttributeName);
    final EClassifier baseType = feature.getEType();
    referencedClass.add(((EClass) baseType));
    String _elementXSDURI = this.getElementXSDURI(baseType);
    final String uri = (_elementXSDURI + IGeneratorConstants.SUFFIX_SUBTPES_ENUM);
    XSDTypeDefinition xsdTypeDefinition = xsdSchema.resolveTypeDefinitionURI(uri);
    xsdAttribute.setTypeDefinition(((XSDSimpleTypeDefinition) xsdTypeDefinition));
    simpleAttributeUse.setContent(xsdAttribute);
    simpleAttributeUse.setUse(XSDAttributeUseCategory.OPTIONAL_LITERAL);
    EList<XSDAttributeGroupContent> _attributeContents = xsdComplexTypeDefinition2.getAttributeContents();
    _attributeContents.add(simpleAttributeUse);
    xsdElement2.setAnonymousTypeDefinition(xsdComplexTypeDefinition2);
    xsdParticle3.setContent(xsdElement2);
    EList<XSDParticle> _contents = xsdModelGroup2.getContents();
    _contents.add(xsdParticle3);
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
    return xsdParticle;
  }
  
  /**
   * create xsd:element for a EReference with rule 5l.EReference_referenced1100Many of RMF
   * EReference & !containment & upperBound>1 && TTFF (1100) && && typeAttributeName == "xsi:type"
   * 
   * 5l.EReference_referenced1100Many_XSITypeTrue ::=
   * 	5l:EReference_referenced1100Many_XSITypeTrue ::=
   * 		<xsd:element name="//roleXmlNamePlural//" minOccurs="//lowerMultiplicity//" maxOccurs="1">
   * 	  		<xsd:complexType>
   * 				<xsd:choice minOccurs="//lowerMultiplicity//" maxOccurs="//upperMultiplicity//">
   * 		   			<xsd:element name="//roleXmlName//" type = "xsd:string"/>
   * 				</xsd:choice>
   *   		</xsd:complexType>
   * 		/xsd:element>
   */
  public XSDParticle create_EReference_referenced1100Many_5l_XSITypeTrue(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema, final ArrayList<EClass> referencedClass) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    final String roleName = this.xsdExtendedMetaData.getXMLWrapperName(feature);
    int lowerBound = feature.getLowerBound();
    if ((lowerBound > 1)) {
      lowerBound = 1;
    }
    xsdParticle.setMinOccurs(lowerBound);
    xsdParticle.setMaxOccurs(1);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    xsdElement.setName(roleName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    int _upperBound = feature.getUpperBound();
    xsdParticle2.setMaxOccurs(_upperBound);
    int _lowerBound = feature.getLowerBound();
    xsdParticle2.setMinOccurs(_lowerBound);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final XSDParticle xsdParticle3 = XSDFactory.eINSTANCE.createXSDParticle();
    final XSDElementDeclaration xsdElement2 = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(feature);
    xsdElement2.setName(_xMLName);
    final XSDTypeDefinition typeDef = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    xsdElement2.setTypeDefinition(typeDef);
    xsdParticle3.setContent(xsdElement2);
    EList<XSDParticle> _contents = xsdModelGroup2.getContents();
    _contents.add(xsdParticle3);
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
    return xsdParticle;
  }
  
  /**
   * To be overriden by custom
   * 
   * create xsd:element for a EReference with rule 5m.EReference_referenced0100Many of RMF
   * EReference & !containment & upperBound>1 && FTFF (0100)
   * 
   * (1) in case typeAttributeName is not "xsi:type"
   * 		5m:EReference_referenced0100Many_XSITypeFalse ::=
   * 			<xsd:element name="//roleXmlName//" minOccurs="//lowerMultiplicity//" maxOccurs="//upperMultiplicity//">
   *   			<xsd:complexType>
   * 					<xsd:simpleContent>
   * 		  				<xsd:extension base="xsd:string">
   * 							<xsd:attribute name="typeAttributeName" type="//typeXmlNsPrefix : typeXmlName//--SUBTYPES-ENUM" use="optional"/>
   * 	     				 </xsd:extension>
   * 					</xsd:simpleContent>
   *   			</xsd:complexType>
   *  		</xsd:element>
   * (2) in case typeAttributeName is "xsi:type":
   * 		5m:EReference_referenced0100Many_XSITypeTrue ::=
   * 			<xsd:element name="roleXmlName" minOccurs="lowerMultiplicity"
   * 						maxOccurs="upperMultiplicity" type="xsd:string"/>
   */
  public XSDParticle create_EReference_referenced0100Many_5m(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema, final ArrayList<EClass> referencedClass) {
    String _xMLTypeAttributeName = this.xsdExtendedMetaData.getXMLTypeAttributeName(feature);
    boolean _equals = _xMLTypeAttributeName.equals("xsi:type");
    boolean _not = (!_equals);
    if (_not) {
      return this.create_EReference_referenced0100Many_5m_XSITypeFalse(feature, eClass, xsdSchema, referencedClass);
    }
    return this.create_EReference_referenced0100Many_5m_XSITypeTrue(feature, eClass, xsdSchema, referencedClass);
  }
  
  /**
   * create xsd:element for a EReference with rule 5n.EReference_referenced0100Single of RMF
   * EReference & !containment & upperBound=1 && FTFF (0100) && typeAttributeName != "xsi:type"
   * 
   * 5m:EReference_referenced0100Many_XSITypeFalse ::=
   * 	<xsd:element name="//roleXmlName//" minOccurs="//lowerMultiplicity//" maxOccurs="//upperMultiplicity//">
   *   <xsd:complexType>
   * 		<xsd:simpleContent>
   * 		  <xsd:extension base="xsd:string">
   * 			<xsd:attribute name="typeAttributeName" type="//typeXmlNsPrefix : typeXmlName//--SUBTYPES-ENUM" use="optional"/>
   * 	      </xsd:extension>
   * 		</xsd:simpleContent>
   *   </xsd:complexType>
   *  </xsd:element>
   */
  public XSDParticle create_EReference_referenced0100Many_5m_XSITypeFalse(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema, final ArrayList<EClass> referencedClass) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    final String roleName = this.xsdExtendedMetaData.getXMLName(feature);
    int _lowerBound = feature.getLowerBound();
    xsdParticle.setMinOccurs(_lowerBound);
    int _upperBound = feature.getUpperBound();
    xsdParticle.setMaxOccurs(_upperBound);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    xsdElement.setName(roleName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    xsdComplexTypeDefinition.setDerivationMethod(XSDDerivationMethod.EXTENSION_LITERAL);
    final XSDSimpleTypeDefinition simpleTypeDefinition = Ecore2XSDFactory.xsdFactory.createXSDSimpleTypeDefinition();
    final XSDTypeDefinition baseTypeDef = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    xsdComplexTypeDefinition.setBaseTypeDefinition(baseTypeDef);
    xsdComplexTypeDefinition.setContent(simpleTypeDefinition);
    final XSDAttributeUse simpleAttributeUse = Ecore2XSDFactory.xsdFactory.createXSDAttributeUse();
    final XSDAttributeDeclaration xsdAttribute = Ecore2XSDFactory.xsdFactory.createXSDAttributeDeclaration();
    final String typeAttributeName = this.xsdExtendedMetaData.getXMLTypeAttributeName(feature);
    xsdAttribute.setName(typeAttributeName);
    final EClassifier baseType = feature.getEType();
    referencedClass.add(((EClass) baseType));
    String _elementXSDURI = this.getElementXSDURI(baseType);
    final String uri = (_elementXSDURI + IGeneratorConstants.SUFFIX_SUBTPES_ENUM);
    XSDTypeDefinition xsdTypeDefinition = xsdSchema.resolveTypeDefinitionURI(uri);
    xsdAttribute.setTypeDefinition(((XSDSimpleTypeDefinition) xsdTypeDefinition));
    simpleAttributeUse.setContent(xsdAttribute);
    simpleAttributeUse.setUse(XSDAttributeUseCategory.OPTIONAL_LITERAL);
    EList<XSDAttributeGroupContent> _attributeContents = xsdComplexTypeDefinition.getAttributeContents();
    _attributeContents.add(simpleAttributeUse);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
    return xsdParticle;
  }
  
  /**
   * create xsd:element for a EReference with rule 5n.EReference_referenced0100Single of RMF
   * EReference & !containment & upperBound=1 && FTFF (0100) && typeAttributeName == "xsi:type"
   * 
   * 5m:EReference_referenced0100Many_XSITypeTrue ::=
   * 	<xsd:element name="roleXmlName" minOccurs="lowerMultiplicity"
   * 		maxOccurs="upperMultiplicity" type="xsd:string"/>
   */
  public XSDParticle create_EReference_referenced0100Many_5m_XSITypeTrue(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema, final ArrayList<EClass> referencedClass) {
    XSDParticle xsdParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
    final String roleName = this.xsdExtendedMetaData.getXMLName(feature);
    int _lowerBound = feature.getLowerBound();
    xsdParticle.setMinOccurs(_lowerBound);
    int _upperBound = feature.getUpperBound();
    xsdParticle.setMaxOccurs(_upperBound);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    xsdElement.setName(roleName);
    final XSDTypeDefinition typeDef = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    xsdElement.setTypeDefinition(typeDef);
    xsdParticle.setContent(xsdElement);
    return xsdParticle;
  }
  
  /**
   * To be overriden by custom
   * 
   * create xsd:element for a EReference with rule 5n.EReference_referenced0100Single of RMF
   * EReference & !containment & upperBound=1 && FTFF (0100)
   * same as 5m:EReference_referenced0100Many
   * 
   * (1) in case typeAttributeName is not "xsi:type"
   * 			<xsd:element name="//roleXmlName//" minOccurs="//lowerMultiplicity//" maxOccurs="//upperMultiplicity//">
   *   			<xsd:complexType>
   * 					<xsd:simpleContent>
   * 		  				<xsd:extension base="xsd:string">
   * 							<xsd:attribute name="typeAttributeName" type="//typeXmlNsPrefix : typeXmlName//--SUBTYPES-ENUM" use="optional"/>
   * 	     				 </xsd:extension>
   * 					</xsd:simpleContent>
   *   			</xsd:complexType>
   *  		</xsd:element>
   * 
   * (2) in case typeAttributeName is "xsi:type":
   * 			<xsd:element name="roleXmlName" minOccurs="lowerMultiplicity"
   * 						maxOccurs="upperMultiplicity" type="xsd:string"/>
   */
  public XSDParticle create_EReference_referenced0100Single_5n(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema, final ArrayList<EClass> referencedClass) {
    return this.create_EReference_referenced0100Many_5m(feature, eClass, xsdSchema, referencedClass);
  }
  
  /**
   * Create xsd:simpleType for a EEnum with rule 6 of RMF
   * 6. EnumSchema ::=
   * 	<xsd:simpleType name=" // Name of Enumeration Class // ">
   * 		<xsd:restriction base="xsd:string">"
   * 			(<xsd:enumeration value=" // (upper case) Name of Literal from Enumeration Class // '/>)
   * 		</xsd:restriction>
   * 	</xsd:simpleType>"
   */
  public XSDSimpleTypeDefinition createEnumSchema6(final EEnum eEnum, final XSDSchema xsdSchema) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(eEnum, xsdSchema);
    final XSDSimpleTypeDefinition _result;
    synchronized (_createCache_createEnumSchema6) {
      if (_createCache_createEnumSchema6.containsKey(_cacheKey)) {
        return _createCache_createEnumSchema6.get(_cacheKey);
      }
      XSDSimpleTypeDefinition _createXSDSimpleTypeDefinition = Ecore2XSDFactory.xsdFactory.createXSDSimpleTypeDefinition();
      _result = _createXSDSimpleTypeDefinition;
      _createCache_createEnumSchema6.put(_cacheKey, _result);
    }
    _init_createEnumSchema6(_result, eEnum, xsdSchema);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, XSDSimpleTypeDefinition> _createCache_createEnumSchema6 = CollectionLiterals.newHashMap();
  
  private void _init_createEnumSchema6(final XSDSimpleTypeDefinition xsdSimpleTypeDefinition, final EEnum eEnum, final XSDSchema xsdSchema) {
    String _xMLName = this.xsdExtendedMetaData.getXMLName(eEnum);
    xsdSimpleTypeDefinition.setName(_xMLName);
    final XSDTypeDefinition baseType = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    xsdSimpleTypeDefinition.setBaseTypeDefinition(((XSDSimpleTypeDefinition) baseType));
    EList<EEnumLiteral> _eLiterals = eEnum.getELiterals();
    final Consumer<EEnumLiteral> _function = new Consumer<EEnumLiteral>() {
      public void accept(final EEnumLiteral it) {
        final XSDEnumerationFacet xsdEnum = Ecore2XSDFactory.xsdFactory.createXSDEnumerationFacet();
        String _name = it.getName();
        String _upperCase = _name.toUpperCase();
        xsdEnum.setLexicalValue(_upperCase);
        EList<XSDConstrainingFacet> _facetContents = xsdSimpleTypeDefinition.getFacetContents();
        _facetContents.add(xsdEnum);
      }
    };
    _eLiterals.forEach(_function);
    EList<XSDSchemaContent> _contents = xsdSchema.getContents();
    _contents.add(xsdSimpleTypeDefinition);
  }
  
  /**
   * Create xsd:complexType for a EDataType with rule 7a of RMF
   * 7a. DatatypeSchema::=
   * 	<xsd:complexType name=" // Name of Datatype // " >
   * 		<xsd:sequence>
   * 			<xsd:any namespace=" // Namespace URI of Datatype // "
   * 					 processContents=" (skip | lax | strict)"
   * 					 minOccurs=" // Minimum // "
   * 					 maxOccurs=" // Maximum // "/>
   * 			</xsd:any>
   * 		</xsd:sequence>
   * 	</xsd:complexType>
   */
  public XSDComplexTypeDefinition createDataTypeSchema7a(final EDataType eDataType, final XSDSchema xsdSchema) {
    XSDComplexTypeDefinition xsdComplexTypeDefinition = Ecore2XSDFactory.xsdFactory.createXSDComplexTypeDefinition();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(eDataType);
    xsdComplexTypeDefinition.setName(_xMLName);
    final XSDParticle xsdParticle = XSDFactory.eINSTANCE.createXSDParticle();
    final XSDModelGroup xsdModelGroup = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup.setCompositor(XSDCompositor.SEQUENCE_LITERAL);
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    final XSDWildcard xsdAny = XSDFactory.eINSTANCE.createXSDWildcard();
    xsdAny.setStringLexicalNamespaceConstraint("##any");
    xsdAny.setProcessContents(XSDProcessContents.LAX_LITERAL);
    xsdParticle2.setMinOccurs(0);
    xsdParticle2.setMaxOccurs(1);
    xsdParticle2.setContent(xsdAny);
    EList<XSDParticle> _contents = xsdModelGroup.getContents();
    _contents.add(xsdParticle2);
    xsdParticle.setContent(xsdModelGroup);
    xsdComplexTypeDefinition.setContent(xsdParticle);
    EList<XSDSchemaContent> _contents_1 = xsdSchema.getContents();
    _contents_1.add(xsdComplexTypeDefinition);
    return xsdComplexTypeDefinition;
  }
  
  /**
   * Create xsd:simpleType for a EDataType with stereotype "primitive" -- rule 7b of RMF
   * 7b. DatatypeSimpleType::=
   * 	<xsd:simpleType name="//xmlName//">
   * 		<xsd:restriction base="xsd://xmlXsdSimpleType//">
   * 			<xsd:pattern value="//xmlXsdPattern//"/>
   * 		</xsd:restriction>
   * 	</xsd:simpleType>
   */
  public XSDSimpleTypeDefinition createDataTypeSchema7b(final EDataType eDataType, final XSDSchema xsdSchema) {
    XSDSimpleTypeDefinition xsdSimpleTypeDefinition = Ecore2XSDFactory.xsdFactory.createXSDSimpleTypeDefinition();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(eDataType);
    xsdSimpleTypeDefinition.setName(_xMLName);
    final XSDTypeDefinition baseType = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    xsdSimpleTypeDefinition.setBaseTypeDefinition(((XSDSimpleTypeDefinition) baseType));
    final List<String> patternValue = ExtendedMetaData.INSTANCE.getPatternFacet(eDataType);
    boolean _and = false;
    boolean _notEquals = (!Objects.equal(patternValue, null));
    if (!_notEquals) {
      _and = false;
    } else {
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(patternValue);
      boolean _not = (!_isNullOrEmpty);
      _and = _not;
    }
    if (_and) {
      final String pattern = patternValue.get(0);
      final XSDPatternFacet xsdPattern = Ecore2XSDFactory.xsdFactory.createXSDPatternFacet();
      xsdPattern.setLexicalValue(pattern);
      EList<XSDConstrainingFacet> _facetContents = xsdSimpleTypeDefinition.getFacetContents();
      _facetContents.add(xsdPattern);
    }
    EList<XSDSchemaContent> _contents = xsdSchema.getContents();
    _contents.add(xsdSimpleTypeDefinition);
    return xsdSimpleTypeDefinition;
  }
  
  /**
   * Create xsd:simpleType for a EDataType with stereotype "primitive" -- rule 7b of RMF
   * 7b. DatatypeSimpleType::=
   * 	<xsd:simpleType name="//xmlName//--SIMPLE">
   * 		<xsd:restriction base="xsd:string">
   * 			<xsd:pattern value="//xmlXsdPattern//"/>
   * 		</xsd:restriction>
   * 	</xsd:simpleType>
   */
  public XSDSimpleTypeDefinition createDataTypeSimpleType7b(final EDataType eDataType, final XSDSchema xsdSchema) {
    XSDSimpleTypeDefinition xsdSimpleTypeDefinition = Ecore2XSDFactory.xsdFactory.createXSDSimpleTypeDefinition();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(eDataType);
    xsdSimpleTypeDefinition.setName(_xMLName);
    final XSDTypeDefinition baseType = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    xsdSimpleTypeDefinition.setBaseTypeDefinition(((XSDSimpleTypeDefinition) baseType));
    final XSDPatternFacet xsdPattern = Ecore2XSDFactory.xsdFactory.createXSDPatternFacet();
    final List<String> patternValue = ExtendedMetaData.INSTANCE.getPatternFacet(eDataType);
    String _get = patternValue.get(0);
    xsdPattern.setLexicalValue(_get);
    EList<XSDConstrainingFacet> _facetContents = xsdSimpleTypeDefinition.getFacetContents();
    _facetContents.add(xsdPattern);
    EList<XSDSchemaContent> _contents = xsdSchema.getContents();
    _contents.add(xsdSimpleTypeDefinition);
    return xsdSimpleTypeDefinition;
  }
  
  /**
   * Create xsd:simpleType for a EDataType with stereotype "primitive" -- rule 7c of RMF
   * 7c. DatatypeComplexType::=
   * 	<xsd:complexType name="//xmlName//">
   *     <xsd:simpleContent>
   * 		<xsd:extension base="EA: //xmlName//--SIMPLE">
   * 		</xsd:extension>
   *    </xsd:simpleContent>
   * 	</xsd:complexType>
   */
  public XSDComplexTypeDefinition createDataTypeComplexType7c(final EDataType eDataType, final XSDSchema xsdSchema) {
    XSDComplexTypeDefinition xsdComplexTypeDefinition = Ecore2XSDFactory.xsdFactory.createXSDComplexTypeDefinition();
    final String xmlName = this.xsdExtendedMetaData.getXMLName(eDataType);
    xsdComplexTypeDefinition.setName(xmlName);
    xsdComplexTypeDefinition.setDerivationMethod(XSDDerivationMethod.EXTENSION_LITERAL);
    final XSDSimpleTypeDefinition simpleTypeDefinition = Ecore2XSDFactory.xsdFactory.createXSDSimpleTypeDefinition();
    final String typeURI = (((this.defaultUserSchemaNamespace + "#") + xmlName) + IGeneratorConstants.SUFFIX_SIMPLE);
    final XSDTypeDefinition baseTypeDef = xsdSchema.resolveTypeDefinitionURI(typeURI);
    xsdComplexTypeDefinition.setBaseTypeDefinition(baseTypeDef);
    xsdComplexTypeDefinition.setContent(simpleTypeDefinition);
    EList<XSDSchemaContent> _contents = xsdSchema.getContents();
    _contents.add(xsdComplexTypeDefinition);
    return xsdComplexTypeDefinition;
  }
  
  /**
   * Create xsd:simpleType for a referenced EClass with rule 8 of RMF
   * 8. ReferencedSimpleType ::=
   * 	<xsd:simpleType name="//xmlName//--SUBTYPES-ENUM">
   * 	   <xsd:restriction base="xsd:string">
   * 		(<xsd:enumeration value="//(sub) typeXmlName//"/>)+
   *      </xsd:restriction>
   * 	</xsd:simpleType>
   */
  public XSDSimpleTypeDefinition createSimpleType8(final EClass eClass, final XSDSchema xsdSchema) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(eClass, xsdSchema);
    final XSDSimpleTypeDefinition _result;
    synchronized (_createCache_createSimpleType8) {
      if (_createCache_createSimpleType8.containsKey(_cacheKey)) {
        return _createCache_createSimpleType8.get(_cacheKey);
      }
      XSDSimpleTypeDefinition _createXSDSimpleTypeDefinition = Ecore2XSDFactory.xsdFactory.createXSDSimpleTypeDefinition();
      _result = _createXSDSimpleTypeDefinition;
      _createCache_createSimpleType8.put(_cacheKey, _result);
    }
    _init_createSimpleType8(_result, eClass, xsdSchema);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, XSDSimpleTypeDefinition> _createCache_createSimpleType8 = CollectionLiterals.newHashMap();
  
  private void _init_createSimpleType8(final XSDSimpleTypeDefinition xsdSimpleTypeDefinition, final EClass eClass, final XSDSchema xsdSchema) {
    String _xMLName = this.xsdExtendedMetaData.getXMLName(eClass);
    String _plus = (_xMLName + IGeneratorConstants.SUFFIX_SUBTPES_ENUM);
    xsdSimpleTypeDefinition.setName(_plus);
    final XSDTypeDefinition baseType = xsdSchema.resolveTypeDefinitionURI((this.defaultXMLSchemaNamespace + "#string"));
    xsdSimpleTypeDefinition.setBaseTypeDefinition(((XSDSimpleTypeDefinition) baseType));
    final XSDEnumerationFacet xsdEnumFacet = XSDFactory.eINSTANCE.createXSDEnumerationFacet();
    String _xMLName_1 = this.xsdExtendedMetaData.getXMLName(eClass);
    xsdEnumFacet.setLexicalValue(_xMLName_1);
    EList<XSDConstrainingFacet> _facetContents = xsdSimpleTypeDefinition.getFacetContents();
    _facetContents.add(xsdEnumFacet);
    List<EClass> _findESubTypesOf = Ecore2XSDUtil.findESubTypesOf(eClass);
    final Consumer<EClass> _function = new Consumer<EClass>() {
      public void accept(final EClass it) {
        final XSDEnumerationFacet xsdEnum = XSDFactory.eINSTANCE.createXSDEnumerationFacet();
        String _xMLName = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLName(it);
        xsdEnum.setLexicalValue(_xMLName);
        EList<XSDConstrainingFacet> _facetContents = xsdSimpleTypeDefinition.getFacetContents();
        _facetContents.add(xsdEnum);
      }
    };
    _findESubTypesOf.forEach(_function);
    EList<XSDSchemaContent> _contents = xsdSchema.getContents();
    _contents.add(xsdSimpleTypeDefinition);
  }
  
  public XSDParticle generateXSDParticleForFeature(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema) {
    final int xmlPersistenceStrategy = this.xsdExtendedMetaData.getXMLPersistenceMappingStrategy(feature);
    XSDParticle xsdParticle = null;
    if ((XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__1001__FEATURE_WRAPPER_ELEMENT__CLASSIFIER_ELEMENT == xmlPersistenceStrategy)) {
      XSDParticle _createXSDParticleForFeatureCase0010 = this.createXSDParticleForFeatureCase0010(feature, eClass);
      xsdParticle = _createXSDParticleForFeatureCase0010;
      this.patternCaseSet.add("0010");
    } else {
      if ((XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__0101__FEATURE_ELEMENT__CLASSIFIER_ELEMENT == xmlPersistenceStrategy)) {
        XSDParticle _createXSDParticleForFeatureCase0012 = this.createXSDParticleForFeatureCase0012(feature, eClass);
        xsdParticle = _createXSDParticleForFeatureCase0012;
        this.patternCaseSet.add("0012");
      } else {
        if ((XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__0100__FEATURE_ELEMENT == xmlPersistenceStrategy)) {
          XSDParticle _createXSDParticleForFeatureCase0013 = this.createXSDParticleForFeatureCase0013(feature, eClass, xsdSchema);
          xsdParticle = _createXSDParticleForFeatureCase0013;
          this.patternCaseSet.add("0013");
        } else {
          if ((XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__1100__FEATURE_WRAPPER_ELEMENT__FEATURE_ELEMENT == xmlPersistenceStrategy)) {
            XSDParticle _createXSDParticleForFeatureCase0023 = this.createXSDParticleForFeatureCase0023(feature, eClass, xsdSchema);
            xsdParticle = _createXSDParticleForFeatureCase0023;
            this.patternCaseSet.add("0023");
          } else {
            if ((XMLPersistenceMappingExtendedMetaData.XML_PERSISTENCE_MAPPING_STRATEGY__1000__FEATURE_WRAPPER_ELEMENT == xmlPersistenceStrategy)) {
              XSDParticle _createXSDParticleForFeatureCase0010_1 = this.createXSDParticleForFeatureCase0010(feature, eClass);
              xsdParticle = _createXSDParticleForFeatureCase0010_1;
              this.patternCaseSet.add("case not allowed, handled as pattern 0010 (the same way in EAST-ADL)");
            } else {
              XSDParticle _createXSDParticleForFeatureCase0012_1 = this.createXSDParticleForFeatureCase0012(feature, eClass);
              xsdParticle = _createXSDParticleForFeatureCase0012_1;
              this.patternCaseSet.add(("unsupported_mapping_strategy" + Integer.valueOf(xmlPersistenceStrategy)));
              String _name = eClass.getName();
              String _plus = ("\n eClass = " + _name);
              String _plus_1 = (_plus + " feature = ");
              String _name_1 = feature.getName();
              String _plus_2 = (_plus_1 + _name_1);
              System.out.print(_plus_2);
            }
          }
        }
      }
    }
    return xsdParticle;
  }
  
  public XSDParticle createXSDParticleForFeatureCase0010(final EStructuralFeature feature, final EClass eClass) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(feature, eClass);
    final XSDParticle _result;
    synchronized (_createCache_createXSDParticleForFeatureCase0010) {
      if (_createCache_createXSDParticleForFeatureCase0010.containsKey(_cacheKey)) {
        return _createCache_createXSDParticleForFeatureCase0010.get(_cacheKey);
      }
      XSDParticle _createXSDParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
      _result = _createXSDParticle;
      _createCache_createXSDParticleForFeatureCase0010.put(_cacheKey, _result);
    }
    _init_createXSDParticleForFeatureCase0010(_result, feature, eClass);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, XSDParticle> _createCache_createXSDParticleForFeatureCase0010 = CollectionLiterals.newHashMap();
  
  private void _init_createXSDParticleForFeatureCase0010(final XSDParticle xsdParticle, final EStructuralFeature feature, final EClass eClass) {
    final String roleName = this.xsdExtendedMetaData.getXMLWrapperName(feature);
    int _lowerBound = feature.getLowerBound();
    boolean _greaterThan = (_lowerBound > 0);
    if (_greaterThan) {
      xsdParticle.setMinOccurs(1);
    } else {
      xsdParticle.setMinOccurs(0);
    }
    xsdParticle.setMaxOccurs(1);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    xsdElement.setName(roleName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    xsdParticle2.setMaxOccurs(1);
    xsdParticle2.setMinOccurs(0);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final EClassifier typeEClassifier = feature.getEType();
    final XSDParticle xsdParticle3 = XSDFactory.eINSTANCE.createXSDParticle();
    final XSDElementDeclaration xsdElement2 = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _name = this.xsdExtendedMetaData.getName(typeEClassifier);
    xsdElement2.setName(_name);
    final String typeeClassifierURI = this.getElementXSDURI(typeEClassifier);
    final XSDTypeDefinition xsdTypeDefinition = xsdElement2.resolveTypeDefinitionURI(typeeClassifierURI);
    xsdElement2.setTypeDefinition(xsdTypeDefinition);
    xsdParticle3.setContent(xsdElement2);
    EList<XSDParticle> _contents = xsdModelGroup2.getContents();
    _contents.add(xsdParticle3);
    if ((typeEClassifier instanceof EClass)) {
      List<EClass> _findESubTypesOf = Ecore2XSDUtil.findESubTypesOf(((EClass)typeEClassifier));
      final Consumer<EClass> _function = new Consumer<EClass>() {
        public void accept(final EClass it) {
          final XSDParticle xsdParticleSubType = XSDFactory.eINSTANCE.createXSDParticle();
          final XSDElementDeclaration xsdElementSubType = XSDFactory.eINSTANCE.createXSDElementDeclaration();
          String _xMLName = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLName(it);
          xsdElementSubType.setName(_xMLName);
          final String subTypeURI = Ecore2XSDFactory.this.getElementXSDURI(it);
          final XSDTypeDefinition xsdTypeDefinitionSubType = xsdElementSubType.resolveTypeDefinitionURI(subTypeURI);
          xsdElementSubType.setTypeDefinition(xsdTypeDefinitionSubType);
          xsdParticleSubType.setContent(xsdElementSubType);
          EList<XSDParticle> _contents = xsdModelGroup2.getContents();
          _contents.add(xsdParticleSubType);
        }
      };
      _findESubTypesOf.forEach(_function);
    }
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
  }
  
  public XSDParticle createXSDParticleForFeatureCase0012(final EStructuralFeature feature, final EClass eClass) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(feature, eClass);
    final XSDParticle _result;
    synchronized (_createCache_createXSDParticleForFeatureCase0012) {
      if (_createCache_createXSDParticleForFeatureCase0012.containsKey(_cacheKey)) {
        return _createCache_createXSDParticleForFeatureCase0012.get(_cacheKey);
      }
      XSDParticle _createXSDParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
      _result = _createXSDParticle;
      _createCache_createXSDParticleForFeatureCase0012.put(_cacheKey, _result);
    }
    _init_createXSDParticleForFeatureCase0012(_result, feature, eClass);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, XSDParticle> _createCache_createXSDParticleForFeatureCase0012 = CollectionLiterals.newHashMap();
  
  private void _init_createXSDParticleForFeatureCase0012(final XSDParticle xsdParticle, final EStructuralFeature feature, final EClass eClass) {
    final String roleName = this.xsdExtendedMetaData.getXMLName(feature);
    xsdParticle.setMinOccurs(0);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    xsdElement.setName(roleName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    xsdParticle2.setMaxOccurs(1);
    xsdParticle2.setMinOccurs(0);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final EClassifier typeEClassifier = feature.getEType();
    final XSDParticle xsdParticle3 = XSDFactory.eINSTANCE.createXSDParticle();
    final XSDElementDeclaration xsdElement2 = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    String _xMLName = this.xsdExtendedMetaData.getXMLName(typeEClassifier);
    xsdElement2.setName(_xMLName);
    final String typeeClassifierURI = this.getElementXSDURI(typeEClassifier);
    final XSDTypeDefinition xsdTypeDefinition = xsdElement2.resolveTypeDefinitionURI(typeeClassifierURI);
    xsdElement2.setTypeDefinition(xsdTypeDefinition);
    xsdParticle3.setContent(xsdElement2);
    EList<XSDParticle> _contents = xsdModelGroup2.getContents();
    _contents.add(xsdParticle3);
    if ((typeEClassifier instanceof EClass)) {
      List<EClass> _findESubTypesOf = Ecore2XSDUtil.findESubTypesOf(((EClass)typeEClassifier));
      final Consumer<EClass> _function = new Consumer<EClass>() {
        public void accept(final EClass it) {
          final XSDParticle xsdParticleSubType = XSDFactory.eINSTANCE.createXSDParticle();
          final XSDElementDeclaration xsdElementSubType = XSDFactory.eINSTANCE.createXSDElementDeclaration();
          String _name = Ecore2XSDFactory.this.xsdExtendedMetaData.getName(it);
          xsdElementSubType.setName(_name);
          final String subTypeURI = Ecore2XSDFactory.this.getElementXSDURI(it);
          final XSDTypeDefinition xsdTypeDefinitionSubType = xsdElementSubType.resolveTypeDefinitionURI(subTypeURI);
          xsdElementSubType.setTypeDefinition(xsdTypeDefinitionSubType);
          xsdParticleSubType.setContent(xsdElementSubType);
          EList<XSDParticle> _contents = xsdModelGroup2.getContents();
          _contents.add(xsdParticleSubType);
        }
      };
      _findESubTypesOf.forEach(_function);
    }
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
  }
  
  public XSDParticle createXSDParticleForFeatureCase0013(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(feature, eClass, xsdSchema);
    final XSDParticle _result;
    synchronized (_createCache_createXSDParticleForFeatureCase0013) {
      if (_createCache_createXSDParticleForFeatureCase0013.containsKey(_cacheKey)) {
        return _createCache_createXSDParticleForFeatureCase0013.get(_cacheKey);
      }
      XSDParticle _createXSDParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
      _result = _createXSDParticle;
      _createCache_createXSDParticleForFeatureCase0013.put(_cacheKey, _result);
    }
    _init_createXSDParticleForFeatureCase0013(_result, feature, eClass, xsdSchema);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, XSDParticle> _createCache_createXSDParticleForFeatureCase0013 = CollectionLiterals.newHashMap();
  
  private void _init_createXSDParticleForFeatureCase0013(final XSDParticle xsdParticle, final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema) {
    final String roleName = this.xsdExtendedMetaData.getXMLName(feature);
    xsdParticle.setMinOccurs(0);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    xsdElement.setName(roleName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    xsdParticle2.setMaxOccurs(1);
    xsdParticle2.setMinOccurs(0);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final EClassifier typeEClassifier = feature.getEType();
    if ((typeEClassifier instanceof EClass)) {
      final XSDParticle xsdParticle3 = XSDFactory.eINSTANCE.createXSDParticle();
      final XSDModelGroupDefinition xsdModelGroupdefinitionRef = XSDFactory.eINSTANCE.createXSDModelGroupDefinition();
      final String refGroupURI = this.getElementXSDURI(typeEClassifier);
      final XSDModelGroupDefinition xsdModelGroupDef = xsdModelGroupdefinitionRef.resolveModelGroupDefinitionURI(refGroupURI);
      xsdModelGroupdefinitionRef.setResolvedModelGroupDefinition(xsdModelGroupDef);
      xsdParticle3.setContent(xsdModelGroupdefinitionRef);
      xsdParticle3.setContent(xsdModelGroupdefinitionRef);
      EList<XSDParticle> _contents = xsdModelGroup2.getContents();
      _contents.add(xsdParticle3);
      List<EClass> _findESubTypesOf = Ecore2XSDUtil.findESubTypesOf(((EClass)typeEClassifier));
      final Consumer<EClass> _function = new Consumer<EClass>() {
        public void accept(final EClass it) {
          final XSDParticle xsdParticleSubType = XSDFactory.eINSTANCE.createXSDParticle();
          final XSDModelGroupDefinition xsdModelGroupdefSubTypeRef = XSDFactory.eINSTANCE.createXSDModelGroupDefinition();
          final String subTypeURI = Ecore2XSDFactory.this.getElementXSDURI(it);
          final XSDModelGroupDefinition subTypeModelGroupDef = xsdModelGroupdefSubTypeRef.resolveModelGroupDefinitionURI(subTypeURI);
          xsdModelGroupdefSubTypeRef.setResolvedModelGroupDefinition(subTypeModelGroupDef);
          xsdParticleSubType.setContent(xsdModelGroupdefSubTypeRef);
          EList<XSDParticle> _contents = xsdModelGroup2.getContents();
          _contents.add(xsdParticleSubType);
        }
      };
      _findESubTypesOf.forEach(_function);
    }
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
  }
  
  public XSDParticle createXSDParticleForFeatureCase0023(final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(feature, eClass, xsdSchema);
    final XSDParticle _result;
    synchronized (_createCache_createXSDParticleForFeatureCase0023) {
      if (_createCache_createXSDParticleForFeatureCase0023.containsKey(_cacheKey)) {
        return _createCache_createXSDParticleForFeatureCase0023.get(_cacheKey);
      }
      XSDParticle _createXSDParticle = Ecore2XSDFactory.xsdFactory.createXSDParticle();
      _result = _createXSDParticle;
      _createCache_createXSDParticleForFeatureCase0023.put(_cacheKey, _result);
    }
    _init_createXSDParticleForFeatureCase0023(_result, feature, eClass, xsdSchema);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, XSDParticle> _createCache_createXSDParticleForFeatureCase0023 = CollectionLiterals.newHashMap();
  
  private void _init_createXSDParticleForFeatureCase0023(final XSDParticle xsdParticle, final EStructuralFeature feature, final EClass eClass, final XSDSchema xsdSchema) {
    final String roleName = this.xsdExtendedMetaData.getXMLWrapperName(feature);
    xsdParticle.setMinOccurs(0);
    final XSDElementDeclaration xsdElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
    xsdElement.setName(roleName);
    final XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
    final XSDParticle xsdParticle2 = XSDFactory.eINSTANCE.createXSDParticle();
    xsdParticle2.setMaxOccurs(1);
    xsdParticle2.setMinOccurs(0);
    final XSDModelGroup xsdModelGroup2 = XSDFactory.eINSTANCE.createXSDModelGroup();
    xsdModelGroup2.setCompositor(XSDCompositor.CHOICE_LITERAL);
    final EClassifier typeEClassifier = feature.getEType();
    if ((typeEClassifier instanceof EClass)) {
      final XSDParticle xsdParticle3 = XSDFactory.eINSTANCE.createXSDParticle();
      final XSDElementDeclaration xsdElement2 = XSDFactory.eINSTANCE.createXSDElementDeclaration();
      String _xMLWrapperName = this.xsdExtendedMetaData.getXMLWrapperName(typeEClassifier);
      xsdElement2.setName(_xMLWrapperName);
      final XSDComplexTypeDefinition xsdComplexTypeDefinition2 = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
      final XSDParticle xsdParticle4 = XSDFactory.eINSTANCE.createXSDParticle();
      xsdParticle4.setMaxOccurs(1);
      xsdParticle4.setMinOccurs(0);
      final XSDModelGroup xsdModelGroup3 = XSDFactory.eINSTANCE.createXSDModelGroup();
      xsdModelGroup3.setCompositor(XSDCompositor.CHOICE_LITERAL);
      final XSDParticle xsdParticle5 = XSDFactory.eINSTANCE.createXSDParticle();
      final XSDModelGroupDefinition xsdModelGroupdefinitionRef = XSDFactory.eINSTANCE.createXSDModelGroupDefinition();
      final String typeURI = this.getElementXSDURI(typeEClassifier);
      final XSDModelGroupDefinition typeModelGroupDef = xsdModelGroupdefinitionRef.resolveModelGroupDefinitionURI(typeURI);
      xsdModelGroupdefinitionRef.setResolvedModelGroupDefinition(typeModelGroupDef);
      xsdParticle5.setContent(xsdModelGroupdefinitionRef);
      List<EClass> _findESubTypesOf = Ecore2XSDUtil.findESubTypesOf(((EClass)typeEClassifier));
      final Consumer<EClass> _function = new Consumer<EClass>() {
        public void accept(final EClass it) {
          final XSDParticle subTypeXSDParticle = XSDFactory.eINSTANCE.createXSDParticle();
          final XSDElementDeclaration subTypeXSDElement = XSDFactory.eINSTANCE.createXSDElementDeclaration();
          String _xMLWrapperName = Ecore2XSDFactory.this.xsdExtendedMetaData.getXMLWrapperName(it);
          subTypeXSDElement.setName(_xMLWrapperName);
          final XSDComplexTypeDefinition subTypeXSDComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
          final XSDParticle xsdParticle6 = XSDFactory.eINSTANCE.createXSDParticle();
          xsdParticle6.setMaxOccurs(1);
          xsdParticle6.setMinOccurs(0);
          final XSDModelGroup xsdModelGroup4 = XSDFactory.eINSTANCE.createXSDModelGroup();
          xsdModelGroup4.setCompositor(XSDCompositor.CHOICE_LITERAL);
          final XSDParticle xsdParticle7 = XSDFactory.eINSTANCE.createXSDParticle();
          final XSDModelGroupDefinition xsdModelGroupdefinitionRef2 = XSDFactory.eINSTANCE.createXSDModelGroupDefinition();
          final String subTypeURI = Ecore2XSDFactory.this.getElementXSDURI(it);
          final XSDModelGroupDefinition subTypeModelGroupDef = xsdModelGroupdefinitionRef2.resolveModelGroupDefinitionURI(subTypeURI);
          xsdModelGroupdefinitionRef2.setResolvedModelGroupDefinition(subTypeModelGroupDef);
          xsdParticle7.setContent(xsdModelGroupdefinitionRef2);
          EList<XSDParticle> _contents = xsdModelGroup4.getContents();
          _contents.add(xsdParticle7);
          xsdParticle6.setContent(xsdModelGroup4);
          subTypeXSDComplexTypeDefinition.setContent(xsdParticle6);
          subTypeXSDElement.setAnonymousTypeDefinition(subTypeXSDComplexTypeDefinition);
          subTypeXSDParticle.setContent(subTypeXSDElement);
          EList<XSDParticle> _contents_1 = xsdModelGroup2.getContents();
          _contents_1.add(subTypeXSDParticle);
        }
      };
      _findESubTypesOf.forEach(_function);
      EList<XSDParticle> _contents = xsdModelGroup3.getContents();
      _contents.add(xsdParticle5);
      xsdParticle4.setContent(xsdModelGroup3);
      xsdComplexTypeDefinition2.setContent(xsdParticle4);
      xsdElement2.setAnonymousTypeDefinition(xsdComplexTypeDefinition2);
      xsdParticle3.setContent(xsdElement2);
      EList<XSDParticle> _contents_1 = xsdModelGroup2.getContents();
      _contents_1.add(xsdParticle3);
    }
    xsdParticle2.setContent(xsdModelGroup2);
    xsdComplexTypeDefinition.setContent(xsdParticle2);
    xsdElement.setAnonymousTypeDefinition(xsdComplexTypeDefinition);
    xsdParticle.setContent(xsdElement);
  }
  
  /**
   * to be override
   */
  public Boolean isXMLPrimitiveXsdType(final EDataType dataType) {
    final Boolean isPrimitive = Boolean.valueOf(true);
    String _xMLXsdSimpleType = this.xsdExtendedMetaData.getXMLXsdSimpleType(dataType);
    final Boolean isXsdTypeDefined = Boolean.valueOf((!Objects.equal(_xMLXsdSimpleType, IGeneratorConstants.UNINITIALIZED_STRING)));
    String _xMLCustomSimpleType = this.xsdExtendedMetaData.getXMLCustomSimpleType(dataType);
    final Boolean isCustomSimpleTypeDefined = Boolean.valueOf(IGeneratorConstants.BOOLEAN_TRUE.equals(_xMLCustomSimpleType));
    Boolean isPatternDefined = Boolean.valueOf(false);
    final List<String> patterns = ExtendedMetaData.INSTANCE.getPatternFacet(dataType);
    boolean _isEmpty = patterns.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      String _get = patterns.get(0);
      boolean _notEquals = (!Objects.equal(_get, null));
      isPatternDefined = Boolean.valueOf(_notEquals);
    }
    if (((((isPrimitive).booleanValue() && (isXsdTypeDefined).booleanValue()) && (!(isCustomSimpleTypeDefined).booleanValue())) && (!(isPatternDefined).booleanValue()))) {
      return Boolean.valueOf(true);
    }
    return Boolean.valueOf(false);
  }
  
  protected String getXsdSimpleType(final EClassifier typeeClassifier) {
    String xsdSimpleType = this.xsdExtendedMetaData.getXMLXsdSimpleType(typeeClassifier);
    boolean _equals = Objects.equal(xsdSimpleType, null);
    if (_equals) {
      final Class<?> instanceClass = typeeClassifier.getInstanceClass();
      String _get = JavaXSDPrimitiveTypeMapping.javaXsdPrimitiveTypeMapping.get(instanceClass);
      xsdSimpleType = _get;
    }
    return xsdSimpleType;
  }
}
