/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.server.json;

import static com.google.common.base.Charsets.UTF_8;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Throwables.propagate;
import static java.util.Objects.requireNonNull;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.lang.reflect.Type;
import java.util.List;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.eclipse.emf.common.util.EMap;
import org.eclipse.epp.logging.aeri.core.IReport;
import org.eclipse.jdt.annotation.Nullable;

import com.google.common.collect.Lists;
import com.google.common.io.Closeables;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.TypeAdapter;
import com.google.gson.reflect.TypeToken;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

public class Json {
    public static final Type T_LIST_STRING = new TypeToken<List<String>>() {
    }.getType();

    @Nullable
    private static Gson gson;

    static synchronized Gson getInstance() {
        if (gson == null) {
            final GsonBuilder builder = new GsonBuilder();

            builder.registerTypeAdapter(UUID.class, new UuidTypeAdapter());

            builder.enableComplexMapKeySerialization();
            builder.setPrettyPrinting();
            gson = builder.create();
        }
        return requireNonNull(gson);
    }

    public static String toJson(IReport report, boolean pretty) {
        checkNotNull(report);
        Gson gson = createGson(pretty);
        String json = gson.toJson(report);
        return json;
    }

    private static Gson createGson(boolean pretty) {
        GsonBuilder builder = new GsonBuilder().setDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS"); //$NON-NLS-1$
        builder.registerTypeAdapter(UUID.class, new UuidTypeAdapter());
        builder.registerTypeHierarchyAdapter(EMap.class, new EMapTypeAdapter());
        builder.addSerializationExclusionStrategy(new EmfFieldExclusionStrategy());
        if (pretty) {
            builder.setPrettyPrinting();
        }
        Gson gson = builder.create();
        return gson;
    }

    public static <T> T deserialize(final CharSequence json, final Type classOfT) {
        return deserialize(json.toString(), classOfT);
    }

    public static <T> T deserialize(final String json, final Type classOfT) {
        try {
            T res = getInstance().fromJson(json, classOfT);
            return requireNonNull(res);
        } catch (final Exception e) {
            throw propagate(e);
        }
    }

    public static <T> T deserialize(final InputStream jsonStream, final Type classOfT) {
        try (Reader reader = new InputStreamReader(jsonStream, UTF_8)) {
            T res = getInstance().fromJson(reader, classOfT);
            return requireNonNull(res);
        } catch (final Exception e) {
            throw propagate(e);
        }
    }

    public static <T> T deserialize(final File jsonFile, final Type classOfT) {
        try (InputStream in = new BufferedInputStream(new FileInputStream(jsonFile))) {
            return deserialize(in, classOfT);
        } catch (final Exception e) {
            throw propagate(e);
        }
    }

    public static String serialize(final Object obj) {
        final StringBuilder sb = new StringBuilder();
        serialize(obj, sb);
        return sb.toString();
    }

    public static void serialize(final Object obj, final Appendable writer) {
        try {
            getInstance().toJson(obj, writer);
        } catch (final Exception e) {
            throw propagate(e);
        }
    }

    public static void serialize(final Object obj, final File jsonFile) {
        try (OutputStream out = new BufferedOutputStream(new FileOutputStream(jsonFile))) {
            serialize(obj, out);
        } catch (final Exception e) {
            throw propagate(e);
        }
    }

    public static void serialize(final Object obj, final OutputStream out) {
        try (Writer writer = new OutputStreamWriter(out, UTF_8)) {
            getInstance().toJson(obj, writer);
        } catch (final Exception e) {
            throw propagate(e);
        }
    }

    public static <T> List<T> deserializeZip(File zip, Class<T> classOfT) throws IOException {
        List<T> res = Lists.newLinkedList();
        ZipInputStream zis = null;
        try {
            FileInputStream fis = new FileInputStream(zip);
            zis = new ZipInputStream(fis);
            ZipEntry entry;
            while ((entry = zis.getNextEntry()) != null) {
                if (!entry.isDirectory()) {
                    res.add(Json.<T>deserialize(zis, classOfT));
                }
            }
        } finally {
            Closeables.close(zis, true);
        }
        return res;
    }

    private static final class EMapTypeAdapter extends TypeAdapter<EMap> {
        @Override
        public void write(JsonWriter out, EMap map) throws IOException {
            if (map == null) {
                out.nullValue();
                return;
            }
            out.beginObject();
            for (Object key : map.keySet()) {
                out.name(key.toString());
                Object value = map.get(key);
                if (value == null) {
                    out.nullValue();
                } else {
                    out.value(value.toString());
                }
            }
            out.endObject();
        }

        @Override
        public EMap read(JsonReader in) throws IOException {
            throw new UnsupportedOperationException();
        }
    }
}
