/**
 * Copyright (c) 2014-2017 EclipseSource Muenchen GmbH and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Stefan Dirix - initial API and implementation
 */
package org.eclipse.emf.ecp.emf2web.json.generator.xtend;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import java.util.Arrays;
import java.util.Collection;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecp.emf2web.util.ReferenceHelper;
import org.eclipse.emf.ecp.view.spi.categorization.model.VCategorization;
import org.eclipse.emf.ecp.view.spi.categorization.model.VCategorizationElement;
import org.eclipse.emf.ecp.view.spi.categorization.model.VCategory;
import org.eclipse.emf.ecp.view.spi.label.model.VLabel;
import org.eclipse.emf.ecp.view.spi.model.VContainedElement;
import org.eclipse.emf.ecp.view.spi.model.VContainer;
import org.eclipse.emf.ecp.view.spi.model.VControl;
import org.eclipse.emf.ecp.view.spi.model.VElement;
import org.eclipse.emf.ecp.view.spi.model.VView;

/**
 * The class which handles the conversion from ecore files to qbForm files.
 */
@SuppressWarnings("all")
public class FormsJsonGenerator extends JsonGenerator {
  private static final String TYPE = "type";
  
  private static final String ELEMENTS = "elements";
  
  private static final String CONTROL = "Control";
  
  private static final String CATEGORIZATION = "Categorization";
  
  private static final String CATEGORY = "Category";
  
  private static final String SCOPE = "scope";
  
  private static final String LABEL = "label";
  
  private ReferenceHelper refHelper;
  
  public FormsJsonGenerator(final ReferenceHelper refHelper) {
    this.refHelper = refHelper;
  }
  
  @Override
  public JsonElement createJsonElement(final EObject object) {
    return this.createJsonFormsElement(object);
  }
  
  private JsonElement _createJsonFormsElement(final VLabel label) {
    JsonObject _xblockexpression = null;
    {
      final JsonObject jsonObj = new JsonObject();
      this.withType(jsonObj, "Label");
      _xblockexpression = this.with(jsonObj, "text", label.getName());
    }
    return _xblockexpression;
  }
  
  private JsonElement _createJsonFormsElement(final VCategorizationElement categorizationElement) {
    return this.withElements(this.withType(new JsonObject(), FormsJsonGenerator.CATEGORIZATION), categorizationElement.getCategorizations());
  }
  
  private JsonElement _createJsonFormsElement(final VCategorization categorization) {
    return this.withElements(this.withLabel(this.withType(new JsonObject(), FormsJsonGenerator.CATEGORIZATION), categorization.getLabel()), categorization.getCategorizations());
  }
  
  private JsonElement _createJsonFormsElement(final VCategory category) {
    JsonObject _xblockexpression = null;
    {
      final JsonObject jsonObj = new JsonObject();
      this.withType(jsonObj, FormsJsonGenerator.CATEGORY);
      this.withLabel(jsonObj, category.getLabel());
      final VContainedElement contained = category.getComposite();
      JsonObject _switchResult = null;
      boolean _matched = false;
      if (contained instanceof VContainer) {
        _matched=true;
        _switchResult = this.withElements(jsonObj, ((VContainer)contained).getChildren());
      }
      if (!_matched) {
        throw new UnsupportedOperationException("Category must contain a VContainer element");
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
  
  private JsonElement _createJsonFormsElement(final VView view) {
    JsonElement _switchResult = null;
    int _size = view.getChildren().size();
    switch (_size) {
      case 0:
        _switchResult = new JsonObject();
        break;
      case 1:
        _switchResult = this.createJsonFormsElement(view.getChildren().get(0));
        break;
      default:
        _switchResult = this.createJsonFormsElement(view.getChildren());
        break;
    }
    return _switchResult;
  }
  
  private JsonElement _createJsonFormsElement(final VControl control) {
    JsonObject _xblockexpression = null;
    {
      final JsonObject jsonObject = new JsonObject();
      this.withType(jsonObject, FormsJsonGenerator.CONTROL);
      this.withLabel(jsonObject, this.getDisplayLabel(control));
      _xblockexpression = this.withScope(jsonObject, this.getRef(control));
    }
    return _xblockexpression;
  }
  
  private JsonElement _createJsonFormsElement(final VContainer container) {
    JsonObject _xblockexpression = null;
    {
      final JsonObject jsonObject = new JsonObject();
      this.withType(jsonObject, this.getType(container));
      String _name = container.getName();
      boolean _tripleNotEquals = (_name != null);
      if (_tripleNotEquals) {
        this.withLabel(jsonObject, this.getDisplayLabel(container.getLabel()));
      }
      _xblockexpression = this.withElements(jsonObject, container.getChildren());
    }
    return _xblockexpression;
  }
  
  private JsonElement _createJsonFormsElement(final Collection<? extends VElement> elements) {
    JsonObject _xblockexpression = null;
    {
      final JsonObject jsonObject = new JsonObject();
      _xblockexpression = this.withVerticalLayout(jsonObject, elements);
    }
    return _xblockexpression;
  }
  
  private JsonElement _createJsonFormsElement(final EObject object) {
    return new JsonObject();
  }
  
  private JsonElement _createJsonFormsElement(final Void x) {
    return new JsonObject();
  }
  
  private JsonObject withType(final JsonObject jsonObject, final String type) {
    return this.with(jsonObject, FormsJsonGenerator.TYPE, type);
  }
  
  private JsonObject withScope(final JsonObject jsonObject, final String ref) {
    return this.with(jsonObject, FormsJsonGenerator.SCOPE, ref);
  }
  
  private JsonObject withLabel(final JsonObject jsonObject, final String label) {
    return this.with(jsonObject, FormsJsonGenerator.LABEL, label);
  }
  
  private JsonObject withElements(final JsonObject jsonObject, final Collection<? extends VElement> elements) {
    return this.with(jsonObject, FormsJsonGenerator.ELEMENTS, elements);
  }
  
  private JsonObject withVerticalLayout(final JsonObject jsonObject, final Collection<? extends VElement> elements) {
    JsonObject _xblockexpression = null;
    {
      this.withType(jsonObject, "VerticalLayout");
      _xblockexpression = this.withElements(jsonObject, elements);
    }
    return _xblockexpression;
  }
  
  private String getType(final VElement vElement) {
    return vElement.eClass().getName();
  }
  
  private String getRef(final VControl control) {
    return this.refHelper.getStringRepresentation(control.getDomainModelReference());
  }
  
  private String getDisplayLabel(final VControl control) {
    return this.refHelper.getLabel(control.getDomainModelReference());
  }
  
  private String getDisplayLabel(final String containerLabel) {
    String _xblockexpression = null;
    {
      boolean _startsWith = containerLabel.trim().startsWith("%");
      if (_startsWith) {
        String label = containerLabel.trim().substring(1);
        final String[] split = label.split("(?<!(^|[A-Z]))(?=[A-Z])|(?<!^)(?=[A-Z][a-z])");
        int _length = (split[0]).length();
        boolean _greaterThan = (_length > 0);
        if (_greaterThan) {
          final char[] charArray = (split[0]).toCharArray();
          charArray[0] = Character.toUpperCase(charArray[0]);
          String _string = new String(charArray);
          split[0] = _string;
        }
        final StringBuilder sb = new StringBuilder();
        for (final String str : split) {
          {
            sb.append(str);
            sb.append(" ");
          }
        }
        return sb.toString().trim();
      }
      _xblockexpression = containerLabel;
    }
    return _xblockexpression;
  }
  
  private JsonElement createJsonFormsElement(final Object categorization) {
    if (categorization instanceof VCategorization) {
      return _createJsonFormsElement((VCategorization)categorization);
    } else if (categorization instanceof VCategory) {
      return _createJsonFormsElement((VCategory)categorization);
    } else if (categorization instanceof VCategorizationElement) {
      return _createJsonFormsElement((VCategorizationElement)categorization);
    } else if (categorization instanceof VLabel) {
      return _createJsonFormsElement((VLabel)categorization);
    } else if (categorization instanceof VControl) {
      return _createJsonFormsElement((VControl)categorization);
    } else if (categorization instanceof VContainer) {
      return _createJsonFormsElement((VContainer)categorization);
    } else if (categorization instanceof VView) {
      return _createJsonFormsElement((VView)categorization);
    } else if (categorization instanceof Collection) {
      return _createJsonFormsElement((Collection<? extends VElement>)categorization);
    } else if (categorization instanceof EObject) {
      return _createJsonFormsElement((EObject)categorization);
    } else if (categorization == null) {
      return _createJsonFormsElement((Void)null);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(categorization).toString());
    }
  }
}
