package org.mandas.docker.client.messages;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link RemovedImage}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableRemovedImage.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableRemovedImage implements RemovedImage {
  private final RemovedImage.Type type;
  private final @Nullable String imageId;

  private ImmutableRemovedImage(
      RemovedImage.Type type,
      @Nullable String imageId) {
    this.type = type;
    this.imageId = imageId;
  }

  /**
   * @return The value of the {@code type} attribute
   */
  @Override
  public RemovedImage.Type type() {
    return type;
  }

  /**
   * @return The value of the {@code imageId} attribute
   */
  @Override
  public @Nullable String imageId() {
    return imageId;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link RemovedImage#type() type} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for type
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableRemovedImage withType(RemovedImage.Type value) {
    if (this.type == value) return this;
    RemovedImage.Type newValue = Objects.requireNonNull(value, "type");
    if (this.type.equals(newValue)) return this;
    return new ImmutableRemovedImage(newValue, this.imageId);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link RemovedImage#imageId() imageId} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for imageId (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableRemovedImage withImageId(@Nullable String value) {
    if (Objects.equals(this.imageId, value)) return this;
    return new ImmutableRemovedImage(this.type, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableRemovedImage} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableRemovedImage
        && equalTo((ImmutableRemovedImage) another);
  }

  private boolean equalTo(ImmutableRemovedImage another) {
    return type.equals(another.type)
        && Objects.equals(imageId, another.imageId);
  }

  /**
   * Computes a hash code from attributes: {@code type}, {@code imageId}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + type.hashCode();
    h += (h << 5) + Objects.hashCode(imageId);
    return h;
  }

  /**
   * Prints the immutable value {@code RemovedImage} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "RemovedImage{"
        + "type=" + type
        + ", imageId=" + imageId
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link RemovedImage} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable RemovedImage instance
   */
  public static ImmutableRemovedImage copyOf(RemovedImage instance) {
    if (instance instanceof ImmutableRemovedImage) {
      return (ImmutableRemovedImage) instance;
    }
    return ImmutableRemovedImage.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableRemovedImage ImmutableRemovedImage}.
   * <pre>
   * ImmutableRemovedImage.builder()
   *    .type(org.mandas.docker.client.messages.RemovedImage.Type) // required {@link RemovedImage#type() type}
   *    .imageId(String | null) // nullable {@link RemovedImage#imageId() imageId}
   *    .build();
   * </pre>
   * @return A new ImmutableRemovedImage builder
   */
  public static ImmutableRemovedImage.Builder builder() {
    return new ImmutableRemovedImage.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableRemovedImage ImmutableRemovedImage}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_TYPE = 0x1L;
    private long initBits = 0x1L;

    private RemovedImage.Type type;
    private String imageId;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code RemovedImage} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(RemovedImage instance) {
      Objects.requireNonNull(instance, "instance");
      type(instance.type());
      @Nullable String imageIdValue = instance.imageId();
      if (imageIdValue != null) {
        imageId(imageIdValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link RemovedImage#type() type} attribute.
     * @param type The value for type 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder type(RemovedImage.Type type) {
      this.type = Objects.requireNonNull(type, "type");
      initBits &= ~INIT_BIT_TYPE;
      return this;
    }

    /**
     * Initializes the value for the {@link RemovedImage#imageId() imageId} attribute.
     * @param imageId The value for imageId (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder imageId(@Nullable String imageId) {
      this.imageId = imageId;
      return this;
    }

    /**
     * Builds a new {@link ImmutableRemovedImage ImmutableRemovedImage}.
     * @return An immutable instance of RemovedImage
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableRemovedImage build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableRemovedImage(type, imageId);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_TYPE) != 0) attributes.add("type");
      return "Cannot build RemovedImage, some of required attributes are not set " + attributes;
    }
  }
}
