package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Image}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableImage.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableImage implements Image {
  private final String created;
  private final String id;
  private final String parentId;
  private final @Nullable List<String> repoTags;
  private final @Nullable List<String> repoDigests;
  private final Long size;
  private final Long virtualSize;
  private final @Nullable Map<String, String> labels;

  private ImmutableImage(
      String created,
      String id,
      String parentId,
      @Nullable List<String> repoTags,
      @Nullable List<String> repoDigests,
      Long size,
      Long virtualSize,
      @Nullable Map<String, String> labels) {
    this.created = created;
    this.id = id;
    this.parentId = parentId;
    this.repoTags = repoTags;
    this.repoDigests = repoDigests;
    this.size = size;
    this.virtualSize = virtualSize;
    this.labels = labels;
  }

  /**
   * @return The value of the {@code created} attribute
   */
  @JsonProperty("Created")
  @Override
  public String created() {
    return created;
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("Id")
  @Override
  public String id() {
    return id;
  }

  /**
   * @return The value of the {@code parentId} attribute
   */
  @JsonProperty("ParentId")
  @Override
  public String parentId() {
    return parentId;
  }

  /**
   * @return The value of the {@code repoTags} attribute
   */
  @JsonProperty("RepoTags")
  @Override
  public @Nullable List<String> repoTags() {
    return repoTags;
  }

  /**
   * @return The value of the {@code repoDigests} attribute
   */
  @JsonProperty("RepoDigests")
  @Override
  public @Nullable List<String> repoDigests() {
    return repoDigests;
  }

  /**
   * @return The value of the {@code size} attribute
   */
  @JsonProperty("Size")
  @Override
  public Long size() {
    return size;
  }

  /**
   * @return The value of the {@code virtualSize} attribute
   */
  @JsonProperty("VirtualSize")
  @Override
  public Long virtualSize() {
    return virtualSize;
  }

  /**
   * @return The value of the {@code labels} attribute
   */
  @JsonProperty("Labels")
  @Override
  public @Nullable Map<String, String> labels() {
    return labels;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Image#created() created} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for created
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableImage withCreated(String value) {
    String newValue = Objects.requireNonNull(value, "created");
    if (this.created.equals(newValue)) return this;
    return new ImmutableImage(
        newValue,
        this.id,
        this.parentId,
        this.repoTags,
        this.repoDigests,
        this.size,
        this.virtualSize,
        this.labels);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Image#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableImage withId(String value) {
    String newValue = Objects.requireNonNull(value, "id");
    if (this.id.equals(newValue)) return this;
    return new ImmutableImage(
        this.created,
        newValue,
        this.parentId,
        this.repoTags,
        this.repoDigests,
        this.size,
        this.virtualSize,
        this.labels);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Image#parentId() parentId} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for parentId
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableImage withParentId(String value) {
    String newValue = Objects.requireNonNull(value, "parentId");
    if (this.parentId.equals(newValue)) return this;
    return new ImmutableImage(
        this.created,
        this.id,
        newValue,
        this.repoTags,
        this.repoDigests,
        this.size,
        this.virtualSize,
        this.labels);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Image#repoTags() repoTags}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableImage withRepoTags(@Nullable String... elements) {
    if (elements == null) {
      return new ImmutableImage(
          this.created,
          this.id,
          this.parentId,
          null,
          this.repoDigests,
          this.size,
          this.virtualSize,
          this.labels);
    }
    @Nullable List<String> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableImage(
        this.created,
        this.id,
        this.parentId,
        newValue,
        this.repoDigests,
        this.size,
        this.virtualSize,
        this.labels);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Image#repoTags() repoTags}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of repoTags elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableImage withRepoTags(@Nullable Iterable<String> elements) {
    if (this.repoTags == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableImage(
        this.created,
        this.id,
        this.parentId,
        newValue,
        this.repoDigests,
        this.size,
        this.virtualSize,
        this.labels);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Image#repoDigests() repoDigests}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableImage withRepoDigests(@Nullable String... elements) {
    if (elements == null) {
      return new ImmutableImage(
          this.created,
          this.id,
          this.parentId,
          this.repoTags,
          null,
          this.size,
          this.virtualSize,
          this.labels);
    }
    @Nullable List<String> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableImage(
        this.created,
        this.id,
        this.parentId,
        this.repoTags,
        newValue,
        this.size,
        this.virtualSize,
        this.labels);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Image#repoDigests() repoDigests}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of repoDigests elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableImage withRepoDigests(@Nullable Iterable<String> elements) {
    if (this.repoDigests == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableImage(
        this.created,
        this.id,
        this.parentId,
        this.repoTags,
        newValue,
        this.size,
        this.virtualSize,
        this.labels);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Image#size() size} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for size
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableImage withSize(Long value) {
    Long newValue = Objects.requireNonNull(value, "size");
    if (this.size.equals(newValue)) return this;
    return new ImmutableImage(
        this.created,
        this.id,
        this.parentId,
        this.repoTags,
        this.repoDigests,
        newValue,
        this.virtualSize,
        this.labels);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Image#virtualSize() virtualSize} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for virtualSize
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableImage withVirtualSize(Long value) {
    Long newValue = Objects.requireNonNull(value, "virtualSize");
    if (this.virtualSize.equals(newValue)) return this;
    return new ImmutableImage(
        this.created,
        this.id,
        this.parentId,
        this.repoTags,
        this.repoDigests,
        this.size,
        newValue,
        this.labels);
  }

  /**
   * Copy the current immutable object by replacing the {@link Image#labels() labels} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to be added to the labels map
   * @return A modified copy of {@code this} object
   */
  public final ImmutableImage withLabels(@Nullable Map<String, ? extends String> entries) {
    if (this.labels == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableImage(
        this.created,
        this.id,
        this.parentId,
        this.repoTags,
        this.repoDigests,
        this.size,
        this.virtualSize,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableImage} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableImage
        && equalTo((ImmutableImage) another);
  }

  private boolean equalTo(ImmutableImage another) {
    return created.equals(another.created)
        && id.equals(another.id)
        && parentId.equals(another.parentId)
        && Objects.equals(repoTags, another.repoTags)
        && Objects.equals(repoDigests, another.repoDigests)
        && size.equals(another.size)
        && virtualSize.equals(another.virtualSize)
        && Objects.equals(labels, another.labels);
  }

  /**
   * Computes a hash code from attributes: {@code created}, {@code id}, {@code parentId}, {@code repoTags}, {@code repoDigests}, {@code size}, {@code virtualSize}, {@code labels}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + created.hashCode();
    h += (h << 5) + id.hashCode();
    h += (h << 5) + parentId.hashCode();
    h += (h << 5) + Objects.hashCode(repoTags);
    h += (h << 5) + Objects.hashCode(repoDigests);
    h += (h << 5) + size.hashCode();
    h += (h << 5) + virtualSize.hashCode();
    h += (h << 5) + Objects.hashCode(labels);
    return h;
  }

  /**
   * Prints the immutable value {@code Image} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Image{"
        + "created=" + created
        + ", id=" + id
        + ", parentId=" + parentId
        + ", repoTags=" + repoTags
        + ", repoDigests=" + repoDigests
        + ", size=" + size
        + ", virtualSize=" + virtualSize
        + ", labels=" + labels
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Image} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Image instance
   */
  public static ImmutableImage copyOf(Image instance) {
    if (instance instanceof ImmutableImage) {
      return (ImmutableImage) instance;
    }
    return ImmutableImage.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableImage ImmutableImage}.
   * <pre>
   * ImmutableImage.builder()
   *    .created(String) // required {@link Image#created() created}
   *    .id(String) // required {@link Image#id() id}
   *    .parentId(String) // required {@link Image#parentId() parentId}
   *    .repoTags(List&amp;lt;String&amp;gt; | null) // nullable {@link Image#repoTags() repoTags}
   *    .repoDigests(List&amp;lt;String&amp;gt; | null) // nullable {@link Image#repoDigests() repoDigests}
   *    .size(Long) // required {@link Image#size() size}
   *    .virtualSize(Long) // required {@link Image#virtualSize() virtualSize}
   *    .labels(Map&amp;lt;String, String&amp;gt; | null) // nullable {@link Image#labels() labels}
   *    .build();
   * </pre>
   * @return A new ImmutableImage builder
   */
  public static ImmutableImage.Builder builder() {
    return new ImmutableImage.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableImage ImmutableImage}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_CREATED = 0x1L;
    private static final long INIT_BIT_ID = 0x2L;
    private static final long INIT_BIT_PARENT_ID = 0x4L;
    private static final long INIT_BIT_SIZE = 0x8L;
    private static final long INIT_BIT_VIRTUAL_SIZE = 0x10L;
    private long initBits = 0x1fL;

    private String created;
    private String id;
    private String parentId;
    private List<String> repoTags = null;
    private List<String> repoDigests = null;
    private Long size;
    private Long virtualSize;
    private Map<String, String> labels = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Image} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Image instance) {
      Objects.requireNonNull(instance, "instance");
      created(instance.created());
      id(instance.id());
      parentId(instance.parentId());
      @Nullable List<String> repoTagsValue = instance.repoTags();
      if (repoTagsValue != null) {
        addAllRepoTags(repoTagsValue);
      }
      @Nullable List<String> repoDigestsValue = instance.repoDigests();
      if (repoDigestsValue != null) {
        addAllRepoDigests(repoDigestsValue);
      }
      size(instance.size());
      virtualSize(instance.virtualSize());
      @Nullable Map<String, String> labelsValue = instance.labels();
      if (labelsValue != null) {
        putAllLabels(labelsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Image#created() created} attribute.
     * @param created The value for created 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Created")
    public final Builder created(String created) {
      this.created = Objects.requireNonNull(created, "created");
      initBits &= ~INIT_BIT_CREATED;
      return this;
    }

    /**
     * Initializes the value for the {@link Image#id() id} attribute.
     * @param id The value for id 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Id")
    public final Builder id(String id) {
      this.id = Objects.requireNonNull(id, "id");
      initBits &= ~INIT_BIT_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link Image#parentId() parentId} attribute.
     * @param parentId The value for parentId 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ParentId")
    public final Builder parentId(String parentId) {
      this.parentId = Objects.requireNonNull(parentId, "parentId");
      initBits &= ~INIT_BIT_PARENT_ID;
      return this;
    }

    /**
     * Adds one element to {@link Image#repoTags() repoTags} list.
     * @param element A repoTags element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder repoTag(String element) {
      if (this.repoTags == null) {
        this.repoTags = new ArrayList<String>();
      }
      this.repoTags.add(Objects.requireNonNull(element, "repoTags element"));
      return this;
    }

    /**
     * Adds elements to {@link Image#repoTags() repoTags} list.
     * @param elements An array of repoTags elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder repoTags(String... elements) {
      if (this.repoTags == null) {
        this.repoTags = new ArrayList<String>();
      }
      for (String element : elements) {
        this.repoTags.add(Objects.requireNonNull(element, "repoTags element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Image#repoTags() repoTags} list.
     * @param elements An iterable of repoTags elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("RepoTags")
    public final Builder repoTags(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.repoTags = null;
        return this;
      }
      this.repoTags = new ArrayList<String>();
      return addAllRepoTags(elements);
    }

    /**
     * Adds elements to {@link Image#repoTags() repoTags} list.
     * @param elements An iterable of repoTags elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllRepoTags(Iterable<String> elements) {
      Objects.requireNonNull(elements, "repoTags element");
      if (this.repoTags == null) {
        this.repoTags = new ArrayList<String>();
      }
      for (String element : elements) {
        this.repoTags.add(Objects.requireNonNull(element, "repoTags element"));
      }
      return this;
    }

    /**
     * Adds one element to {@link Image#repoDigests() repoDigests} list.
     * @param element A repoDigests element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder repoDigest(String element) {
      if (this.repoDigests == null) {
        this.repoDigests = new ArrayList<String>();
      }
      this.repoDigests.add(Objects.requireNonNull(element, "repoDigests element"));
      return this;
    }

    /**
     * Adds elements to {@link Image#repoDigests() repoDigests} list.
     * @param elements An array of repoDigests elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder repoDigests(String... elements) {
      if (this.repoDigests == null) {
        this.repoDigests = new ArrayList<String>();
      }
      for (String element : elements) {
        this.repoDigests.add(Objects.requireNonNull(element, "repoDigests element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Image#repoDigests() repoDigests} list.
     * @param elements An iterable of repoDigests elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("RepoDigests")
    public final Builder repoDigests(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.repoDigests = null;
        return this;
      }
      this.repoDigests = new ArrayList<String>();
      return addAllRepoDigests(elements);
    }

    /**
     * Adds elements to {@link Image#repoDigests() repoDigests} list.
     * @param elements An iterable of repoDigests elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllRepoDigests(Iterable<String> elements) {
      Objects.requireNonNull(elements, "repoDigests element");
      if (this.repoDigests == null) {
        this.repoDigests = new ArrayList<String>();
      }
      for (String element : elements) {
        this.repoDigests.add(Objects.requireNonNull(element, "repoDigests element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Image#size() size} attribute.
     * @param size The value for size 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Size")
    public final Builder size(Long size) {
      this.size = Objects.requireNonNull(size, "size");
      initBits &= ~INIT_BIT_SIZE;
      return this;
    }

    /**
     * Initializes the value for the {@link Image#virtualSize() virtualSize} attribute.
     * @param virtualSize The value for virtualSize 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("VirtualSize")
    public final Builder virtualSize(Long virtualSize) {
      this.virtualSize = Objects.requireNonNull(virtualSize, "virtualSize");
      initBits &= ~INIT_BIT_VIRTUAL_SIZE;
      return this;
    }

    /**
     * Put one entry to the {@link Image#labels() labels} map.
     * @param key The key in the labels map
     * @param value The associated value in the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(String key, String value) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      this.labels.put(
          Objects.requireNonNull(key, "labels key"),
          Objects.requireNonNull(value, "labels value"));
      return this;
    }

    /**
     * Put one entry to the {@link Image#labels() labels} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(Map.Entry<String, ? extends String> entry) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.labels.put(
          Objects.requireNonNull(k, "labels key"),
          Objects.requireNonNull(v, "labels value"));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link Image#labels() labels} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Labels")
    public final Builder labels(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.labels = null;
        return this;
      }
      this.labels = new LinkedHashMap<String, String>();
      return putAllLabels(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link Image#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLabels(Map<String, ? extends String> entries) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.labels.put(
            Objects.requireNonNull(k, "labels key"),
            Objects.requireNonNull(v, "labels value"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableImage ImmutableImage}.
     * @return An immutable instance of Image
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableImage build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableImage(
          created,
          id,
          parentId,
          repoTags == null ? null : createUnmodifiableList(true, repoTags),
          repoDigests == null ? null : createUnmodifiableList(true, repoDigests),
          size,
          virtualSize,
          labels == null ? null : createUnmodifiableMap(false, false, labels));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_CREATED) != 0) attributes.add("created");
      if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
      if ((initBits & INIT_BIT_PARENT_ID) != 0) attributes.add("parentId");
      if ((initBits & INIT_BIT_SIZE) != 0) attributes.add("size");
      if ((initBits & INIT_BIT_VIRTUAL_SIZE) != 0) attributes.add("virtualSize");
      return "Cannot build Image, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>();
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, "value");
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size());
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, "value");
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
