package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link Descriptor}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableDescriptor.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableDescriptor implements Descriptor {
  private final String mediaType;
  private final String digest;
  private final Long size;
  private final List<String> urls;

  private ImmutableDescriptor(
      String mediaType,
      String digest,
      Long size,
      List<String> urls) {
    this.mediaType = mediaType;
    this.digest = digest;
    this.size = size;
    this.urls = urls;
  }

  /**
   * @return The value of the {@code mediaType} attribute
   */
  @JsonProperty("MediaType")
  @Override
  public String mediaType() {
    return mediaType;
  }

  /**
   * @return The value of the {@code digest} attribute
   */
  @JsonProperty("Digest")
  @Override
  public String digest() {
    return digest;
  }

  /**
   * @return The value of the {@code size} attribute
   */
  @JsonProperty("Size")
  @Override
  public Long size() {
    return size;
  }

  /**
   * @return The value of the {@code urls} attribute
   */
  @JsonProperty("URLs")
  @Override
  public List<String> urls() {
    return urls;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Descriptor#mediaType() mediaType} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for mediaType
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDescriptor withMediaType(String value) {
    String newValue = Objects.requireNonNull(value, "mediaType");
    if (this.mediaType.equals(newValue)) return this;
    return new ImmutableDescriptor(newValue, this.digest, this.size, this.urls);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Descriptor#digest() digest} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for digest
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDescriptor withDigest(String value) {
    String newValue = Objects.requireNonNull(value, "digest");
    if (this.digest.equals(newValue)) return this;
    return new ImmutableDescriptor(this.mediaType, newValue, this.size, this.urls);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Descriptor#size() size} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for size
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDescriptor withSize(Long value) {
    Long newValue = Objects.requireNonNull(value, "size");
    if (this.size.equals(newValue)) return this;
    return new ImmutableDescriptor(this.mediaType, this.digest, newValue, this.urls);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Descriptor#urls() urls}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableDescriptor withUrls(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableDescriptor(this.mediaType, this.digest, this.size, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Descriptor#urls() urls}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of urls elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableDescriptor withUrls(Iterable<String> elements) {
    if (this.urls == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableDescriptor(this.mediaType, this.digest, this.size, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableDescriptor} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableDescriptor
        && equalTo((ImmutableDescriptor) another);
  }

  private boolean equalTo(ImmutableDescriptor another) {
    return mediaType.equals(another.mediaType)
        && digest.equals(another.digest)
        && size.equals(another.size)
        && urls.equals(another.urls);
  }

  /**
   * Computes a hash code from attributes: {@code mediaType}, {@code digest}, {@code size}, {@code urls}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + mediaType.hashCode();
    h += (h << 5) + digest.hashCode();
    h += (h << 5) + size.hashCode();
    h += (h << 5) + urls.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Descriptor} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Descriptor{"
        + "mediaType=" + mediaType
        + ", digest=" + digest
        + ", size=" + size
        + ", urls=" + urls
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Descriptor} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Descriptor instance
   */
  public static ImmutableDescriptor copyOf(Descriptor instance) {
    if (instance instanceof ImmutableDescriptor) {
      return (ImmutableDescriptor) instance;
    }
    return ImmutableDescriptor.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableDescriptor ImmutableDescriptor}.
   * <pre>
   * ImmutableDescriptor.builder()
   *    .mediaType(String) // required {@link Descriptor#mediaType() mediaType}
   *    .digest(String) // required {@link Descriptor#digest() digest}
   *    .size(Long) // required {@link Descriptor#size() size}
   *    .url|addAllUrls(String) // {@link Descriptor#urls() urls} elements
   *    .build();
   * </pre>
   * @return A new ImmutableDescriptor builder
   */
  public static ImmutableDescriptor.Builder builder() {
    return new ImmutableDescriptor.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableDescriptor ImmutableDescriptor}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_MEDIA_TYPE = 0x1L;
    private static final long INIT_BIT_DIGEST = 0x2L;
    private static final long INIT_BIT_SIZE = 0x4L;
    private long initBits = 0x7L;

    private String mediaType;
    private String digest;
    private Long size;
    private List<String> urls = new ArrayList<String>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Descriptor} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Descriptor instance) {
      Objects.requireNonNull(instance, "instance");
      mediaType(instance.mediaType());
      digest(instance.digest());
      size(instance.size());
      addAllUrls(instance.urls());
      return this;
    }

    /**
     * Initializes the value for the {@link Descriptor#mediaType() mediaType} attribute.
     * @param mediaType The value for mediaType 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("MediaType")
    public final Builder mediaType(String mediaType) {
      this.mediaType = Objects.requireNonNull(mediaType, "mediaType");
      initBits &= ~INIT_BIT_MEDIA_TYPE;
      return this;
    }

    /**
     * Initializes the value for the {@link Descriptor#digest() digest} attribute.
     * @param digest The value for digest 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Digest")
    public final Builder digest(String digest) {
      this.digest = Objects.requireNonNull(digest, "digest");
      initBits &= ~INIT_BIT_DIGEST;
      return this;
    }

    /**
     * Initializes the value for the {@link Descriptor#size() size} attribute.
     * @param size The value for size 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Size")
    public final Builder size(Long size) {
      this.size = Objects.requireNonNull(size, "size");
      initBits &= ~INIT_BIT_SIZE;
      return this;
    }

    /**
     * Adds one element to {@link Descriptor#urls() urls} list.
     * @param element A urls element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder url(String element) {
      this.urls.add(Objects.requireNonNull(element, "urls element"));
      return this;
    }

    /**
     * Adds elements to {@link Descriptor#urls() urls} list.
     * @param elements An array of urls elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder urls(String... elements) {
      for (String element : elements) {
        this.urls.add(Objects.requireNonNull(element, "urls element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Descriptor#urls() urls} list.
     * @param elements An iterable of urls elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("URLs")
    public final Builder urls(Iterable<String> elements) {
      this.urls.clear();
      return addAllUrls(elements);
    }

    /**
     * Adds elements to {@link Descriptor#urls() urls} list.
     * @param elements An iterable of urls elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllUrls(Iterable<String> elements) {
      for (String element : elements) {
        this.urls.add(Objects.requireNonNull(element, "urls element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableDescriptor ImmutableDescriptor}.
     * @return An immutable instance of Descriptor
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableDescriptor build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableDescriptor(mediaType, digest, size, createUnmodifiableList(true, urls));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_MEDIA_TYPE) != 0) attributes.add("mediaType");
      if ((initBits & INIT_BIT_DIGEST) != 0) attributes.add("digest");
      if ((initBits & INIT_BIT_SIZE) != 0) attributes.add("size");
      return "Cannot build Descriptor, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>();
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
