/*******************************************************************************
 * Copyright (c) 2017 Red Hat, Inc and others.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Red Hat, Inc - initial API and implementation
 *******************************************************************************/
package org.eclipse.reddeer.core.handler;


import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.graphics.Point;
import org.eclipse.reddeer.common.util.Display;
import org.eclipse.reddeer.common.util.ResultRunnable;
import org.eclipse.reddeer.core.exception.CoreLayerException;

/**
 * Contains methods for handling UI operations on {@link StyledText} widgets.
 * 
 * @author Jiri Peterka
 * @author Vlado Pakan
 * 
 */
public class StyledTextHandler extends ControlHandler{
	
	private static StyledTextHandler instance;
	
	/**
	 * Gets instance of StyledTextHandler.
	 * 
	 * @return instance of StyledTextHandler
	 */
	public static StyledTextHandler getInstance(){
		if(instance == null){
			instance = new StyledTextHandler();
		}
		return instance;
	}

	/**
	 * Sets selection within specified {@link StyledText} to specified line and column.
	 * 
	 * @param styledText styled text to handle
	 * @param line line of selection
	 * @param column column of selection
	 */
	public void setSelectionWithinLine(final StyledText styledText, final int line,final int column) {
		Display.syncExec(new Runnable() {

			@Override
			public void run() {
				int offset = styledText.getContent().getOffsetAtLine(line)
						+ column;
				styledText.setSelection(offset);
			}
		});

	}
	
	/**
	 * Sets selection within specified {@link StyledText} to specified start and end.
	 * 
	 * @param styledText styled text to handle
	 * @param start selection start offset.
	 * @param end selection end offset
	 */
	public void setSelection(final StyledText styledText, final int start, final int end) {
		Display.syncExec(new Runnable() {

			@Override
			public void run() {
				styledText.setSelection(start,end);
			}
		});

	}

	/**
	 * Inserts specified text on the current position in specified {@link StyledText}.
	 * 
	 * @param styledText styled text to handle
	 * @param text text to insert
	 */
	public void insertText(final StyledText styledText, final String text) {
		Display.syncExec(new Runnable() {

			@Override
			public void run() {
				styledText.insert(text);
			}
		});
	}

	/**
	 * Inserts specified text on specified position in specified {@link StyledText}.
	 * 
	 * @param styledText styled text to handle
	 * @param line line to select
	 * @param column column to select
	 * @param text text to insert
	 */
	public void insertText(StyledText styledText, int line, int column, String text) {
		setSelectionWithinLine(styledText, line, column);
		insertText(styledText, text);
	}
	
	/**
	 * Gets position of first character of specified text in specified {@link StyledText}.
	 * 
	 * @param styledText styled text to handle
	 * @param text text to get position
	 * @return position of first character of specified text if exists, -1 otherwise 
	 */
	public int getPositionOfText(final StyledText styledText, final String text) {
		return Display.syncExec(new ResultRunnable<Integer>() {

			@Override
			public Integer run() {
				return styledText.getText().indexOf(text);
			}
		});
	}
	
	/**
	 * Selects first occurrence of specified text in specified {@link StyledText}.
	 * 
	 * @param styledText styled text to handle
	 * @param text to select
	 */
	public void selectText(final StyledText styledText, final String text) {
		final int position = getPositionOfText(styledText, text);
		if (position == -1) { 
			throw new CoreLayerException("Unable to find text " + text + " in styled text");
		}
		Display.syncExec(new Runnable() {

			@Override
			public void run() {
				styledText.setSelection(position, position + text.length());
			}
		});
	}

	/**
	 * Selects specified position in specified {@link StyledText}.
	 * 
	 * @param styledText styled text to handle
	 * @param position position to select in specified styled text
	 */
	public void selectPosition(final StyledText styledText, final int position) {
		Display.syncExec(new Runnable() {

			@Override
			public void run() {
				styledText.setSelection(position);
			}
		});
		
	}

	/**
	 * Gets selected text in specified {@link StyledText}.
	 * 
	 * @param styledText styled text to handle
	 * @return selected text
	 */
	public String getSelectionText(final StyledText styledText) {
		return Display.syncExec(new ResultRunnable<String>() {

			@Override
			public String run() {
				return styledText.getSelectionText();
			}
		});
	}

	/**
	 * Gets the current position of the cursor.
	 *
	 * @param styledText the styled text
	 * @return zero based position of the cursor in the styled text.
	 */
	public Point getCursorPosition(final StyledText styledText) {
		return Display.syncExec(new ResultRunnable<Point>() {
			public Point run() {
				styledText.setFocus();
				int offset = styledText.getSelectionRange().x;
				int line = styledText.getContent().getLineAtOffset(offset);
				int offsetAtLine = styledText.getContent().getOffsetAtLine(line);
				int column = offset - offsetAtLine;
				return new Point(line, column);
			}
		});
	}
	
	/**
	 * Returns offset of the line within specified styled text.
	 *
	 * @param styledText            stylerText to get offset from
	 * @param line            line to get offset of
	 * @return the line offset
	 */
	public int getLineOffset(final StyledText styledText, final int line) {
		
		int offset = Display.syncExec(new ResultRunnable<Integer>() {
			public Integer run() {
				styledText.setFocus();
				int offsetAtLine = styledText.getOffsetAtLine(line);
				return offsetAtLine;
			}
		});
		
		return offset;

	}
	
	/**
	 * Gets text of styled text
	 * @param styledText to handle
	 * @return text of specified styled text
	 */
	public String getText(final StyledText styledText){
		return Display.syncExec(new ResultRunnable<String>() {

			@Override
			public String run() {
				return styledText.getText();
			}
		});
	}
	
	/**
	 * Sets text to styled text
	 * @param styledText to handle
	 * @param text to set to specified styled text
	 */
	public void setText(final StyledText styledText, String text){
		Display.syncExec(new Runnable() {
			
			@Override
			public void run() {
				styledText.setText(text);
				
			}
		});
	}
}
