/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat, Inc. All rights reserved.
 *  Copyright (c) Adam Voss. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
const nls = require("vscode-nls");
const localize = nls.loadMessageBundle();
const Yaml = require("yaml-ast-parser-custom-tags");
const jsonParser07_1 = require("./jsonParser07");
const parseUtils_1 = require("../utils/parseUtils");
const recursivelyBuildAst_1 = require("./recursivelyBuildAst");
const documentPositionCalculator_1 = require("../utils/documentPositionCalculator");
const vscode_json_languageservice_1 = require("vscode-json-languageservice");
/**
 * These documents are collected into a final YAMLDocument
 * and passed to the `parseYAML` caller.
 */
class SingleYAMLDocument extends jsonParser07_1.JSONDocument {
    constructor(lines) {
        super(null, []);
        this.lines = lines;
        this.root = null;
        this.errors = [];
        this.warnings = [];
    }
    getSchemas(schema, doc, node) {
        const matchingSchemas = [];
        doc.validate(schema, matchingSchemas, node.start);
        return matchingSchemas;
    }
}
exports.SingleYAMLDocument = SingleYAMLDocument;
function nodeToSingleDoc(yamlNode, startPositions, text) {
    const _doc = new SingleYAMLDocument(startPositions);
    _doc.root = recursivelyBuildAst_1.default(null, yamlNode);
    if (!_doc.root) {
        // TODO: When this is true, consider not pushing the other errors.
        _doc.errors.push({
            message: localize('Invalid symbol', 'Expected a YAML object, array or literal'),
            //@ts-ignore
            code: vscode_json_languageservice_1.ErrorCode.Undefined,
            location: { start: yamlNode.startPosition, end: yamlNode.endPosition }
        });
    }
    const errors = parseUtils_1.formatErrors(yamlNode.errors);
    const warnings = parseUtils_1.formatWarnings(yamlNode.errors, text);
    errors.forEach(e => { return _doc.errors.push(e); });
    warnings.forEach(e => { return _doc.warnings.push(e); });
    return _doc;
}
/**
 * Contains the SingleYAMLDocuments, to be passed
 * to the `parseYAML` caller.
 */
class YAMLDocument {
    constructor(documents) {
        this.documents = documents;
        this.errors = [];
        this.warnings = [];
    }
}
exports.YAMLDocument = YAMLDocument;
/**
 * `yaml-ast-parser-custom-tags` parses the AST and
 * returns YAML AST nodes, which are then formatted
 * for consumption via the language server.
 */
function parse(text, customTags = []) {
    const additionalOptions = parseUtils_1.customTagsToAdditionalOptions(customTags);
    // Parse the AST using `yaml-ast-parser-custom-tags`
    const yamlNodes = [];
    Yaml.loadAll(text, doc => yamlNodes.push(doc), additionalOptions);
    // Generate the SingleYAMLDocs from the AST nodes
    const startPositions = documentPositionCalculator_1.getLineStartPositions(text);
    const yamlDocs = yamlNodes.map(node => nodeToSingleDoc(node, startPositions, text));
    // Consolidate the SingleYAMLDocs
    return new YAMLDocument(yamlDocs);
}
exports.parse = parse;
//# sourceMappingURL=yamlParser07.js.map