/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.di;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages.*;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.IDEConstants.BUNDLE_ID;
import static org.eclipse.epp.logging.aeri.core.IModelPackage.Literals.*;
import static org.eclipse.epp.logging.aeri.core.ResetSendMode.RESTART;
import static org.eclipse.epp.logging.aeri.core.SystemControl.executeHandler;
import static org.eclipse.epp.logging.aeri.core.util.Logs.log;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.preferences.ConfigurationScope;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.e4.core.contexts.IContextFunction;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.epp.internal.logging.aeri.ide.utils.AnonymousId;
import org.eclipse.epp.logging.aeri.core.IModelFactory;
import org.eclipse.epp.logging.aeri.core.ISystemSettings;
import org.eclipse.epp.logging.aeri.core.SystemControl;
import org.eclipse.epp.logging.aeri.core.handlers.ResetSendModeHandler;
import org.eclipse.jdt.annotation.Nullable;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableSet;

/**
 * Context function that computes the {@link ISystemSettings}. Assumes to be placed into the system root context.
 */
public class SystemSettingsCreationFunction implements IContextFunction {

    @Override
    public Object compute(IEclipseContext localContext, @Nullable String contextKey) {
        contextKey = checkNotNull(contextKey);
        ISystemSettings settings = IModelFactory.eINSTANCE.createSystemSettings();
        settings.setReporterId(AnonymousId.getId().toString());
        EClass eClass = settings.eClass();
        ScopedPreferenceStore instanceStore = new ScopedPreferenceStore(InstanceScope.INSTANCE, BUNDLE_ID);
        loadFromPreferences(instanceStore, settings, eClass);
        registerPreferenceStoreChangeListener(instanceStore, settings, eClass);
        registerSettingsChangeListener(instanceStore, settings, new HashSet<EAttribute>());

        // register a listener that sends selected changes to the configuration scope store:
        // convenience to allow users with different workspaces to reuse the settings
        ScopedPreferenceStore configurationStore = new ScopedPreferenceStore(ConfigurationScope.INSTANCE, BUNDLE_ID);
        registerSettingsChangeListener(configurationStore, settings,
                ImmutableSet.of(USER_SETTINGS__REPORTER_NAME, USER_SETTINGS__REPORTER_EMAIL, USER_SETTINGS__ANONYMIZE_STACK_TRACES,
                        USER_SETTINGS__ANONYMIZE_MESSAGES, SYSTEM_SETTINGS__DEBUG_ENABLED));

        if (RESTART == settings.getResetSendMode()) {
            executeHandler(ResetSendModeHandler.class);
        }

        // set this settings object in the root context. Effectively replaces this context function.
        IEclipseContext systemContext = SystemControl.getSystemContext();
        systemContext.set(contextKey, settings);
        return settings;
    }

    private static void registerSettingsChangeListener(final ScopedPreferenceStore store, final ISystemSettings settings,
            final Set<EAttribute> allowedKeys) {
        settings.eAdapters().add(new AdapterImpl() {
            @Override
            public void notifyChanged(Notification msg) {
                Object feature = msg.getFeature();
                if (!(feature instanceof EAttribute)) {
                    return;
                }
                EAttribute attr = (EAttribute) feature;
                EDataType type = attr.getEAttributeType();
                Object value = msg.getNewValue();

                // @Nullable
                String data = EcoreUtil.convertToString(type, value);

                // if empty all keys are allowed:
                if (allowedKeys.isEmpty() || allowedKeys.contains(attr)) {
                    try {
                        // See https://bugs.eclipse.org/bugs/show_bug.cgi?id=494098,
                        // data may be null and null is not valid as value
                        if (data == null) {
                            store.setToDefault(attr.getName());
                        } else {
                            store.putValue(attr.getName(), data);
                        }
                        store.save();
                    } catch (Exception e) {
                        log(ERROR_SAVE_PREFERENCES_FAILED, e, attr.getName(), data);
                    }
                }
            }
        });
    }

    private static void registerPreferenceStoreChangeListener(final ScopedPreferenceStore store, final ISystemSettings settings,
            final EClass eClass) {
        store.addPropertyChangeListener(new IPropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent event) {
                String property = event.getProperty();
                EStructuralFeature feature = eClass.getEStructuralFeature(property);
                if (feature != null && feature instanceof EAttribute) {
                    EAttribute attr = (EAttribute) feature;
                    EDataType type = attr.getEAttributeType();
                    String string = EcoreUtil.convertToString(type, event.getNewValue());
                    Object value = EcoreUtil.createFromString(type, string);
                    settings.eSet(feature, value);
                }
            }
        });
    }

    private static void loadFromPreferences(final ScopedPreferenceStore store, final ISystemSettings settings, final EClass eClass) {
        for (EAttribute attr : eClass.getEAllAttributes()) {
            EDataType type = attr.getEAttributeType();
            String key = attr.getName();
            if (!store.contains(key)) {
                continue;
            }
            String value = store.getString(key);
            try {
                if (attr.isMany()) {
                    List<String> values = convert(value);
                    for (String s : convert(value)) {
                        Object data = EcoreUtil.createFromString(type, s);
                        ((List<Object>) settings.eGet(attr)).add(data);
                    }
                    continue;
                }
                Object data = EcoreUtil.createFromString(type, value);
                settings.eSet(attr, data);
            } catch (Exception e) {
                log(ERROR_FAILED_TO_PARSE_PREFERENCE_VALUE, attr, value);
            }
        }
    }

    static List<String> convert(String string) {
        return Splitter.on(';').omitEmptyStrings().trimResults().splitToList(string);

    }

    static String convert(List<String> strings) {
        return Joiner.on(';').skipNulls().join(strings);
    }
}
