/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marcel Bruch - initial API and implementation.
 */
package org.eclipse.epp.internal.logging.aeri.ide;

import static org.eclipse.e4.core.contexts.ContextInjectionFactory.make;
import static org.eclipse.epp.internal.logging.aeri.ide.IDEWorkflow.CTX_SERVERS;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.IDEConstants.BUNDLE_ID;
import static org.eclipse.epp.logging.aeri.core.Constants.SYSPROP_DISABLE_AERI;
import static org.eclipse.epp.logging.aeri.core.SystemControl.registerHandlers;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.e4.core.contexts.ContextInjectionFactory;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.core.services.events.IEventBroker;
import org.eclipse.epp.internal.logging.aeri.ide.di.ExtensionPointReportProcessorsCreationFunction;
import org.eclipse.epp.internal.logging.aeri.ide.di.ImageRegistryCreationFunction;
import org.eclipse.epp.internal.logging.aeri.ide.di.LogListenerCreationFunction;
import org.eclipse.epp.internal.logging.aeri.ide.di.ReviewDialogCreationFunction;
import org.eclipse.epp.internal.logging.aeri.ide.di.SettingsPersistingRegistryServersCreationFunction;
import org.eclipse.epp.internal.logging.aeri.ide.di.SystemSettingsCreationFunction;
import org.eclipse.epp.internal.logging.aeri.ide.dialogs.ReviewDialog;
import org.eclipse.epp.internal.logging.aeri.ide.dialogs.SetupWizard;
import org.eclipse.epp.internal.logging.aeri.ide.handlers.CreateConfigureDialogHandler;
import org.eclipse.epp.internal.logging.aeri.ide.handlers.OpenConfigureServersDialogHandler;
import org.eclipse.epp.internal.logging.aeri.ide.handlers.OpenReviewDialogHandler;
import org.eclipse.epp.internal.logging.aeri.ide.handlers.OpenServersSetupWizardHandler;
import org.eclipse.epp.internal.logging.aeri.ide.handlers.OpenSystemSetupWizardHandler;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.internal.logging.aeri.ide.notifications.MylynNotificationsSupport;
import org.eclipse.epp.internal.logging.aeri.ide.utils.UploadReportsScheduler;
import org.eclipse.epp.logging.aeri.core.ISystemSettings;
import org.eclipse.epp.logging.aeri.core.SystemControl;
import org.eclipse.epp.logging.aeri.core.util.LogListener;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.ui.IStartup;
import org.eclipse.ui.PlatformUI;
import org.osgi.framework.Bundle;

public class IDEStartup implements IStartup {

    @Override
    public void earlyStartup() {
        if (Boolean.getBoolean(SYSPROP_DISABLE_AERI)) {
            return;
        }
        new Job(Messages.JOB_NAME_INITIALIZING_AERI) {
            @Override
            protected IStatus run(IProgressMonitor monitor) {
                monitor.beginTask(Messages.TASK_NAME_INITIALIZING_AERI, IProgressMonitor.UNKNOWN);
                IEclipseContext aeriContext = initializeIdeContext();
                // create the dispatcher
                IDEWorkflow dispatcher = aeriContext.get(IDEWorkflow.class);
                LogListener listener = aeriContext.get(LogListener.class);
                Platform.addLogListener(listener);
                monitor.done();
                return Status.OK_STATUS;
            }
        }.schedule();
    }

    private static IEclipseContext initializeIdeContext() {
        IEclipseContext context = SystemControl.getSystemContext();
        IEclipseContext parent = PlatformUI.getWorkbench().getService(IEclipseContext.class);
        context.setParent(parent);

        context.set(ReviewDialog.CTX_STATE_REVIEW_IN_PROGRESS, false);
        context.declareModifiable(ReviewDialog.CTX_STATE_REVIEW_IN_PROGRESS);

        context.set(SetupWizard.CTX_STATE_SETUP_IN_PROGRESS, false);
        context.declareModifiable(SetupWizard.CTX_STATE_SETUP_IN_PROGRESS);

        context.set(MylynNotificationsSupport.CTX_STATE_NOTIFICATION_IN_PROGRESS, false);
        context.declareModifiable(MylynNotificationsSupport.CTX_STATE_NOTIFICATION_IN_PROGRESS);

        registerHandlers(OpenReviewDialogHandler.class, OpenServersSetupWizardHandler.class, OpenConfigureServersDialogHandler.class,
                OpenSystemSetupWizardHandler.class, CreateConfigureDialogHandler.class);

        context.set(ISystemSettings.class.getName(), new SystemSettingsCreationFunction());

        Bundle bundle = Platform.getBundle(BUNDLE_ID);
        context.set(Bundle.class, bundle);
        context.set(IDEWorkflow.CTX_BUNDLE_STATE_LOCATION, Platform.getStateLocation(bundle));

        context.set(ILogEventsQueue.class, IIdeFactory.eINSTANCE.createLogEventsQueue());
        context.set(LogListener.class.getName(), new LogListenerCreationFunction());
        LogListener listener = context.get(LogListener.class);
        Platform.addLogListener(listener);

        final IEventBroker broker = context.get(IEventBroker.class);
        context.set(CTX_SERVERS, new SettingsPersistingRegistryServersCreationFunction());
        context.set(IDEWorkflow.CTX_REPORT_PROCESSORS, new ExtensionPointReportProcessorsCreationFunction());
        context.set(MylynNotificationsSupport.class, make(MylynNotificationsSupport.class, context));
        context.set(UploadReportsScheduler.class, new UploadReportsScheduler() {

            @Override
            public void send(ILogEventGroup group, IProgressMonitor monitor) {
                super.send(group, monitor);
                broker.post(IDEWorkflow.TOPIC_NEW_SERVER_RESPONSES, group);
            }
        });
        context.set(IDEWorkflow.class, ContextInjectionFactory.make(IDEWorkflow.class, context));
        IDEWorkflow workflow = context.get(IDEWorkflow.class);

        context.set(ImageRegistry.class.getName(), new ImageRegistryCreationFunction());

        context.set(ReviewDialog.class.getName(), new ReviewDialogCreationFunction());

        context.declareModifiable(IDEWorkflow.CTX_LOG_EVENT_LAST);
        context.declareModifiable(IDEWorkflow.CTX_LOG_EVENT_LAST_DELAYED);
        context.declareModifiable(IDEWorkflow.CTX_LOG_EVENT_LAST_NOTIFIED);

        return context;
    }
}
