(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define(["require", "exports", "../parser/htmlScanner", "vscode-languageserver-types", "../utils/strings", "vscode-uri", "../htmlLanguageTypes"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    var htmlScanner_1 = require("../parser/htmlScanner");
    var vscode_languageserver_types_1 = require("vscode-languageserver-types");
    var strings = require("../utils/strings");
    var vscode_uri_1 = require("vscode-uri");
    var htmlLanguageTypes_1 = require("../htmlLanguageTypes");
    function normalizeRef(url, languageId) {
        var first = url[0];
        var last = url[url.length - 1];
        if (first === last && (first === '\'' || first === '\"')) {
            url = url.substr(1, url.length - 2);
        }
        if (languageId === 'razor' && url[0] === '~') {
            url = url.substr(1);
        }
        return url;
    }
    function validateRef(url, languageId) {
        if (!url.length) {
            return false;
        }
        if (languageId === 'handlebars' && /{{.*}}/.test(url)) {
            return false;
        }
        if (languageId === 'razor' && /@/.test(url)) {
            return false;
        }
        try {
            return !!vscode_uri_1.default.parse(url);
        }
        catch (e) {
            return false;
        }
    }
    function getWorkspaceUrl(documentUri, tokenContent, documentContext, base) {
        if (/^\s*javascript\:/i.test(tokenContent) || /^\s*\#/i.test(tokenContent) || /[\n\r]/.test(tokenContent)) {
            return null;
        }
        tokenContent = tokenContent.replace(/^\s*/g, '');
        if (/^https?:\/\//i.test(tokenContent) || /^file:\/\//i.test(tokenContent)) {
            // Absolute link that needs no treatment
            return tokenContent;
        }
        if (/^\/\//i.test(tokenContent)) {
            // Absolute link (that does not name the protocol)
            var pickedScheme = strings.startsWith(documentUri, 'https://') ? 'https' : 'http';
            return pickedScheme + ':' + tokenContent.replace(/^\s*/g, '');
        }
        if (documentContext) {
            return documentContext.resolveReference(tokenContent, base || documentUri);
        }
        return tokenContent;
    }
    function createLink(document, documentContext, attributeValue, startOffset, endOffset, base) {
        var tokenContent = normalizeRef(attributeValue, document.languageId);
        if (!validateRef(tokenContent, document.languageId)) {
            return null;
        }
        if (tokenContent.length < attributeValue.length) {
            startOffset++;
            endOffset--;
        }
        var workspaceUrl = getWorkspaceUrl(document.uri, tokenContent, documentContext, base);
        if (!workspaceUrl || !isValidURI(workspaceUrl)) {
            return null;
        }
        return {
            range: vscode_languageserver_types_1.Range.create(document.positionAt(startOffset), document.positionAt(endOffset)),
            target: workspaceUrl
        };
    }
    function isValidURI(uri) {
        try {
            vscode_uri_1.default.parse(uri);
            return true;
        }
        catch (e) {
            return false;
        }
    }
    function findDocumentLinks(document, documentContext) {
        var newLinks = [];
        var rootAbsoluteUrl = null;
        var scanner = htmlScanner_1.createScanner(document.getText(), 0);
        var token = scanner.scan();
        var afterHrefOrSrc = false;
        var afterBase = false;
        var base = void 0;
        while (token !== htmlLanguageTypes_1.TokenType.EOS) {
            switch (token) {
                case htmlLanguageTypes_1.TokenType.StartTag:
                    if (!base) {
                        var tagName = scanner.getTokenText().toLowerCase();
                        afterBase = tagName === 'base';
                    }
                    break;
                case htmlLanguageTypes_1.TokenType.AttributeName:
                    var attributeName = scanner.getTokenText().toLowerCase();
                    afterHrefOrSrc = attributeName === 'src' || attributeName === 'href';
                    break;
                case htmlLanguageTypes_1.TokenType.AttributeValue:
                    if (afterHrefOrSrc) {
                        var attributeValue = scanner.getTokenText();
                        if (!afterBase) { // don't highlight the base link itself
                            var link = createLink(document, documentContext, attributeValue, scanner.getTokenOffset(), scanner.getTokenEnd(), base);
                            if (link) {
                                newLinks.push(link);
                            }
                        }
                        if (afterBase && typeof base === 'undefined') {
                            base = normalizeRef(attributeValue, document.languageId);
                            if (base && documentContext) {
                                base = documentContext.resolveReference(base, document.uri);
                            }
                        }
                        afterBase = false;
                        afterHrefOrSrc = false;
                    }
                    break;
            }
            token = scanner.scan();
        }
        return newLinks;
    }
    exports.findDocumentLinks = findDocumentLinks;
});
//# sourceMappingURL=htmlLinks.js.map