/*
 *    /\
 *   /  \ ot 0.0.14
 *  /    \ http://operational-transformation.github.com
 *  \    /
 *   \  / (c) 2012-2014 Tim Baumann <tim@timbaumann.info> (http://timbaumann.info)
 *    \/ ot may be freely distributed under the MIT license.
 */
define('orion/collab/ot',[], function() {
  if (typeof ot === 'undefined') {
    // Export for browsers
    var ot = {};
  }

  ot.TextOperation = (function () {
    'use strict';

    // Constructor for new operations.
    function TextOperation () {
      if (!this || this.constructor !== TextOperation) {
        // => function was called without 'new'
        return new TextOperation();
      }

      // When an operation is applied to an input string, you can think of this as
      // if an imaginary cursor runs over the entire string and skips over some
      // parts, deletes some parts and inserts characters at some positions. These
      // actions (skip/delete/insert) are stored as an array in the "ops" property.
      this.ops = [];
      // An operation's baseLength is the length of every string the operation
      // can be applied to.
      this.baseLength = 0;
      // The targetLength is the length of every string that results from applying
      // the operation on a valid input string.
      this.targetLength = 0;
    }

    TextOperation.prototype.equals = function (other) {
      if (this.baseLength !== other.baseLength) { return false; }
      if (this.targetLength !== other.targetLength) { return false; }
      if (this.ops.length !== other.ops.length) { return false; }
      for (var i = 0; i < this.ops.length; i++) {
        if (this.ops[i] !== other.ops[i]) { return false; }
      }
      return true;
    };

    // Operation are essentially lists of ops. There are three types of ops:
    //
    // * Retain ops: Advance the cursor position by a given number of characters.
    //   Represented by positive ints.
    // * Insert ops: Insert a given string at the current cursor position.
    //   Represented by strings.
    // * Delete ops: Delete the next n characters. Represented by negative ints.

    var isRetain = TextOperation.isRetain = function (op) {
      return typeof op === 'number' && op > 0;
    };

    var isInsert = TextOperation.isInsert = function (op) {
      return typeof op === 'string';
    };

    var isDelete = TextOperation.isDelete = function (op) {
      return typeof op === 'number' && op < 0;
    };


    // After an operation is constructed, the user of the library can specify the
    // actions of an operation (skip/insert/delete) with these three builder
    // methods. They all return the operation for convenient chaining.

    // Skip over a given number of characters.
    TextOperation.prototype.retain = function (n) {
      if (typeof n !== 'number') {
        throw new Error("retain expects an integer");
      }
      if (n === 0) { return this; }
      this.baseLength += n;
      this.targetLength += n;
      if (isRetain(this.ops[this.ops.length-1])) {
        // The last op is a retain op => we can merge them into one op.
        this.ops[this.ops.length-1] += n;
      } else {
        // Create a new op.
        this.ops.push(n);
      }
      return this;
    };

    // Insert a string at the current position.
    TextOperation.prototype.insert = function (str) {
      if (typeof str !== 'string') {
        throw new Error("insert expects a string");
      }
      if (str === '') { return this; }
      this.targetLength += str.length;
      var ops = this.ops;
      if (isInsert(ops[ops.length-1])) {
        // Merge insert op.
        ops[ops.length-1] += str;
      } else if (isDelete(ops[ops.length-1])) {
        // It doesn't matter when an operation is applied whether the operation
        // is delete(3), insert("something") or insert("something"), delete(3).
        // Here we enforce that in this case, the insert op always comes first.
        // This makes all operations that have the same effect when applied to
        // a document of the right length equal in respect to the `equals` method.
        if (isInsert(ops[ops.length-2])) {
          ops[ops.length-2] += str;
        } else {
          ops[ops.length] = ops[ops.length-1];
          ops[ops.length-2] = str;
        }
      } else {
        ops.push(str);
      }
      return this;
    };

    // Delete a string at the current position.
    TextOperation.prototype['delete'] = function (n) {
      if (typeof n === 'string') { n = n.length; }
      if (typeof n !== 'number') {
        throw new Error("delete expects an integer or a string");
      }
      if (n === 0) { return this; }
      if (n > 0) { n = -n; }
      this.baseLength -= n;
      if (isDelete(this.ops[this.ops.length-1])) {
        this.ops[this.ops.length-1] += n;
      } else {
        this.ops.push(n);
      }
      return this;
    };

    // Tests whether this operation has no effect.
    TextOperation.prototype.isNoop = function () {
      return this.ops.length === 0 || (this.ops.length === 1 && isRetain(this.ops[0]));
    };

    // Pretty printing.
    TextOperation.prototype.toString = function () {
      // map: build a new array by applying a function to every element in an old
      // array.
      var map = Array.prototype.map || function (fn) {
        var arr = this;
        var newArr = [];
        for (var i = 0, l = arr.length; i < l; i++) {
          newArr[i] = fn(arr[i]);
        }
        return newArr;
      };
      return map.call(this.ops, function (op) {
        if (isRetain(op)) {
          return "retain " + op;
        } else if (isInsert(op)) {
          return "insert '" + op + "'";
        } else {
          return "delete " + (-op);
        }
      }).join(', ');
    };

    // Converts operation into a JSON value.
    TextOperation.prototype.toJSON = function () {
      return this.ops;
    };

    // Converts a plain JS object into an operation and validates it.
    TextOperation.fromJSON = function (ops) {
      var o = new TextOperation();
      for (var i = 0, l = ops.length; i < l; i++) {
        var op = ops[i];
        if (isRetain(op)) {
          o.retain(op);
        } else if (isInsert(op)) {
          o.insert(op);
        } else if (isDelete(op)) {
          o['delete'](op);
        } else {
          throw new Error("unknown operation: " + JSON.stringify(op));
        }
      }
      return o;
    };

    // Apply an operation to a string, returning a new string. Throws an error if
    // there's a mismatch between the input string and the operation.
    TextOperation.prototype.apply = function (str) {
      var operation = this;
      if (str.length !== operation.baseLength) {
        throw new Error("The operation's base length must be equal to the string's length.");
      }
      var newStr = [], j = 0;
      var strIndex = 0;
      var ops = this.ops;
      for (var i = 0, l = ops.length; i < l; i++) {
        var op = ops[i];
        if (isRetain(op)) {
          if (strIndex + op > str.length) {
            throw new Error("Operation can't retain more characters than are left in the string.");
          }
          // Copy skipped part of the old string.
          newStr[j++] = str.slice(strIndex, strIndex + op);
          strIndex += op;
        } else if (isInsert(op)) {
          // Insert string.
          newStr[j++] = op;
        } else { // delete op
          strIndex -= op;
        }
      }
      if (strIndex !== str.length) {
        throw new Error("The operation didn't operate on the whole string.");
      }
      return newStr.join('');
    };

    // Computes the inverse of an operation. The inverse of an operation is the
    // operation that reverts the effects of the operation, e.g. when you have an
    // operation 'insert("hello "); skip(6);' then the inverse is 'delete("hello ");
    // skip(6);'. The inverse should be used for implementing undo.
    TextOperation.prototype.invert = function (str) {
      var strIndex = 0;
      var inverse = new TextOperation();
      var ops = this.ops;
      for (var i = 0, l = ops.length; i < l; i++) {
        var op = ops[i];
        if (isRetain(op)) {
          inverse.retain(op);
          strIndex += op;
        } else if (isInsert(op)) {
          inverse['delete'](op.length);
        } else { // delete op
          inverse.insert(str.slice(strIndex, strIndex - op));
          strIndex -= op;
        }
      }
      return inverse;
    };

    // Compose merges two consecutive operations into one operation, that
    // preserves the changes of both. Or, in other words, for each input string S
    // and a pair of consecutive operations A and B,
    // apply(apply(S, A), B) = apply(S, compose(A, B)) must hold.
    TextOperation.prototype.compose = function (operation2) {
      var operation1 = this;
      if (operation1.targetLength !== operation2.baseLength) {
        throw new Error("The base length of the second operation has to be the target length of the first operation");
      }

      var operation = new TextOperation(); // the combined operation
      var ops1 = operation1.ops, ops2 = operation2.ops; // for fast access
      var i1 = 0, i2 = 0; // current index into ops1 respectively ops2
      var op1 = ops1[i1++], op2 = ops2[i2++]; // current ops
      while (true) {
        // Dispatch on the type of op1 and op2
        if (typeof op1 === 'undefined' && typeof op2 === 'undefined') {
          // end condition: both ops1 and ops2 have been processed
          break;
        }

        if (isDelete(op1)) {
          operation['delete'](op1);
          op1 = ops1[i1++];
          continue;
        }
        if (isInsert(op2)) {
          operation.insert(op2);
          op2 = ops2[i2++];
          continue;
        }

        if (typeof op1 === 'undefined') {
          throw new Error("Cannot compose operations: first operation is too short.");
        }
        if (typeof op2 === 'undefined') {
          throw new Error("Cannot compose operations: first operation is too long.");
        }

        if (isRetain(op1) && isRetain(op2)) {
          if (op1 > op2) {
            operation.retain(op2);
            op1 = op1 - op2;
            op2 = ops2[i2++];
          } else if (op1 === op2) {
            operation.retain(op1);
            op1 = ops1[i1++];
            op2 = ops2[i2++];
          } else {
            operation.retain(op1);
            op2 = op2 - op1;
            op1 = ops1[i1++];
          }
        } else if (isInsert(op1) && isDelete(op2)) {
          if (op1.length > -op2) {
            op1 = op1.slice(-op2);
            op2 = ops2[i2++];
          } else if (op1.length === -op2) {
            op1 = ops1[i1++];
            op2 = ops2[i2++];
          } else {
            op2 = op2 + op1.length;
            op1 = ops1[i1++];
          }
        } else if (isInsert(op1) && isRetain(op2)) {
          if (op1.length > op2) {
            operation.insert(op1.slice(0, op2));
            op1 = op1.slice(op2);
            op2 = ops2[i2++];
          } else if (op1.length === op2) {
            operation.insert(op1);
            op1 = ops1[i1++];
            op2 = ops2[i2++];
          } else {
            operation.insert(op1);
            op2 = op2 - op1.length;
            op1 = ops1[i1++];
          }
        } else if (isRetain(op1) && isDelete(op2)) {
          if (op1 > -op2) {
            operation['delete'](op2);
            op1 = op1 + op2;
            op2 = ops2[i2++];
          } else if (op1 === -op2) {
            operation['delete'](op2);
            op1 = ops1[i1++];
            op2 = ops2[i2++];
          } else {
            operation['delete'](op1);
            op2 = op2 + op1;
            op1 = ops1[i1++];
          }
        } else {
          throw new Error(
            "This shouldn't happen: op1: " +
            JSON.stringify(op1) + ", op2: " +
            JSON.stringify(op2)
          );
        }
      }
      return operation;
    };

    function getSimpleOp (operation, fn) {
      var ops = operation.ops;
      var isRetain = TextOperation.isRetain;
      switch (ops.length) {
      case 1:
        return ops[0];
      case 2:
        return isRetain(ops[0]) ? ops[1] : (isRetain(ops[1]) ? ops[0] : null);
      case 3:
        if (isRetain(ops[0]) && isRetain(ops[2])) { return ops[1]; }
      }
      return null;
    }

    function getStartIndex (operation) {
      if (isRetain(operation.ops[0])) { return operation.ops[0]; }
      return 0;
    }

    // When you use ctrl-z to undo your latest changes, you expect the program not
    // to undo every single keystroke but to undo your last sentence you wrote at
    // a stretch or the deletion you did by holding the backspace key down. This
    // This can be implemented by composing operations on the undo stack. This
    // method can help decide whether two operations should be composed. It
    // returns true if the operations are consecutive insert operations or both
    // operations delete text at the same position. You may want to include other
    // factors like the time since the last change in your decision.
    TextOperation.prototype.shouldBeComposedWith = function (other) {
      if (this.isNoop() || other.isNoop()) { return true; }

      var startA = getStartIndex(this), startB = getStartIndex(other);
      var simpleA = getSimpleOp(this), simpleB = getSimpleOp(other);
      if (!simpleA || !simpleB) { return false; }

      if (isInsert(simpleA) && isInsert(simpleB)) {
        return startA + simpleA.length === startB;
      }

      if (isDelete(simpleA) && isDelete(simpleB)) {
        // there are two possibilities to delete: with backspace and with the
        // delete key.
        return (startB - simpleB === startA) || startA === startB;
      }

      return false;
    };

    // Decides whether two operations should be composed with each other
    // if they were inverted, that is
    // `shouldBeComposedWith(a, b) = shouldBeComposedWithInverted(b^{-1}, a^{-1})`.
    TextOperation.prototype.shouldBeComposedWithInverted = function (other) {
      if (this.isNoop() || other.isNoop()) { return true; }

      var startA = getStartIndex(this), startB = getStartIndex(other);
      var simpleA = getSimpleOp(this), simpleB = getSimpleOp(other);
      if (!simpleA || !simpleB) { return false; }

      if (isInsert(simpleA) && isInsert(simpleB)) {
        return startA + simpleA.length === startB || startA === startB;
      }

      if (isDelete(simpleA) && isDelete(simpleB)) {
        return startB - simpleB === startA;
      }

      return false;
    };

    // Transform takes two operations A and B that happened concurrently and
    // produces two operations A' and B' (in an array) such that
    // `apply(apply(S, A), B') = apply(apply(S, B), A')`. This function is the
    // heart of OT.
    TextOperation.transform = function (operation1, operation2) {
      if (operation1.baseLength !== operation2.baseLength) {
        throw new Error("Both operations have to have the same base length");
      }

      var operation1prime = new TextOperation();
      var operation2prime = new TextOperation();
      var ops1 = operation1.ops, ops2 = operation2.ops;
      var i1 = 0, i2 = 0;
      var op1 = ops1[i1++], op2 = ops2[i2++];
      while (true) {
        // At every iteration of the loop, the imaginary cursor that both
        // operation1 and operation2 have that operates on the input string must
        // have the same position in the input string.

        if (typeof op1 === 'undefined' && typeof op2 === 'undefined') {
          // end condition: both ops1 and ops2 have been processed
          break;
        }

        // next two cases: one or both ops are insert ops
        // => insert the string in the corresponding prime operation, skip it in
        // the other one. If both op1 and op2 are insert ops, prefer op1.
        if (isInsert(op1)) {
          operation1prime.insert(op1);
          operation2prime.retain(op1.length);
          op1 = ops1[i1++];
          continue;
        }
        if (isInsert(op2)) {
          operation1prime.retain(op2.length);
          operation2prime.insert(op2);
          op2 = ops2[i2++];
          continue;
        }

        if (typeof op1 === 'undefined') {
          throw new Error("Cannot compose operations: first operation is too short.");
        }
        if (typeof op2 === 'undefined') {
          throw new Error("Cannot compose operations: first operation is too long.");
        }

        var minl;
        if (isRetain(op1) && isRetain(op2)) {
          // Simple case: retain/retain
          if (op1 > op2) {
            minl = op2;
            op1 = op1 - op2;
            op2 = ops2[i2++];
          } else if (op1 === op2) {
            minl = op2;
            op1 = ops1[i1++];
            op2 = ops2[i2++];
          } else {
            minl = op1;
            op2 = op2 - op1;
            op1 = ops1[i1++];
          }
          operation1prime.retain(minl);
          operation2prime.retain(minl);
        } else if (isDelete(op1) && isDelete(op2)) {
          // Both operations delete the same string at the same position. We don't
          // need to produce any operations, we just skip over the delete ops and
          // handle the case that one operation deletes more than the other.
          if (-op1 > -op2) {
            op1 = op1 - op2;
            op2 = ops2[i2++];
          } else if (op1 === op2) {
            op1 = ops1[i1++];
            op2 = ops2[i2++];
          } else {
            op2 = op2 - op1;
            op1 = ops1[i1++];
          }
        // next two cases: delete/retain and retain/delete
        } else if (isDelete(op1) && isRetain(op2)) {
          if (-op1 > op2) {
            minl = op2;
            op1 = op1 + op2;
            op2 = ops2[i2++];
          } else if (-op1 === op2) {
            minl = op2;
            op1 = ops1[i1++];
            op2 = ops2[i2++];
          } else {
            minl = -op1;
            op2 = op2 + op1;
            op1 = ops1[i1++];
          }
          operation1prime['delete'](minl);
        } else if (isRetain(op1) && isDelete(op2)) {
          if (op1 > -op2) {
            minl = -op2;
            op1 = op1 + op2;
            op2 = ops2[i2++];
          } else if (op1 === -op2) {
            minl = op1;
            op1 = ops1[i1++];
            op2 = ops2[i2++];
          } else {
            minl = op1;
            op2 = op2 + op1;
            op1 = ops1[i1++];
          }
          operation2prime['delete'](minl);
        } else {
          throw new Error("The two operations aren't compatible");
        }
      }

      return [operation1prime, operation2prime];
    };

    return TextOperation;

  }());

  // Export for CommonJS
  if (typeof module === 'object') {
    module.exports = ot.TextOperation;
  }
  if (typeof ot === 'undefined') {
    // Export for browsers
    var ot = {};
  }

  ot.Selection = (function (global) {
    'use strict';

    var TextOperation = global.ot ? global.ot.TextOperation : ot ? ot.TextOperation : require('./text-operation');

    // Range has `anchor` and `head` properties, which are zero-based indices into
    // the document. The `anchor` is the side of the selection that stays fixed,
    // `head` is the side of the selection where the cursor is. When both are
    // equal, the range represents a cursor.
    function Range (anchor, head) {
      this.anchor = anchor;
      this.head = head;
    }

    Range.fromJSON = function (obj) {
      return new Range(obj.anchor, obj.head);
    };

    Range.prototype.equals = function (other) {
      return this.anchor === other.anchor && this.head === other.head;
    };

    Range.prototype.isEmpty = function () {
      return this.anchor === this.head;
    };

    Range.prototype.transform = function (other) {
      function transformIndex (index) {
        var newIndex = index;
        var ops = other.ops;
        for (var i = 0, l = other.ops.length; i < l; i++) {
          if (TextOperation.isRetain(ops[i])) {
            index -= ops[i];
          } else if (TextOperation.isInsert(ops[i])) {
            newIndex += ops[i].length;
          } else {
            newIndex -= Math.min(index, -ops[i]);
            index += ops[i];
          }
          if (index < 0) { break; }
        }
        return newIndex;
      }

      var newAnchor = transformIndex(this.anchor);
      if (this.anchor === this.head) {
        return new Range(newAnchor, newAnchor);
      }
      return new Range(newAnchor, transformIndex(this.head));
    };

    // A selection is basically an array of ranges. Every range represents a real
    // selection or a cursor in the document (when the start position equals the
    // end position of the range). The array must not be empty.
    function Selection (ranges) {
      this.ranges = ranges || [];
    }

    Selection.Range = Range;

    // Convenience method for creating selections only containing a single cursor
    // and no real selection range.
    Selection.createCursor = function (position) {
      return new Selection([new Range(position, position)]);
    };

    Selection.fromJSON = function (obj) {
      var objRanges = obj.ranges || obj;
      for (var i = 0, ranges = []; i < objRanges.length; i++) {
        ranges[i] = Range.fromJSON(objRanges[i]);
      }
      return new Selection(ranges);
    };

    Selection.prototype.equals = function (other) {
      if (this.position !== other.position) { return false; }
      if (this.ranges.length !== other.ranges.length) { return false; }
      // FIXME: Sort ranges before comparing them?
      for (var i = 0; i < this.ranges.length; i++) {
        if (!this.ranges[i].equals(other.ranges[i])) { return false; }
      }
      return true;
    };

    Selection.prototype.somethingSelected = function () {
      for (var i = 0; i < this.ranges.length; i++) {
        if (!this.ranges[i].isEmpty()) { return true; }
      }
      return false;
    };

    // Return the more current selection information.
    Selection.prototype.compose = function (other) {
      return other;
    };

    // Update the selection with respect to an operation.
    Selection.prototype.transform = function (other) {
      for (var i = 0, newRanges = []; i < this.ranges.length; i++) {
        newRanges[i] = this.ranges[i].transform(other);
      }
      return new Selection(newRanges);
    };

    return Selection;

  }(this));

  // Export for CommonJS
  if (typeof module === 'object') {
    module.exports = ot.Selection;
  }

  if (typeof ot === 'undefined') {
    // Export for browsers
    var ot = {};
  }

  ot.WrappedOperation = (function (global) {
    'use strict';

    // A WrappedOperation contains an operation and corresponing metadata.
    function WrappedOperation (operation, meta) {
      this.wrapped = operation;
      this.meta    = meta;
    }

    WrappedOperation.prototype.apply = function () {
      return this.wrapped.apply.apply(this.wrapped, arguments);
    };

    WrappedOperation.prototype.invert = function () {
      var meta = this.meta;
      return new WrappedOperation(
        this.wrapped.invert.apply(this.wrapped, arguments),
        meta && typeof meta === 'object' && typeof meta.invert === 'function' ?
          meta.invert.apply(meta, arguments) : meta
      );
    };

    // Copy all properties from source to target.
    function copy (source, target) {
      for (var key in source) {
        if (source.hasOwnProperty(key)) {
          target[key] = source[key];
        }
      }
    }

    function composeMeta (a, b) {
      if (a && typeof a === 'object') {
        if (typeof a.compose === 'function') { return a.compose(b); }
        var meta = {};
        copy(a, meta);
        copy(b, meta);
        return meta;
      }
      return b;
    }

    WrappedOperation.prototype.compose = function (other) {
      return new WrappedOperation(
        this.wrapped.compose(other.wrapped),
        composeMeta(this.meta, other.meta)
      );
    };

    function transformMeta (meta, operation) {
      if (meta && typeof meta === 'object') {
        if (typeof meta.transform === 'function') {
          return meta.transform(operation);
        }
      }
      return meta;
    }

    WrappedOperation.transform = function (a, b) {
      var transform = a.wrapped.constructor.transform;
      var pair = transform(a.wrapped, b.wrapped);
      return [
        new WrappedOperation(pair[0], transformMeta(a.meta, b.wrapped)),
        new WrappedOperation(pair[1], transformMeta(b.meta, a.wrapped))
      ];
    };

    return WrappedOperation;

  }(this));

  // Export for CommonJS
  if (typeof module === 'object') {
    module.exports = ot.WrappedOperation;
  }
  if (typeof ot === 'undefined') {
    // Export for browsers
    var ot = {};
  }

  ot.UndoManager = (function () {
    'use strict';

    var NORMAL_STATE = 'normal';
    var UNDOING_STATE = 'undoing';
    var REDOING_STATE = 'redoing';

    // Create a new UndoManager with an optional maximum history size.
    function UndoManager (maxItems) {
      this.maxItems  = maxItems || 50;
      this.state = NORMAL_STATE;
      this.dontCompose = false;
      this.undoStack = [];
      this.redoStack = [];
    }

    // Add an operation to the undo or redo stack, depending on the current state
    // of the UndoManager. The operation added must be the inverse of the last
    // edit. When `compose` is true, compose the operation with the last operation
    // unless the last operation was alread pushed on the redo stack or was hidden
    // by a newer operation on the undo stack.
    UndoManager.prototype.add = function (operation, compose) {
      if (this.state === UNDOING_STATE) {
        this.redoStack.push(operation);
        this.dontCompose = true;
      } else if (this.state === REDOING_STATE) {
        this.undoStack.push(operation);
        this.dontCompose = true;
      } else {
        var undoStack = this.undoStack;
        if (!this.dontCompose && compose && undoStack.length > 0) {
          undoStack.push(operation.compose(undoStack.pop()));
        } else {
          undoStack.push(operation);
          if (undoStack.length > this.maxItems) { undoStack.shift(); }
        }
        this.dontCompose = false;
        this.redoStack = [];
      }
    };

    function transformStack (stack, operation) {
      var newStack = [];
      var Operation = operation.constructor;
      for (var i = stack.length - 1; i >= 0; i--) {
        var pair = Operation.transform(stack[i], operation);
        if (typeof pair[0].isNoop !== 'function' || !pair[0].isNoop()) {
          newStack.push(pair[0]);
        }
        operation = pair[1];
      }
      return newStack.reverse();
    }

    // Transform the undo and redo stacks against a operation by another client.
    UndoManager.prototype.transform = function (operation) {
      this.undoStack = transformStack(this.undoStack, operation);
      this.redoStack = transformStack(this.redoStack, operation);
    };

    // Perform an undo by calling a function with the latest operation on the undo
    // stack. The function is expected to call the `add` method with the inverse
    // of the operation, which pushes the inverse on the redo stack.
    UndoManager.prototype.performUndo = function (fn) {
      this.state = UNDOING_STATE;
      if (this.undoStack.length === 0) { throw new Error("undo not possible"); }
      fn(this.undoStack.pop());
      this.state = NORMAL_STATE;
    };

    // The inverse of `performUndo`.
    UndoManager.prototype.performRedo = function (fn) {
      this.state = REDOING_STATE;
      if (this.redoStack.length === 0) { throw new Error("redo not possible"); }
      fn(this.redoStack.pop());
      this.state = NORMAL_STATE;
    };

    // Is the undo stack not empty?
    UndoManager.prototype.canUndo = function () {
      return this.undoStack.length !== 0;
    };

    // Is the redo stack not empty?
    UndoManager.prototype.canRedo = function () {
      return this.redoStack.length !== 0;
    };

    // Whether the UndoManager is currently performing an undo.
    UndoManager.prototype.isUndoing = function () {
      return this.state === UNDOING_STATE;
    };

    // Whether the UndoManager is currently performing a redo.
    UndoManager.prototype.isRedoing = function () {
      return this.state === REDOING_STATE;
    };

    return UndoManager;

  }());

  // Export for CommonJS
  if (typeof module === 'object') {
    module.exports = ot.UndoManager;
  }

  // translation of https://github.com/djspiewak/cccp/blob/master/agent/src/main/scala/com/codecommit/cccp/agent/state.scala

  if (typeof ot === 'undefined') {
    var ot = {};
  }

  ot.Client = (function (global) {
    'use strict';

    // Client constructor
    function Client (revision) {
      this.revision = revision; // the next expected revision number
      this.state = synchronized_; // start state
    }

    Client.prototype.setState = function (state) {
      this.state = state;
    };

    // Call this method when the user changes the document.
    Client.prototype.applyClient = function (operation) {
      this.setState(this.state.applyClient(this, operation));
    };

    // Call this method with a new operation from the server
    Client.prototype.applyServer = function (operation) {
      this.revision++;
      this.setState(this.state.applyServer(this, operation));
    };

    Client.prototype.serverAck = function () {
      this.revision++;
      this.setState(this.state.serverAck(this));
    };
    
    Client.prototype.serverReconnect = function () {
      if (typeof this.state.resend === 'function') { this.state.resend(this); }
    };

    // Transforms a selection from the latest known server state to the current
    // client state. For example, if we get from the server the information that
    // another user's cursor is at position 3, but the server hasn't yet received
    // our newest operation, an insertion of 5 characters at the beginning of the
    // document, the correct position of the other user's cursor in our current
    // document is 8.
    Client.prototype.transformSelection = function (selection) {
      return this.state.transformSelection(selection);
    };

    // Override this method.
    Client.prototype.sendOperation = function (revision, operation) {
      throw new Error("sendOperation must be defined in child class");
    };

    // Override this method.
    Client.prototype.applyOperation = function (operation) {
      throw new Error("applyOperation must be defined in child class");
    };


    // In the 'Synchronized' state, there is no pending operation that the client
    // has sent to the server.
    function Synchronized () {}
    Client.Synchronized = Synchronized;

    Synchronized.prototype.applyClient = function (client, operation) {
      // When the user makes an edit, send the operation to the server and
      // switch to the 'AwaitingConfirm' state
      client.sendOperation(client.revision, operation);
      return new AwaitingConfirm(operation);
    };

    Synchronized.prototype.applyServer = function (client, operation) {
      // When we receive a new operation from the server, the operation can be
      // simply applied to the current document
      client.applyOperation(operation);
      return this;
    };

    Synchronized.prototype.serverAck = function (client) {
      throw new Error("There is no pending operation.");
    };

    // Nothing to do because the latest server state and client state are the same.
    Synchronized.prototype.transformSelection = function (x) { return x; };

    // Singleton
    var synchronized_ = new Synchronized();


    // In the 'AwaitingConfirm' state, there's one operation the client has sent
    // to the server and is still waiting for an acknowledgement.
    function AwaitingConfirm (outstanding) {
      // Save the pending operation
      this.outstanding = outstanding;
    }
    Client.AwaitingConfirm = AwaitingConfirm;

    AwaitingConfirm.prototype.applyClient = function (client, operation) {
      // When the user makes an edit, don't send the operation immediately,
      // instead switch to 'AwaitingWithBuffer' state
      return new AwaitingWithBuffer(this.outstanding, operation);
    };

    AwaitingConfirm.prototype.applyServer = function (client, operation) {
      // This is another client's operation. Visualization:
      //
      //                   /\
      // this.outstanding /  \ operation
      //                 /    \
      //                 \    /
      //  pair[1]         \  / pair[0] (new outstanding)
      //  (can be applied  \/
      //  to the client's
      //  current document)
      var pair = operation.constructor.transform(this.outstanding, operation);
      client.applyOperation(pair[1]);
      return new AwaitingConfirm(pair[0]);
    };

    AwaitingConfirm.prototype.serverAck = function (client) {
      // The client's operation has been acknowledged
      // => switch to synchronized state
      return synchronized_;
    };

    AwaitingConfirm.prototype.transformSelection = function (selection) {
      return selection.transform(this.outstanding);
    };

    AwaitingConfirm.prototype.resend = function (client) {
      // The confirm didn't come because the client was disconnected.
      // Now that it has reconnected, we resend the outstanding operation.
      client.sendOperation(client.revision, this.outstanding);
    };


    // In the 'AwaitingWithBuffer' state, the client is waiting for an operation
    // to be acknowledged by the server while buffering the edits the user makes
    function AwaitingWithBuffer (outstanding, buffer) {
      // Save the pending operation and the user's edits since then
      this.outstanding = outstanding;
      this.buffer = buffer;
    }
    Client.AwaitingWithBuffer = AwaitingWithBuffer;

    AwaitingWithBuffer.prototype.applyClient = function (client, operation) {
      // Compose the user's changes onto the buffer
      var newBuffer = this.buffer.compose(operation);
      return new AwaitingWithBuffer(this.outstanding, newBuffer);
    };

    AwaitingWithBuffer.prototype.applyServer = function (client, operation) {
      // Operation comes from another client
      //
      //                       /\
      //     this.outstanding /  \ operation
      //                     /    \
      //                    /\    /
      //       this.buffer /  \* / pair1[0] (new outstanding)
      //                  /    \/
      //                  \    /
      //          pair2[1] \  / pair2[0] (new buffer)
      // the transformed    \/
      // operation -- can
      // be applied to the
      // client's current
      // document
      //
      // * pair1[1]
      var transform = operation.constructor.transform;
      var pair1 = transform(this.outstanding, operation);
      var pair2 = transform(this.buffer, pair1[1]);
      client.applyOperation(pair2[1]);
      return new AwaitingWithBuffer(pair1[0], pair2[0]);
    };

    AwaitingWithBuffer.prototype.serverAck = function (client) {
      // The pending operation has been acknowledged
      // => send buffer
      client.sendOperation(client.revision, this.buffer);
      return new AwaitingConfirm(this.buffer);
    };

    AwaitingWithBuffer.prototype.transformSelection = function (selection) {
      return selection.transform(this.outstanding).transform(this.buffer);
    };

    AwaitingWithBuffer.prototype.resend = function (client) {
      // The confirm didn't come because the client was disconnected.
      // Now that it has reconnected, we resend the outstanding operation.
      client.sendOperation(client.revision, this.outstanding);
    };


    return Client;

  }(this));

  if (typeof module === 'object') {
    module.exports = ot.Client;
  }

  /*global ot */

  ot.CodeMirrorAdapter = (function (global) {
    'use strict';

    var TextOperation = ot.TextOperation;
    var Selection = ot.Selection;

    function CodeMirrorAdapter (cm) {
      this.cm = cm;
      this.ignoreNextChange = false;
      this.changeInProgress = false;
      this.selectionChanged = false;

      bind(this, 'onChanges');
      bind(this, 'onChange');
      bind(this, 'onCursorActivity');
      bind(this, 'onFocus');
      bind(this, 'onBlur');

      cm.on('changes', this.onChanges);
      cm.on('change', this.onChange);
      cm.on('cursorActivity', this.onCursorActivity);
      cm.on('focus', this.onFocus);
      cm.on('blur', this.onBlur);
    }

    // Removes all event listeners from the CodeMirror instance.
    CodeMirrorAdapter.prototype.detach = function () {
      this.cm.off('changes', this.onChanges);
      this.cm.off('change', this.onChange);
      this.cm.off('cursorActivity', this.onCursorActivity);
      this.cm.off('focus', this.onFocus);
      this.cm.off('blur', this.onBlur);
    };

    function cmpPos (a, b) {
      if (a.line < b.line) { return -1; }
      if (a.line > b.line) { return 1; }
      if (a.ch < b.ch)     { return -1; }
      if (a.ch > b.ch)     { return 1; }
      return 0;
    }
    function posEq (a, b) { return cmpPos(a, b) === 0; }
    function posLe (a, b) { return cmpPos(a, b) <= 0; }

    function minPos (a, b) { return posLe(a, b) ? a : b; }
    function maxPos (a, b) { return posLe(a, b) ? b : a; }

    function codemirrorDocLength (doc) {
      return doc.indexFromPos({ line: doc.lastLine(), ch: 0 }) +
        doc.getLine(doc.lastLine()).length;
    }

    // Converts a CodeMirror change array (as obtained from the 'changes' event
    // in CodeMirror v4) or single change or linked list of changes (as returned
    // by the 'change' event in CodeMirror prior to version 4) into a
    // TextOperation and its inverse and returns them as a two-element array.
    CodeMirrorAdapter.operationFromCodeMirrorChanges = function (changes, doc) {
      // Approach: Replay the changes, beginning with the most recent one, and
      // construct the operation and its inverse. We have to convert the position
      // in the pre-change coordinate system to an index. We have a method to
      // convert a position in the coordinate system after all changes to an index,
      // namely CodeMirror's `indexFromPos` method. We can use the information of
      // a single change object to convert a post-change coordinate system to a
      // pre-change coordinate system. We can now proceed inductively to get a
      // pre-change coordinate system for all changes in the linked list.
      // A disadvantage of this approach is its complexity `O(n^2)` in the length
      // of the linked list of changes.

      var docEndLength = codemirrorDocLength(doc);
      var operation    = new TextOperation().retain(docEndLength);
      var inverse      = new TextOperation().retain(docEndLength);

      var indexFromPos = function (pos) {
        return doc.indexFromPos(pos);
      };

      function last (arr) { return arr[arr.length - 1]; }

      function sumLengths (strArr) {
        if (strArr.length === 0) { return 0; }
        var sum = 0;
        for (var i = 0; i < strArr.length; i++) { sum += strArr[i].length; }
        return sum + strArr.length - 1;
      }

      function updateIndexFromPos (indexFromPos, change) {
        return function (pos) {
          if (posLe(pos, change.from)) { return indexFromPos(pos); }
          if (posLe(change.to, pos)) {
            return indexFromPos({
              line: pos.line + change.text.length - 1 - (change.to.line - change.from.line),
              ch: (change.to.line < pos.line) ?
                pos.ch :
                (change.text.length <= 1) ?
                  pos.ch - (change.to.ch - change.from.ch) + sumLengths(change.text) :
                  pos.ch - change.to.ch + last(change.text).length
            }) + sumLengths(change.removed) - sumLengths(change.text);
          }
          if (change.from.line === pos.line) {
            return indexFromPos(change.from) + pos.ch - change.from.ch;
          }
          return indexFromPos(change.from) +
            sumLengths(change.removed.slice(0, pos.line - change.from.line)) +
            1 + pos.ch;
        };
      }

      for (var i = changes.length - 1; i >= 0; i--) {
        var change = changes[i];
        indexFromPos = updateIndexFromPos(indexFromPos, change);

        var fromIndex = indexFromPos(change.from);
        var restLength = docEndLength - fromIndex - sumLengths(change.text);

        operation = new TextOperation()
          .retain(fromIndex)
          ['delete'](sumLengths(change.removed))
          .insert(change.text.join('\n'))
          .retain(restLength)
          .compose(operation);

        inverse = inverse.compose(new TextOperation()
          .retain(fromIndex)
          ['delete'](sumLengths(change.text))
          .insert(change.removed.join('\n'))
          .retain(restLength)
        );

        docEndLength += sumLengths(change.removed) - sumLengths(change.text);
      }

      return [operation, inverse];
    };

    // Singular form for backwards compatibility.
    CodeMirrorAdapter.operationFromCodeMirrorChange =
      CodeMirrorAdapter.operationFromCodeMirrorChanges;

    // Apply an operation to a CodeMirror instance.
    CodeMirrorAdapter.applyOperationToCodeMirror = function (operation, cm) {
      cm.operation(function () {
        var ops = operation.ops;
        var index = 0; // holds the current index into CodeMirror's content
        for (var i = 0, l = ops.length; i < l; i++) {
          var op = ops[i];
          if (TextOperation.isRetain(op)) {
            index += op;
          } else if (TextOperation.isInsert(op)) {
            cm.replaceRange(op, cm.posFromIndex(index));
            index += op.length;
          } else if (TextOperation.isDelete(op)) {
            var from = cm.posFromIndex(index);
            var to   = cm.posFromIndex(index - op);
            cm.replaceRange('', from, to);
          }
        }
      });
    };

    CodeMirrorAdapter.prototype.registerCallbacks = function (cb) {
      this.callbacks = cb;
    };

    CodeMirrorAdapter.prototype.onChange = function () {
      // By default, CodeMirror's event order is the following:
      // 1. 'change', 2. 'cursorActivity', 3. 'changes'.
      // We want to fire the 'selectionChange' event after the 'change' event,
      // but need the information from the 'changes' event. Therefore, we detect
      // when a change is in progress by listening to the change event, setting
      // a flag that makes this adapter defer all 'cursorActivity' events.
      this.changeInProgress = true;
    };

    CodeMirrorAdapter.prototype.onChanges = function (_, changes) {
      if (!this.ignoreNextChange) {
        var pair = CodeMirrorAdapter.operationFromCodeMirrorChanges(changes, this.cm);
        this.trigger('change', pair[0], pair[1]);
      }
      if (this.selectionChanged) { this.trigger('selectionChange'); }
      this.changeInProgress = false;
      this.ignoreNextChange = false;
    };

    CodeMirrorAdapter.prototype.onCursorActivity =
    CodeMirrorAdapter.prototype.onFocus = function () {
      if (this.changeInProgress) {
        this.selectionChanged = true;
      } else {
        this.trigger('selectionChange');
      }
    };

    CodeMirrorAdapter.prototype.onBlur = function () {
      if (!this.cm.somethingSelected()) { this.trigger('blur'); }
    };

    CodeMirrorAdapter.prototype.getValue = function () {
      return this.cm.getValue();
    };

    CodeMirrorAdapter.prototype.getSelection = function () {
      var cm = this.cm;

      var selectionList = cm.listSelections();
      var ranges = [];
      for (var i = 0; i < selectionList.length; i++) {
        ranges[i] = new Selection.Range(
          cm.indexFromPos(selectionList[i].anchor),
          cm.indexFromPos(selectionList[i].head)
        );
      }

      return new Selection(ranges);
    };

    CodeMirrorAdapter.prototype.setSelection = function (selection) {
      var ranges = [];
      for (var i = 0; i < selection.ranges.length; i++) {
        var range = selection.ranges[i];
        ranges[i] = {
          anchor: this.cm.posFromIndex(range.anchor),
          head:   this.cm.posFromIndex(range.head)
        };
      }
      this.cm.setSelections(ranges);
    };

    var addStyleRule = (function () {
      var added = {};
      var styleElement = document.createElement('style');
      document.documentElement.getElementsByTagName('head')[0].appendChild(styleElement);
      var styleSheet = styleElement.sheet;

      return function (css) {
        if (added[css]) { return; }
        added[css] = true;
        styleSheet.insertRule(css, (styleSheet.cssRules || styleSheet.rules).length);
      };
    }());

    CodeMirrorAdapter.prototype.setOtherCursor = function (position, color, clientId) {
      var cursorPos = this.cm.posFromIndex(position);
      var cursorCoords = this.cm.cursorCoords(cursorPos);
      var cursorEl = document.createElement('span');
      cursorEl.className = 'other-client';
      cursorEl.style.display = 'inline-block';
      cursorEl.style.padding = '0';
      cursorEl.style.marginLeft = cursorEl.style.marginRight = '-1px';
      cursorEl.style.borderLeftWidth = '2px';
      cursorEl.style.borderLeftStyle = 'solid';
      cursorEl.style.borderLeftColor = color;
      cursorEl.style.height = (cursorCoords.bottom - cursorCoords.top) * 0.9 + 'px';
      cursorEl.style.zIndex = 0;
      cursorEl.setAttribute('data-clientid', clientId);
      return this.cm.setBookmark(cursorPos, { widget: cursorEl, insertLeft: true });
    };

    CodeMirrorAdapter.prototype.setOtherSelectionRange = function (range, color, clientId) {
      var match = /^#([0-9a-fA-F]{6})$/.exec(color);
      if (!match) { throw new Error("only six-digit hex colors are allowed."); }
      var selectionClassName = 'selection-' + match[1];
      var rule = '.' + selectionClassName + ' { background: ' + color + '; }';
      addStyleRule(rule);

      var anchorPos = this.cm.posFromIndex(range.anchor);
      var headPos   = this.cm.posFromIndex(range.head);

      return this.cm.markText(
        minPos(anchorPos, headPos),
        maxPos(anchorPos, headPos),
        { className: selectionClassName }
      );
    };

    CodeMirrorAdapter.prototype.setOtherSelection = function (selection, color, clientId) {
      var selectionObjects = [];
      for (var i = 0; i < selection.ranges.length; i++) {
        var range = selection.ranges[i];
        if (range.isEmpty()) {
          selectionObjects[i] = this.setOtherCursor(range.head, color, clientId);
        } else {
          selectionObjects[i] = this.setOtherSelectionRange(range, color, clientId);
        }
      }
      return {
        clear: function () {
          for (var i = 0; i < selectionObjects.length; i++) {
            selectionObjects[i].clear();
          }
        }
      };
    };

    CodeMirrorAdapter.prototype.trigger = function (event) {
      var args = Array.prototype.slice.call(arguments, 1);
      var action = this.callbacks && this.callbacks[event];
      if (action) { action.apply(this, args); }
    };

    CodeMirrorAdapter.prototype.applyOperation = function (operation) {
      this.ignoreNextChange = true;
      CodeMirrorAdapter.applyOperationToCodeMirror(operation, this.cm);
    };

    CodeMirrorAdapter.prototype.registerUndo = function (undoFn) {
      this.cm.undo = undoFn;
    };

    CodeMirrorAdapter.prototype.registerRedo = function (redoFn) {
      this.cm.redo = redoFn;
    };

    // Throws an error if the first argument is falsy. Useful for debugging.
    function assert (b, msg) {
      if (!b) {
        throw new Error(msg || "assertion error");
      }
    }

    // Bind a method to an object, so it doesn't matter whether you call
    // object.method() directly or pass object.method as a reference to another
    // function.
    function bind (obj, method) {
      var fn = obj[method];
      obj[method] = function () {
        fn.apply(obj, arguments);
      };
    }

    return CodeMirrorAdapter;

  }(this));

  /*global ot */

  ot.SocketIOAdapter = (function () {
    'use strict';

    function SocketIOAdapter (socket) {
      this.socket = socket;

      var self = this;
      socket
        .on('client_left', function (clientId) {
          self.trigger('client_left', clientId);
        })
        .on('set_name', function (clientId, name) {
          self.trigger('set_name', clientId, name);
        })
        .on('ack', function () { self.trigger('ack'); })
        .on('operation', function (clientId, operation, selection) {
          self.trigger('operation', operation);
          self.trigger('selection', clientId, selection);
        })
        .on('selection', function (clientId, selection) {
          self.trigger('selection', clientId, selection);
        })
        .on('reconnect', function () {
          self.trigger('reconnect');
        });
    }

    SocketIOAdapter.prototype.sendOperation = function (revision, operation, selection) {
      this.socket.emit('operation', revision, operation, selection);
    };

    SocketIOAdapter.prototype.sendSelection = function (selection) {
      this.socket.emit('selection', selection);
    };

    SocketIOAdapter.prototype.registerCallbacks = function (cb) {
      this.callbacks = cb;
    };

    SocketIOAdapter.prototype.trigger = function (event) {
      var args = Array.prototype.slice.call(arguments, 1);
      var action = this.callbacks && this.callbacks[event];
      if (action) { action.apply(this, args); }
    };

    return SocketIOAdapter;

  }());
  /*global ot, $ */

  ot.AjaxAdapter = (function () {
    'use strict';

    function AjaxAdapter (path, ownUserName, revision) {
      if (path[path.length - 1] !== '/') { path += '/'; }
      this.path = path;
      this.ownUserName = ownUserName;
      this.majorRevision = revision.major || 0;
      this.minorRevision = revision.minor || 0;
      this.poll();
    }

    AjaxAdapter.prototype.renderRevisionPath = function () {
      return 'revision/' + this.majorRevision + '-' + this.minorRevision;
    };

    AjaxAdapter.prototype.handleResponse = function (data) {
      var i;
      var operations = data.operations;
      for (i = 0; i < operations.length; i++) {
        if (operations[i].user === this.ownUserName) {
          this.trigger('ack');
        } else {
          this.trigger('operation', operations[i].operation);
        }
      }
      if (operations.length > 0) {
        this.majorRevision += operations.length;
        this.minorRevision = 0;
      }

      var events = data.events;
      if (events) {
        for (i = 0; i < events.length; i++) {
          var user = events[i].user;
          if (user === this.ownUserName) { continue; }
          switch (events[i].event) {
            case 'joined':    this.trigger('set_name', user, user); break;
            case 'left':      this.trigger('client_left', user); break;
            case 'selection': this.trigger('selection', user, events[i].selection); break;
          }
        }
        this.minorRevision += events.length;
      }

      var users = data.users;
      if (users) {
        delete users[this.ownUserName];
        this.trigger('clients', users);
      }

      if (data.revision) {
        this.majorRevision = data.revision.major;
        this.minorRevision = data.revision.minor;
      }
    };

    AjaxAdapter.prototype.poll = function () {
      var self = this;
      $.ajax({
        url: this.path + this.renderRevisionPath(),
        type: 'GET',
        dataType: 'json',
        timeout: 5000,
        success: function (data) {
          self.handleResponse(data);
          self.poll();
        },
        error: function () {
          setTimeout(function () { self.poll(); }, 500);
        }
      });
    };

    AjaxAdapter.prototype.sendOperation = function (revision, operation, selection) {
      if (revision !== this.majorRevision) { throw new Error("Revision numbers out of sync"); }
      var self = this;
      $.ajax({
        url: this.path + this.renderRevisionPath(),
        type: 'POST',
        data: JSON.stringify({ operation: operation, selection: selection }),
        contentType: 'application/json',
        processData: false,
        success: function (data) {},
        error: function () {
          setTimeout(function () { self.sendOperation(revision, operation, selection); }, 500);
        }
      });
    };

    AjaxAdapter.prototype.sendSelection = function (obj) {
      $.ajax({
        url: this.path + this.renderRevisionPath() + '/selection',
        type: 'POST',
        data: JSON.stringify(obj),
        contentType: 'application/json',
        processData: false,
        timeout: 1000
      });
    };

    AjaxAdapter.prototype.registerCallbacks = function (cb) {
      this.callbacks = cb;
    };

    AjaxAdapter.prototype.trigger = function (event) {
      var args = Array.prototype.slice.call(arguments, 1);
      var action = this.callbacks && this.callbacks[event];
      if (action) { action.apply(this, args); }
    };

    return AjaxAdapter;

  })();
  /*global ot */

  ot.EditorClient = (function () {
    'use strict';

    var Client = ot.Client;
    var Selection = ot.Selection;
    var UndoManager = ot.UndoManager;
    var TextOperation = ot.TextOperation;
    var WrappedOperation = ot.WrappedOperation;


    function SelfMeta (selectionBefore, selectionAfter) {
      this.selectionBefore = selectionBefore;
      this.selectionAfter  = selectionAfter;
    }

    SelfMeta.prototype.invert = function () {
      return new SelfMeta(this.selectionAfter, this.selectionBefore);
    };

    SelfMeta.prototype.compose = function (other) {
      return new SelfMeta(this.selectionBefore, other.selectionAfter);
    };

    SelfMeta.prototype.transform = function (operation) {
      return new SelfMeta(
        this.selectionBefore.transform(operation),
        this.selectionAfter.transform(operation)
      );
    };


    function OtherMeta (clientId, selection) {
      this.clientId  = clientId;
      this.selection = selection;
    }

    OtherMeta.fromJSON = function (obj) {
      return new OtherMeta(
        obj.clientId,
        obj.selection && Selection.fromJSON(obj.selection)
      );
    };

    OtherMeta.prototype.transform = function (operation) {
      return new OtherMeta(
        this.clientId,
        this.selection && this.selection.transform(operation)
      );
    };


    function OtherClient (id, listEl, editorAdapter, name, selection) {
      this.id = id;
      this.listEl = listEl;
      this.editorAdapter = editorAdapter;
      this.name = name;

      this.li = document.createElement('li');
      if (name) {
        this.li.textContent = name;
        this.listEl.appendChild(this.li);
      }

      this.setColor(name ? hueFromName(name) : Math.random());
      if (selection) { this.updateSelection(selection); }
    }

    OtherClient.prototype.setColor = function (hue) {
      this.hue = hue;
      this.color = hsl2hex(hue, 0.75, 0.5);
      this.lightColor = hsl2hex(hue, 0.5, 0.9);
      if (this.li) { this.li.style.color = this.color; }
    };

    OtherClient.prototype.setName = function (name) {
      if (this.name === name) { return; }
      this.name = name;

      this.li.textContent = name;
      if (!this.li.parentNode) {
        this.listEl.appendChild(this.li);
      }

      this.setColor(hueFromName(name));
    };

    OtherClient.prototype.updateSelection = function (selection) {
      this.removeSelection();
      this.selection = selection;
      this.mark = this.editorAdapter.setOtherSelection(
        selection,
        selection.position === selection.selectionEnd ? this.color : this.lightColor,
        this.id
      );
    };

    OtherClient.prototype.remove = function () {
      if (this.li) { removeElement(this.li); }
      this.removeSelection();
    };

    OtherClient.prototype.removeSelection = function () {
      if (this.mark) {
        this.mark.clear();
        this.mark = null;
      }
    };


    function EditorClient (revision, clients, serverAdapter, editorAdapter) {
      Client.call(this, revision);
      this.serverAdapter = serverAdapter;
      this.editorAdapter = editorAdapter;
      this.undoManager = new UndoManager();
  
      this.initializeClientList();
      this.initializeClients(clients);

      var self = this;

      this.editorAdapter.registerCallbacks({
        change: function (operation, inverse) { self.onChange(operation, inverse); },
        selectionChange: function () { self.onSelectionChange(); },
        blur: function () { self.onBlur(); }
      });
      this.editorAdapter.registerUndo(function () { self.undo(); });
      this.editorAdapter.registerRedo(function () { self.redo(); });

      this.serverAdapter.registerCallbacks({
        client_left: function (clientId) { self.onClientLeft(clientId); },
        set_name: function (clientId, name) { self.getClientObject(clientId).setName(name); },
        ack: function () { self.serverAck(); },
        operation: function (operation) {
          self.applyServer(TextOperation.fromJSON(operation));
        },
        selection: function (clientId, selection) {
          if (selection) {
            self.getClientObject(clientId).updateSelection(
                self.transformSelection(Selection.fromJSON(selection))
            );
          } else {
            self.getClientObject(clientId).removeSelection();
          }
        },
        clients: function (clients) {
          var clientId;
          for (clientId in self.clients) {
            if (self.clients.hasOwnProperty(clientId) && !clients.hasOwnProperty(clientId)) {
              self.onClientLeft(clientId);
            }
          }

          for (clientId in clients) {
            if (clients.hasOwnProperty(clientId)) {
              var clientObject = self.getClientObject(clientId);

              if (clients[clientId].name) {
                clientObject.setName(clients[clientId].name);
              }

              var selection = clients[clientId].selection;
              if (selection) {
                self.clients[clientId].updateSelection(
                    self.transformSelection(Selection.fromJSON(selection))
                );
              } else {
                self.clients[clientId].removeSelection();
              }
            }
          }
        },
        reconnect: function () { self.serverReconnect(); }
      });
    }

    inherit(EditorClient, Client);

    EditorClient.prototype.addClient = function (clientId, clientObj) {
      this.clients[clientId] = new OtherClient(
        clientId,
        this.clientListEl,
        this.editorAdapter,
        clientObj.name || clientId,
        clientObj.selection ? Selection.fromJSON(clientObj.selection) : null
      );
    };

    EditorClient.prototype.initializeClients = function (clients) {
      this.clients = {};
      for (var clientId in clients) {
        if (clients.hasOwnProperty(clientId)) {
          this.addClient(clientId, clients[clientId]);
        }
      }
    };

    EditorClient.prototype.getClientObject = function (clientId) {
      var client = this.clients[clientId];
      if (client) { return client; }
      return this.clients[clientId] = new OtherClient(
        clientId,
        this.clientListEl,
        this.editorAdapter
      );
    };

    EditorClient.prototype.onClientLeft = function (clientId) {
      console.log("User disconnected: " + clientId);
      var client = this.clients[clientId];
      if (!client) { return; }
      client.remove();
      delete this.clients[clientId];
    };

    EditorClient.prototype.initializeClientList = function () {
      this.clientListEl = document.createElement('ul');
    };

    EditorClient.prototype.applyUnredo = function (operation) {
      this.undoManager.add(operation.invert(this.editorAdapter.getValue()));
      this.editorAdapter.applyOperation(operation.wrapped);
      this.selection = operation.meta.selectionAfter;
      this.editorAdapter.setSelection(this.selection);
      this.applyClient(operation.wrapped);
    };

    EditorClient.prototype.undo = function () {
      var self = this;
      if (!this.undoManager.canUndo()) { return; }
      this.undoManager.performUndo(function (o) { self.applyUnredo(o); });
    };

    EditorClient.prototype.redo = function () {
      var self = this;
      if (!this.undoManager.canRedo()) { return; }
      this.undoManager.performRedo(function (o) { self.applyUnredo(o); });
    };

    EditorClient.prototype.onChange = function (textOperation, inverse) {
      var selectionBefore = this.selection;
      this.updateSelection();
      var meta = new SelfMeta(selectionBefore, this.selection);
      var operation = new WrappedOperation(textOperation, meta);

      var compose = this.undoManager.undoStack.length > 0 &&
        inverse.shouldBeComposedWithInverted(last(this.undoManager.undoStack).wrapped);
      var inverseMeta = new SelfMeta(this.selection, selectionBefore);
      this.undoManager.add(new WrappedOperation(inverse, inverseMeta), compose);
      this.applyClient(textOperation);
    };

    EditorClient.prototype.updateSelection = function () {
      this.selection = this.editorAdapter.getSelection();
    };

    EditorClient.prototype.onSelectionChange = function () {
      var oldSelection = this.selection;
      this.updateSelection();
      if (oldSelection && this.selection.equals(oldSelection)) { return; }
      this.sendSelection(this.selection);
    };

    EditorClient.prototype.onBlur = function () {
      this.selection = null;
      this.sendSelection(null);
    };

    EditorClient.prototype.sendSelection = function (selection) {
      if (this.state instanceof Client.AwaitingWithBuffer) { return; }
      this.serverAdapter.sendSelection(selection);
    };

    EditorClient.prototype.sendOperation = function (revision, operation) {
      this.serverAdapter.sendOperation(revision, operation.toJSON(), this.selection);
    };

    EditorClient.prototype.applyOperation = function (operation) {
      this.editorAdapter.applyOperation(operation);
      this.updateSelection();
      this.undoManager.transform(new WrappedOperation(operation, null));
    };

    function rgb2hex (r, g, b) {
      function digits (n) {
        var m = Math.round(255*n).toString(16);
        return m.length === 1 ? '0'+m : m;
      }
      return '#' + digits(r) + digits(g) + digits(b);
    }

    function hsl2hex (h, s, l) {
      if (s === 0) { return rgb2hex(l, l, l); }
      var var2 = l < 0.5 ? l * (1+s) : (l+s) - (s*l);
      var var1 = 2 * l - var2;
      var hue2rgb = function (hue) {
        if (hue < 0) { hue += 1; }
        if (hue > 1) { hue -= 1; }
        if (6*hue < 1) { return var1 + (var2-var1)*6*hue; }
        if (2*hue < 1) { return var2; }
        if (3*hue < 2) { return var1 + (var2-var1)*6*(2/3 - hue); }
        return var1;
      };
      return rgb2hex(hue2rgb(h+1/3), hue2rgb(h), hue2rgb(h-1/3));
    }

    function hueFromName (name) {
      var a = 1;
      for (var i = 0; i < name.length; i++) {
        a = 17 * (a+name.charCodeAt(i)) % 360;
      }
      return a/360;
    }

    // Set Const.prototype.__proto__ to Super.prototype
    function inherit (Const, Super) {
      function F () {}
      F.prototype = Super.prototype;
      Const.prototype = new F();
      Const.prototype.constructor = Const;
    }

    function last (arr) { return arr[arr.length - 1]; }

    // Remove an element from the DOM.
    function removeElement (el) {
      if (el.parentNode) {
        el.parentNode.removeChild(el);
      }
    }

    return EditorClient;
  }());

  return ot;
});

/**
 * @license RequireJS i18n 2.0.6 Copyright (c) 2010-2012, The Dojo Foundation All Rights Reserved.
 * Available via the MIT or new BSD license.
 * see: http://github.com/requirejs/i18n for details
 */
/*jslint regexp: true */
/*global require: false, navigator: false, define: false */

/**
 * This plugin handles i18n! prefixed modules. It does the following:
 *
 * 1) A regular module can have a dependency on an i18n bundle, but the regular
 * module does not want to specify what locale to load. So it just specifies
 * the top-level bundle, like 'i18n!nls/colors'.
 *
 * This plugin will load the i18n bundle at nls/colors, see that it is a root/master
 * bundle since it does not have a locale in its name. It will then try to find
 * the best match locale available in that master bundle, then request all the
 * locale pieces for that best match locale. For instance, if the locale is 'en-us',
 * then the plugin will ask for the 'en-us', 'en' and 'root' bundles to be loaded
 * (but only if they are specified on the master bundle).
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/colors bundle to be that mixed in locale.
 *
 * 2) A regular module specifies a specific locale to load. For instance,
 * i18n!nls/fr-fr/colors. In this case, the plugin needs to load the master bundle
 * first, at nls/colors, then figure out what the best match locale is for fr-fr,
 * since maybe only fr or just root is defined for that locale. Once that best
 * fit is found, all of its locale pieces need to have their bundles loaded.
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/fr-fr/colors bundle to be that mixed in locale.
 */
(function () {
    'use strict';

    //regexp for reconstructing the master bundle name from parts of the regexp match
    //nlsRegExp.exec('foo/bar/baz/nls/en-ca/foo') gives:
    //['foo/bar/baz/nls/en-ca/foo', 'foo/bar/baz/nls/', '/', '/', 'en-ca', 'foo']
    //nlsRegExp.exec('foo/bar/baz/nls/foo') gives:
    //['foo/bar/baz/nls/foo', 'foo/bar/baz/nls/', '/', '/', 'foo', '']
    //so, if match[5] is blank, it means this is the top bundle definition.
    var nlsRegExp = /(^.*(^|\/)nls(\/|$))([^\/]*)\/?([^\/]*)/;

    //Helper function to avoid repeating code. Lots of arguments in the
    //desire to stay functional and support RequireJS contexts without having
    //to know about the RequireJS contexts.
    function addPart(locale, master, needed, toLoad, prefix, suffix) {
        if (master[locale]) {
            needed.push(locale);
            if (master[locale] === true || master[locale] === 1) {
                toLoad.push(prefix + locale + '/' + suffix);
            }
        }
    }

    function addIfExists(req, locale, toLoad, prefix, suffix) {
        var fullName = prefix + locale + '/' + suffix;
        if (require._fileExists(req.toUrl(fullName + '.js'))) {
            toLoad.push(fullName);
        }
    }

    /**
     * Simple function to mix in properties from source into target,
     * but only if target does not already have a property of the same name.
     * This is not robust in IE for transferring methods that match
     * Object.prototype names, but the uses of mixin here seem unlikely to
     * trigger a problem related to that.
     */
    function mixin(target, source, force) {
        var prop;
        for (prop in source) {
            if (source.hasOwnProperty(prop) && (!target.hasOwnProperty(prop) || force)) {
                target[prop] = source[prop];
            } else if (typeof source[prop] === 'object') {
                if (!target[prop] && source[prop]) {
                    target[prop] = {};
                }
                mixin(target[prop], source[prop], force);
            }
        }
    }

    define('i18n',['module'], function (module) {
        var masterConfig = module.config ? module.config() : {};
        masterConfig = masterConfig || {};

        return {
            version: '2.0.6',
            /**
             * Called when a dependency needs to be loaded.
             */
            load: function (name, req, onLoad, config) {
                config = config || {};

                if (config.locale) {
                    masterConfig.locale = config.locale;
                }

                var masterName,
                    match = nlsRegExp.exec(name),
                    prefix = match[1],
                    locale = match[4],
                    suffix = match[5],
                    parts = locale.split('-'),
                    toLoad = [],
                    value = {},
                    i, part, current = '';

                //If match[5] is blank, it means this is the top bundle definition,
                //so it does not have to be handled. Locale-specific requests
                //will have a match[4] value but no match[5]
                if (match[5]) {
                    //locale-specific bundle
                    prefix = match[1];
                    masterName = prefix + suffix;
                } else {
                    //Top-level bundle.
                    masterName = name;
                    suffix = match[4];
                    locale = masterConfig.locale;
                    if (!locale) {
                        locale = masterConfig.locale =
                            typeof navigator === 'undefined' ? 'root' :
                            ((navigator.languages && navigator.languages[0]) ||
                             navigator.language ||
                             navigator.userLanguage || 'root').toLowerCase();
                    }
                    parts = locale.split('-');
                }

                if (config.isBuild) {
                    //Check for existence of all locale possible files and
                    //require them if exist.
                    toLoad.push(masterName);
                    addIfExists(req, 'root', toLoad, prefix, suffix);
                    for (i = 0; i < parts.length; i++) {
                        part = parts[i];
                        current += (current ? '-' : '') + part;
                        addIfExists(req, current, toLoad, prefix, suffix);
                    }

                    if(config.locales) {
                    	var j, k; 
                    	for (j = 0; j < config.locales.length; j++) {
                    		locale = config.locales[j];
                    		parts = locale.split("-");
                    		current = "";
	                    	for (k = 0; k < parts.length; k++) {
		                        part = parts[k];
		                        current += (current ? "-" : "") + part;
		                        addIfExists(req, current, toLoad, prefix, suffix);
	                    	}
                    	}
                    }

                    req(toLoad, function () {
                        onLoad();
                    });
                } else {
                    //First, fetch the master bundle, it knows what locales are available.
                    req([masterName], function (master) {
                        //Figure out the best fit
                        var needed = [],
                            part;

                        //Always allow for root, then do the rest of the locale parts.
                        addPart('root', master, needed, toLoad, prefix, suffix);
                        for (i = 0; i < parts.length; i++) {
                            part = parts[i];
                            current += (current ? '-' : '') + part;
                            addPart(current, master, needed, toLoad, prefix, suffix);
                        }

                        //Load all the parts missing.
                        req(toLoad, function () {
                            var i, partBundle, part;
                            for (i = needed.length - 1; i > -1 && needed[i]; i--) {
                                part = needed[i];
                                partBundle = master[part];
                                if (partBundle === true || partBundle === 1) {
                                    partBundle = req(prefix + part + '/' + suffix);
                                }
                                mixin(value, partBundle);
                            }

                            //All done, notify the loader.
                            onLoad(value);
                        });
                    });
                }
            }
        };
    });
}());

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/nls/messages',["module"],function(module){
    var config = module.config();
    return config && config.root ? config : {root:true};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2012, 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/nls/root/messages',{//Default message bundle
	"Navigator": "Navigator",
	"Sites": "Sites",
	"Shell": "Shell",
	"Get Plugins": "Get Plug-ins",
	"Global": "Global",
	"Editor": "Editor",
	"EditorRelatedLink": "Show Current Folder",
	"EditorRelatedLinkParent": "Show Enclosing Folder",
	"EditorLinkWorkspace": "Edit",
	"EditorRelatedLinkProj": "Show Project",
	"sidebar": "Sidebar",
	"toolbar": "Toolbar",
	"Filter bindings:": "Filter bindings:",
	"Filter bindings": "Type characters to filter by name or key combination",
	"BindingPrompt": "Enter the new binding",
	"NoBinding": "---",
	"orionClientLabel": "Orion client repository",
	"Orion Editor": "Text Editor",
	"Orion Image Viewer": "Image Viewer",
	"Orion Markdown Editor": "Markdown Editor",
	"Orion Markdown Viewer": "Markdown Viewer",
	"Orion JSON Editor": "JSON Editor",
	"Orion System Editor": "System Editor",
	"View on Site": "View on Site",
	"View this file or folder on a web site hosted by Orion": "View this file or folder on a web site hosted by Orion.",
	"ShowAllKeyBindings": "Show a list of all the keybindings on this page",
	"Show Keys": "Show Keys",
	"HideShowBannerFooter": "Hide or show the page banner and footer",
	"Toggle Banner and Footer": "Toggle Banner and Footer",
	"ChooseFileOpenEditor": "Choose a file by name and open an editor on it",
	"FindFile": "Open File...",
	"System Configuration Details": "System Configuration Details",
	"System Config Tooltip": "Go to the System Configuration Details page",
	"System Editor Tooltip": "Open this file type in the desktop's default manner",
	"Background Operations": "Background Operations",
	"Background Operations Tooltip": "Go to the Background Operations page",
	"Operation status is unknown": "Operation status is unknown",
	"Unknown item": "Unknown item",
	"NoSearchAvailableErr": "Can't search: no search service is available",
	"Related": "Related",
	"Options": "Options",
	"LOG: ": "LOG: ",
	"View": "View",
	"SplashTitle": "Setting up Workspace",
	"SplashTitleSettings": "Loading Settings",
	"SplashTitleGit": "Loading Git Repositories",
	"LoadingPage": "Loading Page",
	"LoadingPlugins": "Loading Plugins",
	"AuthenticatingUser": "Authenticating user...",
	"AuthenticatedUser": "Authenticated user",
	"LoadingResources": "Loading Resources",
	"plugin_started": "\"${0}\" started",
	"plugin_lazy activation": "\"${0}\" lazily activated",
	"plugin_starting": "\"${0}\" starting",
	"no parent": "no parent",
	"no tree model": "no tree model",
	"no renderer": "no renderer",
	"could not find table row ": "could not find table row ",
	"Operations": "Operations",
	"Operations running": "Operations running",
	"SomeOpWarning": "Some operations finished with warning",
	"SomeOpErr": "Some operations finished with error",
	"no service registry": "no service registry",
	"Tasks": "Tasks",
	"Close": "Close",
	"Expand all": "Expand all",
	"Collapse all": "Collapse all",
	"Search" : "Search",
	"Advanced search" : "Advanced search",
	"Submit" : "Submit",
	"More" : "More",
	"Recent searches" : "Recent searches",
	"Regular expression" : "Regular expression",
	"Search options" : "Search options",
	"Global search" : "Global search",
	"Orion Home" : "Home",
	"Close notification" : "Close notification",
	"Toggle Side Panel" : "Toggle Side Panel",
	"Open or close the side panel": "Open or close the side panel",
	"Projects" : "Projects",
	"Toggle Sidebar" : "Toggle Sidebar",
	"Sample HTML5 Site": "Sample HTML5 Site",
	"Generate an HTML5 'Hello World' website, including JavaScript, HTML, and CSS files.": "Generate an HTML5 'Hello World' website, including JavaScript, HTML, and CSS files.",
	"Sample Orion Plugin": "Sample Orion Plug-in",
	"Generate a sample plugin for integrating with Orion.": "Generate a sample plug-in for integrating with Orion.",
	"Browser": "Web Browser",
	"OutlineProgress": "Getting outline for ${0} from ${1}",
	"FormatProgress" : "Formatting ${0} from ${1}",
	"outlineTimeout": "Outline service timed out. Try reloading the page and opening the outline again.",
	"UnknownError": "An unknown error occurred.",
	"Filter outline:": "Filter outline:",
	"Filter": "Type characters (* = any string, ? = any character)",
	"TemplateExplorerLabel": "Templates",
	"OpenTemplateExplorer": "Open Template Explorer",
	"Edit": "Edit",
	"CentralNavTooltip": "Toggle Navigation Menu",
	"Wrote: ${0}": "Wrote: ${0}",
	"GenerateHTML": "Generate HTML file",
	"GenerateHTMLTooltip": "Write an HTML file generated from the current Markdown editor content",
	"alt text": "alt text",
	"blockquote": "blockquote",
	"code": "code",
	"code (block)": "code (block)",
	"code (span)": "code (span)",
	"emphasis": "emphasis",
	"fenced code (${0})": "fenced code (${0})",
	"header (${0})": "header (${0})",
	"horizontal rule": "horizontal rule",
	"label": "label",
	"link (auto)": "link (auto)",
	"link (image)": "link (image)",
	"link (inline)": "link (inline)",
	"link label": "link label",
	"link label (optional)": "link label (optional)",
	"link (ref)": "link (ref)",
	"list item (bullet)": "list item (bullet)",
	"list item (numbered)": "list item (numbered)",
	"strikethrough (${0})": "strikethrough (${0})",
	"strong": "strong",
	"table (${0})": "table (${0})",
	"text": "text",
	"title (optional)": "title (optional)",
	"url": "url",
	"workedProgress": "${0} (${1}/${2})",
	"ConfirmLogout": "Do you want to logout?",
	"VerticalPaneOrientation": "Vertical pane orientation",
	"TogglePaneOrientationTooltip": "Toggle split pane orientation",
	"WarningDuplicateLinkId": "Duplicate link ID: ${0} (link IDs are not case-sensitive)",
	"WarningHeaderTooDeep": "Header level cannot exceed 6",
	"WarningLinkHasNoText": "Link has no text",
	"WarningLinkHasNoURL": "Link has no URL",
	"WarningOrderedListItem": "Ordered list item within unordered list",
	"WarningOrderedListShouldStartAt1": "The first item in an ordered list should have index 1",
	"WarningUndefinedLinkId": "Undefined link ID: ${0}",
	"WarningUnorderedListItem": "Unordered list item within ordered list",
	"PageTitleFormat": "${0} - ${1}", // ${0} is the file or resource being edited; ${1} is the task (eg. "Editor")
	// Display names for keys:
	"KeyCTRL": "Ctrl",
	"KeySHIFT": "Shift",
	"KeyALT": "Alt",
	"KeyBKSPC": "Backspace",
	"KeyDEL": "Del",
	"KeyEND": "End",
	"KeyENTER": "Enter",
	"KeyESCAPE": "Esc",
	"KeyHOME": "Home",
	"KeyINSERT": "Ins",
	"KeyPAGEDOWN": "Page Down",
	"KeyPAGEUP": "Page Up",
	"KeySPACE": "Space",
	"KeyTAB": "Tab",
	// Display elapsed time:
	"a year": "a year",
	"years": "${0} years",
	"a month": "a month",
	"months": "${0} months",
	"a day": "a day",
	"days": "${0} days",
	"an hour": "an hour",
	"hours": "${0} hours",
	"a minute": "a minute",
	"minutes": "${0} minutes",
	"timeAgo": "${0} ago", //${0} represent the time elapsed
	"justNow": "just now", //Represent that the time elapsed is less than 1 minute
	"fixAll": "Fix all",
	"nextSplitter" : "Next Splitter",
	"nextSplitterTooltip": "Move focus through the available splitters",
	"Confirm": "Confirm"
});


/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors: IBM Corporation - initial API and implementation
 *******************************************************************************/

/*eslint-env browser, amd*/
define('orion/util',[],function() {

	var userAgent = navigator.userAgent;
	var isIE = (userAgent.indexOf("MSIE") !== -1 || userAgent.indexOf("Trident") !== -1) ? document.documentMode : undefined; //$NON-NLS-1$ //$NON-NLS-0$
	var isFirefox = parseFloat(userAgent.split("Firefox/")[1] || userAgent.split("Minefield/")[1]) || undefined; //$NON-NLS-1$ //$NON-NLS-0$
	var isOpera = userAgent.indexOf("Opera") !== -1 ? parseFloat(userAgent.split("Version/")[1]) : undefined; //$NON-NLS-0$
	var isChrome = parseFloat(userAgent.split("Chrome/")[1]) || undefined; //$NON-NLS-0$
	var isSafari = userAgent.indexOf("Safari") !== -1 && !isChrome; //$NON-NLS-0$
	var isWebkit = parseFloat(userAgent.split("WebKit/")[1]) || undefined; //$NON-NLS-0$
	var isAndroid = userAgent.indexOf("Android") !== -1; //$NON-NLS-0$
	var isIPad = userAgent.indexOf("iPad") !== -1; //$NON-NLS-0$
	var isIPhone = userAgent.indexOf("iPhone") !== -1; //$NON-NLS-0$
	var isIOS = isIPad || isIPhone;
	var isElectron = userAgent.indexOf("Electron") !== -1; //$NON-NLS-0$
	var isMac = navigator.platform.indexOf("Mac") !== -1; //$NON-NLS-0$
	var isWindows = navigator.platform.indexOf("Win") !== -1; //$NON-NLS-0$
	var isLinux = navigator.platform.indexOf("Linux") !== -1; //$NON-NLS-0$
	var isTouch = typeof document !== "undefined" && "ontouchstart" in document.createElement("input"); //$NON-NLS-1$ //$NON-NLS-0$
	
	var platformDelimiter = isWindows ? "\r\n" : "\n"; //$NON-NLS-1$ //$NON-NLS-0$

	function formatMessage(msg) {
		var args = arguments;
		return msg.replace(/\$\{([^\}]+)\}/g, function(str, index) { return args[(index << 0) + 1]; });
	}
	
	var XHTML = "http://www.w3.org/1999/xhtml"; //$NON-NLS-0$
	function createElement(document, tagName) {
		if (document.createElementNS) {
			return document.createElementNS(XHTML, tagName);
		}
		return document.createElement(tagName);
	}
	function confineDialogTab(firstElement, lastElement) {
		lastElement.addEventListener("keydown", function(evt) {
			if(evt.keyCode === 9 && !evt.shiftKey) {
				evt.preventDefault();
				firstElement.focus();
			}
		});
		firstElement.addEventListener("keydown", function(evt) {
			if(evt.keyCode === 9 && evt.shiftKey) {
				evt.preventDefault();
				lastElement.focus();
			}
		});
	}

	return {
		formatMessage: formatMessage,
		
		createElement: createElement,
		confineDialogTab: confineDialogTab,
		
		/** Browsers */
		isIE: isIE,
		isFirefox: isFirefox,
		isOpera: isOpera,
		isChrome: isChrome,
		isSafari: isSafari,
		isWebkit: isWebkit,
		isAndroid: isAndroid,
		isIPad: isIPad,
		isIPhone: isIPhone,
		isIOS: isIOS,
		isElectron: isElectron,
		
		/** OSs */
		isMac: isMac,
		isWindows: isWindows,
		isLinux: isLinux,

		/** Capabilities */
		isTouch: isTouch,

		platformDelimiter: platformDelimiter
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/webui/littlelib',["orion/util"], function(util) {
	/**
	 * @name orion.webui.littlelib
	 * @class A small library of DOM and UI helpers.
	 */

	/**
	 * Holds useful <code>keyCode</code> values.
	 * @name orion.webui.littlelib.KEY
	 * @static
	 */
	var KEY = {
		BKSPC: 8,
		TAB: 9,
		ENTER: 13,
		SHIFT: 16,
		CONTROL: 17,
		ALT: 18,
		ESCAPE: 27,
		SPACE: 32,
		PAGEUP: 33,
		PAGEDOWN: 34,
		END: 35,
		HOME: 36,
		LEFT: 37,
		UP: 38,
		RIGHT: 39,
		DOWN: 40,
		INSERT: 45,
		DEL: 46,
		COMMAND: 91
	};

	/**
	 * Alias for <code>node.querySelector()</code>.
	 * @name orion.webui.littlelib.$
	 * @function
	 * @static
	 * @param {String} selectors Selectors to match on.
	 * @param {Node} [node=document] Node to query under.
	 * @returns {Element}
	 */
	function $(selector, node) {
		if (!node) {
			node = document;
		}
		return node.querySelector(selector);
	}

	/**
	 * Alias for <code>node.querySelectorAll()</code>.
	 * @name orion.webui.littlelib.$$
	 * @function
	 * @static
	 * @param {String} selectors Selectors to match on.
	 * @param {Node} [node=document] Node to query under.
	 * @returns {NodeList}
	 */
	function $$(selector, node) {
		if (!node) {
			node = document;
		}
		return node.querySelectorAll(selector);
	}

	/**
	 * Identical to {@link orion.webui.littlelib.$$}, but returns an Array instead of a NodeList.
	 * @name orion.webui.littlelib.$$array
	 * @function
	 * @static
	 * @param {String} selectors Selectors to match on.
	 * @param {Node} [node=document] Node to query under.
	 * @returns {Element[]}
	 */
	function $$array(selector, node) {
		return Array.prototype.slice.call($$(selector,node));
	}

	/**
	 * Alias for <code>document.getElementById</code>, but returns the input unmodified when passed a Node (or other non-string).
	 * @function
	 * @param {String|Element} elementOrId
	 * @returns {Element}
	 */
	function node(either) {
		var theNode = either;
		if (typeof(either) === "string") { //$NON-NLS-0$
			theNode = document.getElementById(either);
		}	
		return theNode;
	}

	/**
	 * Returns whether <code>child</code> is a descendant of <code>parent</code> in the DOM order.
	 * @function
	 * @param {Node} parent
	 * @param {Node} child
	 * @returns {Boolean}
	 */
	function contains(parent, child) {
		if (!parent || !child) { return false; }
		if (parent === child) { return true; }
		var compare = parent.compareDocumentPosition(child);  // useful to break out for debugging
		return Boolean(compare & 16);
	}

	/**
	 * Returns the bounds of a node. The returned coordinates are absolute (not relative to the viewport).
	 * @function
	 * @param {Node} node
	 * @returns {Object}
	 */
	function bounds(node) {
		var clientRect = node.getBoundingClientRect();
		var scrollLeft = Math.max(document.documentElement.scrollLeft, document.body.scrollLeft);
		var scrollTop = Math.max(document.documentElement.scrollTop, document.body.scrollTop);
		return { 
			left: clientRect.left + scrollLeft,
			top: clientRect.top + scrollTop,
			width: clientRect.width,
			height: clientRect.height
		};
	}

	/**
	 * Removes all children of the given node.
	 * @name orion.webui.littlelib.empty
	 * @function
	 * @static
	 * @param {Node} node
	 */
	function empty(node) {
		while (node.hasChildNodes()) {
			var child = node.firstChild;
			node.removeChild(child);
		}
	}

	function _getTabIndex(node) {
		var result = node.tabIndex;
		if (result === 0 && util.isIE) {
			/*
			 * The default value of tabIndex is 0 on IE, even for elements that are not focusable
			 * by default (http://msdn.microsoft.com/en-us/library/ie/ms534654(v=vs.85).aspx).
			 * Handle this browser difference by treating this value as -1 if the node is a type
			 * that is not focusable by default according to the MS doc and has not had this
			 * attribute value explicitly set on it.
			 */
			var focusableElements = {
				a: true,
				body: true,
				button: true,
				frame: true,
				iframe: true,
				img: true,
				input: true,
				isindex: true,
				object: true,
				select: true,
				textarea: true
			};
			if (!focusableElements[node.nodeName.toLowerCase()] && !node.attributes.tabIndex) {
				result = -1;
			}
		}
		return result;
	}
	
	/* 
	 * Inspired by http://brianwhitmer.blogspot.com/2009/05/jquery-ui-tabbable-what.html
	 */
	function firstTabbable(node) {
		if (_getTabIndex(node) >= 0 && !node.disabled && node.offsetParent) {
			return node;
		}
		if (node.hasChildNodes()) {
			for (var i=0; i<node.childNodes.length; i++) {
				var result = firstTabbable(node.childNodes[i]);
				if (result) {
					return result;
				}
			}
		}
		return null;
	}

	function lastTabbable(node) {
		if (node.hasChildNodes()) {
			for (var i = node.childNodes.length-1; i >= 0; i--) {
				var result = lastTabbable(node.childNodes[i]);
				if (result) {
					return result;
				}
			}
		}
		if (_getTabIndex(node) >= 0 && !node.disabled && node.offsetParent) {
			return node;
		}
		return null;
	}

	/*
	 * Special hook to show the context menu on Shift + F10 (macs only)
	 */
	function installShowContextMenu() {
		if (util.isMac) {
			document.addEventListener("keydown", function (evt) {
				if (evt.keyCode === 121 && evt.shiftKey) {
					var rect, xPos, yPos;
					var focusElement = document.activeElement;
	
					if (focusElement.contentEditable === "true") {
						var selection = window.getSelection();
						var range = selection.getRangeAt(0); //get the text range
						rect = range.getBoundingClientRect();
						xPos = rect.left;
						yPos = rect.top + rect.height;
						
					} else {
						rect = bounds(focusElement);
						xPos = rect.left + (rect.width/2);
						yPos = rect.top + (rect.height/2);
					}
	
					var e = focusElement.ownerDocument.createEvent("MouseEvents");
					e.initMouseEvent("contextmenu", true, true,
					    focusElement.ownerDocument.defaultView, 1, 0, 0, xPos, yPos, false,
					    false, false, false,2, null);
					return !focusElement.dispatchEvent(e);				
				}
			}, true);
		}
	}
	installShowContextMenu();
	
	/* Trap the tabs within the given parent */
	function trapTabs(parentElement) {
		if (parentElement.tabTrapInstalled)
			return;
		
		parentElement.addEventListener("keydown", function (e) { //$NON-NLS-0$
			if(e.keyCode === KEY.TAB) {
				var first = firstTabbable(parentElement);
				var last = lastTabbable(parentElement);
				
				if (first && last) {
					if (e.target === last && !e.shiftKey) {
						// wrap to first tabbable
						first.focus();
						stop(e);
					}
					else if (e.target === first && e.shiftKey) {
						// wrap to last tabbable
						last.focus();
						stop(e);
					}
				}
			} 
		}, true);
		parentElement.tabTrapInstalled = true;
	}
	
	var variableRegEx = /\$\{([^\}]+)\}/;
	// Internal helper
	function processNodes(node, replace) {
		if (node.nodeType === 3) { // TEXT_NODE
			var matches = variableRegEx.exec(node.nodeValue);
			if (matches && matches.length > 1) {
				replace(node, matches);
			}
		} else if (node.nodeType === 1) { // ELEMENT_NODE
			processElementNode(node, replace, "alt"); //$NON-NLS-0$
			//processElementNode(node, replace, "title"); //$NON-NLS-0$
			//processElementNode(node, replace, "placeholder"); //$NON-NLS-0$
			processElementNode(node, replace, "aria-label"); //$NON-NLS-0$
		}
		if (node.hasChildNodes()) {
			for (var i=0; i<node.childNodes.length; i++) {
				processNodes(node.childNodes[i], replace);
			}
		}
	}
	
	function processElementNode(node, replace, attribute) {
		var value = node.getAttribute(attribute);
		if (value) {
			var matches = variableRegEx.exec(value);
			if (matches && matches.length > 1) {
				replace(node, matches, attribute);
			}
		}		
	}

	/**
	 * @name pixelValueOf
	 * @description Returns the pixel value of the given CSS attribute
	 * @param node The element whose style is to be checked
	 * @param attrName The name of the attribute to check. Use the literal CSS name
	 * (i.e. 'padding-left' rather than 'paddingLeft').
	 * @returns returns The value (in pixels) of the attribute. If the value cannot be parsed to an int the zero is returned
	 */
	function pixelValueOf(node, attrName) {
		if (!node || !attrName) {
			return 0;
		}
		
		var doc = node.ownerDocument;
		var win =  doc.defaultView || doc.parentWindow;
		if (win.getComputedStyle) {
			var style = win.getComputedStyle(node, null);
			var value = style.getPropertyValue(attrName);
			// Ensure that value ends in "px"
			if (value.length > 2 && value.indexOf("px", value.length - 2) !== -1) {
				value = value.slice(0,-2);
				var intVal = parseInt(value,10);
				if (intVal !== intVal) {
					return 0;  // value was NaN
				} else {
					return intVal;
				}
			}
		}
		return 0;
	}
	
	/**
	 * Performs substitution of strings into textContent within the given node and its descendants. An occurrence of <code>${n}</code>
	 * in text content will be replaced with the string <code>messages[n]</code>.
	 * <p>This function is recommended for binding placeholder text in template-created DOM elements to actual display strings.</p>
	 * @name orion.webui.littlelib.processTextNodes
	 * @function
	 * @param {Node} node The node to perform replacement under.
	 * @param {String[]} messages The replacement strings.
	 */
	function processTextNodes(node, messages) {
		processNodes(node, function(targetNode, matches, attribute) {
			var replaceText = messages[matches[1]] || matches[1];
			if (targetNode.nodeType === 3) { // TEXT_NODE
				targetNode.parentNode.replaceChild(document.createTextNode(replaceText), targetNode);
			} else if (targetNode.nodeType === 1 && attribute) { // ELEMENT_NODE
				targetNode.setAttribute(attribute, replaceText); //$NON-NLS-0$
			}
		});
	}

	/**
	 * Performs substitution of DOM nodes into textContent within the given node and its descendants. An occurrence of <code>${n}</code>
	 * in text content will be replaced by the DOM node <code>replaceNodes[n]</code>.
	 * <p>This function is recommended for performing rich-text replacement within a localized string. The use of actual DOM nodes
	 * avoids the need for embedded HTML in strings.</p>
	 * @name orion.webui.littlelib.processDOMNodes
	 * @function
	 * @param {Node} node The node to perform replacement under.
	 * @param {Node[]} replaceNodes The replacement nodes.
	 */
	function processDOMNodes(node, replaceNodes) {
		processNodes(node, function(targetNode, matches) {
			var replaceNode = replaceNodes[matches[1]];
			if (replaceNode) {
				var range = document.createRange();
				var start = matches.index;
				range.setStart(targetNode, start);
				range.setEnd(targetNode, start + matches[0].length);
				range.deleteContents();
				range.insertNode(replaceNode);
			}
		});
	}

	/**
	 * Adds auto-dismiss functionality to the document. When a click event occurs whose <code>target</code> is not a descendant of
	 * one of the <code>excludeNodes</code>, the <code>dismissFunction</code> is invoked.
	 * @name orion.webui.littlelib.addAutoDismiss
	 * @function
	 * @static
	 * @param {Node[]} excludeNodes Clicks targeting any descendant of these nodes will not trigger the dismissFunction.
	 * @param {Function} dismissFunction The dismiss handler.
	 */
	
	var autoDismissNodes = null;

	function addAutoDismiss(excludeNodes, dismissFunction) {
		// auto dismissal.  Click anywhere else means close.
		function onclick(event) {
			autoDismissNodes.forEach(function(autoDismissNode) {
				var excludeNodeInDocument = false;
				var excluded = autoDismissNode.excludeNodes.some(function(excludeNode) {
					if(document.body.contains(excludeNode)) {
						excludeNodeInDocument = true;
						return excludeNode.contains(event.target);
					}
					return false;
				});
				if (excludeNodeInDocument && !excluded) {
					try {
						autoDismissNode.dismiss(event);
					} catch (e) {
						if (typeof console !== "undefined" && console) { //$NON-NLS-0$
							console.error(e && e.message);
						}
					}
				}
			});
			autoDismissNodes = autoDismissNodes.filter(function(autoDismissNode) {
				// true if at least one excludeNode is in document.body
				return autoDismissNode.excludeNodes.some(function(excludeNode) {
					return document.body.contains(excludeNode);
				});
			});
		}

		// Hook listener only once
		if (autoDismissNodes === null) {
			autoDismissNodes = [];
			document.addEventListener("click", onclick, true); //$NON-NLS-0$
			if (util.isIOS) {
				document.addEventListener("touchend", function(event){
					function unhook(){
						event.target.removeEventListener("click", unhook);
					}
					if (event.touches.length === 0) {
						// we need a click eventlistener on the target to have ios really trigger a click
						event.target.addEventListener("click", unhook);
					}	
				}, false);
			}
		}
		
		autoDismissNodes.push({excludeNodes: excludeNodes, dismiss: dismissFunction});
	}
	
	/**
	 * Removes all auto-dismiss nodes which trigger the specified dismiss function.
	 * 
	 * @name orion.webui.littlelib.removeAutoDismiss
	 * @function
	 * @static
	 * @param {Function} dismissFunction The dismiss function to look for.
	 */
	function removeAutoDismiss(dismissFunction) {
		autoDismissNodes = autoDismissNodes.filter(function(autoDismissNode) {
			return dismissFunction !== autoDismissNode.dismiss;
		});
	}
	
	/**
	 * Returns the parent of the node that has the vertical scroll bar.
	 * 
	 * @name orion.webui.littlelib.getOffsetParent
	 * @function
	 * @static
	 * @param {Element} node The node to lookup the offset parent
	 */
	function getOffsetParent(node) {
		var offsetParent = node.parentNode, documentElement = document.documentElement;
		while (offsetParent && offsetParent !== documentElement) {
			var style = window.getComputedStyle(offsetParent, null);
			if (!style) { break; }
			var overflow = style.getPropertyValue("overflow-y"); //$NON-NLS-0$
			if (overflow === "auto" || overflow === "scroll") { break; } //$NON-NLS-1$ //$NON-NLS-0$
			offsetParent = offsetParent.parentNode;
		}
		return offsetParent;
	}
	
	/**
	 * Cancels the default behavior of an event and stops its propagation.
	 * @name orion.webui.littlelib.stop
	 * @function
	 * @static
	 * @param {Event} event
	 */
	function stop(event) {
		if (window.document.all) { 
			event.keyCode = 0;
		}
		if (event.preventDefault) {
			event.preventDefault();
			event.stopPropagation();
		}
	}
	
	function setFramesEnabled(enable) {
		var frames = document.getElementsByTagName("iframe"); //$NON-NLS-0$
		for (var i = 0; i<frames.length; i++) {
			frames[i].parentNode.style.pointerEvents = enable ? "" : "none"; //$NON-NLS-0$
		}
	}
	
	/**
	 * Given a non-empty string, returns a string with no spaces in it (valid HTML5 id attribute).
	 * Also removes any punctuation except for '_', '-', and '.' for compatibility with HTML4.
	 * @name orion.webui.littlelib.validId
	 * @function
	 * @static
	 * @param {String} str	A non-empty string
	 * @returns A valid html id string
	 */
	function validId(str) {
		return str.replace(/\s/, '-').replace(/[^A-Za-z0-9_.-]/, '.');
	}
	
	/**
	 * Maps a <code>keyCode</code> to <tt>KEY</tt> name. This is the inverse of {@link orion.webui.littlelib.KEY}.
	 * @private
	 */
	var KEY_CODE = Object.create(null);
	Object.keys(KEY).forEach(function(name) {
		KEY_CODE[KEY[name]] = name;
	});

	/**
	 * @param {Number} keyCode
	 * @returns The name of the <code>lib.KEY</code> entry for keyCode, or null.
	 */
	function keyName(keyCode) {
		return KEY_CODE[keyCode] || null;
	}

	/**
	 * Creates DOM nodes from the specified template string.
	 * 
	 * @param {String} templateString 	A string containing the HTML template to use
	 * @param {Node} parentNode 		Optional. The parent node to insert the new nodes into. 
	 * 									The parent's contents will be completely replaced.
	 * @returns If the template string contains a single node or a wrapper node which
	 * 			wraps all the other nodes that single DOM node will be returned. 
	 * 			Otherwise if the template string contains multiple top-level nodes an
	 * 			{HTMLCollection} object containing all the top-level nodes will be returned.
	 */
	function createNodes(templateString, parentNode) {
		var parent = parentNode;
		var newNodes = null;
		
		if (undefined === parent) {
			parent = document.createElement("div"); //$NON-NLS-0$
		}

		parent.innerHTML = templateString;	
		if (parent.children.length > 1) {
			newNodes = parent.children;
		} else {
			newNodes = parent.firstChild;
		}
		
		return newNodes;
	}

	//return module exports
	return {
		$: $,
		$$: $$,
		$$array: $$array,
		node: node,
		contains: contains,
		bounds: bounds,
		empty: empty,
		firstTabbable: firstTabbable,
		lastTabbable: lastTabbable,
		trapTabs: trapTabs,
		pixelValueOf: pixelValueOf,
		stop: stop,
		processTextNodes: processTextNodes,
		processDOMNodes: processDOMNodes,
		addAutoDismiss: addAutoDismiss,
		setFramesEnabled: setFramesEnabled,
		validId: validId,
		getOffsetParent: getOffsetParent,
		removeAutoDismiss: removeAutoDismiss,
		keyName: keyName,
		KEY: KEY,
		createNodes: createNodes
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors: IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global requirejs*/
define('orion/i18nUtil',[], function() {
	/**
	 * Performs string substitution. Can be invoked in 2 ways:
	 *
	 * i) vargs giving numbered substition values:
	 *   formatMessage("${0} is ${1}", "foo", "bar")  // "foo is bar"
	 *
	 * ii) a map giving the substitutions:
	 *   formatMessage("${thing} is ${1}", {1: "bar", thing: "foo"})  // "foo is bar"
	 */
	function formatMessage(msg) {
		var pattern = /\$\{([^\}]+)\}/g, args = arguments;
		if (args.length === 2 && args[1] && typeof args[1] === "object") {
			return msg.replace(pattern, function(str, key) {
				return args[1][key];
			});
		}
		return msg.replace(pattern, function(str, index) {
			return args[(index << 0) + 1];
		});
	}
	return {
		formatMessage: formatMessage
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 *******************************************************************************/
define ('orion/bidiFormat',[], function() {

    var TextSegment = (function() {
        var TextSegment = function (obj) {
            this.content = "";
            this.actual = "";
            this.textDirection = "";
            this.localGui = "";
            this.isVisible = true;
            this.isSeparator = false;
            this.isParsed = false;
            this.keep = false;
            this.inBounds = false;
            this.inPoints = false;
            var prop = "";
            for (prop in obj) {
                if (obj.hasOwnProperty(prop)) {
                    this[prop] = obj[prop];
                }
            }
        };
        return TextSegment;
    })();

    var tools = (function() {
        function initBounds(bounds) {
            if (!bounds) {
                return false;
            }
            if (typeof(bounds.start) === "undefined") {
                bounds.start = "";
            }
            if (typeof(bounds.end) === "undefined") {
                bounds.end = "";
            }
            if (typeof(bounds.startAfter) !== "undefined") {
                bounds.start = bounds.startAfter;
                bounds.after = true;
            } else {
                bounds.after = false;
            }
            if (typeof(bounds.endBefore) !== "undefined") {
                bounds.end = bounds.endBefore;
                bounds.before = true;
            } else {
                bounds.before = false;
            }
            var startPos = parseInt(bounds.startPos, 10);
            if (!isNaN(startPos)) {
                bounds.usePos = true;
            } else {
                bounds.usePos = false;
            }
            var bLength = parseInt(bounds.length, 10);
            if (!isNaN(bLength)) {
                bounds.useLength = true;
            } else {
                bounds.useLength = false;
            }
            bounds.loops = typeof(bounds.loops) !== "undefined" ? !!bounds.loops : true;
            return true;
        }

        function getBounds(segment, src) {
            var bounds = {};
            for (var prop in src) {
                if (src.hasOwnProperty(prop)) {
                    bounds[prop] = src[prop];
                }
            }
            var content = segment.content;
            var usePos = bounds.usePos && bounds.startPos < content.length;
            if (usePos) {
                bounds.start = "";
                bounds.loops = false;
            }
            bounds.bStart = usePos ? bounds.startPos : bounds.start.length > 0 ? content.indexOf(bounds.start) : 0;
            var useLength = bounds.useLength && bounds.length > 0 && bounds.bStart + bounds.length < content.length;
            if (useLength) {
                bounds.end = "";
            }
            bounds.bEnd = useLength ? bounds.bStart + bounds.length : bounds.end.length > 0 ?
                    content.indexOf(bounds.end, bounds.bStart + bounds.start.length) + 1 : content.length;
            if (!bounds.after) {
                bounds.start = "";
            }
            if (!bounds.before) {
                bounds.end = "";
            }
            return bounds;
        }

        return {
            handleSubcontents: function (segments, args, subs, origContent, locale) { // jshint unused: false
                if (!subs.content || typeof(subs.content) !== "string" || subs.content.length === 0) {
                    return segments;
                }
                var sLoops = true;
                if (typeof(subs.loops) !== "undefined") {
                    sLoops = !!subs.loops;
                }
                for (var j = 0; true; j++) {
                    if (j >= segments.length) {
                        break;
                    }
                    if (segments[j].isParsed || segments.keep || segments[j].isSeparator) {
                        continue;
                    }
                    var content = segments[j].content;
                    var start = content.indexOf(subs.content);
                    if (start < 0) {
                        continue;
                    }
                    var end;
                    var length = 0;
                    if (subs.continued) {
                        do {
                            length++;
                            end = content.indexOf(subs.content, start + length * subs.content.length);
                        } while (end === 0);
                    } else {
                        length = 1;
                    }
                    end = start + length * subs.content.length;
                    segments.splice(j, 1);
                    if (start > 0) {
                        segments.splice(j, 0, new TextSegment({
                            content: content.substring(0, start),
                            localGui: args.dir,
                            keep: true
                        }));
                        j++;
                    }
                    segments.splice(j, 0, new TextSegment({
                        content: content.substring(start, end),
                        textDirection: subs.subDir,
                        localGui: args.dir
                    }));
                    if (end < content.length) {
                        segments.splice(j + 1, 0, new TextSegment({
                            content: content.substring(end, content.length),
                            localGui: args.dir,
                            keep: true
                        }));
                    }
                    if (!sLoops) {
                        break;
                    }
                }
            },

            handleBounds: function (segments, args, aBounds, origContent, locale) {
                for (var i = 0; i < aBounds.length; i++) {
                    if (!initBounds(aBounds[i])) {
                        continue;
                    }
                    for (var j = 0; true; j++) {
                        if (j >= segments.length) {
                            break;
                        }
                        if (segments[j].isParsed || segments[j].inBounds || segments.keep || segments[j].isSeparator) {
                            continue;
                        }
                        var bounds = getBounds(segments[j], aBounds[i]);
                        var start = bounds.bStart;
                        var end = bounds.bEnd;
                        if (start < 0 || end < 0) {
                            continue;
                        }
                        var content = segments[j].content;

                        segments.splice(j, 1);
                        if (start > 0) {
                            segments.splice(j, 0, new TextSegment({
                                content: content.substring(0, start),
                                localGui: args.dir,
                                keep: true
                            }));
                            j++;
                        }
                        if (bounds.start) {
                            segments.splice(j, 0, new TextSegment({
                                content: bounds.start,
                                localGui: args.dir,
                                isSeparator: true
                            }));
                            j++;
                        }
                        segments.splice(j, 0, new TextSegment({
                            content: content.substring(start + bounds.start.length, end - bounds.end.length),
                            textDirection: bounds.subDir,
                            localGui: args.dir,
                            inBounds: true
                        }));
                        if (bounds.end) {
                            j++;
                            segments.splice(j, 0, new TextSegment({
                                content: bounds.end,
                                localGui: args.dir,
                                isSeparator: true
                            }));
                        }
                        if (end + bounds.end.length < content.length) {
                            segments.splice(j + 1, 0, new TextSegment({
                                content: content.substring(end + bounds.end.length, content.length),
                                localGui: args.dir,
                                keep: true
                            }));
                        }
                        if (!bounds.loops) {
                            break;
                        }
                    }
                }
                for (i = 0; i < segments.length; i++) {
                    segments[i].inBounds = false;
                }
                return segments;
            },

            handleCases: function (segments, args, cases, origContent, locale) {
                if (cases.length === 0) {
                    return segments;
                }
                var hArgs = {};
                for (var prop in args) {
                    if (args.hasOwnProperty(prop)) {
                        hArgs[prop] = args[prop];
                    }
                }
                for (var i =  0; i < cases.length; i++) {
                    if (!cases[i].handler || typeof(cases[i].handler.handle) !== "function") {
                        cases[i].handler = args.commonHandler;
                    }
                    if (cases[i].args) {
                        hArgs.cases = cases[i].args.cases;
                        hArgs.points = cases[i].args.points;
                        hArgs.bounds = cases[i].args.bounds;
                        hArgs.subs = cases[i].args.subs;
                    } else {
                        hArgs.cases = [];
                        hArgs.points = [];
                        hArgs.bounds = [];
                        hArgs.subs = {};
                    }
                    cases[i].handler.handle(origContent, segments, hArgs, locale);
                }
                return segments;
            },

            handlePoints: function (segments, args, points, origContent, locale) { //jshint unused: false
                for (var i = 0; i < points.length; i++) {
                    for (var j = 0; true; j++) {
                        if (j >= segments.length) {
                            break;
                        }
                        if (segments[j].isParsed || segments[j].keep || segments[j].isSeparator) {
                            continue;
                        }
                        var content = segments[j].content;
                        var pos = content.indexOf(points[i]);
                        if (pos >= 0) {
                            segments.splice(j, 1);
                            if (pos > 0) {
                                segments.splice(j, 0, new TextSegment({
                                    content: content.substring(0, pos),
                                    textDirection: args.subDir,
                                    localGui: args.dir,
                                    inPoints: true
                                }));
                                j++;
                            }
                            segments.splice(j, 0, new TextSegment({
                                content: points[i],
                                localGui: args.dir,
                                isSeparator: true
                            }));
                            if (pos + points[i].length + 1 <= content.length) {
                                segments.splice(j + 1, 0, new TextSegment({
                                    content: content.substring(pos + points[i].length),
                                    textDirection: args.subDir,
                                    localGui: args.dir,
                                    inPoints: true
                                }));
                            }
                        }
                    }
                }
                for (i = 0; i < segments.length; i++) {
                    if (segments[i].keep) {
                        segments[i].keep = false;
                    } else if(segments[i].inPoints){
                        segments[i].isParsed = true;
                        segments[i].inPoints = false;
                    }
                }
                return segments;
            }
        };
    })();

    var common = (function() {
        return {
            handle: function (content, segments, args, locale) {
                var cases = [];
                if (Array.isArray(args.cases)) {
                    cases = args.cases;
                }
                var points = [];
                if (typeof(args.points) !== "undefined") {
                    if (Array.isArray(args.points)) {
                        points = args.points;
                    } else if (typeof(args.points) === "string") {
                        points = args.points.split("");
                    }
                }
                var subs = {};
                if (typeof(args.subs) === "object") {
                    subs = args.subs;
                }
                var aBounds = [];
                if (Array.isArray(args.bounds)) {
                    aBounds = args.bounds;
                }

                tools.handleBounds(segments, args, aBounds, content, locale);
                tools.handleSubcontents(segments, args, subs, content, locale);
                tools.handleCases(segments, args, cases, content, locale);
                tools.handlePoints(segments, args, points, content, locale);
                return segments;
            }
        };
    })();

    var misc = (function() {
        var isBidiLocale = function (locale) {
            var lang = !locale ? "" : locale.split("-")[0];
            if (!lang || lang.length < 2) {
                return false;
            }
            return ["iw", "he", "ar", "fa", "ur"].some(function (bidiLang) {
                return bidiLang === lang;
            });
        };
        var LRE = "\u202A";
        var RLE = "\u202B";
        var PDF = "\u202C";
        var LRM = "\u200E";
        var RLM = "\u200F";
        var LRO = "\u202D";
        var RLO = "\u202E";

        return {
            LRE: LRE,
            RLE: RLE,
            PDF: PDF,
            LRM: LRM,
            RLM: RLM,
            LRO: LRO,
            RLO: RLO,

            getLocaleDetails: function (locale) {
                if (!locale) {
                    locale = typeof navigator === "undefined" ? "" :
                        (navigator.language ||
                        navigator.userLanguage ||
                        "");
                }
                locale = locale.toLowerCase();
                if (isBidiLocale(locale)) {
                    var full = locale.split("-");
                    return {lang: full[0], country: full[1] ? full[1] : ""};
                }
                return {lang: "not-bidi"};
            },

            removeUcc: function (text) {
                if (text) {
                    return text.replace(/[\u200E\u200F\u202A-\u202E]/g, "");
                }
                return text;
            },

            removeTags: function (text) {
                if (text) {
                    return text.replace(/<[^<]*>/g, "");
                }
                return text;
            },

            getDirection: function (text, dir, guiDir, checkEnd) {
                if (dir !== "auto" && (/^(rtl|ltr)$/i).test(dir)) {
                    return dir;
                }
                guiDir = (/^(rtl|ltr)$/i).test(guiDir) ? guiDir : "ltr";
                var txt = !checkEnd ? text : text.split("").reverse().join("");
                var fdc = /[A-Za-z\u05d0-\u065f\u066a-\u06ef\u06fa-\u07ff\ufb1d-\ufdff\ufe70-\ufefc]/.exec(txt);
                return fdc ? (fdc[0] <= "z" ? "ltr" : "rtl") : guiDir;
            },

            hasArabicChar: function (text) {
                var fdc = /[\u0600-\u065f\u066a-\u06ef\u06fa-\u07ff\ufb1d-\ufdff\ufe70-\ufefc]/.exec(text);
                return !!fdc;
            },

            showMarks: function (text, guiDir) {
                var result = "";
                for (var i = 0; i < text.length; i++) {
                    var c = "" + text.charAt(i);
                    switch (c) {
                    case LRM:
                        result += "<LRM>";
                        break;
                    case RLM:
                        result += "<RLM>";
                        break;
                    case LRE:
                        result += "<LRE>";
                        break;
                    case RLE:
                        result += "<RLE>";
                        break;
                    case LRO:
                        result += "<LRO>";
                        break;
                    case RLO:
                        result += "<RLO>";
                        break;
                    case PDF:
                        result += "<PDF>";
                        break;
                    default:
                        result += c;
                    }
                }
                var mark = typeof(guiDir) === "undefined" || !((/^(rtl|ltr)$/i).test(guiDir)) ? "" :
                    guiDir === "rtl" ? RLO : LRO;
                return mark + result + (mark === "" ? "" : PDF);
            },

            hideMarks: function (text) {
                var txt = text.replace(/<LRM>/g, this.LRM).replace(/<RLM>/g, this.RLM).replace(/<LRE>/g, this.LRE);
                return txt.replace(/<RLE>/g, this.RLE).replace(/<LRO>/g, this.LRO).replace(/<RLO>/g, this.RLO).replace(/<PDF>/g, this.PDF);
            },

            showTags: function (text) {
                return "<xmp>" + text + "</xmp>";
            },

            hideTags: function (text) {
                return text.replace(/<xmp>/g,"").replace(/<\/xmp>/g,"");
            }
        };
    })();

    var stext = (function() {
        var stt = {};

        // args
        //   handler: main handler (default - dbidi/stt/handlers/common)
        //   guiDir: GUI direction (default - "ltr")
        //   dir: main stt direction (default - guiDir)
        //   subDir: direction of subsegments
        //   points: array of delimiters (default - [])
        //   bounds: array of definitions of bounds in which handler works
        //   subs: object defines special handling for some substring if found
        //   cases: array of additional modules with their args for handling special cases (default - [])
        function parseAndDisplayStructure(content, fArgs, isHtml, locale) {
            if (!content || !fArgs) {
                return content;
            }
            return displayStructure(parseStructure(content, fArgs, locale), fArgs, isHtml);
        }

        function checkArguments(fArgs, fullCheck) {
            var args = Array.isArray(fArgs)? fArgs[0] : fArgs;
            if (!args.guiDir) {
                args.guiDir = "ltr";
            }
            if (!args.dir) {
                args.dir = args.guiDir;
            }
            if (!fullCheck) {
                return args;
            }
            if (typeof(args.points) === "undefined") {
                args.points = [];
            }
            if (!args.cases) {
                args.cases = [];
            }
            if (!args.bounds) {
                args.bounds = [];
            }
            args.commonHandler = common;
            return args;
        }

        function parseStructure(content, fArgs, locale) {
            if (!content || !fArgs) {
                return new TextSegment({content: ""});
            }
            var args = checkArguments(fArgs, true);
            var segments = [new TextSegment(
                {
                    content: content,
                    actual: content,
                    localGui: args.dir
                })];
            var parse = common.handle;
            if (args.handler && typeof(args.handler) === "function") {
                parse = args.handler.handle;
            }
            parse(content, segments, args, locale);
            return segments;
        }

        function displayStructure(segments, fArgs, isHtml) {
            var args = checkArguments(fArgs, false);
            if (isHtml) {
                return getResultWithHtml(segments, args);
            }
            else {
                return getResultWithUcc(segments, args);
            }
        }

        function getResultWithUcc(segments, args, isHtml) {
            var result = "";
            var checkedDir = "";
            var prevDir = "";
            var stop = false;
            for (var i = 0; i < segments.length; i++) {
                if (segments[i].isVisible) {
                    var dir = segments[i].textDirection;
                    var lDir = segments[i].localGui;
                    if (lDir !== "" && prevDir === "") {
                        result += (lDir === "rtl" ? misc.RLE : misc.LRE);
                    }
                    else if(prevDir !== "" && (lDir === "" || lDir !== prevDir || stop)) {
                        result += misc.PDF + (i == segments.length - 1 && lDir !== ""? "" : args.dir === "rtl" ? misc.RLM : misc.LRM);
                        if (lDir !== "") {
                            result += (lDir === "rtl" ? misc.RLE : misc.LRE);
                        }
                    }
                    if (dir === "auto") {
                        dir = misc.getDirection(segments[i].content, dir, args.guiDir);
                    }
                    if ((/^(rtl|ltr)$/i).test(dir)) {
                        result += (dir === "rtl" ? misc.RLE : misc.LRE) + segments[i].content + misc.PDF;
                        checkedDir = dir;
                    }
                    else {
                        result += segments[i].content;
                        checkedDir = misc.getDirection(segments[i].content, dir, args.guiDir, true);
                    }
                    if (i < segments.length - 1) {
                        var locDir = lDir && segments[i+1].localGui? lDir : args.dir;
                        result += locDir === "rtl" ? misc.RLM : misc.LRM;
                    }
                    else if(prevDir !== "") {
                        result += misc.PDF;
                    }
                    prevDir = lDir;
                    stop = false;
                }
                else {
                    stop = true;
                }
            }
            var sttDir = args.dir === "auto" ? misc.getDirection(segments[0].actual, args.dir, args.guiDir) : args.dir;
            if (sttDir !== args.guiDir) {
                result = (sttDir === "rtl" ? misc.RLE : misc.LRE) + result + misc.PDF;
            }
            return result;
        }

        function getResultWithHtml(segments, args, isHtml) {
            var result = "";
            var checkedDir = "";
            var prevDir = "";
            for (var i = 0; i < segments.length; i++) {
                if (segments[i].isVisible) {
                    var dir = segments[i].textDirection;
                    var lDir = segments[i].localGui;
                    if (lDir !== "" && prevDir === "") {
                        result += "<bdi dir='" + (lDir === "rtl" ? "rtl" : "ltr") + "'>";
                    }
                    else if(prevDir !== "" && (lDir === "" || lDir !== prevDir || stop)) {
                        result += "</bdi>" + (i == segments.length - 1 && lDir !== ""? "" : "<span style='unicode-bidi: embed; direction: " + (args.dir === "rtl" ? "rtl" : "ltr") + ";'></span>");
                        if (lDir !== "") {
                            result += "<bdi dir='" + (lDir === "rtl" ? "rtl" : "ltr") + "'>";
                        }
                    }

                    if (dir === "auto") {
                        dir = misc.getDirection(segments[i].content, dir, args.guiDir);
                    }
                    if ((/^(rtl|ltr)$/i).test(dir)) {
                        //result += "<span style='unicode-bidi: embed; direction: " + (dir === "rtl" ? "rtl" : "ltr") + ";'>" + segments[i].content + "</span>";
                        result += "<bdi dir='" + (dir === "rtl" ? "rtl" : "ltr") + "'>" + segments[i].content + "</bdi>";
                        checkedDir = dir;
                    }
                    else {
                        result += segments[i].content;
                        checkedDir = misc.getDirection(segments[i].content, dir, args.guiDir, true);
                    }
                    if (i < segments.length - 1) {
                        var locDir = lDir && segments[i+1].localGui? lDir : args.dir;
                        result += "<span style='unicode-bidi: embed; direction: " + (locDir === "rtl" ? "rtl" : "ltr") + ";'></span>";
                    }
                    else if(prevDir !== "") {
                        result += "</bdi>";
                    }
                    prevDir = lDir;
                    stop = false;
                }
                else {
                    stop = true;
                }
            }
            var sttDir = args.dir === "auto" ? misc.getDirection(segments[0].actual, args.dir, args.guiDir) : args.dir;
            if (sttDir !== args.guiDir) {
                result = "<bdi dir='" + (sttDir === "rtl" ? "rtl" : "ltr") + "'>" + result + "</bdi>";
            }
            return result;
        }

        //TBD ?
        function restore(text, isHtml) {
            return text;
        }

        stt.parseAndDisplayStructure = parseAndDisplayStructure;
        stt.parseStructure = parseStructure;
        stt.displayStructure = displayStructure;
        stt.restore = restore;

        return stt;
    })();

    var breadcrumb = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: args.dir ? args.dir : isRtl ? "rtl" : "ltr",
                        subs: {
                            content: ">",
                            continued: true,
                            subDir: isRtl ? "rtl" : "ltr"
                        },
                        cases: [{
                            args: {
                                subs: {
                                    content: "<",
                                    continued: true,
                                    subDir: isRtl ? "ltr" : "rtl"
                                }
                            }
                        }]
                };

                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var comma = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: ","
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var email = (function() {
        function getDir(text, locale) {
            if (misc.getLocaleDetails(locale).lang !== "ar") {
                return "ltr";
            }
            var ind = text.indexOf("@");
            if (ind > 0 && ind < text.length - 1) {
                return misc.hasArabicChar(text.substring(ind + 1)) ? "rtl" : "ltr";
            }
            return "ltr";
        }

        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: getDir(text, locale),
                        points: "<>.:,;@",
                        cases: [{
                            handler: common,
                            args: {
                                bounds: [{
                                    startAfter: "\"",
                                    endBefore: "\""
                                },
                                {
                                    startAfter: "(",
                                    endBefore: ")"
                                }
                                ],
                                points: ""
                            }
                        }]
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var filepath = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        subDir : args.subDir,
                        points: "/\\:.&<>"
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var formula = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: " /%^&[]<>=!?~:.,|()+-*{}",
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();


    var sql = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: "\t!#%&()*+,-./:;<=>?|[]{}",
                        cases: [{
                            handler: common,
                            args: {
                                bounds: [{
                                    startAfter: "/*",
                                    endBefore: "*/"
                                },
                                {
                                    startAfter: "--",
                                    end: "\n"
                                },
                                {
                                    startAfter: "--"
                                }
                                ]
                            }
                        },
                        {
                            handler: common,
                            args: {
                                subs: {
                                    content: " ",
                                    continued: true
                                }
                            }
                        },
                        {
                            handler: common,
                            args: {
                                bounds: [{
                                    startAfter: "'",
                                    endBefore: "'"
                                },
                                {
                                    startAfter: "\"",
                                    endBefore: "\""
                                }
                                ]
                            }
                        }
                        ]
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var underscore = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: "_"
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var url = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: ":?#/@.[]=&<>"
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var word = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: args.dir ? args.dir : isRtl ? "rtl" : "ltr",
                        points: " ,.!?;:",
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var xpath = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: " /[]<>=!:@.|()+-*",
                        cases: [{
                            handler: common,
                            args: {
                                bounds: [{
                                    startAfter: "\"",
                                    endBefore: "\""
                                },
                                {
                                    startAfter: "'",
                                    endBefore: "'"
                                }
                                ],
                                points: ""
                            }
                        }
                        ]
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var custom = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var hArgs = {};
                var prop = "";
                var sArgs = Array.isArray(args)? args[0] : args;
                for (prop in sArgs) {
                    if (sArgs.hasOwnProperty(prop)) {
                        hArgs[prop] = sArgs[prop];
                    }
                }
                hArgs.guiDir = isRtl ? "rtl" : "ltr";
                hArgs.dir = hArgs.dir ? hArgs.dir : hArgs.guiDir;
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, hArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, hArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var message = (function() {
        var params = {msgLang: "en", msgDir: "", phLang: "", phDir: "", phPacking: ["{","}"], phStt: {type: "none", args: {}}, guiDir: ""};
        var parametersChecked = false;

        function getDirectionOfLanguage(lang) {
            if (lang === "he" || lang === "iw" || lang === "ar") {
                return "rtl";
            }
            return "ltr";
        }

        function checkParameters(obj) {
            if (obj.msgDir.length === 0) {
                obj.msgDir = getDirectionOfLanguage(obj.msgLang);
            }
            obj.msgDir = obj.msgDir !== "ltr" && obj.msgDir !== "rtl" && obj.msgDir != "auto"? "ltr" : obj.msgDir;
            if (obj.guiDir.length === 0) {
                obj.guiDir = obj.msgDir;
            }
            obj.guiDir = obj.guiDir !== "rtl"? "ltr" : "rtl";
            if (obj.phDir.length === 0) {
                obj.phDir = obj.phLang.length === 0? obj.msgDir : getDirectionOfLanguage(obj.phLang);
            }
            obj.phDir = obj.phDir !== "ltr" && obj.phDir !== "rtl" && obj.phDir != "auto"? "ltr" : obj.phDir;
            if (typeof (obj.phPacking) === "string") {
                obj.phPacking = obj.phPacking.split("");
            }
            if (obj.phPacking.length < 2) {
                obj.phPacking = ["{","}"];
            }
        }

        return {
            setDefaults: function (args) {
                for (var prop in args) {
                    if (params.hasOwnProperty(prop)) {
                        params[prop] = args[prop];
                    }
                }
                checkParameters(params);
                parametersChecked = true;
            },

            format: function (text) {
                if (!parametersChecked) {
                    checkParameters(params);
                    parametersChecked = true;
                }
                var isHtml = false;
                var hasHtmlArg = false;
                var spLength = params.phPacking[0].length;
                var epLength = params.phPacking[1].length;
                if (arguments.length > 0) {
                    var last = arguments[arguments.length-1];
                    if (typeof (last) === "boolean") {
                        isHtml = last;
                        hasHtmlArg = true;
                    }
                }
                //Message
                var re = new RegExp(params.phPacking[0] + "\\d+" + params.phPacking[1]);
                var m;
                var tSegments = [];
                var offset = 0;
                var txt = text;
                while ((m = re.exec(txt)) != null) {
                    var lastIndex = txt.indexOf(m[0]) + m[0].length;
                    if (lastIndex > m[0].length) {
                        tSegments.push({text: txt.substring(0, lastIndex - m[0].length), ph: false});
                    }
                    tSegments.push({text: m[0], ph: true});
                    offset += lastIndex;
                    txt = txt.substring(lastIndex, txt.length);
                }
                if (offset < text.length) {
                    tSegments.push({text: text.substring(offset, text.length), ph: false});
                }
                //Parameters
                var tArgs = [];
                for (var i = 1; i < arguments.length - (hasHtmlArg? 1 : 0); i++) {
                    var arg = arguments[i];
                    var checkArr = arg;
                    var inLoop = false;
                    var indArr = 0;
                    if (Array.isArray(checkArr)) {
                        arg = checkArr[0];
                        if (typeof(arg) === "undefined") {
                            continue;
                        }
                        inLoop = true;
                    }
                    do {
                        if (typeof (arg) === "string") {
                            tArgs.push({text: arg, dir: params.phDir, stt: params.stt});
                        }
                        else if(typeof (arg) === "boolean") {
                            isHtml = arg;
                        }
                        else if(typeof (arg) === "object") {
                            tArgs.push(arg);
                            if (!arg.hasOwnProperty("text")) {
                                tArgs[tArgs.length-1].text = "{???}";
                            }
                            if (!arg.hasOwnProperty("dir") || arg.dir.length === 0) {
                                tArgs[tArgs.length-1].dir = params.phDir;
                            }
                            if (!arg.hasOwnProperty("stt") || (typeof (arg.stt) === "string" && arg.stt.length === 0) ||
                                (typeof (arg.stt) === "object" && Object.keys(arg.stt).length === 0)) {
                                tArgs[tArgs.length-1].stt = params.phStt;
                            }
                        }
                        else {
                            tArgs.push({text: "" + arg, dir: params.phDir, stt: params.phStt});
                        }
                        if (inLoop) {
                            indArr++;
                            if (indArr == checkArr.length) {
                                inLoop = false;
                            }
                            else {
                                arg = checkArr[indArr];
                            }
                        }
                    } while(inLoop);
                }
                //Indexing
                var segments = [];
                for (i = 0; i < tSegments.length; i++) {
                    var t = tSegments[i];
                    if (!t.ph) {
                        segments.push(new TextSegment({content: t.text, textDirection: params.msgDir}));
                    }
                    else {
                        var ind = parseInt(t.text.substring(spLength, t.text.length - epLength));
                        if (isNaN(ind) || ind >= tArgs.length) {
                            segments.push(new TextSegment({content: t.text, textDirection: params.msgDir}));
                            continue;
                        }
                        var sttType = "none";
                        if (!tArgs[ind].stt) {
                            tArgs[ind].stt = params.phStt;
                        }
                        if (tArgs[ind].stt) {
                            if (typeof (tArgs[ind].stt) === "string") {
                                sttType = tArgs[ind].stt;
                            }
                            else if(tArgs[ind].stt.hasOwnProperty("type")) {
                                sttType = tArgs[ind].stt.type;
                            }
                        }
                        if (sttType.toLowerCase() !== "none") {
                            var sttSegs =  getHandler(sttType).format(tArgs[ind].text, tArgs[ind].stt.args || {},
                                    params.msgDir === "rtl", false, params.msgLang, true);
                            for (var j = 0; j < sttSegs.length; j++) {
                                segments.push(sttSegs[j]);
                            }
                            segments.push(new TextSegment({isVisible: false}));
                        }
                        else {
                            segments.push(new TextSegment({content: tArgs[ind].text, textDirection: (tArgs[ind].dir? tArgs[ind].dir : params.phDir)}));
                        }
                    }
                }
                var result =  stext.displayStructure(segments, {guiDir: params.guiDir, dir: params.msgDir}, isHtml);
                return result;
            }
        };
    })();

    var event = null;

    function getHandler(type) {
        switch (type) {
        case "breadcrumb" :
            return breadcrumb;
        case "comma" :
            return comma;
        case "email" :
            return email;
        case "filepath" :
            return filepath;
        case "formula" :
            return formula;
        case "sql" :
            return sql;
        case "underscore" :
            return underscore;
        case "url" :
            return url;
        case "word" :
            return word;
        case "xpath" :
            return xpath;
        default:
            return custom;
        }
    }

    function isInputEventSupported(element) {
        var agent = window.navigator.userAgent;
        if (agent.indexOf("MSIE") >=0 || agent.indexOf("Trident") >=0 || agent.indexOf("Edge") >=0) {
            return false;
        }
        var checked = document.createElement(element.tagName);
        checked.contentEditable = true;
        var isSupported = ("oninput" in checked);
        if (!isSupported) {
          checked.setAttribute('oninput', 'return;');
          isSupported = typeof checked['oninput'] == 'function';
        }
        checked = null;
        return isSupported;
    }

    function attachElement(element, type, args, isRtl, locale) {
        //if (!element || element.nodeType != 1 || !element.isContentEditable)
        if (!element || element.nodeType != 1) {
            return false;
        }
        if (!event) {
            event = document.createEvent('Event');
            event.initEvent('TF', true, true);
        }
        element.setAttribute("data-tf-type", type);
        var sArgs = args === "undefined"? "{}" : JSON.stringify(Array.isArray(args)? args[0] : args);
        element.setAttribute("data-tf-args", sArgs);
        var dir = "ltr";
        if (isRtl === "undefined") {
            if (element.dir) {
                dir = element.dir;
            }
            else if(element.style && element.style.direction) {
                dir = element.style.direction;
            }
            isRtl = dir.toLowerCase() === "rtl";
        }
        element.setAttribute("data-tf-dir", isRtl);
        element.setAttribute("data-tf-locale", misc.getLocaleDetails(locale).lang);
        if (isInputEventSupported(element)) {
            var ehandler = element.oninput;
            element.oninput = function(event) {
                displayWithStructure(event.target);
            };
        }
        else {
            element.onkeyup = function(e) {
                displayWithStructure(e.target);
                element.dispatchEvent(event);
            };
            element.onmouseup = function(e) {
                displayWithStructure(e.target);
                element.dispatchEvent(event);
            };
        }
        displayWithStructure(element);

        return true;
    }

    function detachElement(element) {
        if (!element || element.nodeType != 1) {
            return;
        }
        element.removeAttribute("data-tf-type");
        element.removeAttribute("data-tf-args");
        element.removeAttribute("data-tf-dir");
        element.removeAttribute("data-tf-locale");
        element.innerHTML = element.textContent || "";
    }

    // This function is called for each user's input into the attached editable html element
    // (see function attachElement()). It controls correct display of the structured text
    // in the element, contents of which is changed dynamically.
    // Arguments, needed to display the structure, added previously as a custom attributes of 
    // the element. The main attribute is the type of the structure ("data-tf-type"), which can 
    // declare one of predefined types of text structure or CUSTOM type. Some types of text structure 
    // additionally require flow direction (e.g. direction, in which seprate segments of the text
    // are displayed) and locale(e.g. language of the text). CUSTOM structure can require some 
    // additional parameters.
    function displayWithStructure(element) {
        var txt = element.textContent || "";
        var selection = document.getSelection();
        if (txt.length === 0 || !selection || selection.rangeCount <= 0) {
            element.dispatchEvent(event);
            return;
        }
        
        var range = selection.getRangeAt(0);
        var tempRange = range.cloneRange(), startNode, startOffset;
        startNode = range.startContainer;
        startOffset = range.startOffset;
        var textOffset = 0;
        if (startNode.nodeType === 3) {
            textOffset += startOffset;
        }
        tempRange.setStart(element,0);
        tempRange.setEndBefore(startNode);
        var div = document.createElement('div');
        div.appendChild(tempRange.cloneContents());
        textOffset += div.textContent.length;

        element.innerHTML = getHandler(element.getAttribute("data-tf-type")).
            format(txt, JSON.parse(element.getAttribute("data-tf-args")), (element.getAttribute("data-tf-dir") === "true"? true : false),
            true, element.getAttribute("data-tf-locale"));
        var parent = element;
        var node = element;
        var newOffset = 0;
        var inEnd = false;
        selection.removeAllRanges();
        range.setStart(element,0);
        range.setEnd(element,0);
        while (node) {
            if (node.nodeType === 3) {
                if (newOffset + node.nodeValue.length >= textOffset) {
                    range.setStart(node, textOffset - newOffset);
                    break;
                }
                else {
                    newOffset += node.nodeValue.length;
                    node = node.nextSibling;
                }
            }
            else if(node.hasChildNodes()) {
                parent = node;
                node = parent.firstChild;
                continue;
            }
            else {
                node = node.nextSibling;
            }
            while (!node) {
                if (parent === element) {
                    inEnd = true;
                    break;
                }
                node = parent.nextSibling;
                parent = parent.parentNode;
            }
            if (inEnd) {
                break;
            }
        }

        selection.addRange(range);
        element.dispatchEvent(event);
    }

    return {
        /**
        * Returns the HTML representation of a given structured text
        * @param text - the structured text
        * @param type - could be one of filepath, url, email
        * @param args - pass additional arguments to the handler. generally null.
        * @param isRtl - indicates if the GUI is mirrored
        * @param locale - the browser locale
        */
        getHtml: function (text, type, args, isRtl, locale) {
            return getHandler(type).format(text, args, isRtl, true, locale);
        },
        /**
        * Handle Structured text correct display for a given HTML element.
        * @param element - the element  : should be of type div contenteditable=true
        * @param type - could be one of filepath, url, email
        * @param args - pass additional arguments to the handler. generally null.
        * @param isRtl - indicates if the GUI is mirrored
        * @param locale - the browser locale
        */
        attach: function (element, type, args, isRtl, locale) {
            return attachElement(element, type, args, isRtl, locale);
        },
        /**
         * Adds UCCs (Unicode Control Chars) to a given structured text for a correct display
         * @param text - the structured text
         * @param type - could be one of filepath, url, email
         * @param args - pass additional arguments to the handler. generally null.
         * @param isRtl - indicates if the GUI is mirrored
         * @param locale - the browser locale
         */
        getString:  function (text, type, args, isRtl, locale) {
			return getHandler(type).format(text, args, isRtl, false, locale);
		}      
    };
});

/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 *******************************************************************************/
define ('orion/bidiUtils',[
	"orion/util",
	"orion/bidiFormat"
],
function(util, bidiFormat) { /* BDL */
	
	var bidiEnabledStorage = "/orion/preferences/bidi/bidiEnabled"; //$NON-NLS-0$
	var bidiLayoutStorage = "/orion/preferences/bidi/bidiLayout"; //$NON-NLS-0$	
	var LRE = "\u202A";	//$NON-NLS-0$
	var PDF = "\u202C"; //$NON-NLS-0$
	var RLE = "\u202B"; //$NON-NLS-0$
		
	function setBrowserLangDirection() {
		
		var lang;
		if (window.dojoConfig) {
			lang = window.dojoConfig.locale;
		}
		if (!lang) {
			lang = navigator.languages ? navigator.languages[0] : navigator.language || navigator.userLanguage;
		}
		var isBidi = lang && "ar iw he".indexOf(lang.substring(0, 2)) !== - 1;

		if (isBidi && isBidiEnabled()) {
			var htmlElement = document.getElementsByTagName("html")[0];
			if (htmlElement){ //should be always true
				htmlElement.setAttribute ("dir", "rtl");
			}
		}
	}
	
	setBrowserLangDirection();
	
	var bidiLayout = getBidiLayout();

	/**
	 * checks if directionality should be applied in Orion.
	 * @returns {Boolean} true if globalization settings exist and bidi is enabled.
	 */		
	function isBidiEnabled() {
		var bidiEnabled = localStorage.getItem(bidiEnabledStorage);
		if (bidiEnabled && bidiEnabled === "true") {		//$NON-NLS-0$
			return true;
		}
		return false;
	}
	
	/**
	 * returns bidiLayout value set in globalization settings.
	 * @returns {String} text direction.
	 */	
	function getBidiLayout() {
		var _bidiLayout = localStorage.getItem(bidiLayoutStorage);
		if (_bidiLayout && (_bidiLayout === "rtl" || _bidiLayout === "ltr" || _bidiLayout === "auto")) {	//$NON-NLS-0$ //$NON-NLS-1$ //$NON-NLS-2$
			return _bidiLayout;
		}
		return "ltr";	//$NON-NLS-0$
	}
	
	/**
	 * returns text direction.
	 * this method is used for handling direction by adding a dir attribute in an HTML element.
	 * if bidiLayout is set to ltr > return ltr
	 * if bidiLayout is set to rtl > return rtl
	 * if bidiLayout is set to auto > check for first strong character in text and return ltr or rtl accordingly.
	 * @param {String} the text on which to set directionality
	 * @returns {String} text direction. rtl or ltr.
	 */	
	function getTextDirection(text) {
		bidiLayout = getBidiLayout();
		if (!isBidiEnabled()) {
			return "";
		}
		if (bidiLayout === "auto" && util.isIE) {	//$NON-NLS-0$
			return checkContextual(text);
		}
		return bidiLayout;
	}
	
	/**
	 * Wraps text by UCC (Unicode control characters) according to text direction
	 * In some cases defining the dir attribute in a different direction than the GUI orientation, 
	 * changes the alignment of the text and/or adjacent elements such as icons.
	 * This doesn't follow the bidi standards (static text should be aligned following GUI direction).
	 * Therefore the only solution is to use UCC (Unicode control characters) to display the text in a correct orientation.
	 * (the text is changed for display purposes only. The original text in the repository remains unchanged)
	 * @param {String} the text to be wrapped
	 * @returns {String} text after adding ucc characters.
	 */		
	function enforceTextDirWithUcc ( text ) {
		if (isBidiEnabled() && text.trim()) {
			bidiLayout = getBidiLayout();
			var dir = bidiLayout === "auto" ? checkContextual( text ) : bidiLayout;	//$NON-NLS-0$
			return ( dir === "ltr" ? LRE : RLE ) + text + PDF;	//$NON-NLS-0$
		}
		return text;	
	}
	
	/**
	 * Finds the first strong (directional) character.
	 * If it is Latin, return ltr. If it is bidi, return rtl. Otherwise, return ltr as default. 
	 * @param {String} the text to be examined
	 * @returns {String} text direction. rtl or ltr.
	 */			
	function checkContextual ( text ) {
		// look for strong (directional) characters
		var fdc = /[A-Za-z\u05d0-\u065f\u066a-\u06ef\u06fa-\u07ff\ufb1d-\ufdff\ufe70-\ufefc]/.exec( text );
		// if found, return the direction that defined by the character, else return ltr as defult.
		return fdc ? fdc[0] <= "z" ? "ltr" : "rtl"  : "ltr";	//$NON-NLS-0$ //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}
	
	function addBidiEventListeners ( input ) {
		if (!input._hasBidiEventListeners) {
			input._hasBidiEventListeners = true;

			var eventTypes = ["keyup", "cut", "paste"];
			for (var i = 0; i < eventTypes.length; ++i) {
				input.addEventListener(eventTypes[i], handleInputEvent.bind(this),
					false);
			}
		}
	}
	
	function handleInputEvent ( event ) {
		var input = event.target;
		if (input) {
			input.dir = getTextDirection(input.value || input.textContent); // resolve dir attribute of the element
		}
	}
	
	function initInputField ( input ) {
		if (isBidiEnabled() && input) {
			input.dir = getTextDirection(input.value || input.textContent); // resolve dir attribute of the element

			if (util.isIE) {
				addBidiEventListeners(input);
			}
		}
	}
	
	function enforceTextDir(range) {
		var comments = [{name:"comment block"}, 
		                {name:"comment line double-slash"},
		                {name:"comment block documentation"},
		                {name:"comment line double-slash jade"},
		                {name:"comment line"},
		                {name:"comment line number-sign php"},
		                {name:"comment block xml"}
		];
		var text = range.text;
		var style = range.style;
		if (isBidiEnabled() && style && style.styleClass && style.styleClass.startsWith("comment") && text.length > 0) {
			for (var i = 0; i < comments.length; i++) {
				if (style.styleClass === comments[i].name) {
					var newStyle = style;
					if (typeof newStyle.attributes === "undefined") {
						newStyle.attributes = {};
					}
					newStyle.attributes.dir = getTextDirection(text);
					range.style = newStyle;		
					return range;
				}
			}
		}
		return range;
	}
	
	function enforceSTT(text, type) {
		if (isBidiEnabled() && text && type) {
			return bidiFormat.getString(text, type, {subDir : getBidiLayout()} , false, 'en');
		}
	    return text;
	}

		
	return {
		isBidiEnabled: isBidiEnabled,
		getTextDirection: getTextDirection,		
		enforceTextDirWithUcc: enforceTextDirWithUcc,
		initInputField: initInputField,
		enforceTextDir: enforceTextDir,
		enforceSTT: enforceSTT
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors: IBM Corporation - initial API and implementation
 *******************************************************************************/

/*eslint-env browser, amd*/
define('orion/urlModifier',[],function() {	
	function modifyUrl(value) {
		return value;
	}
	return modifyUrl;
});

/*******************************************************************************
 * @license
 * Copyright (c) 2009, 2012, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors: IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('orion/uiUtils',[
	'i18n!orion/nls/messages',
	'orion/webui/littlelib',
	'orion/i18nUtil',
	'orion/bidiUtils',
	'orion/urlModifier'
], function(messages, lib, i18nUtil, bidiUtils, urlModifier) {
	/**
	 * This class contains static utility methods. It is not intended to be instantiated.
	 * @class This class contains static utility methods.
	 * @name orion.uiUtils
	 */

	var isMac = navigator.platform.indexOf("Mac") !== -1; //$NON-NLS-0$

	// Maps keyCode to display symbol
	var keySymbols = Object.create(null);
	keySymbols[lib.KEY.DOWN]  = "\u2193"; //$NON-NLS-0$
	keySymbols[lib.KEY.UP]    = "\u2191"; //$NON-NLS-0$
	keySymbols[lib.KEY.RIGHT] = "\u2192"; //$NON-NLS-0$
	keySymbols[lib.KEY.LEFT]  = "\u2190"; //$NON-NLS-0$
	if (isMac) {
		keySymbols[lib.KEY.BKSPC]    = "\u232b"; //$NON-NLS-0$
		keySymbols[lib.KEY.DEL]      = "\u2326"; //$NON-NLS-0$
		keySymbols[lib.KEY.END]      = "\u21f2"; //$NON-NLS-0$
		keySymbols[lib.KEY.ENTER]    = "\u23ce"; //$NON-NLS-0$
		keySymbols[lib.KEY.ESCAPE]   = "\u238b"; //$NON-NLS-0$
		keySymbols[lib.KEY.HOME]     = "\u21f1"; //$NON-NLS-0$
		keySymbols[lib.KEY.PAGEDOWN] = "\u21df"; //$NON-NLS-0$
		keySymbols[lib.KEY.PAGEUP]   = "\u21de"; //$NON-NLS-0$
		keySymbols[lib.KEY.SPACE]    = "\u2423"; //$NON-NLS-0$
		keySymbols[lib.KEY.TAB]      = "\u21e5"; //$NON-NLS-0$
	}

	function getUserKeyStrokeString(binding) {
		var userString = "";

		if (isMac) {
			if (binding.mod4) {
				userString+= "\u2303"; //Ctrl //$NON-NLS-0$
			}
			if (binding.mod3) {
				userString+= "\u2325"; //Alt //$NON-NLS-0$
			}
			if (binding.mod2) {
				userString+= "\u21e7"; //Shift //$NON-NLS-0$
			}
			if (binding.mod1) {
				userString+= "\u2318"; //Command //$NON-NLS-0$
			}
		} else {
			var PLUS = "+"; //$NON-NLS-0$;
			if (binding.mod1)
				userString += messages.KeyCTRL + PLUS;
			if (binding.mod2)
				userString += messages.KeySHIFT + PLUS;
			if (binding.mod3)
				userString += messages.KeyALT + PLUS;
		}
		
		if (binding.alphaKey) {
			return userString+binding.alphaKey;
		}
		if (binding.type === "keypress") {
			return userString+binding.keyCode; 
		}

		// Check if it has a special symbol defined
		var keyCode = binding.keyCode;
		var symbol = keySymbols[keyCode];
		if (symbol) {
			return userString + symbol;
		}

		// Check if it's a known named key from lib.KEY
		var keyName = lib.keyName(keyCode);
		if (keyName) {
			// Some key names are translated, so check for that.
			keyName = messages["Key" + keyName] || keyName; //$NON-NLS-0$
			return userString + keyName;
		}

		var character;
		switch (binding.keyCode) {
			case 59:
				character = binding.mod2 ? ":" : ";"; //$NON-NLS-1$ //$NON-NLS-0$
				break;
			case 61:
				character = binding.mod2 ? "+" : "="; //$NON-NLS-1$ //$NON-NLS-0$
				break;
			case 188:
				character = binding.mod2 ? "<" : ","; //$NON-NLS-1$ //$NON-NLS-0$
				break;
			case 190:
				character = binding.mod2 ? ">" : "."; //$NON-NLS-1$ //$NON-NLS-0$
				break;
			case 191:
				character = binding.mod2 ? "?" : "/"; //$NON-NLS-1$ //$NON-NLS-0$
				break;
			case 192:
				character = binding.mod2 ? "~" : "`"; //$NON-NLS-1$ //$NON-NLS-0$
				break;
			case 219:
				character = binding.mod2 ? "{" : "["; //$NON-NLS-1$ //$NON-NLS-0$
				break;
			case 220:
				character = binding.mod2 ? "|" : "\\"; //$NON-NLS-1$ //$NON-NLS-0$
				break;
			case 221:
				character = binding.mod2 ? "}" : "]"; //$NON-NLS-1$ //$NON-NLS-0$
				break;
			case 222:
				character = binding.mod2 ? '"' : "'"; //$NON-NLS-1$ //$NON-NLS-0$
				break;
			}
		if (character) {
			return userString+character;
		}
		if (binding.keyCode >= 112 && binding.keyCode <= 123) {
			return userString+"F"+ (binding.keyCode - 111); //$NON-NLS-0$
		}
		return userString+String.fromCharCode(binding.keyCode);
	}

	function getUserKeyString(binding) {
		var result = "";
		var keys = binding.getKeys();
		for (var i = 0; i < keys.length; i++) {
			if (i !== 0) {
				result += " "; //$NON-NLS-0$
			}
			result += getUserKeyStrokeString(keys[i]);
		}
		return result;
	}

	/**
	 * @name orion.uiUtils.getUserText
	 * @function
	 * @param {Object} options The options object
	 * @param {String} options.id
	 * @param {Element} options.refNode
	 * @param {Boolean} options.hideRefNode
	 * @param {String} options.initialText
	 * @param {Function} options.onComplete
	 * @param {Function} options.onEditDestroy
	 * @param {String} options.selectTo
	 * @param {Boolean} options.isInitialValid
	 * @param {Boolean} options.insertAsChild
	 */
	function getUserText(options) {
		var id = options.id;
		var refNode = options.refNode;
		var hideRefNode = options.hideRefNode;
		var initialText = options.initialText;
		var onComplete = options.onComplete;
		var onEditDestroy = options.onEditDestroy;
		var selectTo = options.selectTo;
		var isInitialValid = options.isInitialValid;
		var insertAsChild = options.insertAsChild;
		
		var done = false;
		var handler = function(isKeyEvent) {
			return function(event) {
				if (done) {
					return;
				}
				var editBox = lib.node(id),
					newValue = editBox.value;
				if (!editBox) {
					return;
				}
				if (isKeyEvent && event.keyCode === lib.KEY.ESCAPE) {
					if (hideRefNode) {
						refNode.style.display = "";
					}
					done = true;
					editBox.parentNode.removeChild(editBox);
					if (onEditDestroy) {
						onEditDestroy();
					}
					return;
				}
				if (isKeyEvent && event.keyCode !== lib.KEY.ENTER) {
					return;
				} else if (newValue.length === 0 || (!isInitialValid && newValue === initialText)) {
					if (hideRefNode) {
						refNode.style.display = "";
					}
					done = true;
				} else {
					onComplete(newValue);
					if (hideRefNode && refNode.parentNode) {
						refNode.style.display = "";
					}
					done = true;
				}
				// some clients remove temporary dom structures in the onComplete processing, so check that we are still in DOM
				if (editBox.parentNode) {
					editBox.parentNode.removeChild(editBox);
				}
				if (onEditDestroy) {
					onEditDestroy();
				}
			};
		};
	
		// Swap in an editable text field
		var editBox = document.createElement("input"); //$NON-NLS-0$
		editBox.id = id;
		editBox.value = initialText || "";
		if (insertAsChild) {
			refNode.appendChild(editBox);
		} else {
			refNode.parentNode.insertBefore(editBox, refNode.nextSibling);
		}
		editBox.classList.add("userEditBoxPrompt"); //$NON-NLS-0$
		if (hideRefNode) {
			refNode.style.display = "none"; //$NON-NLS-0$
		}	
		bidiUtils.initInputField(editBox);
		editBox.addEventListener("keydown", handler(true), false); //$NON-NLS-0$
		editBox.addEventListener("blur", handler(false), false); //$NON-NLS-0$
		window.setTimeout(function() { 
			editBox.focus(); 
			if (initialText) {
				var box = lib.node(id);
				var end = selectTo ? initialText.indexOf(selectTo) : -1;
				if (end > 0) {
					if(box.createTextRange) {
						var range = box.createTextRange();
						range.collapse(true);
						range.moveStart("character", 0); //$NON-NLS-0$
						range.moveEnd("character", end); //$NON-NLS-0$
						range.select();
					} else if(box.setSelectionRange) {
						box.setSelectionRange(0, end);
					} else if(box.selectionStart !== undefined) {
						box.selectionStart = 0;
						box.selectionEnd = end;
					}
				} else {
					box.select();
				}
			}
		}, 0);
	}
	
	/**
	 * Returns whether the given event should cause a reference
	 * to open in a new window or not.
	 * @param {Object} event The key event
	 * @name orion.util#openInNewWindow
	 * @function
	 */
	function openInNewWindow(event) {
		var isMac = window.navigator.platform.indexOf("Mac") !== -1; //$NON-NLS-0$
		return (isMac && event.metaKey) || (!isMac && event.ctrlKey);
	}
	
	/**
	 * Opens a link in response to some event. Whether the link
	 * is opened in the same window or a new window depends on the event
	 * @param {String} href The link location
	 * @name orion.util#followLink
	 * @function
	 */
	function followLink(href, event) {
		if (event && openInNewWindow(event)) {
			window.open(urlModifier(href));
		} else {
			window.location = urlModifier(href);
		}
	}
	
	function createButton(text, callback) {
		var button = document.createElement("button"); //$NON-NLS-0$
		button.className = "orionButton commandButton commandMargins"; //$NON-NLS-0$
		button.addEventListener("click", function(e) { //$NON-NLS-0$
			callback();
			lib.stop(e);
		}, false);
		if (text) {
			button.appendChild(document.createTextNode(text));
		}
		return button;	
	}
	
	function createDropdownButton(parent, name, populateFunction) {
	}

	/**
	 * Returns whether <code>element</code> or its parent is an HTML5 form element.
	 * @param {Element} element
	 * @param {Element} parentLimit
	 * @function
	 * @returns {Boolean}
	 */
	function isFormElement(element, parentLimit) {
		if (!element || !element.tagName) return false;
		switch (element.tagName.toLowerCase()) {
			case "button": //$NON-NLS-0$
			case "fieldset": //$NON-NLS-0$
			case "form": //$NON-NLS-0$
			case "input": //$NON-NLS-0$
			case "keygen": //$NON-NLS-0$
			case "label": //$NON-NLS-0$
			case "legend": //$NON-NLS-0$
			case "meter": //$NON-NLS-0$
			case "optgroup": //$NON-NLS-0$
			case "output": //$NON-NLS-0$
			case "progress": //$NON-NLS-0$
			case "select": //$NON-NLS-0$
			case "textarea": //$NON-NLS-0$
				return true;
		}
		if (element.parentNode === parentLimit) return false;
		return element.parentNode && isFormElement(element.parentNode, parentLimit);
	}

	/**
	 * Returns the folder name from path.
	 * @param {String} filePath
	 * @param {Boolean} keepTailSlash
	 * @returns {String}
	 */
	function path2FolderName(filePath, keepTailSlash){
		var pathSegs = filePath.split("/");
		pathSegs.splice(pathSegs.length -1, 1);
		return keepTailSlash ? pathSegs.join("/") + "/" : pathSegs.join("/");
	}
	
	function _timeDifference(timeStamp) {
		var commitDate = new Date(timeStamp);
	    var difference = Date.now() - commitDate.getTime();
	    var yearDiff = Math.floor(difference/1000/60/60/24/365);
	    difference -= yearDiff*1000*60*60*24*365;
	    var monthDiff = Math.floor(difference/1000/60/60/24/30);
	    difference -= monthDiff*1000*60*60*24*30;
	    var daysDifference = Math.floor(difference/1000/60/60/24);
	    difference -= daysDifference*1000*60*60*24;
		var hoursDifference = Math.floor(difference/1000/60/60);
	    difference -= hoursDifference*1000*60*60;
	    var minutesDifference = Math.floor(difference/1000/60);
	    difference -= minutesDifference*1000*60;
	    var secondsDifference = Math.floor(difference/1000);
	    return {year: yearDiff, month: monthDiff, day: daysDifference, hour: hoursDifference, minute: minutesDifference, second: secondsDifference};
	}
	
	function _generateTimeString(number, singleTerm, term) {
		if(number > 0) {
			if(number === 1) {
				return messages[singleTerm];
			}
			return i18nUtil.formatMessage(messages[term], number);
		}
		return "";
	}
	
	/**
	 * Returns the time duration passed by now. E.g. "2 minutes", "an hour", "a day", "3 months", "2 years"
	 * @param {String} timeStamp
	 * @returns {String} If the duration is less than 1 minute, it returns empty string "". Otherwise it returns a duration value.
	 */
	function timeElapsed(timeStamp) {
		var diff = _timeDifference(timeStamp);
		var yearStr = _generateTimeString(diff.year, "a year", "years");
		var monthStr = _generateTimeString(diff.month, "a month", "months");
		var dayStr = _generateTimeString(diff.day, "a day", "days");
		var hourStr = _generateTimeString(diff.hour, "an hour", "hours");
		var minuteStr = _generateTimeString(diff.minute, "a minute", "minutes");
		var disPlayStr = "";
		if(yearStr) {
			disPlayStr = diff.year > 0 ? yearStr : yearStr + monthStr;
		} else if(monthStr) {
			disPlayStr = diff.month > 0 ? monthStr : monthStr + dayStr;
		} else if(dayStr) {
			disPlayStr = diff.day > 0 ? dayStr : dayStr + hourStr;
		} else if(hourStr) {
			disPlayStr = diff.hour > 0 ? hourStr : hourStr + minuteStr;
		} else if(minuteStr) {
			disPlayStr = minuteStr;
		}
		return disPlayStr;	
	}
	/**
	 * Returns the displayable time duration passed by now. E.g. "just now", "2 minutes ago", "an hour ago", "a day ago", "3 months ago", "2 years ago"
	 * @param {String} timeStamp
	 * @returns {String} If the duration is less than 1 minute, it returns empty string "just now". Otherwise it returns a duration value.
	 */
	function displayableTimeElapsed(timeStamp) {
		var duration = timeElapsed(timeStamp);
		if(duration) {
			return i18nUtil.formatMessage(messages["timeAgo"], duration);
		}
		return messages["justNow"];
	}
	
	/**
	 * Returns the initial of given name.
	 * For standard names: "First [Midddles] Last", returns capital "FL";
	 * For others, returns substr(0, 2).
	 * 
	 * @param {string} name -
	 * @return {string} - initial
	 */
	function getNameInitial(name) {
		var namePart = name.split(' ');
		if (namePart.length >= 2) {
			return (namePart[0].charAt(0) + namePart[namePart.length - 1].charAt(0)).toUpperCase();
		} else {
			return name.substr(0, 2);
		}
	}
	//return module exports
	return {
		getUserKeyString: getUserKeyString,
		getUserText: getUserText,
		openInNewWindow: openInNewWindow,
		followLink: followLink,
		createButton: createButton,
		createDropdownButton: createDropdownButton,
		isFormElement: isFormElement,
		path2FolderName: path2FolderName,
		timeElapsed: timeElapsed,
		displayableTimeElapsed: displayableTimeElapsed,
		getNameInitial: getNameInitial
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd */
define('orion/collab/collabFileAnnotation',['orion/uiUtils'], function(mUIUtils) {

    'use strict';

    /**
	 * A record of a collaborator annotation in the file tree
	 * 
	 * @constructor
	 * @name {orion.collab.CollabFileAnnotation}
	 * @implements {orion.treetable.TableTree.IAnnotation}
	 * 
	 * @param {string} name - displayed name
	 * @param {string} color - user color
	 * @param {string} location - file location
	 * @param {string} displayedLocation - read friendly location
	 * @param {boolean} editing
	 */
	var CollabFileAnnotation = function(name, color, location, displayedLocation, editing) {
		this.name = name;
		this.color = color;
		// Remove trailing "/"
		if(location.substr(-1) === '/') {
			location = location.substr(0, location.length - 1);
		}
		this.location = location;
		this.displayedLocation = displayedLocation || location;
		this.editing = editing;
	};

	CollabFileAnnotation.prototype = {
		/**
		 * Find the deepest expanded folder item that contains the file having
		 * this annotation.
		 * 
		 * @see IAnnotation for details.
		 * 
		 * @param {orion.explorer.ExplorerModel} model -
		 * @param {Function} callback -
		 */
		findDeepestFitId: function(model, callback) {
			var self = this;
			model.getRoot(function(root) {
				// Find the existing ID reversely
				var location = self.location;
				while (location.length > 0) {
					// Create a fake item
					// NOTE: it's a hack because we don't have any efficient
					//       way to get the actual file node. Instead, we have
					//       to do it recursively starting from the root. If
					//       you find anything wierd happens, change it to the
					//       actual item object.
					var item = {
						Location: location
					};
					var id = model.getId(item);
					// Test if this element exists
					var exists = !!document.getElementById(id);
					if (exists) {
						callback(id);
						return;
					}
					// Not found. This probably means this item is collapsed.
					// Try to find one level upper.
					// Here I assume every url starts from "/"
					location = location.substr(0, location.lastIndexOf('/'));
				}
				// Nothing found
				callback('');
			});
		},

		/**
		 * Get description of this annotation which can be used in for example
		 * tooltip.
		 * 
		 * @return {string} - description
		 */
		getDescription: function() {
			return '<b>' + this.name + '</b> is editing <a href=#' + this.location + '>' + this.displayedLocation + '</a>';
		},

		/**
		 * Generate a new HTML element of this annotation.
		 * 
		 * @return {Element} - the HTML element of this annotation
		 */
		generateHTML: function() {
			var element = document.createElement('div');
			element.innerHTML = mUIUtils.getNameInitial(this.name);
			element.style.backgroundColor = this.color;
			element.classList.add('collabAnnotation');
			if (this.editing) {
				element.classList.add('collabEditing');
			}
			return element;
		}
	};

    return {
		CollabFileAnnotation: CollabFileAnnotation
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd */
define('orion/collab/collabPeer',[], function() {

    'use strict';

	/**
	 * A record of a collaborator
	 * 
	 * @class
	 * @name orion.collabClient.CollabPeer
	 * 
	 * @param {string} id -
	 * @param {string} name -
	 * @param {string} color -
	 */
	var CollabPeer = function(id, name, color) {
		this.id = id;
		this.name = name;
		this.color = color;
	};

    return {
        CollabPeer: CollabPeer
    };
});

/*******************************************************************************
 * @license
 * Copyright (c) 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd */
define('orion/collab/otAdapters',['orion/collab/collabPeer', 'orion/collab/ot', 'orion/uiUtils'], function(mCollabPeer, ot, mUIUtils) {

    'use strict';

    var CollabPeer = mCollabPeer.CollabPeer;
    var TextOperation = ot.TextOperation;
    var Selection = ot.Selection;

    var INPUT_CHANGED_EVENT_INTERVAL = 250;

    var EDITING_FLAG_DURATION = 1000;

	var contextPath = location.pathname.substr(0, location.pathname.lastIndexOf('/edit/edit.html'));

    /**
     * The abstract socket adapter for OT
     *
     * @abstract
     * @class
     * @name orion.collab.OrionSocketAdapter
     *
     * @param {orion.collabClient.CollabClient} collabClient
     */
    var OrionSocketAdapter = function(collabClient) {
        this.collabClient = collabClient;
        this.callbacks = [];
        this.ignoreNextOperation = false;
    };

    OrionSocketAdapter.prototype.constructor = OrionSocketAdapter;
    
    /**
     * Send authenticate message
     */
    OrionSocketAdapter.prototype.authenticate = function() {
        var msg = {
            'type': 'authenticate',
            'token': JSON.parse(localStorage.getItem('orion.user')).jwt,
            'clientId': this.collabClient.getClientId()
        };
        this.send(JSON.stringify(msg));
    };

    /**
     * Send text
     * 
     * Implement this.
     *
     * @param {string} text
     */
    OrionSocketAdapter.prototype.send = function(text) {
        throw new Error('Not implemented.');
    };

    /**
     * Message handler
     *
     * @param {Object} msg
     */
    OrionSocketAdapter.prototype._onMessage = function(msg) {
        if (msg.doc) {
            this._onDocMessage(msg);
        } else {
            this._onSessionMessage(msg);
        }
    };

    /**
     * Document Message handler
     *
     * @param {Object} msg
     */
    OrionSocketAdapter.prototype._onDocMessage = function(msg) {
        if (!this.collabClient.textView) {
            return;
        }
        if (msg.type === "selection" && msg.selection !== null) {
            this.collabClient.otOrionAdapter.destroyCollabAnnotations(msg.clientId);
            if (msg.clientId === this.collabClient.getClientId() && msg.doc !== this.collabClient.currentDoc()) {
                this.sendSelection(null);
            }
        }
        if (msg.doc !== this.collabClient.currentDoc()) {
                return;
        }
        switch(msg.type) {
            case "init-document":
                this.collabClient.startOT(msg.revision, msg.operation, msg.clients);
                this.collabClient.awaitingClients = false;
                break;
            case "ack":
                this.trigger('ack');
                break;
            case "operation":
                if (msg.guid === this.collabClient.guid) {
                    break;
                }
                this.ignoreNextOperation = true;
                try {
                    this.trigger('operation', msg.operation);
                } catch (ex) {
                    this.sendInit();
                } finally {
                    this.ignoreNextOperation = false;
                }
                this.collabClient.editor.markClean();
                break;
            case "selection":
                this.trigger('selection', msg.clientId, msg.selection);
                break;
            case "reconnect":
                this.trigger('reconnect');
                break;
        }
    };

    /**
     * Session Message handler
     *
     * @param {Object} msg
     */
    OrionSocketAdapter.prototype._onSessionMessage = function(msg) {
        var type = msg.type;
        switch (type) {
            case 'file-operation':
                this.collabClient.handleFileOperation(msg);
                break;
        }
    };

    /**
     * Send the initial message
     */
    OrionSocketAdapter.prototype.sendInit = function() {
        var msg = {
            'type': 'join-document',
            'doc': this.collabClient.currentDoc(),
            'clientId': this.collabClient.getClientId()
        };

        this.send(JSON.stringify(msg));
    }

    /**
     * Send OT operation
     * @param {number} revision
     * @param {OT.Operation} operation
     * @param {OT.Selection} selection
     */
    OrionSocketAdapter.prototype.sendOperation = function(revision, operation, selection) {
        if (this.ignoreNextOperation) {
            return;
        }
        var myDoc = this.collabClient.currentDoc();
        var msg = {
            'type': 'operation',
            'revision': revision,
            'operation': operation,
            'selection': selection,
            'doc': myDoc,
            'clientId': this.collabClient.getClientId(),
            'guid': this.collabClient.guid
        };
        this.send(JSON.stringify(msg));
        this.collabClient.editor.markClean();
    };

    /**
     * Send OT selection
     * @param {OT.Selection} selection
     */
    OrionSocketAdapter.prototype.sendSelection = function (selection) {
        var myDoc = this.collabClient.currentDoc();
        var msg = {
            'type': 'selection',
            'selection': selection,
            'doc': myDoc,
            'clientId': this.collabClient.getClientId()
        };
        this.send(JSON.stringify(msg));
    };

    /**
     * Register callbacks.
     * We won't use EventTarget because OT uses registerCallbacks/trigger to
     * perform event operations.
     *
     * @param {Object.<string, (Function|Array.<Function>)>} cb - callbacks
     */
    OrionSocketAdapter.prototype.registerCallbacks = function (cb) {
        this.callbacks = cb;
    };

    /**
     * Trigger an event.
     *
     * @param {Object} event
     */
    OrionSocketAdapter.prototype.trigger = function (event) {
        if (!this.collabClient.textView) return;
        var args = Array.prototype.slice.call(arguments, 1);
        var action = this.callbacks && this.callbacks[event];
        if (action) { action.apply(this, args); }
    };

    /**
     * The socket adapter for OT using CollabSocket as the communitation socket
     *
     * @class
     * @extends orion.collab.OrionSocketAdapter
     * @name orion.collab.OrionCollabSocketAdapter
     *
     * @param {orion.collabClient.CollabClient} client
     * @param {orion.collabClient.CollabSocket} socket
     */
    var OrionCollabSocketAdapter = function(client, socket) {
        OrionSocketAdapter.call(this, client);

        var self = this;
        this.socket = socket;
        this.authenticated = false;

        // Register incoming message handler
        this.socket.addEventListener('message', function(e) {
            self._onMessage(JSON.parse(e.data));
        });
    };

    OrionCollabSocketAdapter.prototype = Object.create(OrionSocketAdapter.prototype);
    OrionCollabSocketAdapter.prototype.constructor = OrionCollabSocketAdapter;

    /**
     * Send text
     *
     * @param {string} text
     */
    OrionCollabSocketAdapter.prototype.send = function(text) {
        this.socket.send(text);
    };

    /**
     * Session Message handler
     *
     * @param {Object} msg
     */
    OrionCollabSocketAdapter.prototype._onDocMessage = function(msg) {
        switch (msg.type) {
            case 'client-joined-doc':
                // Add new client to OT
                this.trigger('client_joined', msg.clientId, this.collabClient.getPeer(msg.clientId));
                break;

            case 'client-left-doc':
                // Clear this client's line annotation
                this.trigger('client_left', msg.clientId);
                break;

            default:
                OrionSocketAdapter.prototype._onDocMessage.call(this, msg);
                break;
        }
    };

    /**
     * Session Message handler
     *
     * @param {Object} msg
     */
    OrionCollabSocketAdapter.prototype._onSessionMessage = function(msg) {
        switch(msg.type) {
            case 'authenticated':
                // Do some initialization
                this.authenticated = true;
                this.collabClient.sendCurrentLocation();
                this.updateClient({
                    name: this.collabClient.getClientDisplayedName()
                });
                this.send(JSON.stringify({
                    type: 'get-clients'
                }));
                if (this.collabClient.textView) {
					this.collabClient.viewInstalled();
                }
                break;

            case 'client-joined':
            case 'client-updated':
                this.collabClient.addOrUpdatePeer(new CollabPeer(msg.clientId, msg.name, msg.color));
                if (msg.location) {
                    this.collabClient.addOrUpdateCollabFileAnnotation(msg.clientId, contextPath, msg.location, msg.editing);
                } else {
                    this.collabClient.addOrUpdateCollabFileAnnotation(msg.clientId, contextPath,'', msg.editing);
                }
                break;

            case 'client-left':
                this.collabClient.removePeer(msg.clientId);
                break;

            default:
                OrionSocketAdapter.prototype._onSessionMessage.call(this, msg);
                break;
        }
    };

    /**
     * Send current location to the server
     * 
     * @param {string} location
     */
    OrionCollabSocketAdapter.prototype.sendLocation = function(location) {
        this.send(JSON.stringify({
            type: 'update-client',
            location: location,
            editing: this.collabClient.editing
        }));
    };

    /**
     * Update this client's info to the server
     * 
     * @param {Object} clientData - fields to update
     */
    OrionCollabSocketAdapter.prototype.updateClient = function(clientData) {
        clientData.type = 'update-client';
        this.send(JSON.stringify(clientData));
    };

    var OrionEditorAdapter = function (editor, collabClient, annotationTypes) {
        this.editor = editor;
        this.orion = editor.getTextView();
        this.model = editor.getModel();
        this.ignoreNextChange = false;
        this.changeInProgress = false;
        this.selectionChanged = false;
        this.myLine = 0;
        this.deleteContent = "";
        this.AT = annotationTypes;
        this.annotations = {};
        this.collabClient = collabClient;
        this.inputChangedRequested = false;
        this.collabClient.editing = false;
        this.editingTimeout = 0;

        this.destroyCollabAnnotations();

        this._onChanging = this.onChanging.bind(this);
        this._onChanged = this.onChanged.bind(this);
        this._onCursorActivity = this.onCursorActivity.bind(this);
        this._onFocus = this.onFocus.bind(this);
        this._onBlur = this.onBlur.bind(this);
        this._selectionListener = this.selectionListener.bind(this);

        this.model.addEventListener('Changing', this._onChanging);
        this.model.addEventListener('Changed', this._onChanged);
        this.orion.addEventListener('cursorActivity', this._onCursorActivity);
        this.orion.addEventListener('Focus', this._onFocus);
        this.orion.addEventListener('blur', this._onBlur);
        this.orion.addEventListener('Selection', this._selectionListener);
    }

    // Removes all event listeners from the Orion instance.
    OrionEditorAdapter.prototype.detach = function () {
        this.model.removeEventListener('Changing', this._onChanging);
        this.model.removeEventListener('Changed', this._onChanged);
        this.orion.removeEventListener('cursorActivity', this._onCursorActivity);
        this.orion.removeEventListener('Focus', this._onFocus);
        this.orion.removeEventListener('blur', this._onBlur);
        this.orion.removeEventListener('Selection', this._selectionListener);
    };

    // Converts a Orion change array (as obtained from the 'changes' event
    // in Orion v4) or single change or linked list of changes (as returned
    // by the 'change' event in Orion prior to version 4) into a
    // TextOperation and its inverse and returns them as a two-element array.
    OrionEditorAdapter.prototype.operationFromOrionChanges = function (changes, doc, deletedText) {
        // Approach: Replay the changes, beginning with the most recent one, and
        // construct the operation and its inverse. We have to convert the position
        // in the pre-change coordinate system to an index. We have a method to
        // convert a position in the coordinate system after all changes to an index,
        // namely Orion's `indexFromPos` method. We can use the information of
        // a single change object to convert a post-change coordinate system to a
        // pre-change coordinate system. We can now proceed inductively to get a
        // pre-change coordinate system for all changes in the linked list.
        // A disadvantage of this approach is its complexity `O(n^2)` in the length
        // of the linked list of changes.

        var docEndLength = this.model.getCharCount() - changes[0].addedCharCount + changes[0].removedCharCount;
        var operation    = new TextOperation().retain(docEndLength);
        var inverse      = new TextOperation().retain(docEndLength);

        for (var i = changes.length - 1; i >= 0; i--) {
            var change = changes[i];

            var fromIndex = change.start;
            var restLength = docEndLength - fromIndex - change.removedCharCount;

            operation = operation.compose(new TextOperation()
                .retain(fromIndex)
                ['delete'](change.removedCharCount)
                .insert(change.text)
                .retain(restLength)
            );

            if (change.addedCharCount && change.removedCharCount) {
            //REPLACE ACTION
            inverse = new TextOperation()
                .retain(fromIndex)
                ['delete'](change.addedCharCount)
                .insert(deletedText)
                .retain(restLength)
                .compose(inverse);
            } else if (change.addedCharCount) {
            //INSERT ACTION
            inverse = new TextOperation()
                .retain(fromIndex)
                ['delete'](change.addedCharCount)
                .retain(restLength)
                .compose(inverse);
            } else {
            //DELETE ACTION
            inverse = new TextOperation()
                .retain(fromIndex)
                .insert(deletedText)
                .retain(restLength)
                .compose(inverse);
            }

            docEndLength += change.removedCharCount - change.text.length;
        }

        return [operation, inverse];
    };

    /**
     *  Apply an operation to a Orion instance.
     * 
     * @throws {Error} operation bound check failed
     * 
     * @param {ot.Operation} operation -
     * @param {TextView} orion - 
     */
    OrionEditorAdapter.prototype.applyOperationToOrion = function (operation, orion) {
        var ops = operation.ops;
        var index = 0; // holds the current index into Orion's content
        var oldLine = this.myLine; // Track the current line before this operation
        var docLength = this.model.getCharCount(); // Track the doc length and verify it at the end
        for (var i = 0, l = ops.length; i < l; i++) {
            var op = ops[i];
            if (TextOperation.isRetain(op)) {
                index += op;
                if (index > docLength || index < 0) {
                    throw new Error('Invalid retain.');
                }
            } else if (TextOperation.isInsert(op)) {
                this.model.setText(op, index, i < (ops.length - 1) ? index : undefined);
                index += op.length;
                docLength += op.length;
                this.requestInputChangedEvent();
            } else if (TextOperation.isDelete(op)) {
                var from = index;
                var to   = index - op;
                docLength += op;
                if (to < 0) {
                    throw new Error('Invalid deletion');
                }
                this.model.setText('', from, to);
                this.requestInputChangedEvent();
            }
        }
        // Verify doc length
        if (index !== docLength) {
            throw new Error('Invalid doc length. Unsynchronized content.');
        }
        // Check if current line is changed
        var deltaLine = this.myLine - oldLine;
        if (deltaLine !== 0) {
            // Try to put the current line at the same position on the screen
            var lineHeight = this.collabClient.textView.getLineHeight();
			var deltaY = deltaLine * lineHeight;
            var originY = this.collabClient.textView.getTopPixel();
            this.collabClient.textView.setTopPixel(originY + deltaY);
        }
    };

    OrionEditorAdapter.prototype.registerCallbacks = function (cb) {
        this.callbacks = cb;

        // Give initial cursor position
        var cursor = this.editor.getSelection().start;
        this.selectionListener({
            newValue: {
                start: cursor
            }
        });
    };

    OrionEditorAdapter.prototype.onChanging = function (change) {
        // By default, Orion's event order is the following:
        // 1. 'ModelChanging', 2. 'ModelChanged'
        // We want to fire save the deleted/replaced text during a 'modelChanging' event if applicable,
        // so that we can use it to create the reverse operation used for the undo-stack after the model has changed.
        if (change.removedCharCount > 0) {
            this.deleteContent = this.model.getText(change.start, change.start + change.removedCharCount);
        }

        this.changeInProgress = true;
    };

    OrionEditorAdapter.prototype.onChanged = function (change) {
        var self = this;
        this.changeInProgress = true;
        if (!this.ignoreNextChange) {
            var pair = this.operationFromOrionChanges([change], this.orion, this.deleteContent);
            this.trigger('change', pair[0], pair[1]);
            // Send editing flag
            this.collabClient.editing = true;
            if (this.editingTimeout) {
                clearTimeout(this.editingTimeout);
            } else {
                this.collabClient.sendCurrentLocation();
                this.collabClient.updateSelfFileAnnotation();
            }
            this.editingTimeout = setTimeout(function() {
                self.editingTimeout = 0;
                self.collabClient.editing = false;
                self.collabClient.sendCurrentLocation();
                self.collabClient.updateSelfFileAnnotation();
            }, EDITING_FLAG_DURATION);
        }
        this.deleteContent = "";
        if (this.selectionChanged) { this.trigger('selectionChange'); }
        this.changeInProgress = false;
        // this.ignoreNextChange = false;
        this.requestInputChangedEvent();
    };

    OrionEditorAdapter.prototype.onCursorActivity = function () {
        if (this.changeInProgress) {
            this.selectionChanged = true;
        } else {
            this.trigger('selectionChange');
        }
    };
    
    OrionEditorAdapter.prototype.onFocus = function () {
        var clientId = this.collabClient.getClientId();
        var peer = this.collabClient.getPeer(clientId);
        var name = peer ? peer.name : undefined;
        var color = peer ? peer.color : color;
        var selection = this.getSelection();
        this.updateLineAnnotation(clientId, selection, name, color);
        this.collabClient.otSocketAdapter.sendSelection(selection);
    };

    OrionEditorAdapter.prototype.onBlur = function () {
        if (!this.orion.somethingSelected()) { this.trigger('blur'); }
    };

    OrionEditorAdapter.prototype.getValue = function () {
        return this.model.getText();
    };

    OrionEditorAdapter.prototype.getSelection = function () {
        return ot.Selection.createCursor(this.editor.getSelection().start);
    };

    OrionEditorAdapter.prototype.setSelection = function (selection) {
      // var ranges = [];
      // for (var i = 0; i < selection.ranges.length; i++) {
      //   var range = selection.ranges[i];
      //   ranges[i] = {
      //     anchor: this.orion.posFromIndex(range.anchor),
      //     head:   this.orion.posFromIndex(range.head)
      //   };
      // }
      // this.orion.setSelections(ranges);
    };

    var addStyleRule = (function () {
        var added = {};
        var styleElement = document.createElement('style');
        document.documentElement.getElementsByTagName('head')[0].appendChild(styleElement);
        var styleSheet = styleElement.sheet;

        return function (css) {
            if (added[css]) { return; }
            added[css] = true;
            styleSheet.insertRule(css, (styleSheet.cssRules || styleSheet.rules).length);
        };
    }());

    OrionEditorAdapter.prototype.selectionListener = function(e) {
        var offset = e.newValue.start;
        var currLine = this.model.getLineAtOffset(offset);
        var lastLine = this.model.getLineCount()-1;
        var lineStartOffset = this.model.getLineStart(currLine);

        if (offset) {
            //decide whether or not it is worth sending (if line has changed or needs updating).
            if (currLine !== this.myLine || currLine === lastLine || currLine === 0) {
                // Send this change
            } else {
                return;
            }
        }

        this.myLine = currLine;

        // Self-tracking
        var clientId = this.collabClient.getClientId();
        var peer = this.collabClient.getPeer(clientId);
        var name = peer ? peer.name : undefined;
        var color = peer ? peer.color : color;
        var selection = ot.Selection.createCursor(offset);
        this.updateLineAnnotation(clientId, selection, name, color);

        if (this.changeInProgress) {
            this.selectionChanged = true;
        } else {
            this.trigger('selectionChange');
        }
    };

    OrionEditorAdapter.prototype.setOtherSelection = function (selection, color, clientId) {
        if (clientId === this.collabClient.getClientId()) {
            // Don't update self by remote
            return {
                clear: function() {
                    // NOOP
                }
            };
        }
        var peer = this.collabClient.getPeer(clientId);
        var name = peer ? peer.name : undefined;
        color = peer ? peer.color : color;
        this.updateLineAnnotation(clientId, selection, name, color);
        var self = this;
        return {
            clear: function() {
                self.destroyCollabAnnotations(clientId);
            }
        };
    };

    OrionEditorAdapter.prototype.updateLineAnnotation = function(id, selection, name, color, force) {
        force = !!force;
        name = name || 'Unknown';
        color = color || '#000000';
        var cursor = selection.ranges[0].head || 0;
        var annotationModel = this.editor.getAnnotationModel();
        var ann = this.AT.createAnnotation(this.AT.ANNOTATION_COLLAB_LINE_CHANGED, cursor, cursor, name + " is editing");
        var initial = mUIUtils.getNameInitial(name);
        ann.html = ann.html.substring(0, ann.html.indexOf('></div>')) + " style='background-color:" + color + "'>" + initial + "</div>";
        ann.peerId = id;
        var peerId = id;

        /*if peer isn't being tracked yet, start tracking
        * else replace previous annotation
        */
        if (!(peerId in this.annotations && this.annotations[peerId]._annotationModel)) {
            this.annotations[peerId] = ann;
            annotationModel.addAnnotation(this.annotations[peerId]);
        } else {
            var currAnn = this.annotations[peerId];
            if (!force && ann.start === currAnn.start) return;
            annotationModel.replaceAnnotations([currAnn], [ann]);
            this.annotations[peerId] = ann;
        }
    };

    /**
     * Update the line annotation of a peer without change its line number
     * i.e. only updates name and color
     * @param {string} id - clientId
     */
    OrionEditorAdapter.prototype.updateLineAnnotationStyle = function(id) {
        var peer = this.collabClient.getPeer(id);
        var name = peer ? peer.name : undefined;
        var color = peer ? peer.color : undefined;
        var annotation = this.annotations[id];
        if (!annotation) {
            return;
        }
        var cursor = annotation.start;
        var selection = ot.Selection.createCursor(cursor);
        this.updateLineAnnotation(id, selection, name, color, true);
    };

    OrionEditorAdapter.prototype.destroyCollabAnnotations = function(peerId) {
      var annotationModel = this.editor.getAnnotationModel();
      var currAnn = null;

      /*If a peer is specified, just remove their annotation
      * Else remove all peers' annotations.
      */
      if (peerId) {
        if (this.annotations[peerId]) {
          //remove that users annotation
          currAnn = this.annotations[peerId];
          annotationModel.removeAnnotation(currAnn);
          delete this.annotations[peerId];
        }
      } else {
        //the session has ended remove everyone's annotation
        annotationModel.removeAnnotations(this.AT.ANNOTATION_COLLAB_LINE_CHANGED);
        this.annotations = {};
      }
    };

    OrionEditorAdapter.prototype.trigger = function (event) {
      var args = Array.prototype.slice.call(arguments, 1);
      var action = this.callbacks && this.callbacks[event];
      if (action) { action.apply(this, args); }
    };

    OrionEditorAdapter.prototype.applyOperation = function (operation) {
      this.ignoreNextChange = true;
      this.applyOperationToOrion(operation, this.model);
      this.ignoreNextChange = false;
    };

    OrionEditorAdapter.prototype.registerUndo = function (undoFn) {
      // this.orion.undo = undoFn;
      this.orion.setAction("undo", undoFn);
    };

    OrionEditorAdapter.prototype.registerRedo = function (redoFn) {
      // this.orion.redo = redoFn;
      this.orion.setAction("redo", redoFn);
    };

    /**
     * Trigger a delayed InputChanged event.
     * In collab mode, client-side auto saving is disabled. As a result, the
     * syntax checker won't work. So here we simulates a InputChanged event.
     */
    OrionEditorAdapter.prototype.requestInputChangedEvent = function() {
        if (!this.inputChangedRequested) {
            this.inputChangedRequested = true;
            var self = this;
            var editor = self.collabClient.editor;
            setTimeout(function() {
                editor.onInputChanged({
                    type: "InputChanged", //$NON-NLS-0$
                    title: editor.getTitle(),
                    message: null,
                    contents: editor.getText(),
                    contentsSaved: true
                });
                self.inputChangedRequested = false;
            }, INPUT_CHANGED_EVENT_INTERVAL);
        }
    };

    return {
        OrionCollabSocketAdapter: OrionCollabSocketAdapter,
        OrionEditorAdapter: OrionEditorAdapter
    };
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('orion/EventTarget',[],function() {
	/**
	 * Creates an Event Target
	 *
	 * @name orion.EventTarget
	 * @class Base for creating an Orion event target
	 */
	function EventTarget() {
		this._namedListeners = {};
	}

	EventTarget.prototype = /** @lends orion.EventTarget.prototype */
	{
		/**
		 * Dispatches a named event along with an arbitrary set of arguments. Any arguments after <code>eventName</code>
		 * will be passed to the event listener(s).
		 * @param {Object} event The event to dispatch. The event object MUST have a type field
		 * @returns {boolean} false if the event has been canceled and any associated default action should not be performed
		 * listeners (if any) have resolved.
		 */
		dispatchEvent: function(event) {
			if (!event.type) {
				throw new Error("unspecified type");
			}
			var listeners = this._namedListeners[event.type];
			if (listeners) {
				listeners.forEach(function(listener) {
					try {
						if (typeof listener === "function") {
							listener(event);
						} else {
							listener.handleEvent(event);
						}
					} catch (e) {
						if (typeof console !== 'undefined') {
							console.log(e); // for now, probably should dispatch an ("error", e)
						}
					}			
				});
			}
			return !event.defaultPrevented;
		},

		/**
		 * Adds an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		addEventListener: function(eventName, listener) {
			if (typeof listener === "function" || listener.handleEvent) {
				this._namedListeners[eventName] = this._namedListeners[eventName] || [];
				this._namedListeners[eventName].push(listener);
			}
		},

		/**
		 * Removes an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		removeEventListener: function(eventName, listener) {
			var listeners = this._namedListeners[eventName];
			if (listeners) {
				for (var i = 0; i < listeners.length; i++) {
					if (listeners[i] === listener) {
						if (listeners.length === 1) {
							delete this._namedListeners[eventName];
						} else {
							listeners.splice(i, 1);
						}
						break;
					}
				}
			}
		}
	};
	EventTarget.prototype.constructor = EventTarget;
	
	EventTarget.attach = function(obj) {
		var eventTarget = new EventTarget();
		obj.dispatchEvent = eventTarget.dispatchEvent.bind(eventTarget);
		obj.addEventListener = eventTarget.addEventListener.bind(eventTarget);
		obj.removeEventListener = eventTarget.removeEventListener.bind(eventTarget);
	};
	
	return EventTarget;
});
!function(t,e){"object"==typeof exports&&"object"==typeof module?module.exports=e():"function"==typeof define&&define.amd?define('socket.io/socket.io',[],e):"object"==typeof exports?exports.io=e():t.io=e()}(this,function(){return function(t){function e(r){if(n[r])return n[r].exports;var o=n[r]={exports:{},id:r,loaded:!1};return t[r].call(o.exports,o,o.exports,e),o.loaded=!0,o.exports}var n={};return e.m=t,e.c=n,e.p="",e(0)}([function(t,e,n){"use strict";function r(t,e){"object"===("undefined"==typeof t?"undefined":o(t))&&(e=t,t=void 0),e=e||{};var n,r=i(t),s=r.source,u=r.id,h=r.path,f=p[u]&&h in p[u].nsps,l=e.forceNew||e["force new connection"]||!1===e.multiplex||f;return l?(c("ignoring socket cache for %s",s),n=a(s,e)):(p[u]||(c("new io instance for %s",s),p[u]=a(s,e)),n=p[u]),r.query&&!e.query&&(e.query=r.query),n.socket(r.path,e)}var o="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function(t){return typeof t}:function(t){return t&&"function"==typeof Symbol&&t.constructor===Symbol&&t!==Symbol.prototype?"symbol":typeof t},i=n(1),s=n(7),a=n(13),c=n(3)("socket.io-client");t.exports=e=r;var p=e.managers={};e.protocol=s.protocol,e.connect=r,e.Manager=n(13),e.Socket=n(37)},function(t,e,n){(function(e){"use strict";function r(t,n){var r=t;n=n||e.location,null==t&&(t=n.protocol+"//"+n.host),"string"==typeof t&&("/"===t.charAt(0)&&(t="/"===t.charAt(1)?n.protocol+t:n.host+t),/^(https?|wss?):\/\//.test(t)||(i("protocol-less url %s",t),t="undefined"!=typeof n?n.protocol+"//"+t:"https://"+t),i("parse %s",t),r=o(t)),r.port||(/^(http|ws)$/.test(r.protocol)?r.port="80":/^(http|ws)s$/.test(r.protocol)&&(r.port="443")),r.path=r.path||"/";var s=r.host.indexOf(":")!==-1,a=s?"["+r.host+"]":r.host;return r.id=r.protocol+"://"+a+":"+r.port,r.href=r.protocol+"://"+a+(n&&n.port===r.port?"":":"+r.port),r}var o=n(2),i=n(3)("socket.io-client:url");t.exports=r}).call(e,function(){return this}())},function(t,e){var n=/^(?:(?![^:@]+:[^:@\/]*@)(http|https|ws|wss):\/\/)?((?:(([^:@]*)(?::([^:@]*))?)?@)?((?:[a-f0-9]{0,4}:){2,7}[a-f0-9]{0,4}|[^:\/?#]*)(?::(\d*))?)(((\/(?:[^?#](?![^?#\/]*\.[^?#\/.]+(?:[?#]|$)))*\/?)?([^?#\/]*))(?:\?([^#]*))?(?:#(.*))?)/,r=["source","protocol","authority","userInfo","user","password","host","port","relative","path","directory","file","query","anchor"];t.exports=function(t){var e=t,o=t.indexOf("["),i=t.indexOf("]");o!=-1&&i!=-1&&(t=t.substring(0,o)+t.substring(o,i).replace(/:/g,";")+t.substring(i,t.length));for(var s=n.exec(t||""),a={},c=14;c--;)a[r[c]]=s[c]||"";return o!=-1&&i!=-1&&(a.source=e,a.host=a.host.substring(1,a.host.length-1).replace(/;/g,":"),a.authority=a.authority.replace("[","").replace("]","").replace(/;/g,":"),a.ipv6uri=!0),a}},function(t,e,n){(function(r){function o(){return!("undefined"==typeof window||!window.process||"renderer"!==window.process.type)||("undefined"!=typeof document&&document.documentElement&&document.documentElement.style&&document.documentElement.style.WebkitAppearance||"undefined"!=typeof window&&window.console&&(window.console.firebug||window.console.exception&&window.console.table)||"undefined"!=typeof navigator&&navigator.userAgent&&navigator.userAgent.toLowerCase().match(/firefox\/(\d+)/)&&parseInt(RegExp.$1,10)>=31||"undefined"!=typeof navigator&&navigator.userAgent&&navigator.userAgent.toLowerCase().match(/applewebkit\/(\d+)/))}function i(t){var n=this.useColors;if(t[0]=(n?"%c":"")+this.namespace+(n?" %c":" ")+t[0]+(n?"%c ":" ")+"+"+e.humanize(this.diff),n){var r="color: "+this.color;t.splice(1,0,r,"color: inherit");var o=0,i=0;t[0].replace(/%[a-zA-Z%]/g,function(t){"%%"!==t&&(o++,"%c"===t&&(i=o))}),t.splice(i,0,r)}}function s(){return"object"==typeof console&&console.log&&Function.prototype.apply.call(console.log,console,arguments)}function a(t){try{null==t?e.storage.removeItem("debug"):e.storage.debug=t}catch(n){}}function c(){var t;try{t=e.storage.debug}catch(n){}return!t&&"undefined"!=typeof r&&"env"in r&&(t=r.env.DEBUG),t}function p(){try{return window.localStorage}catch(t){}}e=t.exports=n(5),e.log=s,e.formatArgs=i,e.save=a,e.load=c,e.useColors=o,e.storage="undefined"!=typeof chrome&&"undefined"!=typeof chrome.storage?chrome.storage.local:p(),e.colors=["lightseagreen","forestgreen","goldenrod","dodgerblue","darkorchid","crimson"],e.formatters.j=function(t){try{return JSON.stringify(t)}catch(e){return"[UnexpectedJSONParseError]: "+e.message}},e.enable(c())}).call(e,n(4))},function(t,e){function n(){throw new Error("setTimeout has not been defined")}function r(){throw new Error("clearTimeout has not been defined")}function o(t){if(u===setTimeout)return setTimeout(t,0);if((u===n||!u)&&setTimeout)return u=setTimeout,setTimeout(t,0);try{return u(t,0)}catch(e){try{return u.call(null,t,0)}catch(e){return u.call(this,t,0)}}}function i(t){if(h===clearTimeout)return clearTimeout(t);if((h===r||!h)&&clearTimeout)return h=clearTimeout,clearTimeout(t);try{return h(t)}catch(e){try{return h.call(null,t)}catch(e){return h.call(this,t)}}}function s(){y&&l&&(y=!1,l.length?d=l.concat(d):m=-1,d.length&&a())}function a(){if(!y){var t=o(s);y=!0;for(var e=d.length;e;){for(l=d,d=[];++m<e;)l&&l[m].run();m=-1,e=d.length}l=null,y=!1,i(t)}}function c(t,e){this.fun=t,this.array=e}function p(){}var u,h,f=t.exports={};!function(){try{u="function"==typeof setTimeout?setTimeout:n}catch(t){u=n}try{h="function"==typeof clearTimeout?clearTimeout:r}catch(t){h=r}}();var l,d=[],y=!1,m=-1;f.nextTick=function(t){var e=new Array(arguments.length-1);if(arguments.length>1)for(var n=1;n<arguments.length;n++)e[n-1]=arguments[n];d.push(new c(t,e)),1!==d.length||y||o(a)},c.prototype.run=function(){this.fun.apply(null,this.array)},f.title="browser",f.browser=!0,f.env={},f.argv=[],f.version="",f.versions={},f.on=p,f.addListener=p,f.once=p,f.off=p,f.removeListener=p,f.removeAllListeners=p,f.emit=p,f.prependListener=p,f.prependOnceListener=p,f.listeners=function(t){return[]},f.binding=function(t){throw new Error("process.binding is not supported")},f.cwd=function(){return"/"},f.chdir=function(t){throw new Error("process.chdir is not supported")},f.umask=function(){return 0}},function(t,e,n){function r(t){var n,r=0;for(n in t)r=(r<<5)-r+t.charCodeAt(n),r|=0;return e.colors[Math.abs(r)%e.colors.length]}function o(t){function n(){if(n.enabled){var t=n,r=+new Date,o=r-(p||r);t.diff=o,t.prev=p,t.curr=r,p=r;for(var i=new Array(arguments.length),s=0;s<i.length;s++)i[s]=arguments[s];i[0]=e.coerce(i[0]),"string"!=typeof i[0]&&i.unshift("%O");var a=0;i[0]=i[0].replace(/%([a-zA-Z%])/g,function(n,r){if("%%"===n)return n;a++;var o=e.formatters[r];if("function"==typeof o){var s=i[a];n=o.call(t,s),i.splice(a,1),a--}return n}),e.formatArgs.call(t,i);var c=n.log||e.log||console.log.bind(console);c.apply(t,i)}}return n.namespace=t,n.enabled=e.enabled(t),n.useColors=e.useColors(),n.color=r(t),"function"==typeof e.init&&e.init(n),n}function i(t){e.save(t),e.names=[],e.skips=[];for(var n=("string"==typeof t?t:"").split(/[\s,]+/),r=n.length,o=0;o<r;o++)n[o]&&(t=n[o].replace(/\*/g,".*?"),"-"===t[0]?e.skips.push(new RegExp("^"+t.substr(1)+"$")):e.names.push(new RegExp("^"+t+"$")))}function s(){e.enable("")}function a(t){var n,r;for(n=0,r=e.skips.length;n<r;n++)if(e.skips[n].test(t))return!1;for(n=0,r=e.names.length;n<r;n++)if(e.names[n].test(t))return!0;return!1}function c(t){return t instanceof Error?t.stack||t.message:t}e=t.exports=o.debug=o["default"]=o,e.coerce=c,e.disable=s,e.enable=i,e.enabled=a,e.humanize=n(6),e.names=[],e.skips=[],e.formatters={};var p},function(t,e){function n(t){if(t=String(t),!(t.length>100)){var e=/^((?:\d+)?\.?\d+) *(milliseconds?|msecs?|ms|seconds?|secs?|s|minutes?|mins?|m|hours?|hrs?|h|days?|d|years?|yrs?|y)?$/i.exec(t);if(e){var n=parseFloat(e[1]),r=(e[2]||"ms").toLowerCase();switch(r){case"years":case"year":case"yrs":case"yr":case"y":return n*u;case"days":case"day":case"d":return n*p;case"hours":case"hour":case"hrs":case"hr":case"h":return n*c;case"minutes":case"minute":case"mins":case"min":case"m":return n*a;case"seconds":case"second":case"secs":case"sec":case"s":return n*s;case"milliseconds":case"millisecond":case"msecs":case"msec":case"ms":return n;default:return}}}}function r(t){return t>=p?Math.round(t/p)+"d":t>=c?Math.round(t/c)+"h":t>=a?Math.round(t/a)+"m":t>=s?Math.round(t/s)+"s":t+"ms"}function o(t){return i(t,p,"day")||i(t,c,"hour")||i(t,a,"minute")||i(t,s,"second")||t+" ms"}function i(t,e,n){if(!(t<e))return t<1.5*e?Math.floor(t/e)+" "+n:Math.ceil(t/e)+" "+n+"s"}var s=1e3,a=60*s,c=60*a,p=24*c,u=365.25*p;t.exports=function(t,e){e=e||{};var i=typeof t;if("string"===i&&t.length>0)return n(t);if("number"===i&&isNaN(t)===!1)return e["long"]?o(t):r(t);throw new Error("val is not a non-empty string or a valid number. val="+JSON.stringify(t))}},function(t,e,n){function r(){}function o(t){var n=""+t.type;return e.BINARY_EVENT!==t.type&&e.BINARY_ACK!==t.type||(n+=t.attachments+"-"),t.nsp&&"/"!==t.nsp&&(n+=t.nsp+","),null!=t.id&&(n+=t.id),null!=t.data&&(n+=JSON.stringify(t.data)),h("encoded %j as %s",t,n),n}function i(t,e){function n(t){var n=d.deconstructPacket(t),r=o(n.packet),i=n.buffers;i.unshift(r),e(i)}d.removeBlobs(t,n)}function s(){this.reconstructor=null}function a(t){var n=0,r={type:Number(t.charAt(0))};if(null==e.types[r.type])return u();if(e.BINARY_EVENT===r.type||e.BINARY_ACK===r.type){for(var o="";"-"!==t.charAt(++n)&&(o+=t.charAt(n),n!=t.length););if(o!=Number(o)||"-"!==t.charAt(n))throw new Error("Illegal attachments");r.attachments=Number(o)}if("/"===t.charAt(n+1))for(r.nsp="";++n;){var i=t.charAt(n);if(","===i)break;if(r.nsp+=i,n===t.length)break}else r.nsp="/";var s=t.charAt(n+1);if(""!==s&&Number(s)==s){for(r.id="";++n;){var i=t.charAt(n);if(null==i||Number(i)!=i){--n;break}if(r.id+=t.charAt(n),n===t.length)break}r.id=Number(r.id)}return t.charAt(++n)&&(r=c(r,t.substr(n))),h("decoded %s as %j",t,r),r}function c(t,e){try{t.data=JSON.parse(e)}catch(n){return u()}return t}function p(t){this.reconPack=t,this.buffers=[]}function u(){return{type:e.ERROR,data:"parser error"}}var h=n(3)("socket.io-parser"),f=n(8),l=n(9),d=n(11),y=n(12);e.protocol=4,e.types=["CONNECT","DISCONNECT","EVENT","ACK","ERROR","BINARY_EVENT","BINARY_ACK"],e.CONNECT=0,e.DISCONNECT=1,e.EVENT=2,e.ACK=3,e.ERROR=4,e.BINARY_EVENT=5,e.BINARY_ACK=6,e.Encoder=r,e.Decoder=s,r.prototype.encode=function(t,n){if(t.type!==e.EVENT&&t.type!==e.ACK||!l(t.data)||(t.type=t.type===e.EVENT?e.BINARY_EVENT:e.BINARY_ACK),h("encoding packet %j",t),e.BINARY_EVENT===t.type||e.BINARY_ACK===t.type)i(t,n);else{var r=o(t);n([r])}},f(s.prototype),s.prototype.add=function(t){var n;if("string"==typeof t)n=a(t),e.BINARY_EVENT===n.type||e.BINARY_ACK===n.type?(this.reconstructor=new p(n),0===this.reconstructor.reconPack.attachments&&this.emit("decoded",n)):this.emit("decoded",n);else{if(!y(t)&&!t.base64)throw new Error("Unknown type: "+t);if(!this.reconstructor)throw new Error("got binary data when not reconstructing a packet");n=this.reconstructor.takeBinaryData(t),n&&(this.reconstructor=null,this.emit("decoded",n))}},s.prototype.destroy=function(){this.reconstructor&&this.reconstructor.finishedReconstruction()},p.prototype.takeBinaryData=function(t){if(this.buffers.push(t),this.buffers.length===this.reconPack.attachments){var e=d.reconstructPacket(this.reconPack,this.buffers);return this.finishedReconstruction(),e}return null},p.prototype.finishedReconstruction=function(){this.reconPack=null,this.buffers=[]}},function(t,e,n){function r(t){if(t)return o(t)}function o(t){for(var e in r.prototype)t[e]=r.prototype[e];return t}t.exports=r,r.prototype.on=r.prototype.addEventListener=function(t,e){return this._callbacks=this._callbacks||{},(this._callbacks["$"+t]=this._callbacks["$"+t]||[]).push(e),this},r.prototype.once=function(t,e){function n(){this.off(t,n),e.apply(this,arguments)}return n.fn=e,this.on(t,n),this},r.prototype.off=r.prototype.removeListener=r.prototype.removeAllListeners=r.prototype.removeEventListener=function(t,e){if(this._callbacks=this._callbacks||{},0==arguments.length)return this._callbacks={},this;var n=this._callbacks["$"+t];if(!n)return this;if(1==arguments.length)return delete this._callbacks["$"+t],this;for(var r,o=0;o<n.length;o++)if(r=n[o],r===e||r.fn===e){n.splice(o,1);break}return this},r.prototype.emit=function(t){this._callbacks=this._callbacks||{};var e=[].slice.call(arguments,1),n=this._callbacks["$"+t];if(n){n=n.slice(0);for(var r=0,o=n.length;r<o;++r)n[r].apply(this,e)}return this},r.prototype.listeners=function(t){return this._callbacks=this._callbacks||{},this._callbacks["$"+t]||[]},r.prototype.hasListeners=function(t){return!!this.listeners(t).length}},function(t,e,n){(function(e){function r(t){if(!t||"object"!=typeof t)return!1;if(o(t)){for(var n=0,i=t.length;n<i;n++)if(r(t[n]))return!0;return!1}if("function"==typeof e.Buffer&&e.Buffer.isBuffer&&e.Buffer.isBuffer(t)||"function"==typeof e.ArrayBuffer&&t instanceof ArrayBuffer||s&&t instanceof Blob||a&&t instanceof File)return!0;if(t.toJSON&&"function"==typeof t.toJSON&&1===arguments.length)return r(t.toJSON(),!0);for(var c in t)if(Object.prototype.hasOwnProperty.call(t,c)&&r(t[c]))return!0;return!1}var o=n(10),i=Object.prototype.toString,s="function"==typeof e.Blob||"[object BlobConstructor]"===i.call(e.Blob),a="function"==typeof e.File||"[object FileConstructor]"===i.call(e.File);t.exports=r}).call(e,function(){return this}())},function(t,e){var n={}.toString;t.exports=Array.isArray||function(t){return"[object Array]"==n.call(t)}},function(t,e,n){(function(t){function r(t,e){if(!t)return t;if(s(t)){var n={_placeholder:!0,num:e.length};return e.push(t),n}if(i(t)){for(var o=new Array(t.length),a=0;a<t.length;a++)o[a]=r(t[a],e);return o}if("object"==typeof t&&!(t instanceof Date)){var o={};for(var c in t)o[c]=r(t[c],e);return o}return t}function o(t,e){if(!t)return t;if(t&&t._placeholder)return e[t.num];if(i(t))for(var n=0;n<t.length;n++)t[n]=o(t[n],e);else if("object"==typeof t)for(var r in t)t[r]=o(t[r],e);return t}var i=n(10),s=n(12),a=Object.prototype.toString,c="function"==typeof t.Blob||"[object BlobConstructor]"===a.call(t.Blob),p="function"==typeof t.File||"[object FileConstructor]"===a.call(t.File);e.deconstructPacket=function(t){var e=[],n=t.data,o=t;return o.data=r(n,e),o.attachments=e.length,{packet:o,buffers:e}},e.reconstructPacket=function(t,e){return t.data=o(t.data,e),t.attachments=void 0,t},e.removeBlobs=function(t,e){function n(t,a,u){if(!t)return t;if(c&&t instanceof Blob||p&&t instanceof File){r++;var h=new FileReader;h.onload=function(){u?u[a]=this.result:o=this.result,--r||e(o)},h.readAsArrayBuffer(t)}else if(i(t))for(var f=0;f<t.length;f++)n(t[f],f,t);else if("object"==typeof t&&!s(t))for(var l in t)n(t[l],l,t)}var r=0,o=t;n(o),r||e(o)}}).call(e,function(){return this}())},function(t,e){(function(e){function n(t){return e.Buffer&&e.Buffer.isBuffer(t)||e.ArrayBuffer&&t instanceof ArrayBuffer}t.exports=n}).call(e,function(){return this}())},function(t,e,n){"use strict";function r(t,e){if(!(this instanceof r))return new r(t,e);t&&"object"===("undefined"==typeof t?"undefined":o(t))&&(e=t,t=void 0),e=e||{},e.path=e.path||"/socket.io",this.nsps={},this.subs=[],this.opts=e,this.reconnection(e.reconnection!==!1),this.reconnectionAttempts(e.reconnectionAttempts||1/0),this.reconnectionDelay(e.reconnectionDelay||1e3),this.reconnectionDelayMax(e.reconnectionDelayMax||5e3),this.randomizationFactor(e.randomizationFactor||.5),this.backoff=new l({min:this.reconnectionDelay(),max:this.reconnectionDelayMax(),jitter:this.randomizationFactor()}),this.timeout(null==e.timeout?2e4:e.timeout),this.readyState="closed",this.uri=t,this.connecting=[],this.lastPing=null,this.encoding=!1,this.packetBuffer=[];var n=e.parser||c;this.encoder=new n.Encoder,this.decoder=new n.Decoder,this.autoConnect=e.autoConnect!==!1,this.autoConnect&&this.open()}var o="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function(t){return typeof t}:function(t){return t&&"function"==typeof Symbol&&t.constructor===Symbol&&t!==Symbol.prototype?"symbol":typeof t},i=n(14),s=n(37),a=n(8),c=n(7),p=n(39),u=n(40),h=n(3)("socket.io-client:manager"),f=n(36),l=n(41),d=Object.prototype.hasOwnProperty;t.exports=r,r.prototype.emitAll=function(){this.emit.apply(this,arguments);for(var t in this.nsps)d.call(this.nsps,t)&&this.nsps[t].emit.apply(this.nsps[t],arguments)},r.prototype.updateSocketIds=function(){for(var t in this.nsps)d.call(this.nsps,t)&&(this.nsps[t].id=this.generateId(t))},r.prototype.generateId=function(t){return("/"===t?"":t+"#")+this.engine.id},a(r.prototype),r.prototype.reconnection=function(t){return arguments.length?(this._reconnection=!!t,this):this._reconnection},r.prototype.reconnectionAttempts=function(t){return arguments.length?(this._reconnectionAttempts=t,this):this._reconnectionAttempts},r.prototype.reconnectionDelay=function(t){return arguments.length?(this._reconnectionDelay=t,this.backoff&&this.backoff.setMin(t),this):this._reconnectionDelay},r.prototype.randomizationFactor=function(t){return arguments.length?(this._randomizationFactor=t,this.backoff&&this.backoff.setJitter(t),this):this._randomizationFactor},r.prototype.reconnectionDelayMax=function(t){return arguments.length?(this._reconnectionDelayMax=t,this.backoff&&this.backoff.setMax(t),this):this._reconnectionDelayMax},r.prototype.timeout=function(t){return arguments.length?(this._timeout=t,this):this._timeout},r.prototype.maybeReconnectOnOpen=function(){!this.reconnecting&&this._reconnection&&0===this.backoff.attempts&&this.reconnect()},r.prototype.open=r.prototype.connect=function(t,e){if(h("readyState %s",this.readyState),~this.readyState.indexOf("open"))return this;h("opening %s",this.uri),this.engine=i(this.uri,this.opts);var n=this.engine,r=this;this.readyState="opening",this.skipReconnect=!1;var o=p(n,"open",function(){r.onopen(),t&&t()}),s=p(n,"error",function(e){if(h("connect_error"),r.cleanup(),r.readyState="closed",r.emitAll("connect_error",e),t){var n=new Error("Connection error");n.data=e,t(n)}else r.maybeReconnectOnOpen()});if(!1!==this._timeout){var a=this._timeout;h("connect attempt will timeout after %d",a);var c=setTimeout(function(){h("connect attempt timed out after %d",a),o.destroy(),n.close(),n.emit("error","timeout"),r.emitAll("connect_timeout",a)},a);this.subs.push({destroy:function(){clearTimeout(c)}})}return this.subs.push(o),this.subs.push(s),this},r.prototype.onopen=function(){h("open"),this.cleanup(),this.readyState="open",this.emit("open");var t=this.engine;this.subs.push(p(t,"data",u(this,"ondata"))),this.subs.push(p(t,"ping",u(this,"onping"))),this.subs.push(p(t,"pong",u(this,"onpong"))),this.subs.push(p(t,"error",u(this,"onerror"))),this.subs.push(p(t,"close",u(this,"onclose"))),this.subs.push(p(this.decoder,"decoded",u(this,"ondecoded")))},r.prototype.onping=function(){this.lastPing=new Date,this.emitAll("ping")},r.prototype.onpong=function(){this.emitAll("pong",new Date-this.lastPing)},r.prototype.ondata=function(t){this.decoder.add(t)},r.prototype.ondecoded=function(t){this.emit("packet",t)},r.prototype.onerror=function(t){h("error",t),this.emitAll("error",t)},r.prototype.socket=function(t,e){function n(){~f(o.connecting,r)||o.connecting.push(r)}var r=this.nsps[t];if(!r){r=new s(this,t,e),this.nsps[t]=r;var o=this;r.on("connecting",n),r.on("connect",function(){r.id=o.generateId(t)}),this.autoConnect&&n()}return r},r.prototype.destroy=function(t){var e=f(this.connecting,t);~e&&this.connecting.splice(e,1),this.connecting.length||this.close()},r.prototype.packet=function(t){h("writing packet %j",t);var e=this;t.query&&0===t.type&&(t.nsp+="?"+t.query),e.encoding?e.packetBuffer.push(t):(e.encoding=!0,this.encoder.encode(t,function(n){for(var r=0;r<n.length;r++)e.engine.write(n[r],t.options);e.encoding=!1,e.processPacketQueue()}))},r.prototype.processPacketQueue=function(){if(this.packetBuffer.length>0&&!this.encoding){var t=this.packetBuffer.shift();this.packet(t)}},r.prototype.cleanup=function(){h("cleanup");for(var t=this.subs.length,e=0;e<t;e++){var n=this.subs.shift();n.destroy()}this.packetBuffer=[],this.encoding=!1,this.lastPing=null,this.decoder.destroy()},r.prototype.close=r.prototype.disconnect=function(){h("disconnect"),this.skipReconnect=!0,this.reconnecting=!1,"opening"===this.readyState&&this.cleanup(),this.backoff.reset(),this.readyState="closed",this.engine&&this.engine.close()},r.prototype.onclose=function(t){h("onclose"),this.cleanup(),this.backoff.reset(),this.readyState="closed",this.emit("close",t),this._reconnection&&!this.skipReconnect&&this.reconnect()},r.prototype.reconnect=function(){if(this.reconnecting||this.skipReconnect)return this;var t=this;if(this.backoff.attempts>=this._reconnectionAttempts)h("reconnect failed"),this.backoff.reset(),this.emitAll("reconnect_failed"),this.reconnecting=!1;else{var e=this.backoff.duration();h("will wait %dms before reconnect attempt",e),this.reconnecting=!0;var n=setTimeout(function(){t.skipReconnect||(h("attempting reconnect"),t.emitAll("reconnect_attempt",t.backoff.attempts),t.emitAll("reconnecting",t.backoff.attempts),t.skipReconnect||t.open(function(e){e?(h("reconnect attempt error"),t.reconnecting=!1,t.reconnect(),t.emitAll("reconnect_error",e.data)):(h("reconnect success"),t.onreconnect())}))},e);this.subs.push({destroy:function(){clearTimeout(n)}})}},r.prototype.onreconnect=function(){var t=this.backoff.attempts;this.reconnecting=!1,this.backoff.reset(),this.updateSocketIds(),this.emitAll("reconnect",t)}},function(t,e,n){t.exports=n(15),t.exports.parser=n(22)},function(t,e,n){(function(e){function r(t,n){if(!(this instanceof r))return new r(t,n);n=n||{},t&&"object"==typeof t&&(n=t,t=null),t?(t=u(t),n.hostname=t.host,n.secure="https"===t.protocol||"wss"===t.protocol,n.port=t.port,t.query&&(n.query=t.query)):n.host&&(n.hostname=u(n.host).host),this.secure=null!=n.secure?n.secure:e.location&&"https:"===location.protocol,n.hostname&&!n.port&&(n.port=this.secure?"443":"80"),this.agent=n.agent||!1,this.hostname=n.hostname||(e.location?location.hostname:"localhost"),this.port=n.port||(e.location&&location.port?location.port:this.secure?443:80),this.query=n.query||{},"string"==typeof this.query&&(this.query=h.decode(this.query)),this.upgrade=!1!==n.upgrade,this.path=(n.path||"/engine.io").replace(/\/$/,"")+"/",this.forceJSONP=!!n.forceJSONP,this.jsonp=!1!==n.jsonp,this.forceBase64=!!n.forceBase64,this.enablesXDR=!!n.enablesXDR,this.timestampParam=n.timestampParam||"t",this.timestampRequests=n.timestampRequests,this.transports=n.transports||["polling","websocket"],this.transportOptions=n.transportOptions||{},this.readyState="",this.writeBuffer=[],this.prevBufferLen=0,this.policyPort=n.policyPort||843,this.rememberUpgrade=n.rememberUpgrade||!1,this.binaryType=null,this.onlyBinaryUpgrades=n.onlyBinaryUpgrades,this.perMessageDeflate=!1!==n.perMessageDeflate&&(n.perMessageDeflate||{}),!0===this.perMessageDeflate&&(this.perMessageDeflate={}),this.perMessageDeflate&&null==this.perMessageDeflate.threshold&&(this.perMessageDeflate.threshold=1024),this.pfx=n.pfx||null,this.key=n.key||null,this.passphrase=n.passphrase||null,this.cert=n.cert||null,this.ca=n.ca||null,this.ciphers=n.ciphers||null,this.rejectUnauthorized=void 0===n.rejectUnauthorized||n.rejectUnauthorized,this.forceNode=!!n.forceNode;var o="object"==typeof e&&e;o.global===o&&(n.extraHeaders&&Object.keys(n.extraHeaders).length>0&&(this.extraHeaders=n.extraHeaders),n.localAddress&&(this.localAddress=n.localAddress)),this.id=null,this.upgrades=null,this.pingInterval=null,this.pingTimeout=null,this.pingIntervalTimer=null,this.pingTimeoutTimer=null,this.open()}function o(t){var e={};for(var n in t)t.hasOwnProperty(n)&&(e[n]=t[n]);return e}var i=n(16),s=n(8),a=n(3)("engine.io-client:socket"),c=n(36),p=n(22),u=n(2),h=n(30);t.exports=r,r.priorWebsocketSuccess=!1,s(r.prototype),r.protocol=p.protocol,r.Socket=r,r.Transport=n(21),r.transports=n(16),r.parser=n(22),r.prototype.createTransport=function(t){a('creating transport "%s"',t);var e=o(this.query);e.EIO=p.protocol,e.transport=t;var n=this.transportOptions[t]||{};this.id&&(e.sid=this.id);var r=new i[t]({query:e,socket:this,agent:n.agent||this.agent,hostname:n.hostname||this.hostname,port:n.port||this.port,secure:n.secure||this.secure,path:n.path||this.path,forceJSONP:n.forceJSONP||this.forceJSONP,jsonp:n.jsonp||this.jsonp,forceBase64:n.forceBase64||this.forceBase64,enablesXDR:n.enablesXDR||this.enablesXDR,timestampRequests:n.timestampRequests||this.timestampRequests,timestampParam:n.timestampParam||this.timestampParam,policyPort:n.policyPort||this.policyPort,pfx:n.pfx||this.pfx,key:n.key||this.key,passphrase:n.passphrase||this.passphrase,cert:n.cert||this.cert,ca:n.ca||this.ca,ciphers:n.ciphers||this.ciphers,rejectUnauthorized:n.rejectUnauthorized||this.rejectUnauthorized,perMessageDeflate:n.perMessageDeflate||this.perMessageDeflate,extraHeaders:n.extraHeaders||this.extraHeaders,forceNode:n.forceNode||this.forceNode,localAddress:n.localAddress||this.localAddress,requestTimeout:n.requestTimeout||this.requestTimeout,protocols:n.protocols||void 0});return r},r.prototype.open=function(){var t;if(this.rememberUpgrade&&r.priorWebsocketSuccess&&this.transports.indexOf("websocket")!==-1)t="websocket";else{if(0===this.transports.length){var e=this;return void setTimeout(function(){e.emit("error","No transports available")},0)}t=this.transports[0]}this.readyState="opening";try{t=this.createTransport(t)}catch(n){return this.transports.shift(),void this.open()}t.open(),this.setTransport(t)},r.prototype.setTransport=function(t){a("setting transport %s",t.name);var e=this;this.transport&&(a("clearing existing transport %s",this.transport.name),this.transport.removeAllListeners()),this.transport=t,t.on("drain",function(){e.onDrain()}).on("packet",function(t){e.onPacket(t)}).on("error",function(t){e.onError(t)}).on("close",function(){e.onClose("transport close")})},r.prototype.probe=function(t){function e(){if(f.onlyBinaryUpgrades){var e=!this.supportsBinary&&f.transport.supportsBinary;h=h||e}h||(a('probe transport "%s" opened',t),u.send([{type:"ping",data:"probe"}]),u.once("packet",function(e){if(!h)if("pong"===e.type&&"probe"===e.data){if(a('probe transport "%s" pong',t),f.upgrading=!0,f.emit("upgrading",u),!u)return;r.priorWebsocketSuccess="websocket"===u.name,a('pausing current transport "%s"',f.transport.name),f.transport.pause(function(){h||"closed"!==f.readyState&&(a("changing transport and sending upgrade packet"),p(),f.setTransport(u),u.send([{type:"upgrade"}]),f.emit("upgrade",u),u=null,f.upgrading=!1,f.flush())})}else{a('probe transport "%s" failed',t);var n=new Error("probe error");n.transport=u.name,f.emit("upgradeError",n)}}))}function n(){h||(h=!0,p(),u.close(),u=null)}function o(e){var r=new Error("probe error: "+e);r.transport=u.name,n(),a('probe transport "%s" failed because of error: %s',t,e),f.emit("upgradeError",r)}function i(){o("transport closed")}function s(){o("socket closed")}function c(t){u&&t.name!==u.name&&(a('"%s" works - aborting "%s"',t.name,u.name),n())}function p(){u.removeListener("open",e),u.removeListener("error",o),u.removeListener("close",i),f.removeListener("close",s),f.removeListener("upgrading",c)}a('probing transport "%s"',t);var u=this.createTransport(t,{probe:1}),h=!1,f=this;r.priorWebsocketSuccess=!1,u.once("open",e),u.once("error",o),u.once("close",i),this.once("close",s),this.once("upgrading",c),u.open()},r.prototype.onOpen=function(){if(a("socket open"),this.readyState="open",r.priorWebsocketSuccess="websocket"===this.transport.name,this.emit("open"),this.flush(),"open"===this.readyState&&this.upgrade&&this.transport.pause){a("starting upgrade probes");for(var t=0,e=this.upgrades.length;t<e;t++)this.probe(this.upgrades[t])}},r.prototype.onPacket=function(t){if("opening"===this.readyState||"open"===this.readyState||"closing"===this.readyState)switch(a('socket receive: type "%s", data "%s"',t.type,t.data),this.emit("packet",t),this.emit("heartbeat"),t.type){case"open":this.onHandshake(JSON.parse(t.data));break;case"pong":this.setPing(),this.emit("pong");break;case"error":var e=new Error("server error");e.code=t.data,this.onError(e);break;case"message":this.emit("data",t.data),this.emit("message",t.data)}else a('packet received with socket readyState "%s"',this.readyState)},r.prototype.onHandshake=function(t){this.emit("handshake",t),this.id=t.sid,this.transport.query.sid=t.sid,this.upgrades=this.filterUpgrades(t.upgrades),this.pingInterval=t.pingInterval,this.pingTimeout=t.pingTimeout,this.onOpen(),"closed"!==this.readyState&&(this.setPing(),this.removeListener("heartbeat",this.onHeartbeat),this.on("heartbeat",this.onHeartbeat))},r.prototype.onHeartbeat=function(t){clearTimeout(this.pingTimeoutTimer);var e=this;e.pingTimeoutTimer=setTimeout(function(){"closed"!==e.readyState&&e.onClose("ping timeout")},t||e.pingInterval+e.pingTimeout)},r.prototype.setPing=function(){var t=this;clearTimeout(t.pingIntervalTimer),t.pingIntervalTimer=setTimeout(function(){a("writing ping packet - expecting pong within %sms",t.pingTimeout),t.ping(),t.onHeartbeat(t.pingTimeout)},t.pingInterval)},r.prototype.ping=function(){var t=this;this.sendPacket("ping",function(){t.emit("ping")})},r.prototype.onDrain=function(){this.writeBuffer.splice(0,this.prevBufferLen),this.prevBufferLen=0,0===this.writeBuffer.length?this.emit("drain"):this.flush()},r.prototype.flush=function(){"closed"!==this.readyState&&this.transport.writable&&!this.upgrading&&this.writeBuffer.length&&(a("flushing %d packets in socket",this.writeBuffer.length),this.transport.send(this.writeBuffer),this.prevBufferLen=this.writeBuffer.length,this.emit("flush"))},r.prototype.write=r.prototype.send=function(t,e,n){return this.sendPacket("message",t,e,n),this},r.prototype.sendPacket=function(t,e,n,r){if("function"==typeof e&&(r=e,e=void 0),"function"==typeof n&&(r=n,n=null),"closing"!==this.readyState&&"closed"!==this.readyState){n=n||{},n.compress=!1!==n.compress;var o={type:t,data:e,options:n};this.emit("packetCreate",o),this.writeBuffer.push(o),r&&this.once("flush",r),this.flush()}},r.prototype.close=function(){function t(){r.onClose("forced close"),a("socket closing - telling transport to close"),r.transport.close()}function e(){r.removeListener("upgrade",e),r.removeListener("upgradeError",e),t()}function n(){r.once("upgrade",e),r.once("upgradeError",e)}if("opening"===this.readyState||"open"===this.readyState){this.readyState="closing";var r=this;this.writeBuffer.length?this.once("drain",function(){this.upgrading?n():t()}):this.upgrading?n():t()}return this},r.prototype.onError=function(t){a("socket error %j",t),r.priorWebsocketSuccess=!1,this.emit("error",t),this.onClose("transport error",t)},r.prototype.onClose=function(t,e){if("opening"===this.readyState||"open"===this.readyState||"closing"===this.readyState){a('socket close with reason: "%s"',t);var n=this;clearTimeout(this.pingIntervalTimer),clearTimeout(this.pingTimeoutTimer),this.transport.removeAllListeners("close"),this.transport.close(),this.transport.removeAllListeners(),this.readyState="closed",this.id=null,this.emit("close",t,e),n.writeBuffer=[],n.prevBufferLen=0}},r.prototype.filterUpgrades=function(t){for(var e=[],n=0,r=t.length;n<r;n++)~c(this.transports,t[n])&&e.push(t[n]);return e}}).call(e,function(){return this}())},function(t,e,n){(function(t){function r(e){var n,r=!1,a=!1,c=!1!==e.jsonp;if(t.location){var p="https:"===location.protocol,u=location.port;u||(u=p?443:80),r=e.hostname!==location.hostname||u!==e.port,a=e.secure!==p}if(e.xdomain=r,e.xscheme=a,n=new o(e),"open"in n&&!e.forceJSONP)return new i(e);if(!c)throw new Error("JSONP disabled");return new s(e)}var o=n(17),i=n(19),s=n(33),a=n(34);e.polling=r,e.websocket=a}).call(e,function(){return this}())},function(t,e,n){(function(e){var r=n(18);t.exports=function(t){var n=t.xdomain,o=t.xscheme,i=t.enablesXDR;try{if("undefined"!=typeof XMLHttpRequest&&(!n||r))return new XMLHttpRequest}catch(s){}try{if("undefined"!=typeof XDomainRequest&&!o&&i)return new XDomainRequest}catch(s){}if(!n)try{return new(e[["Active"].concat("Object").join("X")])("Microsoft.XMLHTTP");
}catch(s){}}}).call(e,function(){return this}())},function(t,e){try{t.exports="undefined"!=typeof XMLHttpRequest&&"withCredentials"in new XMLHttpRequest}catch(n){t.exports=!1}},function(t,e,n){(function(e){function r(){}function o(t){if(c.call(this,t),this.requestTimeout=t.requestTimeout,this.extraHeaders=t.extraHeaders,e.location){var n="https:"===location.protocol,r=location.port;r||(r=n?443:80),this.xd=t.hostname!==e.location.hostname||r!==t.port,this.xs=t.secure!==n}}function i(t){this.method=t.method||"GET",this.uri=t.uri,this.xd=!!t.xd,this.xs=!!t.xs,this.async=!1!==t.async,this.data=void 0!==t.data?t.data:null,this.agent=t.agent,this.isBinary=t.isBinary,this.supportsBinary=t.supportsBinary,this.enablesXDR=t.enablesXDR,this.requestTimeout=t.requestTimeout,this.pfx=t.pfx,this.key=t.key,this.passphrase=t.passphrase,this.cert=t.cert,this.ca=t.ca,this.ciphers=t.ciphers,this.rejectUnauthorized=t.rejectUnauthorized,this.extraHeaders=t.extraHeaders,this.create()}function s(){for(var t in i.requests)i.requests.hasOwnProperty(t)&&i.requests[t].abort()}var a=n(17),c=n(20),p=n(8),u=n(31),h=n(3)("engine.io-client:polling-xhr");t.exports=o,t.exports.Request=i,u(o,c),o.prototype.supportsBinary=!0,o.prototype.request=function(t){return t=t||{},t.uri=this.uri(),t.xd=this.xd,t.xs=this.xs,t.agent=this.agent||!1,t.supportsBinary=this.supportsBinary,t.enablesXDR=this.enablesXDR,t.pfx=this.pfx,t.key=this.key,t.passphrase=this.passphrase,t.cert=this.cert,t.ca=this.ca,t.ciphers=this.ciphers,t.rejectUnauthorized=this.rejectUnauthorized,t.requestTimeout=this.requestTimeout,t.extraHeaders=this.extraHeaders,new i(t)},o.prototype.doWrite=function(t,e){var n="string"!=typeof t&&void 0!==t,r=this.request({method:"POST",data:t,isBinary:n}),o=this;r.on("success",e),r.on("error",function(t){o.onError("xhr post error",t)}),this.sendXhr=r},o.prototype.doPoll=function(){h("xhr poll");var t=this.request(),e=this;t.on("data",function(t){e.onData(t)}),t.on("error",function(t){e.onError("xhr poll error",t)}),this.pollXhr=t},p(i.prototype),i.prototype.create=function(){var t={agent:this.agent,xdomain:this.xd,xscheme:this.xs,enablesXDR:this.enablesXDR};t.pfx=this.pfx,t.key=this.key,t.passphrase=this.passphrase,t.cert=this.cert,t.ca=this.ca,t.ciphers=this.ciphers,t.rejectUnauthorized=this.rejectUnauthorized;var n=this.xhr=new a(t),r=this;try{h("xhr open %s: %s",this.method,this.uri),n.open(this.method,this.uri,this.async);try{if(this.extraHeaders){n.setDisableHeaderCheck&&n.setDisableHeaderCheck(!0);for(var o in this.extraHeaders)this.extraHeaders.hasOwnProperty(o)&&n.setRequestHeader(o,this.extraHeaders[o])}}catch(s){}if("POST"===this.method)try{this.isBinary?n.setRequestHeader("Content-type","application/octet-stream"):n.setRequestHeader("Content-type","text/plain;charset=UTF-8")}catch(s){}try{n.setRequestHeader("Accept","*/*")}catch(s){}"withCredentials"in n&&(n.withCredentials=!0),this.requestTimeout&&(n.timeout=this.requestTimeout),this.hasXDR()?(n.onload=function(){r.onLoad()},n.onerror=function(){r.onError(n.responseText)}):n.onreadystatechange=function(){if(2===n.readyState){var t;try{t=n.getResponseHeader("Content-Type")}catch(e){}"application/octet-stream"===t&&(n.responseType="arraybuffer")}4===n.readyState&&(200===n.status||1223===n.status?r.onLoad():setTimeout(function(){r.onError(n.status)},0))},h("xhr data %s",this.data),n.send(this.data)}catch(s){return void setTimeout(function(){r.onError(s)},0)}e.document&&(this.index=i.requestsCount++,i.requests[this.index]=this)},i.prototype.onSuccess=function(){this.emit("success"),this.cleanup()},i.prototype.onData=function(t){this.emit("data",t),this.onSuccess()},i.prototype.onError=function(t){this.emit("error",t),this.cleanup(!0)},i.prototype.cleanup=function(t){if("undefined"!=typeof this.xhr&&null!==this.xhr){if(this.hasXDR()?this.xhr.onload=this.xhr.onerror=r:this.xhr.onreadystatechange=r,t)try{this.xhr.abort()}catch(n){}e.document&&delete i.requests[this.index],this.xhr=null}},i.prototype.onLoad=function(){var t;try{var e;try{e=this.xhr.getResponseHeader("Content-Type")}catch(n){}t="application/octet-stream"===e?this.xhr.response||this.xhr.responseText:this.xhr.responseText}catch(n){this.onError(n)}null!=t&&this.onData(t)},i.prototype.hasXDR=function(){return"undefined"!=typeof e.XDomainRequest&&!this.xs&&this.enablesXDR},i.prototype.abort=function(){this.cleanup()},i.requestsCount=0,i.requests={},e.document&&(e.attachEvent?e.attachEvent("onunload",s):e.addEventListener&&e.addEventListener("beforeunload",s,!1))}).call(e,function(){return this}())},function(t,e,n){function r(t){var e=t&&t.forceBase64;u&&!e||(this.supportsBinary=!1),o.call(this,t)}var o=n(21),i=n(30),s=n(22),a=n(31),c=n(32),p=n(3)("engine.io-client:polling");t.exports=r;var u=function(){var t=n(17),e=new t({xdomain:!1});return null!=e.responseType}();a(r,o),r.prototype.name="polling",r.prototype.doOpen=function(){this.poll()},r.prototype.pause=function(t){function e(){p("paused"),n.readyState="paused",t()}var n=this;if(this.readyState="pausing",this.polling||!this.writable){var r=0;this.polling&&(p("we are currently polling - waiting to pause"),r++,this.once("pollComplete",function(){p("pre-pause polling complete"),--r||e()})),this.writable||(p("we are currently writing - waiting to pause"),r++,this.once("drain",function(){p("pre-pause writing complete"),--r||e()}))}else e()},r.prototype.poll=function(){p("polling"),this.polling=!0,this.doPoll(),this.emit("poll")},r.prototype.onData=function(t){var e=this;p("polling got data %s",t);var n=function(t,n,r){return"opening"===e.readyState&&e.onOpen(),"close"===t.type?(e.onClose(),!1):void e.onPacket(t)};s.decodePayload(t,this.socket.binaryType,n),"closed"!==this.readyState&&(this.polling=!1,this.emit("pollComplete"),"open"===this.readyState?this.poll():p('ignoring poll - transport state "%s"',this.readyState))},r.prototype.doClose=function(){function t(){p("writing close packet"),e.write([{type:"close"}])}var e=this;"open"===this.readyState?(p("transport open - closing"),t()):(p("transport not open - deferring close"),this.once("open",t))},r.prototype.write=function(t){var e=this;this.writable=!1;var n=function(){e.writable=!0,e.emit("drain")};s.encodePayload(t,this.supportsBinary,function(t){e.doWrite(t,n)})},r.prototype.uri=function(){var t=this.query||{},e=this.secure?"https":"http",n="";!1!==this.timestampRequests&&(t[this.timestampParam]=c()),this.supportsBinary||t.sid||(t.b64=1),t=i.encode(t),this.port&&("https"===e&&443!==Number(this.port)||"http"===e&&80!==Number(this.port))&&(n=":"+this.port),t.length&&(t="?"+t);var r=this.hostname.indexOf(":")!==-1;return e+"://"+(r?"["+this.hostname+"]":this.hostname)+n+this.path+t}},function(t,e,n){function r(t){this.path=t.path,this.hostname=t.hostname,this.port=t.port,this.secure=t.secure,this.query=t.query,this.timestampParam=t.timestampParam,this.timestampRequests=t.timestampRequests,this.readyState="",this.agent=t.agent||!1,this.socket=t.socket,this.enablesXDR=t.enablesXDR,this.pfx=t.pfx,this.key=t.key,this.passphrase=t.passphrase,this.cert=t.cert,this.ca=t.ca,this.ciphers=t.ciphers,this.rejectUnauthorized=t.rejectUnauthorized,this.forceNode=t.forceNode,this.extraHeaders=t.extraHeaders,this.localAddress=t.localAddress}var o=n(22),i=n(8);t.exports=r,i(r.prototype),r.prototype.onError=function(t,e){var n=new Error(t);return n.type="TransportError",n.description=e,this.emit("error",n),this},r.prototype.open=function(){return"closed"!==this.readyState&&""!==this.readyState||(this.readyState="opening",this.doOpen()),this},r.prototype.close=function(){return"opening"!==this.readyState&&"open"!==this.readyState||(this.doClose(),this.onClose()),this},r.prototype.send=function(t){if("open"!==this.readyState)throw new Error("Transport not open");this.write(t)},r.prototype.onOpen=function(){this.readyState="open",this.writable=!0,this.emit("open")},r.prototype.onData=function(t){var e=o.decodePacket(t,this.socket.binaryType);this.onPacket(e)},r.prototype.onPacket=function(t){this.emit("packet",t)},r.prototype.onClose=function(){this.readyState="closed",this.emit("close")}},function(t,e,n){(function(t){function r(t,n){var r="b"+e.packets[t.type]+t.data.data;return n(r)}function o(t,n,r){if(!n)return e.encodeBase64Packet(t,r);var o=t.data,i=new Uint8Array(o),s=new Uint8Array(1+o.byteLength);s[0]=v[t.type];for(var a=0;a<i.length;a++)s[a+1]=i[a];return r(s.buffer)}function i(t,n,r){if(!n)return e.encodeBase64Packet(t,r);var o=new FileReader;return o.onload=function(){t.data=o.result,e.encodePacket(t,n,!0,r)},o.readAsArrayBuffer(t.data)}function s(t,n,r){if(!n)return e.encodeBase64Packet(t,r);if(g)return i(t,n,r);var o=new Uint8Array(1);o[0]=v[t.type];var s=new k([o.buffer,t.data]);return r(s)}function a(t){try{t=d.decode(t,{strict:!1})}catch(e){return!1}return t}function c(t,e,n){for(var r=new Array(t.length),o=l(t.length,n),i=function(t,n,o){e(n,function(e,n){r[t]=n,o(e,r)})},s=0;s<t.length;s++)i(s,t[s],o)}var p,u=n(23),h=n(9),f=n(24),l=n(25),d=n(26);t&&t.ArrayBuffer&&(p=n(28));var y="undefined"!=typeof navigator&&/Android/i.test(navigator.userAgent),m="undefined"!=typeof navigator&&/PhantomJS/i.test(navigator.userAgent),g=y||m;e.protocol=3;var v=e.packets={open:0,close:1,ping:2,pong:3,message:4,upgrade:5,noop:6},b=u(v),w={type:"error",data:"parser error"},k=n(29);e.encodePacket=function(e,n,i,a){"function"==typeof n&&(a=n,n=!1),"function"==typeof i&&(a=i,i=null);var c=void 0===e.data?void 0:e.data.buffer||e.data;if(t.ArrayBuffer&&c instanceof ArrayBuffer)return o(e,n,a);if(k&&c instanceof t.Blob)return s(e,n,a);if(c&&c.base64)return r(e,a);var p=v[e.type];return void 0!==e.data&&(p+=i?d.encode(String(e.data),{strict:!1}):String(e.data)),a(""+p)},e.encodeBase64Packet=function(n,r){var o="b"+e.packets[n.type];if(k&&n.data instanceof t.Blob){var i=new FileReader;return i.onload=function(){var t=i.result.split(",")[1];r(o+t)},i.readAsDataURL(n.data)}var s;try{s=String.fromCharCode.apply(null,new Uint8Array(n.data))}catch(a){for(var c=new Uint8Array(n.data),p=new Array(c.length),u=0;u<c.length;u++)p[u]=c[u];s=String.fromCharCode.apply(null,p)}return o+=t.btoa(s),r(o)},e.decodePacket=function(t,n,r){if(void 0===t)return w;if("string"==typeof t){if("b"===t.charAt(0))return e.decodeBase64Packet(t.substr(1),n);if(r&&(t=a(t),t===!1))return w;var o=t.charAt(0);return Number(o)==o&&b[o]?t.length>1?{type:b[o],data:t.substring(1)}:{type:b[o]}:w}var i=new Uint8Array(t),o=i[0],s=f(t,1);return k&&"blob"===n&&(s=new k([s])),{type:b[o],data:s}},e.decodeBase64Packet=function(t,e){var n=b[t.charAt(0)];if(!p)return{type:n,data:{base64:!0,data:t.substr(1)}};var r=p.decode(t.substr(1));return"blob"===e&&k&&(r=new k([r])),{type:n,data:r}},e.encodePayload=function(t,n,r){function o(t){return t.length+":"+t}function i(t,r){e.encodePacket(t,!!s&&n,!1,function(t){r(null,o(t))})}"function"==typeof n&&(r=n,n=null);var s=h(t);return n&&s?k&&!g?e.encodePayloadAsBlob(t,r):e.encodePayloadAsArrayBuffer(t,r):t.length?void c(t,i,function(t,e){return r(e.join(""))}):r("0:")},e.decodePayload=function(t,n,r){if("string"!=typeof t)return e.decodePayloadAsBinary(t,n,r);"function"==typeof n&&(r=n,n=null);var o;if(""===t)return r(w,0,1);for(var i,s,a="",c=0,p=t.length;c<p;c++){var u=t.charAt(c);if(":"===u){if(""===a||a!=(i=Number(a)))return r(w,0,1);if(s=t.substr(c+1,i),a!=s.length)return r(w,0,1);if(s.length){if(o=e.decodePacket(s,n,!1),w.type===o.type&&w.data===o.data)return r(w,0,1);var h=r(o,c+i,p);if(!1===h)return}c+=i,a=""}else a+=u}return""!==a?r(w,0,1):void 0},e.encodePayloadAsArrayBuffer=function(t,n){function r(t,n){e.encodePacket(t,!0,!0,function(t){return n(null,t)})}return t.length?void c(t,r,function(t,e){var r=e.reduce(function(t,e){var n;return n="string"==typeof e?e.length:e.byteLength,t+n.toString().length+n+2},0),o=new Uint8Array(r),i=0;return e.forEach(function(t){var e="string"==typeof t,n=t;if(e){for(var r=new Uint8Array(t.length),s=0;s<t.length;s++)r[s]=t.charCodeAt(s);n=r.buffer}e?o[i++]=0:o[i++]=1;for(var a=n.byteLength.toString(),s=0;s<a.length;s++)o[i++]=parseInt(a[s]);o[i++]=255;for(var r=new Uint8Array(n),s=0;s<r.length;s++)o[i++]=r[s]}),n(o.buffer)}):n(new ArrayBuffer(0))},e.encodePayloadAsBlob=function(t,n){function r(t,n){e.encodePacket(t,!0,!0,function(t){var e=new Uint8Array(1);if(e[0]=1,"string"==typeof t){for(var r=new Uint8Array(t.length),o=0;o<t.length;o++)r[o]=t.charCodeAt(o);t=r.buffer,e[0]=0}for(var i=t instanceof ArrayBuffer?t.byteLength:t.size,s=i.toString(),a=new Uint8Array(s.length+1),o=0;o<s.length;o++)a[o]=parseInt(s[o]);if(a[s.length]=255,k){var c=new k([e.buffer,a.buffer,t]);n(null,c)}})}c(t,r,function(t,e){return n(new k(e))})},e.decodePayloadAsBinary=function(t,n,r){"function"==typeof n&&(r=n,n=null);for(var o=t,i=[];o.byteLength>0;){for(var s=new Uint8Array(o),a=0===s[0],c="",p=1;255!==s[p];p++){if(c.length>310)return r(w,0,1);c+=s[p]}o=f(o,2+c.length),c=parseInt(c);var u=f(o,0,c);if(a)try{u=String.fromCharCode.apply(null,new Uint8Array(u))}catch(h){var l=new Uint8Array(u);u="";for(var p=0;p<l.length;p++)u+=String.fromCharCode(l[p])}i.push(u),o=f(o,c)}var d=i.length;i.forEach(function(t,o){r(e.decodePacket(t,n,!0),o,d)})}}).call(e,function(){return this}())},function(t,e){t.exports=Object.keys||function(t){var e=[],n=Object.prototype.hasOwnProperty;for(var r in t)n.call(t,r)&&e.push(r);return e}},function(t,e){t.exports=function(t,e,n){var r=t.byteLength;if(e=e||0,n=n||r,t.slice)return t.slice(e,n);if(e<0&&(e+=r),n<0&&(n+=r),n>r&&(n=r),e>=r||e>=n||0===r)return new ArrayBuffer(0);for(var o=new Uint8Array(t),i=new Uint8Array(n-e),s=e,a=0;s<n;s++,a++)i[a]=o[s];return i.buffer}},function(t,e){function n(t,e,n){function o(t,r){if(o.count<=0)throw new Error("after called too many times");--o.count,t?(i=!0,e(t),e=n):0!==o.count||i||e(null,r)}var i=!1;return n=n||r,o.count=t,0===t?e():o}function r(){}t.exports=n},function(t,e,n){var r;(function(t,o){!function(i){function s(t){for(var e,n,r=[],o=0,i=t.length;o<i;)e=t.charCodeAt(o++),e>=55296&&e<=56319&&o<i?(n=t.charCodeAt(o++),56320==(64512&n)?r.push(((1023&e)<<10)+(1023&n)+65536):(r.push(e),o--)):r.push(e);return r}function a(t){for(var e,n=t.length,r=-1,o="";++r<n;)e=t[r],e>65535&&(e-=65536,o+=w(e>>>10&1023|55296),e=56320|1023&e),o+=w(e);return o}function c(t,e){if(t>=55296&&t<=57343){if(e)throw Error("Lone surrogate U+"+t.toString(16).toUpperCase()+" is not a scalar value");return!1}return!0}function p(t,e){return w(t>>e&63|128)}function u(t,e){if(0==(4294967168&t))return w(t);var n="";return 0==(4294965248&t)?n=w(t>>6&31|192):0==(4294901760&t)?(c(t,e)||(t=65533),n=w(t>>12&15|224),n+=p(t,6)):0==(4292870144&t)&&(n=w(t>>18&7|240),n+=p(t,12),n+=p(t,6)),n+=w(63&t|128)}function h(t,e){e=e||{};for(var n,r=!1!==e.strict,o=s(t),i=o.length,a=-1,c="";++a<i;)n=o[a],c+=u(n,r);return c}function f(){if(b>=v)throw Error("Invalid byte index");var t=255&g[b];if(b++,128==(192&t))return 63&t;throw Error("Invalid continuation byte")}function l(t){var e,n,r,o,i;if(b>v)throw Error("Invalid byte index");if(b==v)return!1;if(e=255&g[b],b++,0==(128&e))return e;if(192==(224&e)){if(n=f(),i=(31&e)<<6|n,i>=128)return i;throw Error("Invalid continuation byte")}if(224==(240&e)){if(n=f(),r=f(),i=(15&e)<<12|n<<6|r,i>=2048)return c(i,t)?i:65533;throw Error("Invalid continuation byte")}if(240==(248&e)&&(n=f(),r=f(),o=f(),i=(7&e)<<18|n<<12|r<<6|o,i>=65536&&i<=1114111))return i;throw Error("Invalid UTF-8 detected")}function d(t,e){e=e||{};var n=!1!==e.strict;g=s(t),v=g.length,b=0;for(var r,o=[];(r=l(n))!==!1;)o.push(r);return a(o)}var y="object"==typeof e&&e,m=("object"==typeof t&&t&&t.exports==y&&t,"object"==typeof o&&o);m.global!==m&&m.window!==m||(i=m);var g,v,b,w=String.fromCharCode,k={version:"2.1.2",encode:h,decode:d};r=function(){return k}.call(e,n,e,t),!(void 0!==r&&(t.exports=r))}(this)}).call(e,n(27)(t),function(){return this}())},function(t,e){t.exports=function(t){return t.webpackPolyfill||(t.deprecate=function(){},t.paths=[],t.children=[],t.webpackPolyfill=1),t}},function(t,e){!function(){"use strict";for(var t="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",n=new Uint8Array(256),r=0;r<t.length;r++)n[t.charCodeAt(r)]=r;e.encode=function(e){var n,r=new Uint8Array(e),o=r.length,i="";for(n=0;n<o;n+=3)i+=t[r[n]>>2],i+=t[(3&r[n])<<4|r[n+1]>>4],i+=t[(15&r[n+1])<<2|r[n+2]>>6],i+=t[63&r[n+2]];return o%3===2?i=i.substring(0,i.length-1)+"=":o%3===1&&(i=i.substring(0,i.length-2)+"=="),i},e.decode=function(t){var e,r,o,i,s,a=.75*t.length,c=t.length,p=0;"="===t[t.length-1]&&(a--,"="===t[t.length-2]&&a--);var u=new ArrayBuffer(a),h=new Uint8Array(u);for(e=0;e<c;e+=4)r=n[t.charCodeAt(e)],o=n[t.charCodeAt(e+1)],i=n[t.charCodeAt(e+2)],s=n[t.charCodeAt(e+3)],h[p++]=r<<2|o>>4,h[p++]=(15&o)<<4|i>>2,h[p++]=(3&i)<<6|63&s;return u}}()},function(t,e){(function(e){function n(t){for(var e=0;e<t.length;e++){var n=t[e];if(n.buffer instanceof ArrayBuffer){var r=n.buffer;if(n.byteLength!==r.byteLength){var o=new Uint8Array(n.byteLength);o.set(new Uint8Array(r,n.byteOffset,n.byteLength)),r=o.buffer}t[e]=r}}}function r(t,e){e=e||{};var r=new i;n(t);for(var o=0;o<t.length;o++)r.append(t[o]);return e.type?r.getBlob(e.type):r.getBlob()}function o(t,e){return n(t),new Blob(t,e||{})}var i=e.BlobBuilder||e.WebKitBlobBuilder||e.MSBlobBuilder||e.MozBlobBuilder,s=function(){try{var t=new Blob(["hi"]);return 2===t.size}catch(e){return!1}}(),a=s&&function(){try{var t=new Blob([new Uint8Array([1,2])]);return 2===t.size}catch(e){return!1}}(),c=i&&i.prototype.append&&i.prototype.getBlob;t.exports=function(){return s?a?e.Blob:o:c?r:void 0}()}).call(e,function(){return this}())},function(t,e){e.encode=function(t){var e="";for(var n in t)t.hasOwnProperty(n)&&(e.length&&(e+="&"),e+=encodeURIComponent(n)+"="+encodeURIComponent(t[n]));return e},e.decode=function(t){for(var e={},n=t.split("&"),r=0,o=n.length;r<o;r++){var i=n[r].split("=");e[decodeURIComponent(i[0])]=decodeURIComponent(i[1])}return e}},function(t,e){t.exports=function(t,e){var n=function(){};n.prototype=e.prototype,t.prototype=new n,t.prototype.constructor=t}},function(t,e){"use strict";function n(t){var e="";do e=s[t%a]+e,t=Math.floor(t/a);while(t>0);return e}function r(t){var e=0;for(u=0;u<t.length;u++)e=e*a+c[t.charAt(u)];return e}function o(){var t=n(+new Date);return t!==i?(p=0,i=t):t+"."+n(p++)}for(var i,s="0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz-_".split(""),a=64,c={},p=0,u=0;u<a;u++)c[s[u]]=u;o.encode=n,o.decode=r,t.exports=o},function(t,e,n){(function(e){function r(){}function o(t){i.call(this,t),this.query=this.query||{},a||(e.___eio||(e.___eio=[]),a=e.___eio),this.index=a.length;var n=this;a.push(function(t){n.onData(t)}),this.query.j=this.index,e.document&&e.addEventListener&&e.addEventListener("beforeunload",function(){n.script&&(n.script.onerror=r)},!1)}var i=n(20),s=n(31);t.exports=o;var a,c=/\n/g,p=/\\n/g;s(o,i),o.prototype.supportsBinary=!1,o.prototype.doClose=function(){this.script&&(this.script.parentNode.removeChild(this.script),this.script=null),this.form&&(this.form.parentNode.removeChild(this.form),this.form=null,this.iframe=null),i.prototype.doClose.call(this)},o.prototype.doPoll=function(){var t=this,e=document.createElement("script");this.script&&(this.script.parentNode.removeChild(this.script),this.script=null),e.async=!0,e.src=this.uri(),e.onerror=function(e){t.onError("jsonp poll error",e)};var n=document.getElementsByTagName("script")[0];n?n.parentNode.insertBefore(e,n):(document.head||document.body).appendChild(e),this.script=e;var r="undefined"!=typeof navigator&&/gecko/i.test(navigator.userAgent);r&&setTimeout(function(){var t=document.createElement("iframe");document.body.appendChild(t),document.body.removeChild(t)},100)},o.prototype.doWrite=function(t,e){function n(){r(),e()}function r(){if(o.iframe)try{o.form.removeChild(o.iframe)}catch(t){o.onError("jsonp polling iframe removal error",t)}try{var e='<iframe src="javascript:0" name="'+o.iframeId+'">';i=document.createElement(e)}catch(t){i=document.createElement("iframe"),i.name=o.iframeId,i.src="javascript:0"}i.id=o.iframeId,o.form.appendChild(i),o.iframe=i}var o=this;if(!this.form){var i,s=document.createElement("form"),a=document.createElement("textarea"),u=this.iframeId="eio_iframe_"+this.index;s.className="socketio",s.style.position="absolute",s.style.top="-1000px",s.style.left="-1000px",s.target=u,s.method="POST",s.setAttribute("accept-charset","utf-8"),a.name="d",s.appendChild(a),document.body.appendChild(s),this.form=s,this.area=a}this.form.action=this.uri(),r(),t=t.replace(p,"\\\n"),this.area.value=t.replace(c,"\\n");try{this.form.submit()}catch(h){}this.iframe.attachEvent?this.iframe.onreadystatechange=function(){"complete"===o.iframe.readyState&&n()}:this.iframe.onload=n}}).call(e,function(){return this}())},function(t,e,n){(function(e){function r(t){var e=t&&t.forceBase64;e&&(this.supportsBinary=!1),this.perMessageDeflate=t.perMessageDeflate,this.usingBrowserWebSocket=h&&!t.forceNode,this.protocols=t.protocols,this.usingBrowserWebSocket||(l=o),i.call(this,t)}var o,i=n(21),s=n(22),a=n(30),c=n(31),p=n(32),u=n(3)("engine.io-client:websocket"),h=e.WebSocket||e.MozWebSocket;if("undefined"==typeof window)try{o=n(35)}catch(f){}var l=h;l||"undefined"!=typeof window||(l=o),t.exports=r,c(r,i),r.prototype.name="websocket",r.prototype.supportsBinary=!0,r.prototype.doOpen=function(){if(this.check()){var t=this.uri(),e=this.protocols,n={agent:this.agent,perMessageDeflate:this.perMessageDeflate};n.pfx=this.pfx,n.key=this.key,n.passphrase=this.passphrase,n.cert=this.cert,n.ca=this.ca,n.ciphers=this.ciphers,n.rejectUnauthorized=this.rejectUnauthorized,this.extraHeaders&&(n.headers=this.extraHeaders),this.localAddress&&(n.localAddress=this.localAddress);try{this.ws=this.usingBrowserWebSocket?e?new l(t,e):new l(t):new l(t,e,n)}catch(r){return this.emit("error",r)}void 0===this.ws.binaryType&&(this.supportsBinary=!1),this.ws.supports&&this.ws.supports.binary?(this.supportsBinary=!0,this.ws.binaryType="nodebuffer"):this.ws.binaryType="arraybuffer",this.addEventListeners()}},r.prototype.addEventListeners=function(){var t=this;this.ws.onopen=function(){t.onOpen()},this.ws.onclose=function(){t.onClose()},this.ws.onmessage=function(e){t.onData(e.data)},this.ws.onerror=function(e){t.onError("websocket error",e)}},r.prototype.write=function(t){function n(){r.emit("flush"),setTimeout(function(){r.writable=!0,r.emit("drain")},0)}var r=this;this.writable=!1;for(var o=t.length,i=0,a=o;i<a;i++)!function(t){s.encodePacket(t,r.supportsBinary,function(i){if(!r.usingBrowserWebSocket){var s={};if(t.options&&(s.compress=t.options.compress),r.perMessageDeflate){var a="string"==typeof i?e.Buffer.byteLength(i):i.length;a<r.perMessageDeflate.threshold&&(s.compress=!1)}}try{r.usingBrowserWebSocket?r.ws.send(i):r.ws.send(i,s)}catch(c){u("websocket closed before onclose event")}--o||n()})}(t[i])},r.prototype.onClose=function(){i.prototype.onClose.call(this)},r.prototype.doClose=function(){"undefined"!=typeof this.ws&&this.ws.close()},r.prototype.uri=function(){var t=this.query||{},e=this.secure?"wss":"ws",n="";this.port&&("wss"===e&&443!==Number(this.port)||"ws"===e&&80!==Number(this.port))&&(n=":"+this.port),this.timestampRequests&&(t[this.timestampParam]=p()),this.supportsBinary||(t.b64=1),t=a.encode(t),t.length&&(t="?"+t);var r=this.hostname.indexOf(":")!==-1;return e+"://"+(r?"["+this.hostname+"]":this.hostname)+n+this.path+t},r.prototype.check=function(){return!(!l||"__initialize"in l&&this.name===r.prototype.name)}}).call(e,function(){return this}())},function(t,e){},function(t,e){var n=[].indexOf;t.exports=function(t,e){if(n)return t.indexOf(e);for(var r=0;r<t.length;++r)if(t[r]===e)return r;return-1}},function(t,e,n){"use strict";function r(t,e,n){this.io=t,this.nsp=e,this.json=this,this.ids=0,this.acks={},this.receiveBuffer=[],this.sendBuffer=[],this.connected=!1,this.disconnected=!0,n&&n.query&&(this.query=n.query),this.io.autoConnect&&this.open()}var o="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function(t){return typeof t}:function(t){return t&&"function"==typeof Symbol&&t.constructor===Symbol&&t!==Symbol.prototype?"symbol":typeof t},i=n(7),s=n(8),a=n(38),c=n(39),p=n(40),u=n(3)("socket.io-client:socket"),h=n(30);t.exports=e=r;var f={connect:1,connect_error:1,connect_timeout:1,connecting:1,disconnect:1,error:1,reconnect:1,reconnect_attempt:1,reconnect_failed:1,reconnect_error:1,reconnecting:1,ping:1,pong:1},l=s.prototype.emit;s(r.prototype),r.prototype.subEvents=function(){if(!this.subs){var t=this.io;this.subs=[c(t,"open",p(this,"onopen")),c(t,"packet",p(this,"onpacket")),c(t,"close",p(this,"onclose"))]}},r.prototype.open=r.prototype.connect=function(){return this.connected?this:(this.subEvents(),this.io.open(),"open"===this.io.readyState&&this.onopen(),this.emit("connecting"),this)},r.prototype.send=function(){var t=a(arguments);return t.unshift("message"),this.emit.apply(this,t),this},r.prototype.emit=function(t){if(f.hasOwnProperty(t))return l.apply(this,arguments),this;var e=a(arguments),n={type:i.EVENT,data:e};return n.options={},n.options.compress=!this.flags||!1!==this.flags.compress,"function"==typeof e[e.length-1]&&(u("emitting packet with ack id %d",this.ids),this.acks[this.ids]=e.pop(),n.id=this.ids++),this.connected?this.packet(n):this.sendBuffer.push(n),delete this.flags,this},r.prototype.packet=function(t){t.nsp=this.nsp,this.io.packet(t)},r.prototype.onopen=function(){if(u("transport is open - connecting"),"/"!==this.nsp)if(this.query){var t="object"===o(this.query)?h.encode(this.query):this.query;u("sending connect packet with query %s",t),this.packet({type:i.CONNECT,query:t})}else this.packet({type:i.CONNECT})},r.prototype.onclose=function(t){u("close (%s)",t),this.connected=!1,this.disconnected=!0,delete this.id,this.emit("disconnect",t)},r.prototype.onpacket=function(t){if(t.nsp===this.nsp)switch(t.type){case i.CONNECT:this.onconnect();break;case i.EVENT:this.onevent(t);break;case i.BINARY_EVENT:this.onevent(t);break;case i.ACK:this.onack(t);break;case i.BINARY_ACK:this.onack(t);break;case i.DISCONNECT:this.ondisconnect();break;case i.ERROR:this.emit("error",t.data)}},r.prototype.onevent=function(t){var e=t.data||[];u("emitting event %j",e),null!=t.id&&(u("attaching ack callback to event"),e.push(this.ack(t.id))),this.connected?l.apply(this,e):this.receiveBuffer.push(e)},r.prototype.ack=function(t){var e=this,n=!1;return function(){if(!n){n=!0;var r=a(arguments);u("sending ack %j",r),e.packet({type:i.ACK,id:t,data:r})}}},r.prototype.onack=function(t){var e=this.acks[t.id];"function"==typeof e?(u("calling ack %s with %j",t.id,t.data),e.apply(this,t.data),delete this.acks[t.id]):u("bad ack %s",t.id)},r.prototype.onconnect=function(){this.connected=!0,this.disconnected=!1,this.emit("connect"),this.emitBuffered()},r.prototype.emitBuffered=function(){var t;for(t=0;t<this.receiveBuffer.length;t++)l.apply(this,this.receiveBuffer[t]);for(this.receiveBuffer=[],t=0;t<this.sendBuffer.length;t++)this.packet(this.sendBuffer[t]);this.sendBuffer=[]},r.prototype.ondisconnect=function(){u("server disconnect (%s)",this.nsp),this.destroy(),this.onclose("io server disconnect")},r.prototype.destroy=function(){if(this.subs){for(var t=0;t<this.subs.length;t++)this.subs[t].destroy();this.subs=null}this.io.destroy(this)},r.prototype.close=r.prototype.disconnect=function(){return this.connected&&(u("performing disconnect (%s)",this.nsp),this.packet({type:i.DISCONNECT})),this.destroy(),this.connected&&this.onclose("io client disconnect"),this},r.prototype.compress=function(t){return this.flags=this.flags||{},this.flags.compress=t,this}},function(t,e){function n(t,e){var n=[];e=e||0;for(var r=e||0;r<t.length;r++)n[r-e]=t[r];return n}t.exports=n},function(t,e){"use strict";function n(t,e,n){return t.on(e,n),{destroy:function(){t.removeListener(e,n)}}}t.exports=n},function(t,e){var n=[].slice;t.exports=function(t,e){if("string"==typeof e&&(e=t[e]),"function"!=typeof e)throw new Error("bind() requires a function");var r=n.call(arguments,2);return function(){return e.apply(t,r.concat(n.call(arguments)))}}},function(t,e){function n(t){t=t||{},this.ms=t.min||100,this.max=t.max||1e4,this.factor=t.factor||2,this.jitter=t.jitter>0&&t.jitter<=1?t.jitter:0,this.attempts=0}t.exports=n,n.prototype.duration=function(){var t=this.ms*Math.pow(this.factor,this.attempts++);if(this.jitter){var e=Math.random(),n=Math.floor(e*this.jitter*t);t=0==(1&Math.floor(10*e))?t-n:t+n}return 0|Math.min(t,this.max)},n.prototype.reset=function(){this.attempts=0},n.prototype.setMin=function(t){this.ms=t},n.prototype.setMax=function(t){this.max=t},n.prototype.setJitter=function(t){this.jitter=t}}])});
//# sourceMappingURL=socket.io.js.map;
/*******************************************************************************
 * @license
 * Copyright (c) 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd */
/* Initializes websocket connection */

define('orion/collab/collabSocket',['orion/EventTarget','socket.io/socket.io'], function(EventTarget, io) {
    'use strict;'

    var DEBUG = false;

    /**
     * Collab socket client
     * 
     * @class
     * @constructor
     * 
     * @param {string} sessionId
     */
	function CollabSocket(hubUrl, sessionId) {
        var self = this;
		
		this.socket = io.connect( hubUrl+ "?sessionId=" +sessionId, { path: "/socket.io/" });

        this.socket.on('connect', function() {
            self.dispatchEvent({
                type: 'ready'
            });
        });
        
        this.socket.on('disconnect', function() {
            self.dispatchEvent({
                type: 'close'
            });
        });
        
        this.socket.on('error', function(e) {
           self.dispatchEvent({
                type: 'error',
                error: e
            });
            console.error(e);
        });

        this.socket.on('message', function(data) {
            self.dispatchEvent({
                type: 'message',
                data: data
            });
            if (DEBUG) {
                var msgObj = JSON.parse(data);
                console.log('CollabSocket In: ' + msgObj.type, msgObj);
            }
        });

        EventTarget.attach(this);
	}

	CollabSocket.prototype.constructor = CollabSocket;

    /**
     * Send message
     * 
     * @param {string} message
     */
    CollabSocket.prototype.send = function(message) {
        this.socket.send(message);
        if (DEBUG) {
            var msgObj = JSON.parse(message);
            console.log('CollabSocket Out: ' + msgObj.type, msgObj);
        }
    };

    /**
     * Close this socket
     */
    CollabSocket.prototype.close = function() {
        this.socket.close();
    };

	return {
        CollabSocket: CollabSocket
    };
});

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd, node*/
(function(root, factory) { // UMD
    if (typeof define === "function" && define.amd) { //$NON-NLS-0$
        define('orion/Deferred',factory);
    } else if (typeof exports === "object") { //$NON-NLS-0$
        module.exports = factory();
    } else {
        root.orion = root.orion || {};
        root.orion.Deferred = factory();
    }
}(this, function() {
    var queue = [],
        running = false;

    function run() {
        var fn;
        while ((fn = queue.shift())) {
            fn();
        }
        running = false;
    }

	var runAsync = (function() {
		if (typeof process !== "undefined" && typeof process.nextTick === "function") {
			var nextTick = process.nextTick;
    		return function() {
    			nextTick(run);
    		};
		} else if (typeof MutationObserver === "function") {
			var div = document.createElement("div");
			var observer = new MutationObserver(run);
			observer.observe(div, {
            	attributes: true
        	});
        	return function() {
        		div.setAttribute("class", "_tick");
        	};
		}
		return function() {
			setTimeout(run, 0);
		};
	})();

    function enqueue(fn) {
        queue.push(fn);
        if (!running) {
            running = true;
            runAsync();
        }
    }

    function noReturn(fn) {
        return function(result) {
            fn(result);
        };
    }
    
    function settleDeferred(fn, result, deferred) {
    	try {
    		var listenerResult = fn(result);
    		var listenerThen = listenerResult && (typeof listenerResult === "object" || typeof listenerResult === "function") && listenerResult.then;
    		if (typeof listenerThen === "function") {
    			if (listenerResult === deferred.promise) {
    				deferred.reject(new TypeError());
    			} else {
    				var listenerResultCancel = listenerResult.cancel;
    				if (typeof listenerResultCancel === "function") {
    					deferred._parentCancel = listenerResultCancel.bind(listenerResult);
    				} else {
    					delete deferred._parentCancel;
    				}
    				listenerThen.call(listenerResult, noReturn(deferred.resolve), noReturn(deferred.reject), noReturn(deferred.progress));
    			}
    		} else {
    			deferred.resolve(listenerResult);
    		}
    	} catch (e) {
    		deferred.reject(e);
    	}
    }


    /**
     * @name orion.Promise
     * @class Interface representing an eventual value.
     * @description Promise is an interface that represents an eventual value returned from the single completion of an operation.
     *
     * <p>For a concrete class that implements Promise and provides additional API, see {@link orion.Deferred}.</p>
     * @see orion.Deferred
     * @see orion.Deferred#promise
     */
    /**
     * @name then
     * @function
     * @memberOf orion.Promise.prototype
     * @description Adds handlers to be called on fulfillment or progress of this promise.
     * @param {Function} [onResolve] Called when this promise is resolved.
     * @param {Function} [onReject] Called when this promise is rejected.
     * @param {Function} [onProgress] May be called to report progress events on this promise.
     * @returns {orion.Promise} A new promise that is fulfilled when the given <code>onResolve</code> or <code>onReject</code>
     * callback is finished. The callback's return value gives the fulfillment value of the returned promise.
     */
    /**
     * Cancels this promise.
     * @name cancel
     * @function
     * @memberOf orion.Promise.prototype
     * @param {Object} reason The reason for canceling this promise.
     * @param {Boolean} [strict]
     */

    /**
     * @name orion.Deferred
     * @borrows orion.Promise#then as #then
     * @borrows orion.Promise#cancel as #cancel
     * @class Provides abstraction over asynchronous operations.
     * @description Deferred provides abstraction over asynchronous operations.
     *
     * <p>Because Deferred implements the {@link orion.Promise} interface, a Deferred may be used anywhere a Promise is called for.
     * However, in most such cases it is recommended to use the Deferred's {@link #promise} field instead, which exposes a 
     * simplified, minimally <a href="https://github.com/promises-aplus/promises-spec">Promises/A+</a>-compliant interface to callers.</p>
     */
    function Deferred() {
        var result, state, listeners = [],
            _this = this;

        function notify() {
            var listener;
            while ((listener = listeners.shift())) {
                var deferred = listener.deferred;
                var methodName = state === "fulfilled" ? "resolve" : "reject"; //$NON-NLS-0$ //$NON-NLS-1$ //$NON-NLS-2$
                var fn = listener[methodName];
                if (typeof fn === "function") { //$NON-NLS-0$
                	settleDeferred(fn, result, deferred);
                } else {
                    deferred[methodName](result);
                }
            }
        }

        function _reject(error) {
            delete _this._parentCancel;
            state = "rejected";
            result = error;
            if (listeners.length) {
                enqueue(notify);
            }
        }

        function _resolve(value) {
            function once(fn) {
                return function(result) {
                    if (!state || state === "assumed") {
                          fn(result);
                    }
                };
            }
            delete _this._parentCancel;
            try {
                var valueThen = value && (typeof value === "object" || typeof value === "function") && value.then;
                if (typeof valueThen === "function") {
                    if (value === _this) {
                        _reject(new TypeError());
                    } else {
                        state = "assumed";
                        var valueCancel = value && value.cancel;
                        if (typeof valueCancel !== "function") {
                            var deferred = new Deferred();
                            value = deferred.promise;
                            try {
                                valueThen(deferred.resolve, deferred.reject, deferred.progress);
                            } catch (thenError) {
                                deferred.reject(thenError);
                            }
                            valueCancel = value.cancel;
                            valueThen = value.then;
                        }
                        result = value;
                        valueThen.call(value, once(_resolve), once(_reject));
                        _this._parentCancel = valueCancel.bind(value);
                    }
                } else {
                    state = "fulfilled";
                    result = value;
                    if (listeners.length) {
                        enqueue(notify);
                    }
                }
            } catch (error) {
                once(_reject)(error);
            }
        }

        function cancel() {
            var parentCancel = _this._parentCancel;
            if (parentCancel) {
                delete _this._parentCancel;
                parentCancel();
            } else if (!state) {
                var cancelError = new Error("Cancel");
                cancelError.name = "Cancel";
                _reject(cancelError);
            }
        }


        /**
         * Resolves this Deferred.
         * @name resolve
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} value
         * @returns {orion.Promise}
         */
        this.resolve = function(value) {
            if (!state) {
                _resolve(value);
            }
            return _this;
        };

        /**
         * Rejects this Deferred.
         * @name reject
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} error
         * @param {Boolean} [strict]
         * @returns {orion.Promise}
         */
        this.reject = function(error) {
            if (!state) {
                _reject(error);
            }
            return _this;
        };

        /**
         * Notifies listeners of progress on this Deferred.
         * @name progress
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} update The progress update.
         * @returns {orion.Promise}
         */
        this.progress = function(update) {
            if (!state) {
                listeners.forEach(function(listener) {
                    if (listener.progress) {
                        try {
                            listener.progress(update);
                        } catch (ignore) {
                            // ignore
                        }
                    }
                });
            }
            return _this.promise;
        };

        this.cancel = function() {
            if (_this._parentCancel) {
                setTimeout(cancel, 0);
            } else {
                cancel();
            }
            return _this;
        };

        // Note: "then" ALWAYS returns before having onResolve or onReject called as per http://promises-aplus.github.com/promises-spec/
        this.then = function(onFulfill, onReject, onProgress) {
        	var deferred = new Deferred();
            deferred._parentCancel = _this.promise.cancel;
            listeners.push({
                resolve: onFulfill,
                reject: onReject,
                progress: onProgress,
                deferred: deferred
            });
            if (state === "fulfilled" || state === "rejected") {
                enqueue(notify);
            }
            return deferred.promise;
        };

        /**
         * The promise exposed by this Deferred.
         * @name promise
         * @field
         * @memberOf orion.Deferred.prototype
         * @type orion.Promise
         */
        this.promise = {
            then: _this.then,
            cancel: _this.cancel
        };
    }

    /**
     * Returns a promise that represents the outcome of all the input promises.
     * <p>When <code>all</code> is called with a single parameter, the returned promise has <dfn>eager</dfn> semantics,
     * meaning that if any input promise rejects, the returned promise immediately rejects, without waiting for the rest of the
     * input promises to fulfill.</p>
     *
     * To obtain <dfn>lazy</dfn> semantics (meaning the returned promise waits for every input promise to fulfill), pass the
     * optional parameter <code>optOnError</code>.
     * @name all
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {orion.Promise[]} promises The input promises.
     * @param {Function} [optOnError] Handles a rejected input promise. <code>optOnError</code> is invoked for every rejected
     * input promise, and is passed the reason the input promise was rejected. <p><code>optOnError</code> can return a value, which
     * allows it to act as a transformer: the return value serves as the final fulfillment value of the rejected promise in the 
     * results array generated by <code>all</code>.
     * @returns {orion.Promise} A new promise. The returned promise is generally fulfilled to an <code>Array</code> whose elements
     * give the fulfillment values of the input promises. <p>However, if an input promise rejects and eager semantics is used, the 
     * returned promise will instead be fulfilled to a single error value.</p>
     */
    Deferred.all = function(promises, optOnError) {
        var count = promises.length,
            result = [],
            rejected = false,
            deferred = new Deferred();

        deferred.then(undefined, function() {
            rejected = true;
            promises.forEach(function(promise) {
                if (promise.cancel) {
                    promise.cancel();
                }
            });
        });

        function onResolve(i, value) {
            if (!rejected) {
                result[i] = value;
                if (--count === 0) {
                    deferred.resolve(result);
                }
            }
        }

        function onReject(i, error) {
            if (!rejected) {
                if (optOnError) {
                    try {
                        onResolve(i, optOnError(error));
                        return;
                    } catch (e) {
                        error = e;
                    }
                }
                deferred.reject(error);
            }
        }

        if (count === 0) {
            deferred.resolve(result);
        } else {
            promises.forEach(function(promise, i) {
                promise.then(onResolve.bind(undefined, i), onReject.bind(undefined, i));
            });
        }
        return deferred.promise;
    };

    /**
     * Applies callbacks to a promise or to a regular object.
     * @name when
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {Object|orion.Promise} value Either a {@link orion.Promise}, or a normal value.
     * @param {Function} onResolve Called when the <code>value</code> promise is resolved. If <code>value</code> is not a promise,
     * this function is called immediately.
     * @param {Function} onReject Called when the <code>value</code> promise is rejected. If <code>value</code> is not a promise, 
     * this function is never called.
     * @param {Function} onProgress Called when the <code>value</code> promise provides a progress update. If <code>value</code> is
     * not a promise, this function is never called.
     * @returns {orion.Promise} A new promise.
     */
    Deferred.when = function(value, onResolve, onReject, onProgress) {
        var promise, deferred;
        if (value && typeof value.then === "function") { //$NON-NLS-0$
            promise = value;
        } else {
            deferred = new Deferred();
            deferred.resolve(value);
            promise = deferred.promise;
        }
        return promise.then(onResolve, onReject, onProgress);
    };

    return Deferred;
}));
/*******************************************************************************
 * Copyright (c) 2014 SAP AG and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAP AG - initial API and implementation
 *******************************************************************************/
define('orion/xsrfUtils',[],function(){
	var XSRF_TOKEN = "x-csrf-token";//$NON-NLS-0$

	/**
	 * extracts value of xsrf cookie if available
	 */
	function getCSRFToken() {
		if (typeof document === "undefined") return null;
		var cookies = document.cookie.split(";");//$NON-NLS-0$

		var i,n,v;
		for(i = 0; i<cookies.length; i++) {
			n = cookies[i].substr(0, cookies[i].indexOf("=")).trim();//$NON-NLS-0$
			v = cookies[i].substr(cookies[i].indexOf("=") + 1).trim();//$NON-NLS-0$

			if(n == XSRF_TOKEN) {
				return v;
			}
		}
	}

	/**
	 * adds xsrf nonce to header if set in cookies
	 * @param {Object} request header
	 */
	function setNonceHeader(headers) {
		var token = getCSRFToken();
		if (token) {
			headers[XSRF_TOKEN] = token;
		}
	}

	/**
	 * adds xsrf nonce to an XMLHTTPRequest object if set in cookies
	 * @param {Object} XMLHttpRequest object
	 */
	function addCSRFNonce(request) {
		var token = getCSRFToken();
		if(token) {
			request.setRequestHeader(XSRF_TOKEN, token);
		}
	}

	return {
		XSRF_TOKEN: XSRF_TOKEN,
		getCSRFToken: getCSRFToken,
		setNonceHeader: setNonceHeader,
		addCSRFNonce: addCSRFNonce
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global StopIteration*/
// URL Shim -- see http://url.spec.whatwg.org/ and http://dvcs.w3.org/hg/url/raw-file/tip/Overview.html

(function() {
    try {
        var testURL;
        if (typeof self.URL === "function" && self.URL.length !== 0 &&
                (testURL = new self.URL("http://www.w3.org?q")).protocol === "http:" && testURL.query) {
            return;
        }
    } catch (e) {}

    //[1]scheme, [2]authority, [3]path, [4]query, [5]fragment
    var _URI_RE = /^(?:([^:\/?#]+):)?(?:\/\/([^\/?#]*))?([^?#]*)(?:\?([^#]*))?(?:#(.*))?$/;
    //[ userinfo "@" ] host [ ":" port ]
    var _AUTHORITY_RE = /^(?:(.*)@)?(\[[^\]]*\]|[^:]*)(?::(.*))?$/;

    var _NO_WS_RE = /^\S*$/;
    var _SCHEME_RE = /^([a-zA-Z](?:[a-zA-Z0-9+-.])*)$/;
    var _PORT_RE = /^\d*$/;
    var _HOST_RE = /^(\[[^\]\/?#\s]*\]|[^:\/?#\s]*)$/;
    var _HOSTPORT_RE = /^(\[[^\]\/?#\s]*\]|[^:\/?#\s]*)(?::(\d*))?$/;
    var _PATH_RE = /^([^?#\s]*)$/;
    var _QUERY_RE = /^([^\s]*)$/;
    var _FRAGMENT_RE = _NO_WS_RE;
    var _USERNAME_PASSWORD_RE = /([^:]*):?(.*)/;

    var STOP_ITERATION = typeof StopIteration !== "undefined" ? StopIteration : new Error("Stop Iteration");
    var DEFAULT_PORTS = {
        "ftp:": "21",
            "gopher:": "70",
            "http:": "80",
            "https:": "443",
            "ws:": "80",
            "wss:": "443"
    };

    function _checkString(txt) {
        if (typeof txt !== "string") {
            throw new TypeError();
        }
    }

    function _parseQuery(query) {
        return query ? query.split("&") : [];
    }

    function _stringifyQuery(pairs) {
        if (pairs.length === 0) {
            return "";
        }
        return pairs.join("&");
    }

    function _parsePair(pair) {
        var parsed = /([^=]*)(?:=?)(.*)/.exec(pair);
        var key = parsed[1] ? decodeURIComponent(parsed[1]) : "";
        var value = parsed[2] ? decodeURIComponent(parsed[2]) : "";
        return [key, value];
    }

    function _stringifyPair(entry) {
        var pair = encodeURIComponent(entry[0]);
        if (entry[1]) {
            pair += "=" + encodeURIComponent(entry[1]);
        }
        return pair;
    }

    function _createMapIterator(url, kind) {
        var query = "";
        var pairs = [];
        var index = 0;
        return {
            next: function() {
                if (query !== url.query) {
                    query = url.query;
                    pairs = _parseQuery(query);
                }
                if (index < pairs.length) {
                    var entry = _parsePair(pairs[index++]);
                    switch (kind) {
                        case "keys":
                            return entry[0];
                        case "values":
                            return entry[1];
                        case "keys+values":
                            return [entry[0], entry[1]];
                        default:
                            throw new TypeError();
                    }
                }
                throw STOP_ITERATION;
            }
        };
    }

    // See http://url.spec.whatwg.org/#interface-urlquery
    function URLQuery(url) {
        Object.defineProperty(this, "_url", {
            get: function() {
                return url._url;
            }
        });
    }

    Object.defineProperties(URLQuery.prototype, {
        get: {
            value: function(key) {
                _checkString(key);
                var result;
                var pairs = _parseQuery(this._url.query);
                pairs.some(function(pair) {
                    var entry = _parsePair(pair);
                    if (entry[0] === key) {
                        result = entry[1];
                        return true;
                    }
                });
                return result;
            },
            enumerable: true
        },
        set: {
            value: function(key, value) {
                _checkString(key);
                _checkString(value);
                var pairs = _parseQuery(this._url.query);
                var found = pairs.some(function(pair, i) {
                    var entry = _parsePair(pair);
                    if (entry[0] === key) {
                        entry[1] = value;
                        pairs[i] = _stringifyPair(entry);
                        return true;
                    }
                });
                if (!found) {
                    pairs.push(_stringifyPair([key, value]));
                }
                this._url.query = _stringifyQuery(pairs);
            },
            enumerable: true
        },
        has: {
            value: function(key) {
                _checkString(key);
                var pairs = _parseQuery(this._url.query);
                return pairs.some(function(pair) {
                    var entry = _parsePair(pair);
                    if (entry[0] === key) {
                        return true;
                    }
                });
            },
            enumerable: true
        },
            "delete": {
            value: function(key) {
                _checkString(key);
                var pairs = _parseQuery(this._url.query);
                var filtered = pairs.filter(function(pair) {
                    var entry = _parsePair(pair);
                    return entry[0] !== key;
                });
                if (filtered.length !== pairs.length) {
                    this._url.query = _stringifyQuery(filtered);
                    return true;
                }
                return false;
            },
            enumerable: true
        },
        clear: {
            value: function() {
                this._url.query = "";
            },
            enumerable: true
        },
        forEach: {
            value: function(callback, thisArg) {
                if (typeof callback !== "function") {
                    throw new TypeError();
                }
                var iterator = _createMapIterator(this._url, "keys+values");
                try {
                    while (true) {
                        var entry = iterator.next();
                        callback.call(thisArg, entry[1], entry[0], this);
                    }
                } catch (e) {
                    if (e !== STOP_ITERATION) {
                        throw e;
                    }
                }
            },
            enumerable: true
        },
        keys: {
            value: function() {
                return _createMapIterator(this._url, "keys");
            },
            enumerable: true
        },
        values: {
            value: function() {
                return _createMapIterator(this._url, "values");
            },
            enumerable: true
        },
        items: {
            value: function() {
                return _createMapIterator(this._url, "keys+values");
            }
        },
        size: {
            get: function() {
                return _parseQuery(this._url.query).length;
            },
            enumerable: true
        },
        getAll: {
            value: function(key) {
                _checkString(key);
                var result = [];
                var pairs = _parseQuery(this._url.query);
                pairs.forEach(function(pair) {
                    var entry = _parsePair(pair);
                    if (entry[0] === key) {
                        result.push(entry[1]);
                    }
                });
                return result;
            },
            enumerable: true
        },
        append: {
            value: function(key, value) {
                _checkString(key);
                _checkString(value);
                var pairs = _parseQuery(this._url.query);
                pairs.push(_stringifyPair([key, value]));
                this._url.query = _stringifyQuery(pairs);
            },
            enumerable: true
        }
    });

    function _makeAbsoluteURL(url, base) {
        if (!url.scheme && base) {
            url.scheme = base.scheme;
            if (!url.host && base.host) {
                url.userinfo = base.userinfo;
                url.host = base.host;
                url.port = base.port;
                url.pathRelative = true;
            }
        }
        if (url.pathRelative) {
            if (!url.path) {
                url.path = base.path;
            } else if (url.path[0] !== "/") {
                var basePath = /^(.*\/)[^\/]*$/.exec(base.path)[1] || "/";
                url.path = basePath + url.path;
            }
        }
    }

    function _normalizeScheme(scheme) {
        return scheme.toLowerCase();
    }

    function _normalizePort(port) {
        return port ? (/[1-9]\d*$/).exec(port)[0] : "";
    }

    function _normalizePath(path) {
        var result = [];
        path.split("/").forEach(function(segment) {
            if (segment === "..") {
            	if (result.length > 1) {
                	result.pop();
            	}
            } else if (segment !== ".") {
                result.push(segment);
            }
        });
        return result.join("/");
    }


    function _normalizeURL(url) {
        if (url.scheme) {
            url.scheme = _normalizeScheme(url.scheme);
        }
        if (url.port) {
            url.port = _normalizePort(url.port);
        }
        if (url.host && url.path) {
            url.path = _normalizePath(url.path);
        }
    }

    function _encodeWhitespace(text) {
        return text.replace(/\s/g, function(c) {
            return "%" + c.charCodeAt(0).toString(16);
        });
    }

    function _parseURL(input, base) {
        if (typeof input !== "string") {
            throw new TypeError();
        }

        input = _encodeWhitespace(input);

        var parsedURI = _URI_RE.exec(input);
        if (!parsedURI) {
            return null;
        }
        var url = {};
        url.scheme = parsedURI[1] || "";
        if (url.scheme && !_SCHEME_RE.test(url.scheme)) {
            return null;
        }
        var authority = parsedURI[2];
        if (authority) {
            var parsedAuthority = _AUTHORITY_RE.exec(authority);
            url.userinfo = parsedAuthority[1];
            url.host = parsedAuthority[2];
            url.port = parsedAuthority[3];
            if (url.port && !_PORT_RE.test(url.port)) {
                return null;
            }
        }
        url.path = parsedURI[3];
        url.query = parsedURI[4];
        url.fragment = parsedURI[5];

        _makeAbsoluteURL(url, base);
        _normalizeURL(url);
        return url;
    }

    function _serialize(url) {
        var result = (url.scheme ? url.scheme + ":" : "");
        if (url.host) {
            result += "//";
            if (url.userinfo) {
                result += url.userinfo + "@";
            }
            result += url.host;
            if (url.port) {
                result += ":" + url.port;
            }
        }
        result += url.path;
        if (url.query) {
            result += "?" + url.query;
        }
        if (url.fragment) {
            result += "#" + url.fragment;
        }
        return result;
    }

    // See http://url.spec.whatwg.org/#api
    function URL(input, base) {
        var baseURL;
        if (base) {
            base = base.href || base;
            baseURL = _parseURL(base);
            if (!baseURL || !baseURL.scheme) {
                throw new SyntaxError();
            }
            Object.defineProperty(this, "_baseURL", {
                value: baseURL
            });
        }

        var url = _parseURL(input, baseURL);
        if (!url) {
            throw new SyntaxError();
        }

        Object.defineProperty(this, "_input", {
            value: input,
            writable: true
        });

        Object.defineProperty(this, "_url", {
            value: url,
            writable: true
        });

        var query = new URLQuery(this);
        Object.defineProperty(this, "query", {
            get: function() {
                return this._url ? query : null;
            },
            enumerable: true
        });
    }

    Object.defineProperties(URL.prototype, {
    	toString: {
    		value: function() {
    			return this.href;
    		}		
    	},
        href: {
            get: function() {
                return this._url ? _serialize(this._url) : this._input;
            },
            set: function(value) {
                _checkString(value);
                this._input = value;
                this._url = _parseURL(this._input, this._baseURL);
            },
            enumerable: true
        },
        origin: {
            get: function() {
                return (this._url && this._url.host ? this.protocol + "//" + this.host : "");
            },
            enumerable: true
        },
        protocol: {
            get: function() {
                return this._url ? this._url.scheme + ":" : ":";
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var scheme = (value.slice(-1) === ":") ? value.substring(0, value.length - 1) : value;
                if (scheme === "" || _SCHEME_RE.test(scheme)) {
                    this._url.scheme = _normalizeScheme(scheme);
                }

            },
            enumerable: true
        },
        _userinfo: { // note: not part of spec so not enumerable
            get: function() {
                return this._url ? this._url.userinfo : "";
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                this._url.userinfo = value;
            }
        },
        username: {
            get: function() {
                if (!this._url) {
                    return "";
                }
                var parsed = _USERNAME_PASSWORD_RE.exec(this._userinfo);
                var username = decodeURIComponent(parsed[1] || "");
                return username;
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var parsed = _USERNAME_PASSWORD_RE.exec(this._userinfo);
                var userpass = [encodeURIComponent(value || "")];
                if (parsed[2]) {
                    userpass.push(parsed[2]);
                }
                this._userinfo = userpass.join(":");
            },
            enumerable: true
        },
        password: {
            get: function() {
                if (!this._url) {
                    return "";
                }
                var parsed = _USERNAME_PASSWORD_RE.exec(this._userinfo);
                var password = decodeURIComponent(parsed[2] || "");
                return password;
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var parsed = _USERNAME_PASSWORD_RE.exec(this._userinfo);
                var userpass = [parsed[1] || ""];
                if (value) {
                    userpass.push(encodeURIComponent(value));
                }
                this._userinfo = userpass.join(":");
            },
            enumerable: true
        },
        host: {
            get: function() {
                var result = "";
                if (this._url && this._url.host) {
                    result += this._url.host;
                    if (this._url.port) {
                        result += ":" + this._url.port;
                    }
                }
                return result;
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var result = _HOSTPORT_RE.exec(value);
                if (result) {
                    this._url.host = result[1];
                    this._url.port = _normalizePort(result[2]);
                }
            },
            enumerable: true
        },
        hostname: {
            get: function() {
                return this._url ? this._url.host : "";
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var result = _HOST_RE.exec(value);
                if (result) {
                    this._url.host = value;
                }
            },
            enumerable: true
        },
        port: {
            get: function() {
                var port = this._url ? this._url.port || "" : "";
                if (port && port === DEFAULT_PORTS[this.protocol]) {
                    port = "";
                }
                return port;
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var result = _PORT_RE.exec(value);
                if (result) {
                    this._url.port = _normalizePort(value);
                }
            },
            enumerable: true
        },
        pathname: {
            get: function() {
                return this._url ? this._url.path : "";
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                var result = _PATH_RE.exec(value);
                if (result) {
                    if (this._url.host && value && value[0] !== "/") {
                        value = "/" + value;
                    }
                    this._url.path = value ? _normalizePath(value) : "";
                }
            },
            enumerable: true
        },
        search: {
            get: function() {
                return (this._url && this._url.query ? "?" + this._url.query : "");
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                if (value && value[0] === "?") {
                    value = value.substring(1);
                }
                var result = _QUERY_RE.exec(value);
                if (result) {
                    this._url.query = value;
                }
            },
            enumerable: true
        },
        hash: {
            get: function() {
                return (this._url && this._url.fragment ? "#" + this._url.fragment : "");
            },
            set: function(value) {
                _checkString(value);
                if (!this._url) {
                    return;
                }
                if (value && value[0] === "#") {
                    value = value.substring(1);
                }
                var result = _FRAGMENT_RE.exec(value);
                if (result) {
                    this._url.fragment = value;
                }
            },
            enumerable: true
        }
    });

	var _URL = self.URL || self.webkitURL;
    if (_URL && _URL.createObjectURL) {
        Object.defineProperty(URL, "createObjectURL", {
            value: _URL.createObjectURL.bind(_URL),
            enumerable: false
        });

        Object.defineProperty(URL, "revokeObjectURL", {
            value: _URL.revokeObjectURL.bind(_URL),
            enumerable: false
        });
    }
    self.URL = URL;
}());
define("orion/URL-shim", function(){});

/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global URL*/
/**
 * @name orion.xhr
 * @namespace Provides a promise-based API to {@link XMLHttpRequest}.
 */
define('orion/xhr',[
	'orion/Deferred',
	'orion/xsrfUtils',
	'orion/urlModifier',
	'orion/URL-shim', // no exports, must come last
], function(Deferred, xsrfUtils, urlModifier) {

	/**
	 * @name orion.xhr.Result
	 * @class Wraps an XHR response or failure.
	 * @property {Object} args Arguments passed to the {@link orion.xhr.xhr} call.
	 * @property {Object|ArrayBuffer|Blob|Document|String} response The <code>response</code> object returned by the XMLHttpRequest.
	 * It is typed according to the <code>responseType</code> passed to the XHR call (by default it is a {@link String}).
	 * @property {String} [responseText] The <code>response</code> returned by the XMLHttpRequest, if it is a {@link String}.
	 * If the <code>response</code> is not a String, this property is <code>null</code>.
	 * @property {Number} status The HTTP status code returned by the XMLHttpRequest.
	 * @property {String} url The URL that the XHR request was made to.
	 * @property {XMLHttpRequest} xhr The underlying XMLHttpRequest object.
	 * @property {String|Error} error <i>Optional</i>. If a timeout occurred or an error was thrown while performing the
	 * XMLHttpRequest, this field contains information about the error.
	 */

	/**
	 * @param {String} url
	 * @param {Object} options
	 * @param {XMLHttpRequest} xhr
	 * @param {String|Error} [error]
	 */
	function makeResult(url, options, xhr, error) {
		var response = typeof xhr.response !== 'undefined' ? xhr.response : xhr.responseText; //$NON-NLS-0$
		var responseText = typeof response === 'string' ? response : null; //$NON-NLS-0$
		var status;
		try {
			status = xhr.status;
		} catch (e) {
			status = 0;
		}
		var result = {
			args: options,
			response: response,
			responseText: responseText,
			status: status,
			url: url,
			xhr: xhr
		};
		if (typeof error !== 'undefined') { //$NON-NLS-0$
			result.error = error;
		}
		return result;
	}

	function isSameOrigin(url) {
		return new URL(location.href).origin === new URL(url, location.href).origin;
	}

	/**
	 * Wrapper for {@link XMLHttpRequest} that returns a promise.
	 * @name xhr
	 * @function
	 * @memberOf orion.xhr
	 * @param {String} method One of 'GET', 'POST', 'PUT', 'DELETE'.
	 * @param {String} url The URL to request.
	 * @param {Object} [options]
	 * @param {Object|ArrayBuffer|Blob|Document} [options.data] The raw data to send as the request body. (Only allowed for POST and PUT).
	 * @param {Object} [options.headers] A map of header names and values to set on the request.
	 * @param {Boolean} [options.log=false] If <code>true</code>, failed requests will be logged to the JavaScript console.
	 * @param {String} [options.responseType=''] Determines the type of the response object returned. Value must be one of the following:
	 * <ul><li><code>'arraybuffer'</code>
	 * <li><code>'blob'</code>
	 * <li><code>'document'</code>
	 * <li><code>'json'</code>
	 * <li><code>'text'</code>
	 * <li><code>''</code> (same as <code>'text'</code>)</ul>
	 * @param {Number} [options.timeout=0] Timeout in milliseconds. Defaults to 0 (no timeout).
	 * @returns {Deferred} A deferred for the result. The deferred will resolve on 2xx, 3xx status codes or reject on 4xx, 5xx status codes.
	 * In both cases a {@link orion.xhr.Result} is provided to the listener.
	 */
	// TODO: upload progress, user/password
	function _xhr(method, url, options/*, XMLHttpRequestImpl */) {
		options = options || {};
		url = urlModifier(url);

		var xhr = (arguments.length > 3 && arguments[3]) ? arguments[3] : new XMLHttpRequest(); //$NON-NLS-0$
		var d = new Deferred();
		var headers = options.headers || {};
		if (isSameOrigin(url)) {
			xsrfUtils.setNonceHeader(headers);
		}
		var log = options.log || false;
		var data;
		if (typeof headers['X-Requested-With'] === 'undefined') { //$NON-NLS-1$ //$NON-NLS-0$
			headers['X-Requested-With'] = 'XMLHttpRequest'; //$NON-NLS-1$ //$NON-NLS-0$
		}
		if (typeof options.data !== 'undefined' && (method === 'POST' || method === 'PUT')) { //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
			data = options.data;
		}
		
		var cancelled = false;
		var aborted = false;
		d.promise.then(undefined, function(error) {
			cancelled = true;
			if (!aborted && error instanceof Error && error.name === "Cancel") {
				xhr.abort();
			}
		});
		
		var eventSource = options.upload ? xhr.upload : xhr;
		eventSource.onabort = function() {
			aborted = true;
			if (!cancelled) {
				var cancelError = new Error("Cancel");
				cancelError.name = "Cancel";
				d.reject(cancelError);
			}
		};
		xhr.onload = function() {
			var result = makeResult(url, options, xhr);
			if (200 <= xhr.status && xhr.status < 400) {
				d.resolve(result);
			} else {
				d.reject(result);
				if(log && typeof console !== 'undefined') { //$NON-NLS-0$
					console.log(new Error(xhr.statusText));
				}
			}
		};
		eventSource.onerror = function() {
			var result = makeResult(url, options, xhr);
			d.reject(result);
			if (log && typeof console !== 'undefined') { //$NON-NLS-0$
				console.log(new Error(xhr.statusText));
			}
		};
		eventSource.onprogress = function(progressEvent) {
			progressEvent.xhr = xhr;
			d.progress(progressEvent);
		};
	
		try {
			console.log(method + " " + url);
			xhr.open(method, url, true /* async */);
			if (typeof options.responseType === 'string') { //$NON-NLS-0$
				xhr.responseType = options.responseType;
			}
			if (typeof options.timeout === 'number' && localStorage.noTimeout !== "true") { //$NON-NLS-0$
				if (typeof xhr.timeout === 'number') { //$NON-NLS-0$
					// Browser supports XHR timeout
					xhr.timeout = options.timeout;
					xhr.addEventListener('timeout', function(e) { //$NON-NLS-0$
						d.reject(makeResult(url, options, xhr, 'Timeout exceeded')); //$NON-NLS-0$
					});
				} else {
					// Use our own timer
					var timeoutId = setTimeout(function() {
						d.reject(makeResult(url, options, xhr, 'Timeout exceeded')); //$NON-NLS-0$
					}, options.timeout);
					d.promise.then(clearTimeout.bind(null, timeoutId), clearTimeout.bind(null, timeoutId));
				}
			}
			Object.keys(headers).forEach(function(key) {
				xhr.setRequestHeader(key, headers[key]);
			});
			xhr.send(data || null);
		} catch (e) {
			d.reject(makeResult(url, options, xhr, e));
		}

		return d.promise;
	}
	return _xhr;
});
/*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
/*global URL*/
define('orion/collab/shareProjectClient',["orion/xhr", "orion/URL-shim"], function(xhr) {
	var contextPath = location.pathname.substr(0, location.pathname.lastIndexOf('/edit/edit.html'));
	return {
		shareProjectUrl: contextPath + "/sharedWorkspace/project/",
		unshareProjectUrl: contextPath + "/sharedWorskpace/project/",
		addUserUrl: contextPath + "/sharedWorkspace/user/",
		removeUserUrl: contextPath + "/sharedWorkspace/user/",
		shareProject: function(project) {
			return xhr("POST", this.shareProjectUrl + project, {
				headers: {
					"Orion-Version": "1",
					"X-Create-Options" : "no-overwrite",
					"Content-Type": "application/json;charset=UTF-8"
				}
			});
		},
		unshareProject: function(project) {
			return xhr("DELETE", this.unshareProjectUrl + project, {
				headers: {
					"Orion-Version": "1",
					"X-Create-Options" : "no-overwrite",
					"Content-Type": "application/json;charset=UTF-8"
				}
			});
		},
		addUser: function(username, project) {
			return xhr("POST", this.addUserUrl + project + '/' + username, {
				headers: {
					"Orion-Version": "1",
					"X-Create-Options" : "no-overwrite",
					"Content-Type": "application/json;charset=UTF-8"
				}
			}).then(function() {
				window.location.reload();
			});
		},
		removeUser: function(username, project) {
			return xhr("DELETE", this.removeUserUrl + project + '/' + username, {
				headers: {
					"Orion-Version": "1",
					"X-Create-Options" : "no-overwrite",
					"Content-Type": "application/json;charset=UTF-8"
				}
			}).then(function() {
				window.location.reload();
			});
		}
	}
});

/*******************************************************************************
 * @license
 * Copyright (c) 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd */
define('orion/collab/collabFileCommands',['orion/collab/shareProjectClient'], function(shareProjectClient) {

    'use strict';

    var mCommands;
    var mCommandRegistry;

    var CollabFileCommands = Object.create(null);

    CollabFileCommands.init = function(callback) {
        require(['orion/commands', 'orion/commandRegistry'], function(_mCommands, _mCommandRegistry) {
            mCommands = _mCommands;
            mCommandRegistry = _mCommandRegistry;
            callback();
        });
    };

    CollabFileCommands.createFileCommands = function(serviceRegistry, commandService, fileClient) {
		var shareProjectCommand = new mCommands.Command({
			name: "Share",
			tooltip: "Share project with a friend",
			description: "Add a user so that they can collaborate with you on the project.",
			imageClass: "core-sprite-link", //$NON-NLS-0$
			id: "orion.collab.shareProject", //$NON-NLS-0$
			parameters: new mCommandRegistry.ParametersDescription([new mCommandRegistry.CommandParameter('username', 'text', "Username:", "Enter username here")]), //$NON-NLS-5$ //$NON-NLS-4$ //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
			callback: function(data) {
				var username = data.parameters.parameterTable.username.value;
				var project = encodeURIComponent(data.items[0].Location);
				shareProjectClient.addUser(username, project);
			},
			visibleWhen: function(item) {
				if (Array.isArray(item)) {
					if (item.length === 1) {
						return !item[0].Parents || !item[0].Parents.length;
					}
				}
				return false;
			}
		});
		commandService.addCommand(shareProjectCommand);

		var unshareProjectCommand = new mCommands.Command({
			name: "Unshare",
			tooltip: "Unshare a project",
			description: "Remove a user from the sharing list of this project.",
			imageClass: "core-sprite-link", //$NON-NLS-0$
			id: "orion.collab.unshareProject", //$NON-NLS-0$
			parameters: new mCommandRegistry.ParametersDescription([new mCommandRegistry.CommandParameter('username', 'text', "Username:", "Enter username here")]), //$NON-NLS-5$ //$NON-NLS-4$ //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
			callback: function(data) {
				var username = data.parameters.parameterTable.username.value;
				var project = encodeURIComponent(data.items[0].Location);
				shareProjectClient.removeUser(username, project);
			},
			visibleWhen: function(item) {
				if (Array.isArray(item)) {
					if (item.length === 1) {
						return !item[0].Parents || !item[0].Parents.length;
					}
				}
				return false;
			}
		});
		commandService.addCommand(unshareProjectCommand);

    };

    return CollabFileCommands;
});

/*******************************************************************************
 * @license
 * Copyright (c) 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd */
define('orion/collab/collabFileEditingAnnotation',['orion/collab/collabFileAnnotation'], function(mCollabFileAnnotation) {

    'use strict';

	var CollabFileAnnotation = mCollabFileAnnotation.CollabFileAnnotation;

    /**
	 * An annotation shows that a file is under editing
	 * 
	 * @constructor
	 * @name {orion.collab.CollabFileEditingAnnotation}
	 * @extends {orion.collab.CollabFileAnnotation}
	 * 
	 * @param {string} location - file location
	 * @param {Array.<string>} users - list of users that is modifying this file
	 */
	var CollabFileEditingAnnotation = function(location, users) {
		// Remove trailing "/"
		if(location.substr(-1) === '/') {
			location = location.substr(0, location.length - 1);
		}
		this.location = location;
		console.assert(Array.isArray(users));
		this.users = users;
	};

	CollabFileEditingAnnotation.prototype = Object.create(CollabFileAnnotation.prototype);

	/**
	 * Get description of this annotation which can be used in for example
	 * tooltip.
	 * 
	 * @return {string} - description
	 */
	CollabFileEditingAnnotation.prototype.getDescription = function() {
		return '<b>' + this.users.join('</b>, <b>') + '</b> ' + (this.users.length > 1 ? 'are' : 'is') + ' modifying this file.';
	};

	/**
	 * Generate a new HTML element of this annotation.
	 * 
	 * @return {Element} - the HTML element of this annotation
	 */
	CollabFileEditingAnnotation.prototype.generateHTML = function() {
		var element = document.createElement('div');
		element.innerHTML = '···';
		element.classList.add('editingAnnotation');
		return element;
	};

    return {
		CollabFileEditingAnnotation: CollabFileEditingAnnotation
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2016, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd */
define('orion/collab/collabClient',['orion/collab/ot', 'orion/collab/collabFileAnnotation', 'orion/collab/otAdapters',
	'orion/collab/collabPeer', 'orion/collab/collabSocket', 'orion/collab/collabFileCommands',
	'orion/collab/collabFileEditingAnnotation'],
	function(ot, mCollabFileAnnotation, mOtAdapters, mCollabPeer, mCollabSocket, mCollabFileCommands,
		mCollabFileEditingAnnotation) {

    'use strict';

	var mAnnotations;
	var mTreeTable;
	var AT;

	var contextPath = location.pathname.substr(0, location.pathname.lastIndexOf('/edit/edit.html'));
	// This is a workaround of split window. Basically this guid let the collab client ignore all file operations and text operations from the same guid.
	// However, when there are two split windows on a same file, a text operation will still be sent twice, which needs to be fixed. (TODO)
	var guid = Math.floor(Math.random() * 0x100000000).toString(16) + '.' + Math.floor(Math.random() * 0x100000000).toString(16) + '.' + Math.floor(Math.random() * 0x100000000).toString(16) + '.' + Math.floor(Math.random() * 0x100000000).toString(16) + '.' + Math.floor(Math.random() * 0x100000000).toString(16) + '.' + Math.floor(Math.random() * 0x100000000).toString(16) + '.' + Math.floor(Math.random() * 0x100000000).toString(16) + '.' + Math.floor(Math.random() * 0x100000000).toString(16);

	function init(callback) {
		require(['orion/editor/annotations', 'orion/webui/treetable'], function(_mAnnotations, _mTreeTable) {
			mAnnotations = _mAnnotations;
			mTreeTable = _mTreeTable;
			AT = mAnnotations.AnnotationType;
			mCollabFileCommands.init(function() {
				callback();
			});
		});
	}

	var CollabFileAnnotation = mCollabFileAnnotation.CollabFileAnnotation;
	var CollabFileEditingAnnotation = mCollabFileEditingAnnotation.CollabFileEditingAnnotation;
	var OrionCollabSocketAdapter = mOtAdapters.OrionCollabSocketAdapter;
	var OrionEditorAdapter = mOtAdapters.OrionEditorAdapter;
	var CollabPeer = mCollabPeer.CollabPeer;
	
	// ms to delay updating collaborator annotation.
	// We need this delay because the annotation is updated asynchronizedly and is transferd in multiple
	// packages. We don't want the UI to refresh too frequently.
	var COLLABORATOR_ANNOTATION_UPDATE_DELAY = 50;

	var SOCKET_RECONNECT_MAX_ATTEMPT = 5;
	var SOCKET_RECONNECT_DELAY = 100;
	var SOCKET_PING_TIMEOUT = 25000;

	/**
	 * Creates a new collaboration client.
	 * @class 
	 * @name orion.collabClient.CollabClient
	 */
	function CollabClient(editor, inputManager, fileClient, serviceRegistry, commandRegistry, preferences) {
		mCollabFileCommands.createFileCommands(serviceRegistry, commandRegistry, fileClient);
		this.editor = editor;
		this.inputManager = inputManager;
		this.fileClient = fileClient;
		this.preferences = preferences;
		this.textView = this.editor.getTextView();
		var self = this;
		this.collabMode = false;
		this.clientId = '';
		this.clientDisplayedName = '';
		this.fileClient.addEventListener('Changed', self.sendFileOperation.bind(self));
		this.serviceRegistry = serviceRegistry;
		this.editor.addEventListener('InputContentsSet', function(event) {self.viewInstalled.call(self, event);});
		this.editor.addEventListener('TextViewUninstalled', function(event) {self.viewUninstalled.call(self, event);});
		this.projectSessionID = '';
		this.inputManager.addEventListener('InputChanged', function(e) {
			self.onInputChanged(e)
		});
		this.ot = null;
		this.otOrionAdapter = null;
		this.otSocketAdapter = null;
		this.socketReconnectAttempt = 0;
		this.socketIntentionalyClosing = false;
		this.socketPingTimeout = 0;
		this.awaitingClients = false;
		this.collabFileAnnotations = {};
		// Timeout id to indicate whether a delayed update has already been assigned
		this.collabFileAnnotationsUpdateTimeoutId = 0;
		/**
		 * A map of clientid -> peer
		 * @type {Object.<string, CollabPeer>}
		 */
		this.peers = {};
		this.editing = false;
		this.guid = guid;
		// Initialize current project
		var file = this.inputManager.getInput();
		var metadata = this.inputManager.getFileMetadata();
		if (metadata) {
			this.onInputChanged({
				metadata: metadata,
				input: {
					resource: file
				}
			});
		}
	}

	CollabClient.prototype = {

		/**
		 * getter of clientId
		 * @return {string}
		 */
		getClientId: function() {
			return this.clientId;
		},

		/**
		 * getter of clientDisplayedName
		 * @return {string}
		 */
		getClientDisplayedName: function() {
			return this.clientDisplayedName;
		},

		/**
		 * Initialize client name and id
		 * @param {function} callback
		 */
		initClientInfo: function(callback) {
			var self = this;
			var userService = this.serviceRegistry.getService("orion.core.user"); 
			var authServices = this.serviceRegistry.getServiceReferences("orion.core.auth");
			var authService = this.serviceRegistry.getService(authServices[0]);
			authService.getUser().then(function(jsonData) {
				userService.getUserInfo(contextPath + jsonData.Location).then(function(accountData) {
					var username = accountData.UserName;
					self.clientDisplayedName = accountData.FullName || username;
					var MASK = 0xFFFFFF + 1;
					var MAGIC = 161803398 / 2 % MASK;
					self.clientId = username + '.' + guid.substr(0, 4);
					callback();
				}, function(err) {
					console.error(err);
				});
			});
		},

		/**
		 * Input changed handler
		 */
		onInputChanged: function(e) {
			this.location = e.input.resource;
			this.updateSelfFileAnnotation();
			this.destroyOT();
			this.sendCurrentLocation();
			if (e.metadata.Attributes) {
				var projectSessionID = e.metadata.Attributes.hubID;
				if (this.projectSessionID !== projectSessionID) {
					this.projectSessionID = projectSessionID;
					this.projectChanged(projectSessionID);
				}
			}
		},

		/**
		 * Send current location to collab peers
		 */
		sendCurrentLocation: function() {
			if (this.otSocketAdapter && this.otSocketAdapter.authenticated) {
				this.otSocketAdapter.sendLocation(this.currentDoc());
			}
		},

		/**
		 * Reset the record of collaborator file annotation and request to update UI
		 */
		resetCollabFileAnnotation: function() {
			this.collabFileAnnotations = {};
			this._requestFileAnnotationUpdate();
		},

		/**
		 * Add or update a record of collaborator file annotation and request to update UI
		 * 
		 * @param {string} clientId
		 * @param {string} url
		 * @param {boolean} editing
		 */
		addOrUpdateCollabFileAnnotation: function(clientId, contextP, url, editing) {
			var peer = this.getPeer(clientId);
			// Peer might be loading. Once it is loaded, this annotation will be automatically updated,
			// so we can safely leave it blank.
			var name = (peer && peer.name) ? peer.name : 'Unknown';
			var color = (peer && peer.color) ? peer.color : '#000000';
			if (url) {
				url = contextP + this.getFileSystemPrefix() + url;
			}
			this.collabFileAnnotations[clientId] = new CollabFileAnnotation(name, color, url, this.projectRelativeLocation(url), editing);
			this._requestFileAnnotationUpdate();
		},

		/**
		 * Remove a collaborator's file annotation by id and request to update UI
		 * 
		 * @param {string} clientId -
		 */
		removeCollabFileAnnotation: function(clientId) {
			if (this.collabFileAnnotations.hasOwnProperty(clientId)) {
				delete this.collabFileAnnotations[clientId];
				this._requestFileAnnotationUpdate();
			}
		},

		/**
		 * Request a file annotation UI update
		 */
		_requestFileAnnotationUpdate: function() {
			var self = this;
			if (!this.collabFileAnnotationsUpdateTimeoutId) {
				// No delayed update is assigned. Assign one.
				// This is necessary because we don't want duplicate UI action within a short period.
				this.collabFileAnnotationsUpdateTimeoutId = setTimeout(function() {
					self.collabFileAnnotationsUpdateTimeoutId = 0;
					var annotations = [];
					var editingFileUsers = {}; // map from location to list of usernames indicating all users that are typing on this file
					for (var key in self.collabFileAnnotations) {
						if (self.collabFileAnnotations.hasOwnProperty(key)) {
							var annotation = self.collabFileAnnotations[key];
							annotations.push(annotation);
							if (annotation.editing) {
								if (!editingFileUsers[annotation.location]) {
									editingFileUsers[annotation.location] = [];
								}
								editingFileUsers[annotation.location].push(annotation.name);
							}
						}
					}
					// Add editing annotations
					for (var location in editingFileUsers) {
						if (editingFileUsers.hasOwnProperty(location)) {
							annotations.push(new CollabFileEditingAnnotation(location, editingFileUsers[location]));
						}
					}
					self.fileClient.dispatchEvent({
						type: 'AnnotationChanged',
						removeTypes: [CollabFileAnnotation],
						annotations: annotations
					});
				}, COLLABORATOR_ANNOTATION_UPDATE_DELAY);
			}
		},

		/**
		 * Determine whether a client has a file annotation
		 * 
		 * @return {boolean} -
		 */
		collabHasFileAnnotation: function(clientId) {
			return !!this.collabFileAnnotations[clientId];
		},

		/**
		 * Get the client's file annotation
		 * 
		 * @return {CollabFileAnnotation} -
		 */
		getCollabFileAnnotation: function(clientId) {
			return this.collabFileAnnotations[clientId];
		},

		/**
		 * Update the current client's file annotation
		 */
		updateSelfFileAnnotation: function() {
			if (this.collabMode) {
				this.addOrUpdateCollabFileAnnotation(this.getClientId(), contextPath, this.currentDoc(), this.editing);
			}
		},

		/**
		 * Add or update peer record
		 * 
		 * @param {CollabPeer} peer -
		 */
		addOrUpdatePeer: function(peer) {
			if (this.peers[peer.id]) {
				// Update
				this.peers[peer.id] = peer;
			} else {
				// Add
				this.peers[peer.id] = peer;
			}
			if (this.collabHasFileAnnotation(peer.id)) {
				var annotation = this.getCollabFileAnnotation(peer.id);
				this.addOrUpdateCollabFileAnnotation(peer.id, contextPath, annotation.location);
			}
			if (this.otOrionAdapter && this.textView) {
				// Make sure we have view installed
				this.otOrionAdapter.updateLineAnnotationStyle(peer.id);
			}
		},

		/**
		 * Get peer by id
		 * 
		 * @return {CollabPeer} -
		 */
		getPeer: function(clientId) {
			return this.peers[clientId];
		},

		/**
		 * Get all peers
		 */
		getAllPeers: function(clientId) {
			return this.peers;
		},

		/**
		 * Remove a peer by its ID if it exists
		 * This method also removes all the removing client's annotation
		 */
		removePeer: function(clientId) {
			if (this.peers.hasOwnProperty(clientId)) {
				delete this.peers[clientId];
				this.removeCollabFileAnnotation(clientId);
			}
		},

		/**
		 * Remove all peers
		 */
		clearPeers: function() {
			this.peers = {};
		},

		startOT: function(revision, operation, clients) {
			if (this.ot) {
				this.otOrionAdapter.detach();
			}
			var selection = this.textView.getSelection();
			this.textView.getModel().setText(operation[0], 0);
			this.textView.setSelection(selection.start, selection.end);
			this.otOrionAdapter = new OrionEditorAdapter(this.editor, this, AT);
			this.ot = new ot.EditorClient(revision, clients, this.otSocketAdapter, this.otOrionAdapter, this.getClientId());
			// Give initial cursor position
			this.otOrionAdapter.onFocus();
			this.editor.markClean();
		},

		destroyOT: function() {
			if (this.ot && this.otOrionAdapter) {
				this.otOrionAdapter.detach();
				//reset to regular undo/redo behaviour
				if (this.textView) {
					this.editor.getTextActions().init();
				}
				this.ot = null;
				if (this.otSocketAdapter) {
					var msg = {
				      'type': 'leave-document',
				      'clientId': this.getClientId()
				    };
					this.otSocketAdapter.send(JSON.stringify(msg));
				}
			}
		},

		/**
		 *  peal /shardworkspace/tree/file/user-orionContent/testfolder/testfile of /file/user-orionContent/testfolder/testfile to /user-orionContent/testfolder/testfile
		 */
		currentDoc: function() {
			var workspace = this.getFileSystemPrefix();
			return this.location.substring(this.location.indexOf(workspace) + workspace.length);
		},

		getFileSystemPrefix: function() {
			return this.location.substr(contextPath.length).indexOf('/sharedWorkspace') === 0 ? '/sharedWorkspace/tree/file' : '/file';
		},

		viewInstalled: function(event) {
			var self = this;
			var ruler = this.editor._annotationRuler;
			ruler.addAnnotationType(AT.ANNOTATION_COLLAB_LINE_CHANGED, 1);
			ruler = this.editor._overviewRuler;
			ruler.addAnnotationType(AT.ANNOTATION_COLLAB_LINE_CHANGED, 1);
			this.textView = this.editor.getTextView();
			if (this.viewFocusHandlerTarget) {
				this.viewFocusHandlerTarget.removeEventListener('Focus', this.viewFocusHandler);
				this.viewFocusHandlerTarget = this.viewFocusHandler = null;
			}
			this.textView.addEventListener('Focus', this.viewFocusHandler = function() {
				self.sendCurrentLocation();
			});
			this.viewFocusHandlerTarget = this.textView;
			if (this.otSocketAdapter) {
				this.otSocketAdapter.sendInit();
			}
		},

		viewUninstalled: function(event) {
			if (this.viewFocusHandlerTarget) {
				this.viewFocusHandlerTarget.removeEventListener('Focus', this.viewFocusHandler);
				this.viewFocusHandlerTarget = this.viewFocusHandler = null;
			}
			this.textView = null;
			this.destroyOT();
		},

		socketConnected: function() {
			var self = this;
			this.otSocketAdapter = new OrionCollabSocketAdapter(this, this.socket);
			this.socketReconnectAttempt = 0;
			if (!this.clientId) {
				this.initClientInfo(function() {
					self.otSocketAdapter.authenticate();
				});
			} else {
				this.otSocketAdapter.authenticate();
			}
			this.inputManager.syncEnabled = false;
			this.issueNextPing();
		},

		socketDisconnected: function() {
			this.socket = null;
			this.otSocketAdapter = null;
			this.inputManager.syncEnabled = true;
			this.destroyOT();
			this.resetCollabFileAnnotation();
			if (!this.socketIntentionalyClosing) {
				if (this.socketReconnectAttempt < SOCKET_RECONNECT_MAX_ATTEMPT) {
					var self = this;
					this.socketReconnectAttempt++;
					setTimeout(function() {
						self.projectChanged(self.projectSessionID);
					}, SOCKET_RECONNECT_DELAY);
				} else {
					console.error('Network error. Cannot enable collaboration feature.');
				}
			}
		},

		issueNextPing: function() {
			var self = this;
			if (this.socketPingTimeout) {
				clearTimeout(this.socketPingTimeout);
			}
			this.socketPingTimeout = setTimeout(function() {
				self.socketPingTimeout = 0;
				if (self.socket) {
					self.socket.send(JSON.stringify({
						type: 'ping'
					}));
					self.issueNextPing();
				}
			}, SOCKET_PING_TIMEOUT);
		},

		projectChanged: function(projectSessionID) {
			var self = this;
			if (this.socket) {
				// Close the current session
				this.socketIntentionalyClosing = true;
				this.socket.close();
				setTimeout(function() {
					// Polling until 'close' event is triggered
					// 'close' event won't wait for any IO operations thus
					// this.socket should be null in the next event loop 
					self.projectChanged(projectSessionID);
				}, 0);
				return;
			}
			this.socketIntentionalyClosing = false;
			// Initialize collab socket
			if (projectSessionID) {
				this.preferences.get("/collab").then(function(collabOptions) {
					self.socket = new mCollabSocket.CollabSocket(collabOptions.hubUrl, projectSessionID);
					self.socket.addEventListener('ready', function onReady() {
						self.socket.removeEventListener('ready', onReady);
						self.socketConnected();
					});
					self.socket.addEventListener('close', function onClose() {
						self.socket.removeEventListener('close', onClose);
						self.socketDisconnected();
					});
				});
				this.collabMode = true;
			} else {
				this.collabMode = false;
			}
			this.clearPeers();
			this.resetCollabFileAnnotation();
		},

		sendFileOperation: function(evt) {
			if (!this.otSocketAdapter) return;
			if (!this.ignoreNextFileOperation) {
				var operation = evt.created ? 'created' : evt.moved ? 'moved' : evt.deleted ? 'deleted' : evt.copied ? 'copied' : '';
				if (operation) {
				    var msg = {
						'type': 'file-operation',
						'operation': operation,
						'data': evt[operation],
						'clientId': this.getClientId(),
						'guid': this.fileClient.guid || this.guid
				    };
					this.otSocketAdapter.send(JSON.stringify(msg));
				}
			}
			this.ignoreNextFileOperation = false;
		},

		handleFileOperation: function(msg) {
			if (!this.ignoreNextFileOperation) {
				if (msg.guid !== guid) {
					var evt = this.makeFileClientEvent(msg.operation, msg.data);
					this.dispatchFileClientEvent(evt);
				}
			}
			this.ignoreNextFileOperation = false;
		},

		/**
		 * Make a event for FileClient by data received from peers. The event
		 * That this method made also prevent UI changes (e.g. expanding the
		 * file tree).
		 * 
		 * @param {stirng} operation
		 * @param {Array} data
		 */
		makeFileClientEvent: function(operation, data) {
			/**
			** we can't trigger the event directly since the user might be on a seperate file system.
			*/
			data = data[0];
			var evt = {
				type: 'Changed',
				silent: true
			};

			var evtData = {'select': false};

			switch (operation) {
				case 'created':
					var parentLocation = this.transformLocation(data.parent);
					var result = data.result;
					if (result) {
						result.Parents = []; //is parents even needed for this operation?
						result.Location = this.transformLocation(result.Location);
					}
					evt.created = [{'parent': parentLocation, 'result': result, 'eventData': evtData}];
					break;
				case 'deleted':
					var deleteLocation = this.transformLocation(data.deleteLocation);
					evt.deleted = [{'deleteLocation': deleteLocation, 'eventData': evtData}];
					break;
				case 'moved':
					var sourceLocation = this.transformLocation(data.source);
					var targetLocation = this.transformLocation(data.target);
					var result = data.result;
					result.Parents = []; //is parents even needed for this operation?
					result.Location = this.transformLocation(result.Location);
					evt.moved = [{'source': sourceLocation, 'target': targetLocation, 'result': result, 'eventData': evtData}];
					break;
				case 'copied':
					var sourceLocation = this.transformLocation(data.source);
					var targetLocation = this.transformLocation(data.target);
					var result = data.result;
					result.Parents = []; //is parents even needed for this operation?
					result.Location = this.transformLocation(result.Location);
					evt.copied = [{'source': sourceLocation, 'target': targetLocation, 'result': result, 'eventData': evtData}];
					break;
			}

			return evt;
		},

		/**
		 * For example we need to convert a '/file/web-orionContent/potato.js' to '/sharedWorkspace/tree/file/web-orionContent/potato.js'
		 * and vice-versa, depending on our file system and the sender's filesystem.
		 */
		transformLocation: function(location) { //Location = "/file/orders-api-uselessTesting_X/app.js"
			var filePath = location.replace(/^.*\/file/, "");
			var loc = this.getFileSystemPrefix();
			return contextPath + loc + filePath;
		},

		/*
		 *  Generate the location string for the tooltip for each collabrator.
		 */
		projectRelativeLocation: function(location) {
			if (location.substr(contextPath.length).indexOf('/file/') === 0) {
				// Local workspace
				return location.substr(contextPath.length).split("/").slice(3).join('/')
			} else {
				// Shared workspace
				return location.substr(contextPath.length).split('/').slice(5).join('/');
			}
		},

		dispatchFileClientEvent: function(evt) {
			this.ignoreNextFileOperation = true;
			this.fileClient.dispatchEvent(evt);
		}
	};

	CollabClient.prototype.constructor = CollabClient;

	return {
		CollabClient: CollabClient,
		init: init
	};
});

