/*****************************************************************************
 * Copyright (c) 2013 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Camille Letavernier (camille.letavernier@cea.fr) - Initial API and implementation
 *
 *****************************************************************************/
package org.eclipse.papyrus.infra.gmfdiag.hyperlink.providers;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.papyrus.infra.core.sashwindows.di.service.IPageManager;
import org.eclipse.papyrus.infra.core.services.ServiceException;
import org.eclipse.papyrus.infra.core.services.ServicesRegistry;
import org.eclipse.papyrus.infra.emf.utils.ServiceUtilsForEObject;
import org.eclipse.papyrus.infra.gmfdiag.common.helper.SemanticElementHelper;
import org.eclipse.papyrus.infra.gmfdiag.hyperlink.Activator;
import org.eclipse.papyrus.infra.widgets.providers.IHierarchicContentProvider;
import org.eclipse.papyrus.infra.widgets.util.IRevealSemanticElement;


public class SpecificViewContentProvider implements IHierarchicContentProvider, IRevealSemanticElement {

	private Viewer viewer;
	
	private EObject[] roots;

	private EObject[] getRoots(ServicesRegistry context) {
		try {
			IPageManager pageManager = context.getService(IPageManager.class);
			List<Diagram> result = new LinkedList<Diagram>();
			for (Object page : pageManager.allPages()) {
				if (page instanceof Diagram) {
					result.add((Diagram) page);
				}
			}
			return result.toArray(new Diagram[result.size()]);
		} catch (ServiceException ex) {
			Activator.log.error(ex);
		}
		return new EObject[0];
	}

	public Object[] getElements(Object inputElement) {
		if (roots == null) {
			if (inputElement instanceof EObject) {
				try {
					ServicesRegistry registry = ServiceUtilsForEObject.getInstance().getServiceRegistry((EObject) inputElement);
					roots = getRoots(registry);
				} catch (ServiceException ex) {
					Activator.log.error(ex);
				}
			}
		}
		return roots == null ? new Object[0] : roots;
	}

	public Object[] getChildren(Object parentElement) {
		if (parentElement instanceof EObject) {
			List<Object> validChildren = new LinkedList<Object>();
			for (Object childElement : ((EObject) parentElement).eContents()) {
				if (isValidValue(childElement)) {
					validChildren.add(childElement);
				} else {
					// Go deeper, to find e.g. TopView(Class)::AttributeCompartment(Class)::TopView(Property)
					validChildren.addAll(Arrays.asList(getChildren(childElement)));
				}
			}
			return validChildren.toArray();

		}
		return new Object[0];
	}

	public Object getParent(Object element) {
		if (element instanceof EObject) {
			return ((EObject) element).eContainer();
		}
		return null;
	}

	public boolean hasChildren(Object element) {
		return getChildren(element).length > 0;
	}

	public void dispose() {
		// Nothing
	}

	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		this.viewer = viewer;
	}

	public boolean isValidValue(Object element) {
		if (element instanceof View && ((View) element).getElement() == null) {
			return false;
		}
		return element instanceof View && SemanticElementHelper.findTopView((View) element) == element;
	}

	/**
	 * @see org.eclipse.papyrus.infra.widgets.util.IRevealSemanticElement#revealSemanticElement(java.util.List)
	 *
	 * @param elementList
	 */
	public void revealSemanticElement(List<?> elementList) {
		if (viewer != null) {
			viewer.setSelection(new StructuredSelection(elementList), true);
		}
	}
}
