/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.birt.core.btree;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.birt.core.btree.BTreeConstants;
import org.eclipse.birt.core.btree.BTreeCursor;
import org.eclipse.birt.core.btree.BTreeFile;
import org.eclipse.birt.core.btree.BTreeNode;
import org.eclipse.birt.core.btree.BTreeOption;
import org.eclipse.birt.core.btree.BTreeSerializer;
import org.eclipse.birt.core.btree.BTreeUtils;
import org.eclipse.birt.core.btree.BTreeValue;
import org.eclipse.birt.core.btree.BTreeValues;
import org.eclipse.birt.core.btree.ExternalValueList;
import org.eclipse.birt.core.btree.IndexEntry;
import org.eclipse.birt.core.btree.IndexNode;
import org.eclipse.birt.core.btree.LeafEntry;
import org.eclipse.birt.core.btree.LeafNode;
import org.eclipse.birt.core.btree.NodeFile;
import org.eclipse.birt.core.btree.NodeInputStream;
import org.eclipse.birt.core.btree.NodeOutputStream;
import org.eclipse.birt.core.btree.ValueNode;
import org.eclipse.birt.core.i18n.CoreMessages;

public class BTree<K, V>
implements BTreeConstants {
    protected static Logger logger = Logger.getLogger(BTree.class.getName());
    protected NodeFile file;
    protected boolean shareFile;
    private int version;
    private boolean allowDuplicate;
    private boolean allowNullKey;
    private int keySize;
    private boolean hasValue;
    private int valueSize;
    private int headNodeId;
    private int rootNodeId;
    private int freeNodeId;
    private int totalBlocks;
    private int totalLevels;
    private int totalKeys;
    private int totalValues;
    private int cacheSize;
    protected boolean readOnly;
    protected BTreeSerializer<K> keySerializer;
    protected BTreeSerializer<V> valueSerializer;
    protected Comparator<K> comparator;
    private LinkedHashMap<Integer, BTreeNode<K, V>> nodeCaches = new LinkedHashMap<Integer, BTreeNode<K, V>>(8, 0.75f, true){
        private static final long serialVersionUID = 1L;

        @Override
        protected boolean removeEldestEntry(Map.Entry<Integer, BTreeNode<K, V>> arg) {
            if (BTree.this.file == null) {
                return false;
            }
            if (this.size() >= BTree.this.cacheSize) {
                BTreeNode node = arg.getValue();
                if (node.isLocked()) {
                    for (Map.Entry entry : this.entrySet()) {
                        BTreeNode value = (BTreeNode)entry.getValue();
                        if (value.isLocked()) continue;
                        if (value.isDirty()) {
                            try {
                                BTree.this.writeNode(value);
                            }
                            catch (IOException ex) {
                                logger.log(Level.WARNING, "failed to write node " + value.getNodeId() + " type " + value.getNodeType(), ex);
                                return false;
                            }
                        }
                        this.remove(entry.getKey());
                        break;
                    }
                    return false;
                }
                if (node.isDirty()) {
                    try {
                        BTree.this.writeNode(node);
                    }
                    catch (IOException ex) {
                        logger.log(Level.WARNING, "failed to write node " + node.getNodeId() + " type " + node.getNodeType(), ex);
                        return false;
                    }
                }
                return true;
            }
            return false;
        }
    };
    private final BTreeValue<K> NULL_KEY = new BTreeValue();

    public BTree() throws IOException {
        this(new BTreeOption());
    }

    public BTree(BTreeOption<K, V> option) throws IOException {
        if (option.file != null) {
            this.shareFile = option.shareFile;
            this.file = option.file instanceof NodeFile ? (NodeFile)option.file : new ReusableBTreeFile(option.file);
        }
        this.comparator = option.comparator;
        this.keySerializer = option.keySerializer;
        this.valueSerializer = option.valueSerializer;
        this.readOnly = option.readOnly;
        this.version = 0;
        this.rootNodeId = -1;
        this.freeNodeId = -1;
        this.totalLevels = 0;
        this.totalKeys = 0;
        this.totalValues = 0;
        this.allowDuplicate = option.allowDuplicate;
        this.allowNullKey = option.allowNullKey;
        this.keySize = option.keySize;
        this.hasValue = option.hasValue;
        this.valueSize = option.valueSize;
        this.headNodeId = option.headNodeId;
        this.cacheSize = option.cacheSize;
        if (this.file != null) {
            if (this.file.getTotalBlock() > this.headNodeId) {
                byte[] bytes = new byte[4096];
                this.file.readBlock(this.headNodeId, bytes);
                DataInputStream input = new DataInputStream(new ByteArrayInputStream(bytes));
                this.readTreeHead(input);
            } else {
                ByteArrayOutputStream buffer = new ByteArrayOutputStream(4096);
                DataOutputStream output = new DataOutputStream(buffer);
                this.writeTreeHead(output);
                this.file.writeBlock(this.headNodeId, buffer.toByteArray());
            }
            this.totalBlocks = this.file.getTotalBlock();
        }
    }

    public void close() throws IOException {
        if (this.file == null) {
            return;
        }
        try {
            if (!this.readOnly) {
                ByteArrayOutputStream buffer = new ByteArrayOutputStream(4096);
                DataOutputStream output = new DataOutputStream(buffer);
                this.writeTreeHead(output);
                this.file.writeBlock(this.headNodeId, buffer.toByteArray());
                for (BTreeNode<K, V> node : this.nodeCaches.values()) {
                    if (!node.isDirty()) continue;
                    this.writeNode(node);
                }
            }
            if (!this.shareFile) {
                this.file.close();
            }
        }
        finally {
            this.file = null;
        }
    }

    LeafEntry<K, V> getFirstEntry() throws IOException {
        int nodeId = this.rootNodeId;
        while (nodeId != -1) {
            BTreeNode<K, V> node = this.loadBTreeNode(nodeId);
            try {
                if (node.getNodeType() == 2) {
                    LeafEntry leafEntry = ((LeafNode)node).getFirstEntry();
                    return leafEntry;
                }
                nodeId = ((IndexNode)node).getFirstChild();
            }
            finally {
                node.unlock();
            }
        }
        return null;
    }

    LeafEntry<K, V> getLastEntry() throws IOException {
        int nodeId = this.rootNodeId;
        while (nodeId != -1) {
            BTreeNode<K, V> node = this.loadBTreeNode(nodeId);
            try {
                if (node.getNodeType() == 2) {
                    LeafEntry leafEntry = ((LeafNode)node).getLastEntry();
                    return leafEntry;
                }
                nodeId = ((IndexNode)node).getLastChild();
            }
            finally {
                node.unlock();
            }
        }
        return null;
    }

    protected LeafEntry<K, V> findEntry(K k) throws IOException {
        if (k == null && !this.allowNullKey) {
            throw new NullPointerException("k can not be null");
        }
        if (this.rootNodeId != -1) {
            BTreeValue<K> key = this.createKey(k);
            BTreeNode<K, V> root = this.loadBTreeNode(this.rootNodeId);
            try {
                int nodeType = root.getNodeType();
                if (nodeType == 1) {
                    LeafEntry leafEntry = ((IndexNode)root).find(key);
                    return leafEntry;
                }
                if (nodeType == 2) {
                    LeafEntry leafEntry = ((LeafNode)root).find(key);
                    return leafEntry;
                }
            }
            finally {
                root.unlock();
            }
        }
        return null;
    }

    void removeEntry(LeafEntry<K, V> entry) throws IOException {
        throw new UnsupportedOperationException("setEntryValue");
    }

    protected LeafEntry<K, V> insertEntry(K k, V v) throws IOException {
        BTreeValue<K> key = this.createKey(k);
        BTreeValue[] values = new BTreeValue[1];
        if (this.hasValue()) {
            values[0] = this.createValue(v);
        }
        return this.insertEntry(key, values);
    }

    LeafEntry<K, V> insertEntry(K k, V[] vs) throws IOException {
        if (!this.allowNullKey && k == null) {
            throw new NullPointerException("key can not be null");
        }
        assert (vs != null && vs.length > 0);
        BTreeValue<K> key = this.createKey(k);
        if (!this.hasValue() || vs == null || vs.length == 0) {
            BTreeValue[] values = new BTreeValue[1];
            return this.insertEntry(key, values);
        }
        BTreeValue[] values = new BTreeValue[vs.length];
        int i = 0;
        while (i < values.length) {
            values[i] = this.createValue(vs[i]);
            ++i;
        }
        return this.insertEntry(key, values);
    }

    private LeafEntry<K, V> insertEntry(BTreeValue<K> key, BTreeValue<V>[] values) throws IOException {
        if (this.rootNodeId == -1) {
            LeafNode<K, V> root = this.createLeafNode();
            try {
                root.setPrevNodeId(-1);
                root.setNextNodeId(-1);
                this.rootNodeId = root.getNodeId();
                ++this.totalLevels;
                LeafEntry<K, V> leafEntry = root.insert(key, values);
                return leafEntry;
            }
            finally {
                root.unlock();
            }
        }
        BTreeNode<K, V> root = this.loadBTreeNode(this.rootNodeId);
        try {
            int nodeType = root.getNodeType();
            if (nodeType == 1) {
                IndexEntry splitEntry;
                IndexNode indexNode = (IndexNode)root;
                LeafEntry<K, V> insertEntry = indexNode.insert(key, values);
                if (indexNode.needSplit() && (splitEntry = indexNode.split()) != null) {
                    this.insertIndex(splitEntry.getKey(), splitEntry.getChildNodeId());
                }
                LeafEntry<K, V> leafEntry = insertEntry;
                return leafEntry;
            }
            if (nodeType == 2) {
                IndexEntry splitEntry;
                LeafNode leafNode = (LeafNode)root;
                LeafEntry<K, V> insertEntry = leafNode.insert(key, values);
                if (leafNode.needSplit() && (splitEntry = leafNode.split()) != null) {
                    this.insertIndex(splitEntry.getKey(), splitEntry.getChildNodeId());
                }
                LeafEntry<K, V> leafEntry = insertEntry;
                return leafEntry;
            }
            throw new IOException(CoreMessages.getFormattedString("error.UnsupportedNodeType", nodeType));
        }
        finally {
            root.unlock();
        }
    }

    protected void insertIndex(BTreeValue<K> key, int childNodeId) throws IOException {
        IndexNode<K, V> newRoot = this.createIndexNode();
        try {
            newRoot.setPrevNodeId(-1);
            newRoot.setNextNodeId(-1);
            newRoot.setFirstChild(this.rootNodeId);
            newRoot.insertIndex(key, childNodeId);
            this.rootNodeId = newRoot.getNodeId();
            ++this.totalLevels;
        }
        finally {
            newRoot.unlock();
        }
    }

    public int getTotalKeys() {
        return this.totalKeys;
    }

    public int getTotalValues() {
        return this.totalValues;
    }

    public V getValue(K key) throws IOException {
        K entryKey;
        if (!this.hasValue()) {
            return null;
        }
        LeafEntry<K, V> entry = this.findEntry(key);
        if (entry != null && this.comparator.compare(key, entryKey = this.getKey(entry.getKey())) == 0) {
            BTreeValues<V> values = entry.getValues();
            BTreeValues.Value<V> value = values.getFirstValue();
            return this.getValue(value.getValue());
        }
        return null;
    }

    public Collection<V> getValues(K key) throws IOException {
        K entryKey;
        if (!this.hasValue()) {
            return null;
        }
        LeafEntry<K, V> entry = this.findEntry(key);
        if (entry != null && this.comparator.compare(key, entryKey = this.getKey(entry.getKey())) == 0) {
            BTreeValues<V> values = entry.getValues();
            ArrayList<V> list = new ArrayList<V>(values.getValueCount());
            BTreeValues.Value<V> value = values.getFirstValue();
            while (value != null) {
                list.add(this.getValue(value.getValue()));
                value = value.getNext();
            }
            return list;
        }
        return null;
    }

    public boolean exist(K key) throws IOException {
        K entryKey;
        LeafEntry<K, V> entry = this.findEntry(key);
        return entry != null && this.comparator.compare(key, entryKey = this.getKey(entry.getKey())) == 0;
    }

    public void insert(K k, V v) throws IOException {
        if (this.readOnly) {
            throw new IOException(CoreMessages.getString("error.ReadOnlyTree"));
        }
        this.insertEntry(k, v);
    }

    public void insert(K k, V[] vs) throws IOException {
        if (this.readOnly) {
            throw new IOException(CoreMessages.getString("error.ReadOnlyTree"));
        }
        this.insertEntry(k, vs);
    }

    public void remove(K key) throws IOException {
        K entryKey;
        LeafEntry<K, V> entry = this.findEntry(key);
        if (entry != null && this.comparator.compare(key, entryKey = this.getKey(entry.getKey())) == 0) {
            this.removeEntry(entry);
        }
    }

    public BTreeCursor<K, V> createCursor() {
        return new BTreeCursor(this);
    }

    int compare(BTreeValue<K> k1, BTreeValue<K> k2) throws IOException {
        K key2;
        K key1 = this.getKey(k1);
        if (key1 == (key2 = this.getKey(k2))) {
            return 0;
        }
        if (key1 == null) {
            return -1;
        }
        if (key2 == null) {
            return 1;
        }
        return this.comparator.compare(key1, key2);
    }

    private void writeNode(BTreeNode<K, V> node) throws IOException {
        NodeOutputStream out = new NodeOutputStream(this.file, node.getUsedBlocks());
        Throwable throwable = null;
        Object var4_5 = null;
        try {
            try {
                DataOutputStream output = new DataOutputStream(out);
                output.writeInt(node.getNodeType());
                node.write(output);
                node.setDirty(false);
            }
            finally {
                if (out != null) {
                    out.close();
                }
            }
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    synchronized BTreeNode<K, V> loadBTreeNode(int nodeId) throws IOException {
        BTreeNode<K, V> node = this.nodeCaches.get(nodeId);
        if (node != null) {
            node.lock();
            return node;
        }
        if (this.file == null) {
            throw new IOException(CoreMessages.getFormattedString("error.CannotLoadNode", nodeId));
        }
        NodeInputStream in = new NodeInputStream(this.file, nodeId);
        Throwable throwable = null;
        Object var5_6 = null;
        try {
            try {
                DataInputStream input = new DataInputStream(in);
                int nodeType = input.readInt();
                switch (nodeType) {
                    case 1: {
                        node = new IndexNode(this, nodeId);
                        break;
                    }
                    case 2: {
                        node = new LeafNode(this, nodeId);
                        break;
                    }
                    case 3: {
                        node = new ValueNode(this, nodeId);
                        break;
                    }
                    default: {
                        throw new IOException(CoreMessages.getFormattedString("error.UnsupportedNodeType", nodeType, nodeId));
                    }
                }
                node.read(input);
                node.setUsedBlocks(in.getUsedBlocks());
                node.setDirty(false);
                node.lock();
                this.nodeCaches.put(nodeId, node);
                return node;
            }
            finally {
                if (in != null) {
                    in.close();
                }
            }
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    IndexNode<K, V> loadIndexNode(int nodeId) throws IOException {
        BTreeNode<K, V> node = this.loadBTreeNode(nodeId);
        if (node instanceof IndexNode) {
            IndexNode indexNode = (IndexNode)node;
            return indexNode;
        }
        node.unlock();
        throw new IOException(CoreMessages.getFormattedString("error.UnsupportedNodeType", node.getNodeType(), node.getNodeId()));
    }

    LeafNode<K, V> loadLeafNode(int nodeId) throws IOException {
        BTreeNode<K, V> node = this.loadBTreeNode(nodeId);
        if (node instanceof LeafNode) {
            LeafNode leafNode = (LeafNode)node;
            return leafNode;
        }
        node.unlock();
        throw new IOException(CoreMessages.getFormattedString("error.UnsupportedNodeType", node.getNodeType(), node.getNodeId()));
    }

    ValueNode<K, V> loadValueNode(int nodeId) throws IOException {
        BTreeNode<K, V> node = this.loadBTreeNode(nodeId);
        if (node instanceof ValueNode) {
            return (ValueNode)node;
        }
        node.unlock();
        throw new IOException(CoreMessages.getFormattedString("error.UnsupportedNodeType", node.getNodeType(), node.getNodeId()));
    }

    protected int allocBlock() throws IOException {
        ++this.totalBlocks;
        if (this.file != null) {
            return this.file.allocBlock();
        }
        return this.totalBlocks;
    }

    protected void releaseBlock(int blockId) throws IOException {
        if (this.file != null) {
            this.file.freeBlock(blockId);
        }
    }

    public LeafNode<K, V> createLeafNode() throws IOException {
        int nodeId = this.allocBlock();
        LeafNode valueNode = new LeafNode(this, nodeId);
        this.nodeCaches.put(nodeId, valueNode);
        valueNode.lock();
        return valueNode;
    }

    public IndexNode<K, V> createIndexNode() throws IOException {
        int nodeId = this.allocBlock();
        IndexNode indexNode = new IndexNode(this, nodeId);
        this.nodeCaches.put(nodeId, indexNode);
        indexNode.lock();
        return indexNode;
    }

    public ValueNode<K, V> createValueNode() throws IOException {
        int nodeId = this.allocBlock();
        ValueNode valueNode = new ValueNode(this, nodeId);
        this.nodeCaches.put(nodeId, valueNode);
        valueNode.lock();
        return valueNode;
    }

    ExternalValueList<K, V> createExternalValueList(BTreeValues<V> values) throws IOException {
        ExternalValueList list = new ExternalValueList(this);
        BTreeValues.Value<V> value = values.getFirstValue();
        while (value != null) {
            list.append(value.getValue());
            value = value.getNext();
        }
        return list;
    }

    BTreeValue<K> createKey(K key) throws IOException {
        if (key == null) {
            assert (this.allowNullKey);
            return this.NULL_KEY;
        }
        byte[] keyBytes = this.keySerializer.getBytes(key);
        int keySize = this.getKeySize();
        if (keySize != 0 && keySize != keyBytes.length) {
            throw new IOException(CoreMessages.getFormattedString("error.KeySizeError", keyBytes.length, keySize));
        }
        return new BTreeValue<K>(key, keyBytes);
    }

    protected K getKey(BTreeValue<K> key) throws IOException {
        if (key == this.NULL_KEY) {
            return null;
        }
        K k = key.getValue();
        if (k != null) {
            return k;
        }
        byte[] keyBytes = key.getBytes();
        if (keyBytes != null) {
            try {
                k = this.keySerializer.getObject(keyBytes);
                key.setValue(k);
            }
            catch (ClassNotFoundException ce) {
                throw new IOException(ce.getMessage());
            }
        }
        return k;
    }

    int writeKey(DataOutput out, BTreeValue<K> key) throws IOException {
        int size = 0;
        if (this.allowNullKey) {
            if (key == this.NULL_KEY) {
                out.writeBoolean(true);
                return 1;
            }
            out.writeBoolean(false);
            size = 1;
        }
        byte[] bytes = key.getBytes();
        int keySize = this.getKeySize();
        if (keySize != 0 && keySize != bytes.length) {
            throw new IOException(CoreMessages.getString("error.MismatchKeyLength"));
        }
        if (keySize == 0) {
            out.writeInt(bytes.length);
            out.write(bytes);
            return size + 4 + bytes.length;
        }
        out.write(bytes);
        return size + bytes.length;
    }

    BTreeValue<K> readKey(DataInput in) throws IOException {
        boolean isNull;
        if (this.allowNullKey && (isNull = in.readBoolean())) {
            return this.NULL_KEY;
        }
        int keySize = this.getKeySize();
        if (keySize == 0) {
            keySize = in.readInt();
        }
        byte[] keyBytes = new byte[keySize];
        in.readFully(keyBytes);
        return new BTreeValue(keyBytes);
    }

    protected V getValue(BTreeValue<V> value) throws IOException {
        V v = value.getValue();
        if (v != null) {
            return v;
        }
        byte[] valueBytes = value.getBytes();
        if (valueBytes != null) {
            try {
                v = this.valueSerializer.getObject(valueBytes);
                value.setValue(v);
            }
            catch (ClassNotFoundException ex) {
                throw new IOException(ex.getMessage());
            }
        }
        return v;
    }

    private BTreeValue<V> createValue(V value) throws IOException {
        byte[] valueBytes = this.valueSerializer.getBytes(value);
        int valueSize = this.getValueSize();
        if (valueSize != 0 && valueSize != valueBytes.length) {
            throw new IOException(CoreMessages.getFormattedString("error.ValueSizeError", valueBytes.length, valueSize));
        }
        return new BTreeValue<V>(value, valueBytes);
    }

    int writeValue(DataOutput out, BTreeValue<V> value) throws IOException {
        byte[] bytes = value.getBytes();
        if (this.valueSize != 0 && this.valueSize != bytes.length) {
            throw new IOException(CoreMessages.getString("error.MismatchValueLength"));
        }
        if (this.valueSize == 0) {
            out.writeInt(bytes.length);
            out.write(bytes);
            return bytes.length + 4;
        }
        out.write(bytes);
        return this.valueSize;
    }

    BTreeValue<V> readValue(DataInput in) throws IOException {
        int size = this.getValueSize();
        if (size == 0) {
            size = in.readInt();
        }
        byte[] bytes = new byte[size];
        in.readFully(bytes);
        return new BTreeValue(bytes);
    }

    void lockEntry(LeafEntry<K, V> entry) {
        entry.getNode().lock();
    }

    void unlockEntry(LeafEntry<K, V> entry) {
        entry.getNode().unlock();
    }

    int getKeySize() {
        return this.keySize;
    }

    int getValueSize() {
        return this.valueSize;
    }

    int getKeySize(BTreeValue<K> key) {
        if (this.allowNullKey) {
            if (key == this.NULL_KEY) {
                return 1;
            }
            if (this.keySize == 0) {
                return 5 + key.getBytes().length;
            }
            return this.keySize + 1;
        }
        if (this.keySize == 0) {
            return 4 + key.getBytes().length;
        }
        return this.keySize;
    }

    int getValueSize(BTreeValue<V> value) {
        if (this.valueSize == 0) {
            return 4 + value.getBytes().length;
        }
        return this.valueSize;
    }

    boolean hasValue() {
        return this.hasValue;
    }

    boolean allowDuplicate() {
        return this.allowDuplicate;
    }

    int getRootNodeId() {
        return this.rootNodeId;
    }

    protected void readTreeHead(DataInput in) throws IOException {
        long tag = in.readLong();
        if (tag != 0x4254524545L) {
            throw new IOException(CoreMessages.getFormattedString("error.InvaldMagicTag", Long.toHexString(tag)));
        }
        this.version = in.readInt();
        if (this.version != 0) {
            throw new IOException(CoreMessages.getFormattedString("error.UnsupportedVersion", this.version));
        }
        this.readV0(in);
    }

    private void readV0(DataInput in) throws IOException {
        this.allowDuplicate = in.readBoolean();
        this.keySize = in.readInt();
        this.hasValue = in.readBoolean();
        this.valueSize = in.readInt();
        this.rootNodeId = in.readInt();
        this.freeNodeId = in.readInt();
        this.totalLevels = in.readInt();
        this.totalKeys = in.readInt();
        this.totalValues = in.readInt();
        this.allowNullKey = in.readBoolean();
    }

    protected void writeTreeHead(DataOutput out) throws IOException {
        out.writeLong(0x4254524545L);
        out.writeInt(0);
        out.writeBoolean(this.allowDuplicate);
        out.writeInt(this.keySize);
        out.writeBoolean(this.hasValue);
        out.writeInt(this.valueSize);
        out.writeInt(this.rootNodeId);
        out.writeInt(this.freeNodeId);
        out.writeInt(this.totalLevels);
        out.writeInt(this.totalKeys);
        out.writeInt(this.totalValues);
        out.writeBoolean(this.allowNullKey);
    }

    void increaseTotalKeys() {
        ++this.totalKeys;
    }

    void increaseTotalValues(int count) {
        this.totalValues += count;
    }

    public void dump() throws IOException {
        System.out.println("BTREE:" + this.rootNodeId);
        System.out.println("keySize:" + this.keySize);
        System.out.println("hasValue:" + this.hasValue);
        System.out.println("allowDuplicate:" + this.allowDuplicate);
        System.out.println("valueSize:" + this.valueSize);
        System.out.println("rootNodeId");
        System.out.println("freeNodeId");
        System.out.println("totalLevles:" + this.totalLevels);
        System.out.println("totalKeys:" + this.totalKeys);
        System.out.println("totalValues:" + this.totalValues);
    }

    public void dumpAll() throws IOException {
        this.dump();
        if (this.rootNodeId != -1) {
            BTreeNode<K, V> rootNode = this.loadBTreeNode(this.rootNodeId);
            try {
                rootNode.dumpAll();
            }
            finally {
                rootNode.unlock();
            }
        }
    }

    protected class ReusableBTreeFile
    implements NodeFile {
        BTreeFile file;

        ReusableBTreeFile(BTreeFile file) {
            this.file = file;
            BTree.this.freeNodeId = -1;
        }

        @Override
        public int getTotalBlock() throws IOException {
            return this.file.getTotalBlock();
        }

        @Override
        public int allocBlock() throws IOException {
            if (BTree.this.freeNodeId != -1) {
                int blockId = BTree.this.freeNodeId;
                byte[] bytes = new byte[4];
                this.file.readBlock(BTree.this.freeNodeId, bytes);
                BTree.this.freeNodeId = BTreeUtils.bytesToInteger(bytes);
                return blockId;
            }
            return this.file.allocBlock();
        }

        @Override
        public void freeBlock(int blockId) throws IOException {
            byte[] bytes = new byte[8];
            BTreeUtils.integerToBytes(BTree.this.freeNodeId, bytes);
            this.file.writeBlock(blockId, bytes);
            BTree.this.freeNodeId = blockId;
        }

        @Override
        public Object lock() throws IOException {
            return this.file.lock();
        }

        @Override
        public void readBlock(int blockId, byte[] bytes) throws IOException {
            this.file.readBlock(blockId, bytes);
        }

        @Override
        public void unlock(Object lock) throws IOException {
            this.file.unlock(lock);
        }

        @Override
        public void writeBlock(int blockId, byte[] bytes) throws IOException {
            this.file.writeBlock(blockId, bytes);
        }

        @Override
        public void close() throws IOException {
            this.file.close();
        }
    }
}

