/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.kura.core.keystore.crl;

import com.eclipsesource.json.Json;
import com.eclipsesource.json.JsonArray;
import com.eclipsesource.json.JsonValue;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.Writer;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertStore;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.X509CRL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import org.eclipse.kura.core.keystore.crl.StoredCRL;
import org.eclipse.kura.core.keystore.util.MappingCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CRLStore
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(CRLStore.class);
    private final File storeFile;
    private final Map<Set<URI>, StoredCRL> crls = new HashMap<Set<URI>, StoredCRL>();
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
    private final long storeDelayMs;
    private Optional<ScheduledFuture<?>> storeTask = Optional.empty();

    public CRLStore(File storeFile, long storeDelayMs) {
        this.storeFile = storeFile;
        this.storeDelayMs = storeDelayMs;
        if (storeFile.exists()) {
            this.load();
        } else {
            File parent = storeFile.getParentFile();
            if (!parent.isDirectory() && !parent.mkdirs()) {
                logger.warn("failed to create directory: {}", (Object)parent);
            }
        }
    }

    public synchronized void storeCRL(StoredCRL crl) {
        this.crls.put(crl.getDistributionPoints(), crl);
        this.requestStore();
    }

    public synchronized boolean removeCRLs(Predicate<StoredCRL> predicate) {
        boolean changed = false;
        Iterator<StoredCRL> iter = this.crls.values().iterator();
        while (iter.hasNext()) {
            StoredCRL next = iter.next();
            if (!predicate.test(next)) continue;
            logger.info("removing CRL: {}", (Object)next.getCrl().getIssuerX500Principal());
            changed = true;
            iter.remove();
        }
        if (changed) {
            this.requestStore();
        }
        return changed;
    }

    public Optional<StoredCRL> getCrl(Set<URI> distributionPoints) {
        return Optional.ofNullable(this.crls.get(distributionPoints));
    }

    public synchronized List<StoredCRL> getCRLs() {
        return new ArrayList<StoredCRL>(this.crls.values());
    }

    public CertStore getCertStore() throws InvalidAlgorithmParameterException, NoSuchAlgorithmException {
        return CertStore.getInstance("Collection", new CollectionCertStoreParameters(new MappingCollection<StoredCRL, X509CRL>(this.crls.values(), StoredCRL::getCrl)));
    }

    private void requestStore() {
        Optional<ScheduledFuture<?>> currentTask = this.storeTask;
        if (currentTask.isPresent()) {
            currentTask.get().cancel(false);
        }
        this.storeTask = Optional.of(this.executor.schedule(this::store, this.storeDelayMs, TimeUnit.MILLISECONDS));
    }

    private synchronized void load() {
        try {
            Throwable throwable = null;
            Object var2_4 = null;
            try (InputStreamReader in = new InputStreamReader(new FileInputStream(this.storeFile));){
                JsonArray array = Json.parse((Reader)in).asArray();
                for (JsonValue value : array) {
                    StoredCRL crl = StoredCRL.fromJson(value.asObject());
                    logger.info("loaded CRL for {}", (Object)crl.getCrl().getIssuerX500Principal());
                    this.crls.put(crl.getDistributionPoints(), crl);
                }
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (Exception e) {
            logger.warn("failed to load CRLs", (Throwable)e);
        }
    }

    private void store() {
        try {
            logger.info("storing CRLs...");
            List<StoredCRL> currentCrls = this.getCRLs();
            JsonArray array = new JsonArray();
            for (StoredCRL crl : currentCrls) {
                array.add((JsonValue)crl.toJson());
            }
            File tmpFile = new File(this.storeFile.getParent(), String.valueOf(this.storeFile.getName()) + "_");
            Object object = null;
            Object var5_7 = null;
            try (FileWriter out = new FileWriter(tmpFile);){
                array.writeTo((Writer)out);
            }
            catch (Throwable throwable) {
                if (object == null) {
                    object = throwable;
                } else if (object != throwable) {
                    ((Throwable)object).addSuppressed(throwable);
                }
                throw object;
            }
            Files.move(tmpFile.toPath(), this.storeFile.toPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
            logger.info("storing CRLs...done");
        }
        catch (Exception e) {
            logger.warn("failed to store CRLs", (Throwable)e);
        }
    }

    @Override
    public void close() throws IOException {
        this.executor.shutdown();
    }
}

