/*******************************************************************************
 * Copyright (c) Philipps University of Marburg. All rights reserved. 
 * This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Philipps University of Marburg - initial API and implementation
 *******************************************************************************/
package pum.simuref.generator.managers;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.refactor.refactoring.core.ExtensionPointTags;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import pum.simuref.generator.core.SimuRefInfo;


/**
 * @author alex
 *
 */
public abstract class XMLPluginFileManager {
	
	private static final String PLUGINXML = "plugin.xml";
	private static final String PLUGIN = "plugin";
	private static final String EXTENSION = "extension";
	private static final String POINT = "point";
	
	private static final String ID_TAG = "id";
	private static final String SIMUREF_EMF_REFACTORING_ID = "emfRefactoringId";
	private static final String SIMUREF_JAVA_REFACTORING_ID = "javaRefactoringId";
	private static final String CLASS_TAG = "class";
	private static final String NAME_TAG = "name";
	private static final String SIMUREF_MAPPING = "mapping";
	
	private static final String SIMUREF_PARTICIPANT_M2C_EXTPOINT_TAG = "emfRefactoringParticipant";
	
	private static final String SIMUREF_PARTICIPANT_M2C_EXTPOINT_NAME 
		= "pum.simuref.modeltocode.participant.emfRefactoringParticipant";
	// emfRefactoringParticipant - class, emfRefactoringId, id, name
	
	private static final String SIMUREF_LISTENER_M2C_EXTPOINT_NAME
		= "pum.simuref.modeltocode.listener";
	// mapping - class, emfRefactoringId, id
	
	private static final String SIMUREF_LISTENER_C2M_EXTPOINT_NAME
		= "pum.simuref.codetomodel.listener";
	// mapping - class, javaRefactoringId, id
	
	private static final String LTK_RENAME_PARTICIPANT_C2M_EXTPOINT_NAME
	 	= "org.eclipse.ltk.core.refactoring.renameParticipants";
	
	private static final String LTK_RENAME_TAG
		= "renameParticipant";
	
	private static final String LTK_MOVE_PARTICIPANT_C2M_EXTPOINT_NAME
		= "org.eclipse.ltk.core.refactoring.moveParticipants";
	
	private static final String LTK_MOVE_TAG
		= "moveParticipant";
	private static final String INSTANCE_TAG = "instanceof";
	
	private static final String VALUE_TAG = "value";

	/**
	 * Static method for checking whether the project with the given name
	 * owns a plugin.xml file.
	 * @param projectName Name of the project to be checked.
	 * @return true if the project with the given name owns a plugin.xml 
	 * file; false otherwise.
	 */
	public static boolean pluginXmlExists(String projectName) {
		IProject project = ResourcesPlugin.getWorkspace()
											.getRoot().getProject(projectName);
		java.net.URI uri = project.getLocationURI();
		String path = uri.getPath() + "/" + XMLPluginFileManager.PLUGINXML;
		File file = new File(path);
		return file.exists();
	}

	/**
	 * Static method for creating a plugin.xml file within the project
	 * with the given name.
	 * @param projectName Name of the project the plugin.xml file has
	 * to be generated into.
	 */
	public static void createPluginXml(String projectName) {
		final DocumentBuilder builder = 
							XMLPluginFileManager.newDocumentBuilder();
		if (null != builder){
			final Document doc = builder.newDocument();			
			final Element root = doc.createElement
										(XMLPluginFileManager.PLUGIN);
			doc.appendChild(root);
			final TransformerFactory transformerFactory = 
										TransformerFactory.newInstance();
			Transformer transformer = null;
			try {
				transformer = transformerFactory.newTransformer();
				transformer.setOutputProperty(OutputKeys.INDENT,"yes");
			} catch (final TransformerConfigurationException e) {
				e.printStackTrace();
			}
			final DOMSource source = new DOMSource(doc);
			IProject project = ResourcesPlugin.getWorkspace().getRoot()
												.getProject(projectName);
			java.net.URI uri = project.getLocationURI();
			StreamResult result = new StreamResult
						(uri + "/" + XMLPluginFileManager.PLUGINXML);			
			try {
				transformer.transform(source, result);
			} catch (final TransformerException e) {
				e.printStackTrace();
			}			
		}
	}	
	
	private static DocumentBuilder newDocumentBuilder() {
		try {
			return DocumentBuilderFactory.newInstance()
												.newDocumentBuilder();
		} catch (final ParserConfigurationException e) {
			e.printStackTrace();
			return null;
		}
	}
	
	private static Transformer createTransformer(){
		final TransformerFactory transformerFactory = TransformerFactory.newInstance();
		try {
			Transformer transformer = transformerFactory.newTransformer();
			transformer.setOutputProperty(OutputKeys.INDENT,"yes");
			return transformer;
		} catch (final TransformerConfigurationException e) {
			e.printStackTrace();
			return null;
		}		
	}
	
	/**
	 * Abstract method for getting a list of RefactoringConfig out from
	 * the model refactoring extension point served in the plugin.xml 
	 * file of the project with the given name.
	 * @param projectName Name of the project the plugin.xml file has to
	 * be analyzed for serving the model refactoring extension point.
	 * @return List of RefactoringConfig out from the model refactoring
	 * extension point served in the plugin.xml file of the project with
	 * the given name.
	 */
	public static List<SimuRefInfo> getSimuRefConfig
												(String projectName) {
		final DocumentBuilder builder = XMLPluginFileManager
												.newDocumentBuilder();
		List<SimuRefInfo> simuRefConfigs = 
									new ArrayList<SimuRefInfo>();
		if (null != builder){
			Document doc = null;
			try {
				IProject project = ResourcesPlugin.getWorkspace()
									.getRoot().getProject(projectName);
				
				String path = project.getLocation().toString() + "/" + XMLPluginFileManager.PLUGINXML;
				System.out.println("Path: " + path);
				final File file = new File(path);
				System.out.println("exists: " + file.exists());
				if (! file.exists()) createPluginFile(path);
				
				java.net.URI uri = project.getLocationURI();
				path = uri + "/" + XMLPluginFileManager.PLUGINXML;
				doc = builder.parse(path);
			} catch (final SAXException e) {
				e.printStackTrace();
			} catch (final IOException e) {
				e.printStackTrace();
			}

			if (doc != null) {
				// emfRefactoringParticipant - class, emfRefactoringId, id, name
				Element root = doc.getDocumentElement();
				System.out.println(root);
				NodeList extPointList = root.getElementsByTagName(ExtensionPointTags.EXTENSION_POINT_TAG);
				for (int j=0; j < extPointList.getLength(); j++){
					Element extPoint = (Element) extPointList.item(j);
					if (extPoint.getAttribute("point").equals(SIMUREF_PARTICIPANT_M2C_EXTPOINT_NAME)) { 			// PARTICIPANT MC
						NodeList simuRefactorings = extPoint.getElementsByTagName
								(SIMUREF_PARTICIPANT_M2C_EXTPOINT_TAG);
						for (int i=0; i < simuRefactorings.getLength(); i++){
							Element simuRefactoring = (Element) simuRefactorings.item(i);
							String mappingClass = simuRefactoring.getAttribute
											(CLASS_TAG);
							String pluginId = simuRefactoring.getAttribute
											(ID_TAG);
							String name = simuRefactoring.getAttribute
											(NAME_TAG);
							String emfRefactoringId = simuRefactoring.getAttribute
											(SIMUREF_EMF_REFACTORING_ID);
							SimuRefInfo refactoringConfig = new SimuRefInfo(pluginId, projectName, mappingClass, name, emfRefactoringId, null, SimuRefInfo.PARTICIPANT_MC);
							simuRefConfigs.add(refactoringConfig);
						}
					}
					else if (extPoint.getAttribute("point").equals(SIMUREF_LISTENER_M2C_EXTPOINT_NAME)) {			// LISTENER MC 
						NodeList simuRefactorings = extPoint.getElementsByTagName
												(SIMUREF_MAPPING);
						for (int i=0; i < simuRefactorings.getLength(); i++){
							Element simuRefactoring = (Element) simuRefactorings.item(i);
							String mappingClass = simuRefactoring.getAttribute
											(CLASS_TAG);
							String pluginId = simuRefactoring.getAttribute
											(ID_TAG);
							String emfRefactoringId = simuRefactoring.getAttribute
											(SIMUREF_EMF_REFACTORING_ID);
							SimuRefInfo refactoringConfig = new SimuRefInfo(pluginId, projectName, mappingClass, null, emfRefactoringId, null, SimuRefInfo.LISTENER_MC);
							simuRefConfigs.add(refactoringConfig);
						}
					}
					else if (extPoint.getAttribute("point").equals(SIMUREF_LISTENER_C2M_EXTPOINT_NAME)) {			// LISTENER CM
						NodeList simuRefactorings = extPoint.getElementsByTagName
											(SIMUREF_MAPPING);
						for (int i=0; i < simuRefactorings.getLength(); i++){
							Element simuRefactoring = (Element) simuRefactorings.item(i);
							String mappingClass = simuRefactoring.getAttribute
											(CLASS_TAG);
							String pluginId = simuRefactoring.getAttribute
											(ID_TAG);
							String javaRefactoringId = simuRefactoring.getAttribute
											(SIMUREF_JAVA_REFACTORING_ID);
							SimuRefInfo refactoringConfig = new SimuRefInfo(pluginId, projectName, mappingClass, null, null, javaRefactoringId, SimuRefInfo.LISTENER_CM);
							simuRefConfigs.add(refactoringConfig);
						}
					}
					else if (extPoint.getAttribute("point").equals(LTK_RENAME_PARTICIPANT_C2M_EXTPOINT_NAME)) {			// RENAME PARTICIPANT CM
						NodeList ltkParticipant = extPoint.getElementsByTagName
											(LTK_RENAME_TAG);
						for (int i=0; i < ltkParticipant.getLength(); i++){
							Element ltkParti = (Element) ltkParticipant.item(i);
							String implClass = ltkParti.getAttribute
											(CLASS_TAG);
							String id = ltkParti.getAttribute
											(ID_TAG);
							String name = ltkParti.getAttribute
											(NAME_TAG);
							Element ins = (Element) ltkParti.getElementsByTagName(INSTANCE_TAG).item(0);
							System.out.println("----> " + ins.getAttribute(VALUE_TAG));
							SimuRefInfo refactoringConfig = new SimuRefInfo(id, projectName, implClass, name, null, null, SimuRefInfo.PARTICIPANT_CM_RENAME);
							refactoringConfig.setJdtImport(ins.getAttribute(VALUE_TAG));
							simuRefConfigs.add(refactoringConfig);
						}
					}
					else if (extPoint.getAttribute("point").equals(LTK_MOVE_PARTICIPANT_C2M_EXTPOINT_NAME)) {			// MOVE PARTICIPANT CM
						NodeList ltkParticipant = extPoint.getElementsByTagName
											(LTK_MOVE_TAG);
						for (int i=0; i < ltkParticipant.getLength(); i++){
							Element ltkParti = (Element) ltkParticipant.item(i);
							String implClass = ltkParti.getAttribute
											(CLASS_TAG);
							String id = ltkParti.getAttribute
											(ID_TAG);
							String name = ltkParti.getAttribute
											(NAME_TAG);
							SimuRefInfo refactoringConfig = new SimuRefInfo(id, projectName, implClass, name, null, null, SimuRefInfo.PARTICIPANT_CM_MOVE);
							Element ins = (Element) ltkParti.getElementsByTagName(INSTANCE_TAG).item(0);
							refactoringConfig.setJdtImportWithId(ins.getAttribute(VALUE_TAG));
							simuRefConfigs.add(refactoringConfig);
						}
					}
				}
			}
		}
		return simuRefConfigs;
	}

	/**
	 * Abstract method for saving the given RefactoringConfig list as model
	 * refactoring extension point servings in the plugin.xml file of the
	 * project with the given name. 
	 * @param projectName Name of the project the model refactoring 
	 * extension point servings shall be stored in the plugin.xml file.
	 * @param simuRefConfigs List of RefactoringConfig to be saved as
	 * model refactoring extension point servings.
	 */
	public static void saveSimuRefConfig
			(String projectName, List<SimuRefInfo> simuRefConfigs) {
		final DocumentBuilder builder = 
							XMLPluginFileManager.newDocumentBuilder();
		if (null != builder){
			final Document doc = builder.newDocument();			
			Element root = doc.createElement(XMLPluginFileManager.PLUGIN);
			doc.appendChild(root);
			
			final Element extensionParticipantMC = doc.createElement(XMLPluginFileManager.EXTENSION);
			extensionParticipantMC.setAttribute(XMLPluginFileManager.POINT, 
					SIMUREF_PARTICIPANT_M2C_EXTPOINT_NAME);
			
			final Element extensionListenerMC = doc.createElement(XMLPluginFileManager.EXTENSION);
			extensionListenerMC.setAttribute(XMLPluginFileManager.POINT, 
					SIMUREF_LISTENER_M2C_EXTPOINT_NAME);
			
			final Element extensionListenerCM = doc.createElement(XMLPluginFileManager.EXTENSION);
			extensionListenerCM.setAttribute(XMLPluginFileManager.POINT, 
					SIMUREF_LISTENER_C2M_EXTPOINT_NAME);
			
			final Element extensionRenameParticipantCM = doc.createElement(XMLPluginFileManager.EXTENSION);
			extensionRenameParticipantCM.setAttribute(XMLPluginFileManager.POINT, 
					LTK_RENAME_PARTICIPANT_C2M_EXTPOINT_NAME);
			
			final Element extensionMoveParticipantCM = doc.createElement(XMLPluginFileManager.EXTENSION);
			extensionMoveParticipantCM.setAttribute(XMLPluginFileManager.POINT, 
					LTK_MOVE_PARTICIPANT_C2M_EXTPOINT_NAME);
			
			for (SimuRefInfo simuref : simuRefConfigs){
				if (simuref.getTechnique() == SimuRefInfo.PARTICIPANT_MC) {				// Participant Model -> Code
					Element simuRefactoring = doc.createElement
											(SIMUREF_PARTICIPANT_M2C_EXTPOINT_TAG);
					if (simuref.getPackageName() == null){
						simuRefactoring.setAttribute(CLASS_TAG, 
								simuref.getMappingClass());
					} else {
						simuRefactoring.setAttribute(CLASS_TAG, 
								simuref.getPackageName() + "." + simuref.getMappingClass());
					}
					simuRefactoring.setAttribute(CLASS_TAG, 
									simuref.getMappingClass());
					simuRefactoring.setAttribute(SIMUREF_EMF_REFACTORING_ID, 
									simuref.getEmfRefactoringId());
					simuRefactoring.setAttribute(ID_TAG, simuref.getPluginId());
					simuRefactoring.setAttribute(NAME_TAG, simuref.getName());
					extensionParticipantMC.appendChild(simuRefactoring);
				} else if (simuref.getTechnique() == SimuRefInfo.LISTENER_MC) { 				// Listen Model -> Code
					Element simuRefactoring = doc.createElement
											(SIMUREF_MAPPING);
					if (simuref.getPackageName() == null){
						simuRefactoring.setAttribute(CLASS_TAG, 
								simuref.getMappingClass());
					} else {
						simuRefactoring.setAttribute(CLASS_TAG, 
								simuref.getPackageName() + "." + simuref.getMappingClass());
					}
					simuRefactoring.setAttribute(CLASS_TAG, 
									simuref.getMappingClass());
					simuRefactoring.setAttribute(SIMUREF_EMF_REFACTORING_ID, 
									simuref.getEmfRefactoringId());
					simuRefactoring.setAttribute(ID_TAG, simuref.getPluginId());
					extensionListenerMC.appendChild(simuRefactoring);
				} else if (simuref.getTechnique() == SimuRefInfo.LISTENER_CM) { 				// Listen Code -> Model
					Element simuRefactoring = doc.createElement
											(SIMUREF_MAPPING);
					if (simuref.getPackageName() == null){
						simuRefactoring.setAttribute(CLASS_TAG, 
								simuref.getMappingClass());
					} else {
						simuRefactoring.setAttribute(CLASS_TAG, 
								simuref.getPackageName() + "." + simuref.getMappingClass());
					}
					simuRefactoring.setAttribute(CLASS_TAG, 
									simuref.getMappingClass());
					simuRefactoring.setAttribute(SIMUREF_JAVA_REFACTORING_ID, 
									simuref.getJavaRefactoringId());
					simuRefactoring.setAttribute(ID_TAG, simuref.getPluginId());
					extensionListenerCM.appendChild(simuRefactoring);
				}
				else if ((simuref.getTechnique() == SimuRefInfo.PARTICIPANT_CM_RENAME) || (simuref.getTechnique() == SimuRefInfo.PARTICIPANT_CM_MOVE))  { 				// Participant Code -> Model
					Element simuRefactoring = null;
					if (simuref.getTechnique() == SimuRefInfo.PARTICIPANT_CM_RENAME)
						simuRefactoring = doc.createElement(LTK_RENAME_TAG);
					if (simuref.getTechnique() == SimuRefInfo.PARTICIPANT_CM_MOVE)
						simuRefactoring = doc.createElement(LTK_MOVE_TAG);
					if (simuref.getPackageName() == null){
						simuRefactoring.setAttribute(CLASS_TAG, 
								simuref.getMappingClass());
					} else {
						simuRefactoring.setAttribute(CLASS_TAG, 
								simuref.getPackageName() + "." + simuref.getMappingClass());
					}
					simuRefactoring.setAttribute(CLASS_TAG, 
									simuref.getMappingClass());
					simuRefactoring.setAttribute(ID_TAG, simuref.getPluginId());
					simuRefactoring.setAttribute(NAME_TAG, simuref.getName());
					Element enable = doc.createElement("enablement");
					Element wi = doc.createElement("with");
					wi.setAttribute("variable", "affectedNatures");
					Element iter = doc.createElement("iterate");
					iter.setAttribute("operator", "or");
					Element eq = doc.createElement("equals");
					eq.setAttribute("value", "org.eclipse.jdt.core.javanature");
					iter.appendChild(eq);
					wi.appendChild(iter);
					enable.appendChild(wi);
					
					Element wi2 = doc.createElement("with");
					wi2.setAttribute("variable", "element");
					Element or = doc.createElement("or");
					Element ins = doc.createElement("instanceof");
					if ((simuref.getJdtType() == null) || (simuref.getJdtType().isEmpty())) {
						ins.setAttribute("value", simuref.getJdtImport());
					} else {
						ins.setAttribute("value", "org.eclipse.jdt.core." + simuref.getJdtType());
					}
					
					or.appendChild(ins);
					wi2.appendChild(or);
					enable.appendChild(wi2);
					simuRefactoring.appendChild(enable);
					
					if (simuref.getTechnique() == SimuRefInfo.PARTICIPANT_CM_RENAME)
						extensionRenameParticipantCM.appendChild(simuRefactoring);
					if (simuref.getTechnique() == SimuRefInfo.PARTICIPANT_CM_MOVE)
						extensionMoveParticipantCM.appendChild(simuRefactoring);
					
				}
			}
			root.appendChild(extensionParticipantMC);
			root.appendChild(extensionListenerMC);			
			root.appendChild(extensionListenerCM);
			root.appendChild(extensionRenameParticipantCM);
			root.appendChild(extensionMoveParticipantCM);
			Transformer transformer = XMLPluginFileManager.createTransformer();
			final DOMSource source = new DOMSource(doc);
			IProject project = ResourcesPlugin.getWorkspace()
											.getRoot().getProject(projectName);
			java.net.URI uri = project.getLocationURI();
			StreamResult result = new StreamResult(uri + "/" 
											+ XMLPluginFileManager.PLUGINXML);			
			try {
				transformer.transform(source, result);
			} catch (final TransformerException e) {
				e.printStackTrace();
			}			
		}
	}
	
	private static void createPluginFile(String path) {
		final DocumentBuilder builder = XMLPluginFileManager.newDocumentBuilder();
		Document doc = null;
		Element root;
		if (builder != null) {
			doc = builder.newDocument();
			root = doc.createElement(XMLPluginFileManager.PLUGIN);
			doc.appendChild(root);
			final Element extension = doc.createElement(ExtensionPointTags.EXTENSION_POINT_TAG);
			extension.setAttribute(XMLPluginFileManager.POINT, ExtensionPointTags.EXTENSION_POINT_NAME);
			root.appendChild(extension);
		}
		final Transformer transformer = XMLPluginFileManager.createTransformer();
		final DOMSource source = new DOMSource(doc);
		final StreamResult result = new StreamResult(path);
		try {
			transformer.transform(source, result);
		} catch (final TransformerException e) {
			e.printStackTrace();
		}
	}

}
