/*
 * Copyright © 2007-2018 Advanced Micro Devices, Inc.
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NON-INFRINGEMENT. IN NO EVENT SHALL THE COPYRIGHT HOLDERS, AUTHORS
 * AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 */

/**
****************************************************************************************************
* @file  addrobject.cpp
* @brief Contains the Object base class implementation.
****************************************************************************************************
*/

#include "addrinterface.h"
#include "addrobject.h"

namespace Addr
{

/**
****************************************************************************************************
*   Object::Object
*
*   @brief
*       Constructor for the Object class.
****************************************************************************************************
*/
Object::Object()
{
    m_client.handle = NULL;
    m_client.callbacks.allocSysMem = NULL;
    m_client.callbacks.freeSysMem = NULL;
    m_client.callbacks.debugPrint = NULL;
}

/**
****************************************************************************************************
*   Object::Object
*
*   @brief
*       Constructor for the Object class.
****************************************************************************************************
*/
Object::Object(const Client* pClient)
{
    m_client = *pClient;
}

/**
****************************************************************************************************
*   Object::~Object
*
*   @brief
*       Destructor for the Object class.
****************************************************************************************************
*/
Object::~Object()
{
}

/**
****************************************************************************************************
*   Object::ClientAlloc
*
*   @brief
*       Calls instanced allocSysMem inside Client
****************************************************************************************************
*/
VOID* Object::ClientAlloc(
    size_t         objSize,    ///< [in] Size to allocate
    const Client*  pClient)    ///< [in] Client pointer
{
    VOID* pObjMem = NULL;

    if (pClient->callbacks.allocSysMem != NULL)
    {
        ADDR_ALLOCSYSMEM_INPUT allocInput = {0};

        allocInput.size        = sizeof(ADDR_ALLOCSYSMEM_INPUT);
        allocInput.flags.value = 0;
        allocInput.sizeInBytes = static_cast<UINT_32>(objSize);
        allocInput.hClient     = pClient->handle;

        pObjMem = pClient->callbacks.allocSysMem(&allocInput);
    }

    return pObjMem;
}

/**
****************************************************************************************************
*   Object::Alloc
*
*   @brief
*       A wrapper of ClientAlloc
****************************************************************************************************
*/
VOID* Object::Alloc(
    size_t objSize      ///< [in] Size to allocate
    ) const
{
    return ClientAlloc(objSize, &m_client);
}

/**
****************************************************************************************************
*   Object::ClientFree
*
*   @brief
*       Calls freeSysMem inside Client
****************************************************************************************************
*/
VOID Object::ClientFree(
    VOID*          pObjMem,    ///< [in] User virtual address to free.
    const Client*  pClient)    ///< [in] Client pointer
{
    if (pClient->callbacks.freeSysMem != NULL)
    {
        if (pObjMem != NULL)
        {
            ADDR_FREESYSMEM_INPUT freeInput = {0};

            freeInput.size      = sizeof(ADDR_FREESYSMEM_INPUT);
            freeInput.hClient   = pClient->handle;
            freeInput.pVirtAddr = pObjMem;

            pClient->callbacks.freeSysMem(&freeInput);
        }
    }
}

/**
****************************************************************************************************
*   Object::Free
*
*   @brief
*       A wrapper of ClientFree
****************************************************************************************************
*/
VOID Object::Free(
    VOID* pObjMem       ///< [in] User virtual address to free.
    ) const
{
    ClientFree(pObjMem, &m_client);
}

/**
****************************************************************************************************
*   Object::operator new
*
*   @brief
*       Placement new operator. (with pre-allocated memory pointer)
*
*   @return
*       Returns pre-allocated memory pointer.
****************************************************************************************************
*/
VOID* Object::operator new(
    size_t objSize,     ///< [in] Size to allocate
    VOID*  pMem)        ///< [in] Pre-allocated pointer
{
    return pMem;
}

/**
****************************************************************************************************
*   Object::operator delete
*
*   @brief
*       Frees Object object memory.
****************************************************************************************************
*/
VOID Object::operator delete(
    VOID* pObjMem)      ///< [in] User virtual address to free.
{
    Object* pObj = static_cast<Object*>(pObjMem);
    ClientFree(pObjMem, &pObj->m_client);
}

/**
****************************************************************************************************
*   Object::DebugPrint
*
*   @brief
*       Print debug message
*
*   @return
*       N/A
****************************************************************************************************
*/
VOID Object::DebugPrint(
    const CHAR* pDebugString,     ///< [in] Debug string
    ...
    ) const
{
#if DEBUG
    if (m_client.callbacks.debugPrint != NULL)
    {
        ADDR_DEBUGPRINT_INPUT debugPrintInput = {0};

        debugPrintInput.size         = sizeof(ADDR_DEBUGPRINT_INPUT);
        debugPrintInput.pDebugString = const_cast<CHAR*>(pDebugString);
        debugPrintInput.hClient      = m_client.handle;
        va_start(debugPrintInput.ap, pDebugString);

        m_client.callbacks.debugPrint(&debugPrintInput);

        va_end(debugPrintInput.ap);
    }
#endif
}

} // Addr
