package WWW::AzimuthAero::PriceCrawler;
$WWW::AzimuthAero::PriceCrawler::VERSION = '0.4';

# ABSTRACT: Crawler for https://azimuth.aero/

use parent 'WWW::AzimuthAero';
use feature 'say';


sub prepare_requests {
    my ( $self, %params ) = @_;

    my $iata_map = $self->route_map->route_map_iata( $params{cities} );

    my $n = 0;
    if ( $params{verbose} ) {
        say 'Cities total: ' . scalar @{ $params{cities} } if $params{verbose};
        for my $x ( values %$iata_map ) {
            $n += scalar @$x;
        }
        say
'Amount of WWW::AzimuthAero::get_schedule_dates HTTP requests will be performed: '
          if $params{verbose};
    }

    my @get_requests;
    while ( ( $from, $cities ) = each(%$iata_map) ) {
        for my $to (@$cities) {
            $n-- if $params{verbose};
            my @dates = $self->get_schedule_dates(
                from                => $from,
                to                  => $to,
                only_direct_flights => $params{only_direct_flights}
            );
            say "$n : get_schedule_dates : $from -> $to | dates available : "
              . scalar @dates
              if $params{verbose};
            for my $date (@dates) {
                push @get_requests, { from => $from, to => $to, date => $date };
            }
        }
    }

    if ( $params{verbose} ) {
        say 'Amount of WWW::AzimuthAero::get HTTP requests will be performed: '
          . scalar @get_requests
          if $params{verbose};
        say 'Total HTTP requests: ' . $n + scalar @get_requests
          if $params{verbose};
    }

    return @get_requests;

}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

WWW::AzimuthAero::PriceCrawler - Crawler for https://azimuth.aero/

=head1 VERSION

version 0.4

=head1 SYNOPSIS

    my $azo_price_crawler = WWW::AzimuthAero::PriceCrawler->new();
    $azo_price_crawler->prepare_requests(
      cities => [ 'ROV', 'MOW', 'LED' ],
      only_direct_flights => 1,
      verbose => 1
    )

=head1 DESCRIPTION

Wrappper under L<WWW::AzimuthAero>

If you need to use proxy you can easily find them using L<proxybroker|https://github.com/constverum/ProxyBroker>

Just run

  docker run -it nrdhm/proxybroker proxybroker find --types HTTP

and run your script based on I<WWW::AzimuthAero::PriceCrawler> like

  docker-compose run -e MOJO_PROXY=1 -e HTTPS_PROXY='https://200.116.226.210:36964>' ...

=head1 METHODS

This module has same methods as L<WWW::AzimuthAero> and some more :

=head2 prepare_requests

Return arrray of hashes with params (from, to, date) for L<WWW::AzimuthAero::get> method

    my @l = $azo_price_crawler->prepare_requests( max_date => '18.12.2019', verbose => 1, cities => [ qw/ROV LED/ ] );

In fact, combines L<WWW::AzimuthAero::RouteMap/route_map_iata> and L<WWW::AzimuthAero/get_schedule_dates>

=head3 params

B<max_date> - C<'%d.%m.%Y'> format, if no specified will looks forward for 2 months, default max_date of L<WWW::AzimuthAero/get_schedule_dates>

B<verbose> - print amount of L<WWW::AzimuthAero/get_schedule_dates> requests and future amount of L<WWW::AzimuthAero/get> requests

B<cities> - filter for L<WWW::AzimuthAero::RouteMap/route_map_iata>

B<only_direct_flights> - leave only direct flights. Cities considered without direct flights if L<WWW::AzimuthAero/get_schedule_dates> has no C<available_to> propery in response

=head1 AUTHOR

Pavel Serikov <pavelsr@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2019 by Pavel Serikov.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
