# ABSTRACT: ponapi demo running command line utility
package PONAPI::CLI::Command::demo;

use PONAPI::CLI -command;

use strict;
use warnings;

sub abstract    { "Run a DEMO PONAPI server" }
sub description { "This tool will run a demo server with a mock DB" }

sub opt_spec {
    return (
        [ "server|s", "Run a PONAPI demo server" ],
        [ "query|q",  "Execute a query from the demo server" ],
        [ "port=i",   "Specify a port for the server (default=5000)" ],
    );
}

sub validate_args {
    my ( $self, $opt, $args ) = @_;

    $self->usage_error("one of server (-s) or query (-q) is required.\n")
        unless $opt->{server} xor $opt->{query};

    $self->{port} = $opt->{port} || 5000;
}

sub execute {
    my ( $self, $opt, $args ) = @_;

    $self->run_server() if $opt->{server};
    $self->run_query()  if $opt->{query};
}

sub run_server {
    my $self = shift;

    require Plack::Runner;
    require Plack::Middleware::MethodOverride;
    require PONAPI::Server;

    require File::Temp;
    File::Temp->import('tempdir');

    require Path::Class;
    Path::Class->import('file');

    my $dir  = tempdir( CLEANUP => 1 );
    my $conf = file( $dir . '/server.yml' );
    $conf->spew(<<"DEFAULT_CONF");
server:
  spec_version: "1.0"
  sort_allowed: "false"
  send_version_header: "true"
  send_document_self_link: "true"
  links_type: "relative"
  respond_to_updates_with_200: "false"

repository:
  class:  "Test::PONAPI::Repository::MockDB"
  args:   []
DEFAULT_CONF

    my $app = Plack::Middleware::MethodOverride->wrap(
        PONAPI::Server->new(
            'repository.class' => 'Test::PONAPI::Repository::MockDB',
            'ponapi.config_dir' => $dir
        )->to_app()
    );

    my $runner = Plack::Runner->new;
    $runner->parse_options( '-port', $self->{port} );
    $runner->run($app);
}

sub run_query {
    my $self = shift;

    require JSON::XS;
    require HTTP::Tiny;

    my $url = 'http://localhost:' . $self->{port} . '/articles/2?include=comments,authors';

    my $res = HTTP::Tiny->new->get( $url, {
        headers => {
            'Content-Type' => 'application/vnd.api+json',
        },
    });

    print "\nGET $url\n\n";
    print $res->{protocol} . " " . $res->{status} . " " . $res->{reason} . "\n";

    print "Content-Type: " . $res->{headers}{'content-type'} . "\n\n";

    my $json = JSON::XS->new;
    print $json->pretty(1)->encode( $json->decode($res->{content}) );
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

PONAPI::CLI::Command::demo - ponapi demo running command line utility

=head1 VERSION

version 0.002010

=head1 AUTHORS

=over 4

=item *

Mickey Nasriachi <mickey@cpan.org>

=item *

Stevan Little <stevan@cpan.org>

=item *

Brian Fraser <hugmeir@cpan.org>

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by Mickey Nasriachi, Stevan Little, Brian Fraser.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
