#!/usr/bin/perl
use strict;
use warnings;
use Test::More 'no_plan';
use Bio::Phylo::IO 'parse_tree';
use Bio::Phylo::Util::Logger ':levels';

SKIP: {
	eval { require Statistics::R };
	skip 'Statistics::R not installed', 2, if $@;

	eval { require Math::Random };
	skip 'Math::Random not installed', 2 if $@;

	my $tree = parse_tree(
		'-format' => 'nexus',
		'-handle' => \*DATA,
	);
	$tree->generize( 
		'-delim'     => '_', 
		'-monotypic' => 1,
		'-polypara'  => 1,
	);
	$tree->get_logger->VERBOSE( '-level' => ERROR );

	my $rep = $tree->replicate( '-genera' => 1 )->first;
	ok( $rep );
	ok( $tree->calc_number_of_terminals == $rep->calc_number_of_terminals );
	
	# check for duplicate terminal labels in replicated tree
	my %cnt, my %dup;
	for ( map {$_->get_name} @{$rep->get_terminals} ) {
		$dup{$_}++ if $cnt{$_}++;
	}
	ok ( ! scalar(keys%dup) );
};

__DATA__
#NEXUS                                                                                                                                                                       

BEGIN TAXA;                                                                                                                                                                  
[! Taxa block written by Bio::Phylo::Taxa 0.58 on Tue Aug 11 19:06:39 2015 ]                                                                                                 
        DIMENSIONS NTAX=100;                                                                                                                                                 
        TAXLABELS                                                                                                                                                            
                                                                                         
                Allenopithecus_nigroviridis                                                                                                                                 
                Allocebus_trichotis                                                                                                                                          
                Alouatta_caraya                                                                                                                                              
                Alouatta_palliata                                                                                                                                            
                Aotus_azarai_infulatus                                                                                                                                       
                Aotus_trivirgatus                                                                                                                                            
                Arctocebus_calabarensis                                                                                                                                      
                Ateles_belzebuth_chamek                                                                                                                                      
                Ateles_fusciceps                                                                                                                                             
                Avahi_laniger                                                                                                                                                
                Brachyteles_arachnoides                                                                                                                                      
                Cacajao_calvus
                Cacajao_melanocephalus
                Callicebus_moloch
                Callicebus_personatus
                Callimico_goeldii
                Callithrix_humeralifera
                Callithrix_pygmaea
                Cebus_albifrons
                Cebus_apella
                Cercocebus_galeritus
                Cercocebus_torquatus
                Cercopithecus_lhoesti
                Cercopithecus_mona
                Cheirogaleus_major
                Cheirogaleus_medius
                Chiropotes_albinasus
                Chiropotes_satanas
                Chlorocebus_aethiops
                Colobus_angolensis
                Colobus_guereza
                Daubentonia_madagascariensis
                Erythrocebus_patas
                Eulemur_coronatus
                Eulemur_rubriventer
                Euoticus_elegantulus
                Galago_alleni
                Galago_senegalensis
                Galagoides_demidoff
                Galagoides_zanzibaricus
                Gorilla_gorilla
                Hapalemur_aureus
                Hapalemur_simus
                Homo_sapiens
                Hoolock_hoolock
                Hylobates_agilis
                Hylobates_lar
                Lagothrix_lagotricha
                Lemur_catta
                Leontopithecus_chrysopygus
                Leontopithecus_rosalia
                Lepilemur_mustelinus
                Lepilemur_ruficaudatus
                Lophocebus_albigena
                Loris_tardigradus
                Macaca_ochreata
                Macaca_sylvanus
                Mandrillus_leucophaeus
                Mandrillus_sphinx
                Microcebus_murinus
                Microcebus_rufus
                Miopithecus_talapoin
                Mirza_coquereli
                Nasalis_larvatus
                Nomascus_concolor
                Nomascus_gabriellae
                Nycticebus_coucang
                Nycticebus_pygmaeus
                Otolemur_crassicaudatus
                Otolemur_garnettii
                Pan_paniscus
                Pan_troglodytes
                Papio_hamadryas
                Perodicticus_potto
                Phaner_furcifer
                Piliocolobus_badius
                Pithecia_irrorata
                Pithecia_pithecia
                Pongo_pygmaeus
                Presbytis_comata
                Presbytis_melalophos
                Procolobus_verus
                Propithecus_diadema
                Propithecus_tattersalli
                Pygathrix_nemaeus
                Rhinopithecus_bieti
                Rhinopithecus_roxellana
                Saguinus_fuscicollis
                Saguinus_imperator
                Saimiri_sciureus
                Saimiri_ustus
                Semnopithecus_entellus
                Simias_concolor
                Symphalangus_syndactylus
                Tarsius_syrichta
                Tarsius_tarsier
                Theropithecus_gelada
                Trachypithecus_auratus
                Trachypithecus_francoisi
                Varecia_variegata
        ;
END;
BEGIN TREES;
[! Trees block written by Bio::Phylo::Forest 0.58 on Tue Aug 11 19:06:39 2015 ]
        TRANSLATE
                1 Daubentonia_madagascariensis,
                2 Tarsius_tarsier,
                3 Tarsius_syrichta,
                4 Galagoides_zanzibaricus,
                5 Phaner_furcifer,
                6 Arctocebus_calabarensis,
                7 Perodicticus_potto,
                8 Loris_tardigradus,
                9 Galagoides_demidoff,
                10 Lepilemur_mustelinus,
                11 Lepilemur_ruficaudatus,
                12 Avahi_laniger,
                13 Nycticebus_pygmaeus,
                14 Nycticebus_coucang,
                15 Chiropotes_albinasus,
                16 Callicebus_personatus,
                17 Callicebus_moloch,
                18 Hoolock_hoolock,
                19 Pongo_pygmaeus,
                20 Propithecus_tattersalli,
                21 Propithecus_diadema,
                22 Galago_senegalensis,
                23 Euoticus_elegantulus,
                24 Galago_alleni,
                25 Alouatta_caraya,
                26 Alouatta_palliata,
                27 Gorilla_gorilla,
                28 Miopithecus_talapoin,
                29 Cheirogaleus_major,
                30 Cheirogaleus_medius,
                31 Allocebus_trichotis,
                32 Eulemur_coronatus,
                33 Eulemur_rubriventer,
                34 Otolemur_crassicaudatus,
                35 Otolemur_garnettii,
                36 Aotus_azarai_infulatus,
                37 Aotus_trivirgatus,
                38 Saimiri_ustus,
                39 Saimiri_sciureus,
                40 Cebus_albifrons,
                41 Cebus_apella,
                42 Brachyteles_arachnoides,
                43 Lagothrix_lagotricha,
                44 Ateles_belzebuth_chamek,
                45 Ateles_fusciceps,
                46 Cacajao_melanocephalus,
                47 Pithecia_irrorata,
                48 Pithecia_pithecia,
                49 Nomascus_gabriellae,
                50 Nomascus_concolor,
                51 Symphalangus_syndactylus,
                52 Homo_sapiens,
                53 Colobus_angolensis,
                54 Colobus_guereza,
                55 Procolobus_verus,
                56 Piliocolobus_badius,
                57 Presbytis_comata,
                58 Presbytis_melalophos,
                59 Macaca_sylvanus,
                60 Macaca_ochreata,
                61 Allenopithecus_nigroviridis,
                62 Mirza_coquereli,
                63 Varecia_variegata,
                64 Hapalemur_simus,
                65 Hapalemur_aureus,
                66 Lemur_catta,
                67 Saguinus_fuscicollis,
                68 Saguinus_imperator,
                69 Cacajao_calvus,
                70 Chiropotes_satanas,
                71 Hylobates_lar,
                72 Hylobates_agilis,
                73 Pan_paniscus,
                74 Pan_troglodytes,
                75 Semnopithecus_entellus,
                76 Pygathrix_nemaeus,
                77 Cercocebus_galeritus,
                78 Theropithecus_gelada,
                79 Cercopithecus_mona,
                80 Microcebus_murinus,
                81 Microcebus_rufus,
                82 Callimico_goeldii,
                83 Leontopithecus_chrysopygus,
                84 Leontopithecus_rosalia,
                85 Trachypithecus_auratus,
                86 Trachypithecus_francoisi,
                87 Cercocebus_torquatus,
                88 Papio_hamadryas,
                89 Lophocebus_albigena,
                90 Erythrocebus_patas,
                91 Callithrix_pygmaea,
                92 Callithrix_humeralifera,
                93 Nasalis_larvatus,
                94 Simias_concolor,
                95 Rhinopithecus_bieti,
                96 Rhinopithecus_roxellana,
                97 Mandrillus_sphinx,
                98 Mandrillus_leucophaeus,
                99 Cercopithecus_lhoesti,
                100 Chlorocebus_aethiops;
        TREE TREE1 = [&R] (((((5[&length_median=35.121061,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2642642725946185E-6,length_95%_HPD={30.965221,38.278559},length=35.09906269369368,length_range={29.416339,40.118377},height_95%_HPD={0.0,2.00000000916134E-6}]:35.114601,(10[&length_median=23.377896,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2432432519680724E-6,length_95%_HPD={19.835378,26.259975},length=23.316889027027024,length_range={17.778512,29.357709},height_95%_HPD={0.0,2.999999992425728E-6}]:23.316889,11[&length_median=23.377896,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2432432519680724E-6,length_95%_HPD={19.835378,26.259975},length=23.316889027027024,length_range={17.778512,29.357709},height_95%_HPD={0.0,2.999999992425728E-6}]:23.316889)[&length_median=11.69497,height_range={17.778512999999997,29.35771000000001},height_median=23.37789700000001,height=23.31689027027028,length_95%_HPD={8.083474,14.920275},length=11.795132972972969,posterior=1.0,length_range={7.741614,16.83651},height_95%_HPD={19.835379000000017,26.25997600000001}]:11.797712)[&length_median=4.160589,height_range={29.416339999999998,40.11837700000001},height_median=35.097337000000024,height=35.04906085714288,length_95%_HPD={2.284887,6.041923},length=4.255153255319149,posterior=0.987987987987988,length_range={1.873844,8.806778},height_95%_HPD={30.96522200000002,38.00112200000002}]:4.729601,((12[&length_median=22.500371,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2732732819649591E-6,length_95%_HPD={19.731919,25.450856},length=22.653673654654668,length_range={19.206478,26.608344},height_95%_HPD={1.4210854715202004E-14,2.000000023372195E-6}]:22.653674,(20[&length_median=11.077221,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.294294303615711E-6,length_95%_HPD={9.075096,13.265564},length=11.143531807807808,length_range={8.16782,15.009853},height_95%_HPD={0.0,2.999999992425728E-6}]:11.143532,21[&length_median=11.077221,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.294294303615711E-6,length_95%_HPD={9.075096,13.265564},length=11.143531807807808,length_range={8.16782,15.009853},height_95%_HPD={0.0,2.999999992425728E-6}]:11.143532)[&length_median=11.424292,height_range={8.167821000000018,15.009855000000009},height_median=11.07722200000002,height=11.143533102102113,length_95%_HPD={9.074319,14.308201},length=11.510141825825825,posterior=1.0,length_range={8.287583,15.959555},height_95%_HPD={9.075097,13.26556500000001}]:11.510142)[&length_median=14.486132,height_range={19.20647900000003,26.60834600000001},height_median=22.500372999999996,height=22.65367492792795,length_95%_HPD={11.243305,18.099578},length=14.512274717717714,posterior=1.0,length_range={9.967769,21.41005},height_95%_HPD={19.731921000000007,25.450857}]:14.838735,(((29[&length_median=23.183281,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2072072162363948E-6,length_95%_HPD={19.926608,27.185155},length=23.34193990690692,length_range={18.455813,29.936217},height_95%_HPD={0.0,2.000000023372195E-6}]:23.341940,30[&length_median=23.183281,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2072072162363948E-6,length_95%_HPD={19.926608,27.185155},length=23.34193990690692,length_range={18.455813,29.936217},height_95%_HPD={0.0,2.000000023372195E-6}]:23.341940)[&length_median=7.42034,height_range={18.455814000000018,29.93621800000001},height_median=23.183282000000005,height=23.341941114114114,length_95%_HPD={4.221182,10.618905},length=7.424704183183182,posterior=1.0,length_range={3.160578,13.888756},height_95%_HPD={19.926609000000013,27.18515500000003}]:7.424704,(31[&length_median=24.383899,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2372372462332816E-6,length_95%_HPD={21.246128,26.914213},length=24.37000007807809,length_range={19.824251,28.773579},height_95%_HPD={0.0,2.0000000375830496E-6}]:24.370000,((80[&length_median=11.046336,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.237237246147931E-6,length_95%_HPD={9.234633,13.04162},length=11.109405051051056,length_range={8.74732,14.367654},height_95%_HPD={0.0,2.999999992425728E-6}]:11.109405,81[&length_median=11.046336,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.237237246147931E-6,length_95%_HPD={9.234633,13.04162},length=11.109405051051056,length_range={8.74732,14.367654},height_95%_HPD={0.0,2.999999992425728E-6}]:11.109405)[&length_median=8.40914,height_range={8.747321000000014,14.36765600000001},height_median=11.046337000000008,height=11.1094062882883,length_95%_HPD={6.561707,10.963902},length=8.52646993093093,posterior=1.0,length_range={5.651568,11.843887},height_95%_HPD={9.234634,13.041621000000006}]:8.526470,62[&length_median=19.515672,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2192192284314555E-6,length_95%_HPD={17.031943,22.453269},length=19.635875000000016,length_range={15.572059,24.296862},height_95%_HPD={0.0,2.999999992425728E-6}]:19.635875)[&length_median=4.786039,height_range={15.572060000000015,24.296863000000016},height_median=19.515671999999995,height=19.63587621921923,length_95%_HPD={2.279556,7.056497},length=4.734125096096094,posterior=1.0,length_range={1.697036,8.188825},height_95%_HPD={17.031945000000007,22.453270000000018}]:4.734125)[&length_median=6.251683,height_range={19.82425200000001,28.77358200000001},height_median=24.38390100000003,height=24.370001315315328,length_95%_HPD={4.104063,9.442746},length=6.396643981981978,posterior=1.0,length_range={2.663895,10.674217},height_95%_HPD={21.246128000000006,26.914214000000015}]:6.396644)[&length_median=4.357671,height_range={26.792620000000014,36.465492},height_median=30.386362000000013,height=30.76664529729733,length_95%_HPD={2.82429,5.739166},length=4.364443036036037,posterior=1.0,length_range={2.260214,6.665498},height_95%_HPD={27.827913000000024,34.63987300000003}]:4.875147,(((63[&length_median=18.230961,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2072072163217452E-6,length_95%_HPD={10.618816,23.688412},length=18.231224114114127,length_range={8.736985,32.399338},height_95%_HPD={0.0,2.999999992425728E-6}]:18.366874,64[&length_median=17.515711,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2012012105442792E-6,length_95%_HPD={11.405049,23.543506},length=17.508670648648643,length_range={8.736985,31.397692},height_95%_HPD={0.0,2.999999992425728E-6}]:18.366874)[&length_median=2.80919,height_range={8.736987000000013,27.845365000000022},height_median=16.00090200000001,height=15.906565377906984,length_95%_HPD={0.13021,7.891105},length=3.2739901569767427,posterior=0.5165165165165165,length_range={0.110486,9.457673},height_95%_HPD={10.820663000000025,20.48616600000001}]:1.780173,(65[&length_median=13.993023,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2672672765288948E-6,length_95%_HPD={11.565069,16.590596},length=14.229168024024027,length_range={11.067516,19.337478},height_95%_HPD={0.0,3.000000006636583E-6}]:14.229168,66[&length_median=13.993023,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2672672765288948E-6,length_95%_HPD={11.565069,16.590596},length=14.229168024024027,length_range={11.067516,19.337478},height_95%_HPD={0.0,3.000000006636583E-6}]:14.229168)[&length_median=4.838408,height_range={11.067516000000012,19.337479000000023},height_median=13.993023999999991,height=14.229169291291306,length_95%_HPD={2.824336,7.072854},length=4.855509144144149,posterior=1.0,length_range={2.161427,11.915343},height_95%_HPD={11.565071000000003,16.590596000000005}]:5.917879)[&length_median=2.0878445,height_range={15.976826000000017,32.39934000000002},height_median=18.821895500000004,height=19.273486719047632,length_95%_HPD={0.41046,3.951585},length=2.13005139047619,posterior=0.6306306306306306,length_range={0.083375,5.671528},height_95%_HPD={16.650340000000014,22.80046400000002}]:1.343276,(32[&length_median=11.324763,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2132132233794685E-6,length_95%_HPD={9.286333,13.325014},length=11.384793570570572,length_range={7.499498,15.685659},height_95%_HPD={0.0,2.0000000375830496E-6}]:11.384794,33[&length_median=11.324763,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2132132233794685E-6,length_95%_HPD={9.286333,13.325014},length=11.384793570570572,length_range={7.499498,15.685659},height_95%_HPD={0.0,2.0000000375830496E-6}]:11.384794)[&length_median=9.396777,height_range={7.499499,15.685660000000027},height_median=11.324764000000002,height=11.384794783783795,length_95%_HPD={6.770448,11.979787},length=9.505453231231234,posterior=1.0,length_range={5.557052,18.221074},height_95%_HPD={9.28633400000001,13.325015000000022}]:10.105529)[&length_median=13.55542,height_range={18.126636000000012,32.91920200000003},height_median=21.189011000000015,height=21.49032394294297,length_95%_HPD={10.605318,20.10165},length=14.151468237237243,posterior=1.0,length_range={6.991051,21.423171},height_95%_HPD={18.41150400000003,24.223718000000005}]:14.151468)[&length_median=2.350552,height_range={31.283322000000013,39.47376600000001},height_median=34.12695100000002,height=34.191688750000004,length_95%_HPD={1.496775,3.62632},length=2.4454595357142863,posterior=0.7567567567567568,length_range={1.052052,4.666147},height_95%_HPD={31.283322000000013,36.36067000000002}]:1.850618)[&length_median=2.933047,height_range={32.870578000000016,41.409405000000014},height_median=36.595680500000014,height=36.637148285714346,length_95%_HPD={1.513852,5.105783},length=3.08378617063492,posterior=0.7567567567567568,length_range={0.256301,6.513966},height_95%_HPD={34.14399200000001,39.01372400000001}]:2.351793)[&length_median=8.151705,height_range={36.213706000000016,43.79854100000003},height_median=39.908228000000015,height=39.844203183183254,length_95%_HPD={6.497341,10.366568},length=8.246656309309309,posterior=1.0,length_range={4.73055,12.072236},height_95%_HPD={37.231883000000025,42.462883999999995}]:8.291612,1[&length_median=48.002959,height_range={1.4210854715202004E-14,2.000000023372195E-6},height_median=1.0000000116860974E-6,height=1.237237245593153E-6,length_95%_HPD={45.843091,50.372486},length=48.135813894894916,length_range={45.255241,51.6},height_95%_HPD={9.999999974752427E-7,2.00000000916134E-6}]:48.135814)[&length_median=3.702277,height_range={45.25524200000001,51.600002},height_median=47.90755300000001,height=48.02856478637774,length_95%_HPD={1.586397,5.498803},length=3.5948999256965912,posterior=0.96996996996997,length_range={1.053856,7.578703},height_95%_HPD={46.101198000000025,50.01360500000001}]:3.486945,(((6[&length_median=21.119875,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2342342431098345E-6,length_95%_HPD={18.668661,24.616598},length=21.229410006006,length_range={17.357527,25.585373},height_95%_HPD={1.4210854715202004E-14,2.000000023372195E-6}]:21.229410,7[&length_median=21.119875,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2342342431098345E-6,length_95%_HPD={18.668661,24.616598},length=21.229410006006,length_range={17.357527,25.585373},height_95%_HPD={1.4210854715202004E-14,2.000000023372195E-6}]:21.229410)[&length_median=11.249169,height_range={17.357529000000014,25.585374000000016},height_median=21.119878,height=21.229411240240264,length_95%_HPD={8.496855,14.121695},length=11.250717294294295,posterior=1.0,length_range={6.10888,16.746321},height_95%_HPD={18.668661999999998,24.616599000000015}]:11.961472,(8[&length_median=22.600399,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2402402490580016E-6,length_95%_HPD={19.222752,25.520552},length=22.657919687687684,length_range={18.07612,28.319643},height_95%_HPD={1.4210854715202004E-14,2.000000023372195E-6}]:22.657920,(13[&length_median=10.825499,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2672672769983226E-6,length_95%_HPD={9.070005,12.542418},length=10.861906789789797,length_range={8.046393,14.899741},height_95%_HPD={0.0,2.000000023372195E-6}]:10.861907,14[&length_median=10.825499,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2672672769983226E-6,length_95%_HPD={9.070005,12.542418},length=10.861906789789797,length_range={8.046393,14.899741},height_95%_HPD={0.0,2.000000023372195E-6}]:10.861907)[&length_median=11.748936,height_range={8.046395000000018,14.899742000000018},height_median=10.825500000000005,height=10.861908057057068,length_95%_HPD={8.720718,14.904088},length=11.796012870870872,posterior=1.0,length_range={5.013415,16.446065},height_95%_HPD={9.070007000000004,12.54241900000001}]:11.796013)[&length_median=10.460189,height_range={18.076122000000012,28.31964400000001},height_median=22.6004,height=22.657920927927947,length_95%_HPD={7.83298,13.273094},length=10.53296256756756,posterior=1.0,length_range={7.442656,14.318014},height_95%_HPD={19.222753000000004,25.52055300000002}]:10.532963)[&length_median=2.9028315,height_range={28.467210000000016,39.263235000000016},height_median=31.841851,height=32.17904762500001,length_95%_HPD={1.587403,4.975232},length=2.993503479166666,posterior=0.5765765765765766,length_range={0.869223,5.538958},height_95%_HPD={28.895816000000003,35.804883}]:1.725984,((((22[&length_median=16.842043,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.312312321076135E-6,length_95%_HPD={12.629187,22.085971},length=17.297960933933926,length_range={12.629187,26.203343},height_95%_HPD={0.0,2.000000023372195E-6}]:17.297961,23[&length_median=16.842043,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.312312321076135E-6,length_95%_HPD={12.629187,22.085971},length=17.297960933933926,length_range={12.629187,26.203343},height_95%_HPD={0.0,2.000000023372195E-6}]:17.297961)[&length_median=7.132466,height_range={12.629187000000002,26.20334400000001},height_median=16.842043999999994,height=17.29796224624626,length_95%_HPD={3.851895,10.781364},length=7.3603237087087034,posterior=1.0,length_range={3.764542,14.95167},height_95%_HPD={12.629187000000002,22.085973000000017}]:7.727964,(24[&length_median=15.579851,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2552552645898856E-6,length_95%_HPD={10.734926,21.360135},length=15.804177993993996,length_range={9.234487,25.753713},height_95%_HPD={0.0,2.9999999782148734E-6}]:15.828779,(34[&length_median=9.248909,height_range={0.0,3.999999989900971E-6},height_median=1.0000000116860974E-6,height=1.2432432524801751E-6,length_95%_HPD={7.072803,12.029173},length=9.394892042042045,length_range={6.678304,13.503746},height_95%_HPD={0.0,2.999999992425728E-6}]:9.401658,35[&length_median=9.240669,height_range={0.0,3.999999989900971E-6},height_median=1.0000000116860974E-6,height=1.2402402494420788E-6,length_95%_HPD={7.072803,12.029173},length=9.383823936936944,length_range={6.678304,13.985812},height_95%_HPD={0.0,2.999999992425728E-6}]:9.401659)[&length_median=6.372121,height_range={6.678305999999992,13.503746000000014},height_median=9.223553000000024,height=9.327814730650166,length_95%_HPD={1.425958,10.931936},length=6.605349925696595,posterior=0.96996996996997,length_range={1.00382,15.458243},height_95%_HPD={7.072804999999988,11.968876000000009}]:6.427121)[&length_median=7.9061565,height_range={9.517326000000025,25.75371400000001},height_median=15.55100500000001,height=15.773050596969714,length_95%_HPD={2.713003,12.749878},length=8.072989851515146,posterior=0.990990990990991,length_range={2.254407,17.536993},height_95%_HPD={10.987251000000015,21.36013600000002}]:9.197146)[&length_median=1.409853,height_range={20.10705800000001,34.040315000000014},height_median=22.196004000000013,height=22.87522220833335,length_95%_HPD={0.121106,3.505076},length=1.5903898437499997,posterior=0.2882882882882883,length_range={0.121106,5.276593},height_95%_HPD={20.10705800000001,26.18479100000001}]:0.813902,9[&length_median=24.568155,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.318318327024302E-6,length_95%_HPD={19.684072,32.619484},length=24.972548357357354,length_range={15.387162,39.707946},height_95%_HPD={0.0,2.000000023372195E-6}]:25.839827)[&length_median=1.0689265,height_range={20.264133000000015,35.22867099999999},height_median=23.556480000000004,height=24.18135771311475,length_95%_HPD={0.113493,2.633245},length=1.2346803278688525,posterior=0.3663663663663664,length_range={0.027745,4.546974},height_95%_HPD={20.959167000000015,29.412799000000007}]:0.452345,4[&length_median=24.09918,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2642642731493965E-6,length_95%_HPD={17.340375,32.837346},length=24.552880969969983,length_range={15.387162,39.715636},height_95%_HPD={1.4210854715202004E-14,2.000000023372195E-6}]:26.292172)[&length_median=7.857903,height_range={20.247574000000014,39.921849000000016},height_median=25.717443000000017,height=26.292173228228236,length_95%_HPD={5.503031,11.466947},length=7.91393929429429,posterior=1.0,length_range={3.194483,11.466947},height_95%_HPD={21.775000000000006,33.22897400000001}]:8.624694)[&length_median=16.890777,height_range={30.393468,43.116332000000014},height_median=34.592213000000015,height=34.9168674834835,length_95%_HPD={12.068906,19.581914},length=16.660936981981973,posterior=1.0,length_range={8.483669,24.157571},height_95%_HPD={32.09200200000001,39.53109600000001}]:16.705893)[&length_median=25.9,fossil_age={51.6,51.6},fossil_name=F_8,height_range={51.600000000000016,59.17870500000001},height_median=51.60000100000001,height=51.622760105105215,fossil_id=8,length_95%_HPD={25.9,25.9},length=25.877241132131974,posterior=1.0,length_range={18.321297,25.9},height_95%_HPD={51.600001,51.60000200000001}]:25.877241,((2[&length_median=17.87821,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2132132215871085E-6,length_95%_HPD={12.033989,25.254569},length=18.227399780780786,length_range={10.509087,28.493581},height_95%_HPD={0.0,2.00000000916134E-6}]:18.227400,3[&length_median=17.87821,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2132132215871085E-6,length_95%_HPD={12.033989,25.254569},length=18.227399780780786,length_range={10.509087,28.493581},height_95%_HPD={0.0,2.00000000916134E-6}]:18.227400)[&length_median=43.090784,height_range={10.509089000000003,28.493581999999996},height_median=17.878211000000007,height=18.227400993994003,length_95%_HPD={33.278427,48.153651},length=42.259125228228235,posterior=1.0,length_range={32.052478,51.560799},height_95%_HPD={12.033991999999984,25.254569000000018}]:42.259125,((((((36[&length_median=5.816427,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2642642719118147E-6,length_95%_HPD={4.549293,7.193623},length=5.880679744744744,length_range={4.092625,8.798442},height_95%_HPD={0.0,2.999999992425728E-6}]:5.880680,37[&length_median=5.816427,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2642642719118147E-6,length_95%_HPD={4.549293,7.193623},length=5.880679744744744,length_range={4.092625,8.798442},height_95%_HPD={0.0,2.999999992425728E-6}]:5.880680)[&length_median=12.918091,height_range={4.092627000000007,8.798444000000003},height_median=5.816428000000016,height=5.880681009009015,length_95%_HPD={11.026498,15.867712},length=13.160169642642634,posterior=1.0,length_range={10.525336,20.414285},height_95%_HPD={4.5492940000000175,7.193624000000028}]:13.160170,(38[&length_median=3.465325,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2342342417869023E-6,length_95%_HPD={2.696141,4.331587},length=3.4814678468468436,length_range={2.449604,5.026872},height_95%_HPD={0.0,2.9999999782148734E-6}]:3.481468,39[&length_median=3.465325,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2342342417869023E-6,length_95%_HPD={2.696141,4.331587},length=3.4814678468468436,length_range={2.449604,5.026872},height_95%_HPD={0.0,2.9999999782148734E-6}]:3.481468)[&length_median=15.332894,height_range={2.4496050000000054,5.026873000000009},height_median=3.4653260000000046,height=3.4814690810810895,length_95%_HPD={13.552863,18.861887},length=15.559381570570556,posterior=1.0,length_range={12.374897,23.581985},height_95%_HPD={2.696142000000009,4.331588000000011}]:15.559382)[&length_median=5.94345,height_range={16.619602,27.429929},height_median=18.783637000000013,height=19.04085065165165,length_95%_HPD={4.192227,8.186383},length=6.1320009309309285,posterior=1.0,length_range={3.899806,11.48878},height_95%_HPD={16.619602,22.102293000000017}]:6.145587,(((((91[&length_median=3.694374,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2732732818369335E-6,length_95%_HPD={2.590148,4.955694},length=3.700859102102099,length_range={2.153025,5.787711},height_95%_HPD={0.0,3.000000006636583E-6}]:3.700859,92[&length_median=3.694374,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2732732818369335E-6,length_95%_HPD={2.590148,4.955694},length=3.700859102102099,length_range={2.153025,5.787711},height_95%_HPD={0.0,3.000000006636583E-6}]:3.700859)[&length_median=8.296772,height_range={2.1530260000000254,5.787711000000002},height_median=3.694375000000008,height=3.7008603753753846,length_95%_HPD={6.22216,10.096517},length=8.29435014414415,posterior=1.0,length_range={5.777047,11.142357},height_95%_HPD={2.5901480000000134,4.955695000000006}]:8.294350,82[&length_median=12.021954,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2432432517973713E-6,length_95%_HPD={9.970334,13.730374},length=11.979380909909901,length_range={5.952154,15.373524},height_95%_HPD={0.0,3.000000006636583E-6}]:11.995209)[&length_median=2.3423365,height_range={9.745587999999998,15.373526000000012},height_median=12.02298900000001,height=11.997536448795195,length_95%_HPD={1.354009,3.246206},length=2.3717994487951795,posterior=0.996996996996997,length_range={1.189467,4.176937},height_95%_HPD={9.921112000000008,13.631478000000016}]:2.375543,(83[&length_median=0.803765,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.258258266048998E-6,length_95%_HPD={0.22512,1.486044},length=0.878406057057057,length_range={0.179596,3.22465},height_95%_HPD={0.0,2.999999992425728E-6}]:0.878406,84[&length_median=0.803765,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.258258266048998E-6,length_95%_HPD={0.22512,1.486044},length=0.878406057057057,length_range={0.179596,3.22465},height_95%_HPD={0.0,2.999999992425728E-6}]:0.878406)[&length_median=13.419747,height_range={0.17959899999999607,3.224650000000011},height_median=0.8037670000000077,height=0.8784073153153231,length_95%_HPD={11.620808,15.510994},length=13.492346105105094,posterior=1.0,length_range={10.804176,17.866901},height_95%_HPD={0.22512300000001062,1.4860440000000068}]:13.492346)[&length_median=0.876573,height_range={11.530569999999997,18.541952000000016},height_median=14.309810000000002,height=14.369335897590366,length_95%_HPD={0.227827,1.692073},length=0.9261309246987951,posterior=0.996996996996997,length_range={0.227827,2.415008},height_95%_HPD={12.548338000000001,16.547755000000016}]:0.925231,(67[&length_median=7.000322,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.309309317056508E-6,length_95%_HPD={4.849983,9.342259},length=7.049513744744746,length_range={3.917131,10.483465},height_95%_HPD={0.0,3.000000006636583E-6}]:7.049514,68[&length_median=7.000322,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.309309317056508E-6,length_95%_HPD={4.849983,9.342259},length=7.049513744744746,length_range={3.917131,10.483465},height_95%_HPD={0.0,3.000000006636583E-6}]:7.049514)[&length_median=8.233538,height_range={3.9171329999999926,10.483467000000019},height_median=7.00032400000002,height=7.049515054054058,length_95%_HPD={5.05629,10.958084},length=8.246469186186179,posterior=1.0,length_range={4.131619,14.309818},height_95%_HPD={4.849983999999992,9.34226000000001}]:8.246469)[&length_median=8.6207745,height_range={12.428145,20.026145000000014},height_median=15.227849999999997,height=15.295466822289153,length_95%_HPD={6.22214,13.14119},length=9.102746915662642,posterior=0.996996996996997,length_range={6.22214,20.725039},height_95%_HPD={13.326351999999986,17.535302000000016}]:9.107481,(40[&length_median=9.737014,height_range={0.0,3.999999989900971E-6},height_median=1.0000000116860974E-6,height=1.2942943020794023E-6,length_95%_HPD={8.011687,11.520285},length=9.798047567567568,length_range={7.229218,13.14972},height_95%_HPD={0.0,2.000000023372195E-6}]:9.798048,41[&length_median=9.737014,height_range={0.0,3.999999989900971E-6},height_median=1.0000000116860974E-6,height=1.2942943020794023E-6,length_95%_HPD={8.011687,11.520285},length=9.798047567567568,length_range={7.229218,13.14972},height_95%_HPD={0.0,2.000000023372195E-6}]:9.798048)[&length_median=14.083678,height_range={7.229219000000015,13.149721000000014},height_median=9.737015000000014,height=9.798048861861863,length_95%_HPD={11.670233,19.323039},length=14.598714774774786,posterior=1.0,length_range={11.205725,26.697226},height_95%_HPD={8.011689000000018,11.520286000000013}]:14.605416)[&length_median=0.8241935,height_range={20.894697999999998,34.58504599999998},height_median=23.693450500000004,height=23.863981639610415,length_95%_HPD={0.133319,1.52281},length=0.8452584967532467,posterior=0.924924924924925,length_range={0.030729,2.593239},height_95%_HPD={21.640679999999996,26.418324000000027}]:0.782973)[&length_median=1.0485955,height_range={22.243303999999995,38.934188999999996},height_median=24.568032499999997,height=25.189274789156638,length_95%_HPD={0.570101,1.514049},length=1.0604385210843381,posterior=0.996996996996997,length_range={0.446598,2.020776},height_95%_HPD={22.243303999999995,30.209270999999994}]:1.061655,(((42[&length_median=6.881251,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2162162231315715E-6,length_95%_HPD={5.550708,8.042043},length=6.897635930930935,length_range={5.093039,10.081699},height_95%_HPD={0.0,2.000000023372195E-6}]:6.897636,43[&length_median=6.881251,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2162162231315715E-6,length_95%_HPD={5.550708,8.042043},length=6.897635930930935,length_range={5.093039,10.081699},height_95%_HPD={0.0,2.000000023372195E-6}]:6.897636)[&length_median=3.681144,height_range={5.093041000000014,10.081700000000012},height_median=6.881253000000015,height=6.89763714714716,length_95%_HPD={1.676615,5.609196},length=3.73750577777778,posterior=1.0,length_range={0.997371,7.499085},height_95%_HPD={5.550709999999995,8.042044000000004}]:3.737506,(44[&length_median=2.666759,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2222222294211405E-6,length_95%_HPD={1.745071,3.531646},length=2.6739595855855867,length_range={1.67645,4.116367},height_95%_HPD={0.0,2.000000023372195E-6}]:2.673960,45[&length_median=2.666759,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2222222294211405E-6,length_95%_HPD={1.745071,3.531646},length=2.6739595855855867,length_range={1.67645,4.116367},height_95%_HPD={0.0,2.000000023372195E-6}]:2.673960)[&length_median=7.958119,height_range={1.676450999999986,4.116368999999992},height_median=2.6667599999999965,height=2.6739608078078154,length_95%_HPD={6.211131,9.902241},length=7.961182117117116,posterior=1.0,length_range={5.549876,11.277772},height_95%_HPD={1.7450709999999958,3.5316480000000325}]:7.961182)[&length_median=7.038609,height_range={8.493421000000012,13.493960000000015},height_median=10.61935099999998,height=10.635142924924935,length_95%_HPD={4.543501,9.201199},length=7.155007780780783,posterior=1.0,length_range={3.917194,12.74202},height_95%_HPD={8.837670000000017,12.448154000000017}]:7.155008,(25[&length_median=5.538216,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2192192275352756E-6,length_95%_HPD={4.364675,6.697666},length=5.5386281351351325,length_range={3.832445,7.37186},height_95%_HPD={0.0,2.000000023372195E-6}]:5.538628,26[&length_median=5.538216,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2192192275352756E-6,length_95%_HPD={4.364675,6.697666},length=5.5386281351351325,length_range={3.832445,7.37186},height_95%_HPD={0.0,2.000000023372195E-6}]:5.538628)[&length_median=12.063674,height_range={3.8324460000000045,7.371861999999993},height_median=5.53821600000002,height=5.538629354354365,length_95%_HPD={9.503261,14.834012},length=12.25152135135135,posterior=1.0,length_range={8.966723,17.83301},height_95%_HPD={4.364676000000017,6.69766700000001}]:12.251521)[&length_median=8.145078,height_range={14.615151000000004,23.839517000000015},height_median=17.632122000000017,height=17.79015070570572,length_95%_HPD={5.674335,11.279786},length=8.433465255255259,posterior=1.0,length_range={5.304715,18.262013},height_95%_HPD={15.877443000000014,20.691474999999997}]:8.457942)[&length_median=1.5598405,height_range={23.400667000000006,39.685153},height_median=25.634752000000006,height=26.251884875757593,length_95%_HPD={0.63839,2.170478},length=1.5159701090909086,posterior=0.990990990990991,length_range={0.162033,2.751553},height_95%_HPD={23.400667000000006,31.103926000000016}]:1.631579,((15[&length_median=12.853367,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2132132211603561E-6,length_95%_HPD={6.370254,28.10473},length=15.66915757357358,length_range={5.952154,33.2464},height_95%_HPD={1.4210854715202004E-14,2.000000023372195E-6}]:16.488219,(((69[&length_median=3.286417,height_range={0.0,3.999999989900971E-6},height_median=1.0000000116860974E-6,height=1.2462462538539373E-6,length_95%_HPD={0.898436,5.576629},length=3.3100853873873897,length_range={0.831472,8.788183},height_95%_HPD={0.0,2.000000023372195E-6}]:3.585110,70[&length_median=3.064025,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2282282352412818E-6,length_95%_HPD={0.831472,5.745518},length=3.2942555795795805,length_range={0.831472,8.788183},height_95%_HPD={0.0,2.000000023372195E-6}]:3.585110)[&length_median=1.5694905,height_range={0.8314739999999858,8.788184000000001},height_median=2.5680870000000127,height=2.825051366666674,length_95%_HPD={0.020298,3.226707},length=1.6928630000000002,posterior=0.5405405405405406,length_range={0.020298,5.227839},height_95%_HPD={0.898438000000013,5.064782000000008}]:0.915061,46[&length_median=3.90639,height_range={0.0,3.999999989900971E-6},height_median=1.0000000116860974E-6,height=1.234234241914928E-6,length_95%_HPD={1.29324,6.12789},length=3.9342917177177217,length_range={1.209943,8.851448},height_95%_HPD={0.0,2.000000023372195E-6}]:4.500171)[&length_median=6.968545,height_range={1.320993999999999,8.85145},height_median=4.392111000000014,height=4.500172651651663,length_95%_HPD={4.338442,11.439512},length=7.297306120120116,posterior=1.0,length_range={2.646718,12.33753},height_95%_HPD={2.4362239999999957,6.375271999999981}]:7.660518,(47[&length_median=2.819438,height_range={0.0,3.0000000350582923E-6},height_median=1.0000000116860974E-6,height=1.2072072144013595E-6,length_95%_HPD={1.463637,4.382361},length=2.857050426426425,length_range={0.763298,4.667317},height_95%_HPD={0.0,2.000000023372195E-6}]:2.857050,48[&length_median=2.819438,height_range={0.0,3.0000000350582923E-6},height_median=1.0000000116860974E-6,height=1.2072072144013595E-6,length_95%_HPD={1.463637,4.382361},length=2.857050426426425,length_range={0.763298,4.667317},height_95%_HPD={0.0,2.000000023372195E-6}]:2.857050)[&length_median=8.938886,height_range={0.763298000000006,4.667318000000009},height_median=2.819439000000017,height=2.8570516336336413,length_95%_HPD={6.325947,12.788278},length=9.086504423423426,posterior=1.0,length_range={4.620517,14.231998},height_95%_HPD={1.4636380000000173,4.3823620000000005}]:9.303639)[&length_median=2.872942,height_range={8.457872000000009,17.10961300000001},height_median=12.169183999999987,height=12.363683267206492,length_95%_HPD={0.006546,9.887463},length=4.278710372469634,posterior=0.7417417417417418,length_range={0.006546,11.244296},height_95%_HPD={9.848696000000004,15.738866000000009}]:4.327530)[&length_median=10.033088,height_range={8.46691700000001,19.087573000000013},height_median=12.231077999999997,height=12.59165108713693,length_95%_HPD={4.357007,13.277995},length=9.681501946058093,posterior=0.7237237237237237,length_range={0.995931,15.128644},height_95%_HPD={9.848331000000002,17.484539000000012}]:7.006733,(16[&length_median=12.28013,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2762762838934993E-6,length_95%_HPD={10.323933,14.820829},length=12.417957918918916,length_range={9.135744,16.865446},height_95%_HPD={0.0,2.000000023372195E-6}]:12.417958,17[&length_median=12.28013,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2762762838934993E-6,length_95%_HPD={10.323933,14.820829},length=12.417957918918916,length_range={9.135744,16.865446},height_95%_HPD={0.0,2.000000023372195E-6}]:12.417958)[&length_median=10.006315,height_range={9.135745000000014,16.86544700000001},height_median=12.280131000000011,height=12.417959195195197,length_95%_HPD={6.949587,12.360797},length=9.881334930930937,posterior=1.0,length_range={5.050623,15.741699},height_95%_HPD={10.323934000000008,14.82083}]:11.076994)[&length_median=5.3878509999999995,height_range={18.055194000000014,29.82876500000001},height_median=22.22509100000001,height=22.425018723880594,length_95%_HPD={0.350498,8.347814},length=5.448176507462686,posterior=0.8048048048048048,length_range={0.178558,13.438561},height_95%_HPD={19.736056000000012,26.254708999999977}]:4.384719)[&length_median=22.488426,height_range={24.458217999999995,40.343089},height_median=27.311575999999988,height=27.879671543543576,length_95%_HPD={17.462397,24.836859},length=21.920329693693688,posterior=1.0,length_range={9.456912,25.341784},height_95%_HPD={24.963141999999998,32.33760400000003}]:21.920330,(((((49[&length_median=3.179947,height_range={0.0,3.0000000350582923E-6},height_median=1.0000000116860974E-6,height=1.2522522578390433E-6,length_95%_HPD={2.470098,3.872958},length=3.196057114114117,length_range={2.115892,4.351424},height_95%_HPD={0.0,2.9999999640040187E-6}]:3.196057,50[&length_median=3.179947,height_range={0.0,3.0000000350582923E-6},height_median=1.0000000116860974E-6,height=1.2522522578390433E-6,length_95%_HPD={2.470098,3.872958},length=3.196057114114117,length_range={2.115892,4.351424},height_95%_HPD={0.0,2.9999999640040187E-6}]:3.196057)[&length_median=4.96068,height_range={2.115893,4.35142500000002},height_median=3.1799490000000077,height=3.196058366366372,length_95%_HPD={3.822092,6.291562},length=5.027059495495495,posterior=1.0,length_range={3.186137,8.311223},height_95%_HPD={2.470100000000002,3.8729580000000254}]:5.194007,(51[&length_median=7.357726,height_range={0.0,3.0000000350582923E-6},height_median=1.0000000116860974E-6,height=1.2492492556544516E-6,length_95%_HPD={6.244225,8.840505},length=7.403799600600605,length_range={5.525946,9.737542},height_95%_HPD={0.0,2.000000023372195E-6}]:7.410270,(71[&length_median=1.645528,height_range={0.0,3.0000000350582923E-6},height_median=1.0000000116860974E-6,height=1.2342342408907221E-6,length_95%_HPD={1.188333,2.27201},length=1.6798998558558558,length_range={0.964477,2.59716},height_95%_HPD={0.0,2.0000000375830496E-6}]:1.679900,72[&length_median=1.645528,height_range={0.0,3.0000000350582923E-6},height_median=1.0000000116860974E-6,height=1.2342342408907221E-6,length_95%_HPD={1.188333,2.27201},length=1.6798998558558558,length_range={0.964477,2.59716},height_95%_HPD={0.0,2.0000000375830496E-6}]:1.679900)[&length_median=5.537851,height_range={0.964478999999983,2.5971610000000283},height_median=1.6455290000000247,height=1.6799010900900955,length_95%_HPD={4.31179,6.97168},length=5.582610111111109,posterior=1.0,length_range={3.707748,7.803347},height_95%_HPD={1.1883350000000092,2.2720110000000204}]:5.730370)[&length_median=1.407243,height_range={5.525947000000002,8.790706000000014},height_median=6.994918000000013,height=7.0336613920704885,length_95%_HPD={0.149836,2.536502},length=1.4341039339207047,posterior=0.6816816816816816,length_range={0.139603,3.505039},height_95%_HPD={6.062606000000017,7.9767109999999946}]:0.979794)[&length_median=2.133627,height_range={7.166712000000004,12.29574500000001},height_median=8.260442000000012,height=8.310114782334388,length_95%_HPD={0.841524,3.514775},length=2.111006132492113,posterior=0.9519519519519519,length_range={0.117421,3.962334},height_95%_HPD={7.3577279999999945,9.248685000000009}]:2.009576,18[&length_median=10.350965,height_range={0.0,2.000000023372195E-6},height_median=1.0000000116860974E-6,height=1.237237244014169E-6,length_95%_HPD={9.087071,11.791267},length=10.384732645645634,length_range={8.243453,14.518473},height_95%_HPD={9.999999974752427E-7,2.00000000916134E-6}]:10.399640)[&length_median=8.414984,height_range={8.302016000000009,14.518474000000012},height_median=10.385017000000005,height=10.399641540540554,length_95%_HPD={7.008733,9.659853},length=8.474949705705704,posterior=1.0,length_range={6.418198,20.309151},height_95%_HPD={9.096962000000019,11.738891999999993}]:8.484444,((27[&height_range={1.4210854715202004E-14,2.000000023372195E-6},height_median=1.0000000116860974E-6,height=1.237237245593153E-6,length=6.299999999999973,height_95%_HPD={9.999999974752427E-7,2.00000000916134E-6}]:6.300000,((73[&length_median=2.636784,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.189189194551122E-6,length_95%_HPD={2.1071,3.079505},length=2.6337435285285276,length_range={1.904352,3.391097},height_95%_HPD={0.0,2.00000000916134E-6}]:2.633744,74[&length_median=2.636784,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.189189194551122E-6,length_95%_HPD={2.1071,3.079505},length=2.6337435285285276,length_range={1.904352,3.391097},height_95%_HPD={0.0,2.00000000916134E-6}]:2.633744)[&length_median=2.884733,height_range={1.904354000000012,3.3910980000000137},height_median=2.6367850000000033,height=2.633744717717722,length_95%_HPD={2.352783,3.439496},length=2.885248987987987,posterior=1.0,length_range={2.053792,3.574561},height_95%_HPD={2.1071019999999976,3.079505999999995}]:2.885249,52[&length_median=5.531599,height_range={0.0,2.000000023372195E-6},height_median=1.0000000116860974E-6,height=1.2372372428192623E-6,length_95%_HPD={5.172888,5.897939},length=5.518992468468473,length_range={4.972852,5.947241},height_95%_HPD={9.999999974752427E-7,2.00000000916134E-6}]:5.518992)[&length_median=0.768401,height_range={4.972853000000015,5.947242000000003},height_median=5.531600000000012,height=5.518993705705713,length_95%_HPD={0.402061,1.127112},length=0.7810075315315318,posterior=1.0,length_range={0.352759,1.327148},height_95%_HPD={5.172889000000012,5.8979400000000055}]:0.781008)[&length_median=8.9,fossil_age={6.3,6.3},fossil_name=F_3,height_range={6.300000000000011,6.3000020000000205},height_median=6.300001000000009,height=6.3000012372372645,fossil_id=3,length_95%_HPD={8.9,8.9},length=8.921621621621675,posterior=1.0,length_range={8.9,12.5},height_95%_HPD={6.300000999999995,6.300002000000006}]:8.921622,19[&length_median=15.2,height_range={1.4210854715202004E-14,2.000000023372195E-6},height_median=1.0000000116860974E-6,height=1.237237245593153E-6,length_95%_HPD={15.2,15.2},length=15.212127546546451,length_range={15.2,17.376355},height_95%_HPD={9.999999974752427E-7,2.00000000916134E-6}]:15.221622)[&length_median=3.6,fossil_age={15.2,15.2},fossil_name=F_4,height_range={15.20000000000001,15.200002000000019},height_median=15.200001000000007,height=15.200001238670747,fossil_id=4,length_95%_HPD={3.6,3.6},length=3.6845921450151056,posterior=0.993993993993994,length_range={3.6,17.6},height_95%_HPD={15.200000999999993,15.200002000000005}]:3.662462)[&length_median=14.0,fossil_age={18.8,18.8},fossil_name=F_5,height_range={18.80000000000001,32.800001000000016},height_median=18.80000100000001,height=18.88408532132123,fossil_id=5,length_95%_HPD={14.0,14.0},length=13.957356936936938,posterior=1.0,length_range={6.814784,14.0},height_95%_HPD={18.800000999999995,18.800002000000006}]:13.957357,((((53[&length_median=6.336392,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.1651651701610004E-6,length_95%_HPD={5.398624,7.442761},length=6.345156177177175,length_range={5.020915,8.025732},height_95%_HPD={0.0,2.00000000916134E-6}]:6.345156,54[&length_median=6.336392,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.1651651701610004E-6,length_95%_HPD={5.398624,7.442761},length=6.345156177177175,length_range={5.020915,8.025732},height_95%_HPD={0.0,2.00000000916134E-6}]:6.345156)[&length_median=4.723158,height_range={5.0209159999999855,8.025734000000014},height_median=6.336393000000015,height=6.345157342342349,length_95%_HPD={3.029061,5.749296},length=4.640939627627628,posterior=1.0,length_range={2.081862,6.823847},height_95%_HPD={5.398624000000012,7.442761000000004}]:4.917328,(55[&length_median=9.441427,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2072072135905298E-6,length_95%_HPD={8.124883,11.085941},length=9.457476699699695,length_range={7.11322,12.32331},height_95%_HPD={0.0,2.00000000916134E-6}]:9.733865,56[&length_median=9.601071,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.207207213419829E-6,length_95%_HPD={8.059844,11.793675},length=9.733865054054048,length_range={7.11322,15.116624},height_95%_HPD={0.0,2.00000000916134E-6}]:9.733865)[&length_median=1.908622,height_range={7.11322100000001,11.274975000000026},height_median=9.259842500000019,height=9.306920636363644,length_95%_HPD={0.482565,3.158951},length=1.9281445000000001,posterior=0.7927927927927928,length_range={0.482565,4.074756},height_95%_HPD={8.022650999999982,10.774874000000011}]:1.528619)[&length_median=1.628737,height_range={9.88926600000002,15.116625000000006},height_median=11.228654000000006,height=11.262485324324334,length_95%_HPD={0.996148,2.423383},length=1.66272208108108,posterior=1.0,length_range={0.627199,3.389095},height_95%_HPD={10.188652000000005,12.194503000000012}]:1.662722,((((85[&length_median=1.868387,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2072072115421185E-6,length_95%_HPD={0.893735,2.971259},length=1.9202113723723726,length_range={0.859506,3.926271},height_95%_HPD={0.0,2.9999999782148734E-6}]:1.920211,86[&length_median=1.868387,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2072072115421185E-6,length_95%_HPD={0.893735,2.971259},length=1.9202113723723726,length_range={0.859506,3.926271},height_95%_HPD={0.0,2.9999999782148734E-6}]:1.920211)[&length_median=4.047572,height_range={0.8595080000000053,3.9262719999999973},height_median=1.868387999999996,height=1.9202125795795826,length_95%_HPD={2.429873,5.300402},length=3.954126132132131,posterior=1.0,length_range={1.850223,5.563245},height_95%_HPD={0.8937360000000041,2.971260000000001}]:3.954126,75[&length_median=5.90203,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.243243247615198E-6,length_95%_HPD={4.992586,6.774103},length=5.87433746846847,length_range={4.443997,7.508054},height_95%_HPD={0.0,2.9999999782148734E-6}]:5.874337)[&length_median=3.797893,height_range={4.443998000000008,7.508055999999996},height_median=5.902030999999994,height=5.874338711711717,length_95%_HPD={2.844258,4.708881},length=3.7927591891891907,posterior=1.0,length_range={2.522938,5.498476},height_95%_HPD={4.992587,6.774103999999994}]:3.792759,(((93[&length_median=2.179576,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2732732785082647E-6,length_95%_HPD={1.664615,3.068338},length=2.236578573573575,length_range={1.276078,3.667618},height_95%_HPD={0.0,2.999999992425728E-6}]:2.236579,94[&length_median=2.179576,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2732732785082647E-6,length_95%_HPD={1.664615,3.068338},length=2.236578573573575,length_range={1.276078,3.667618},height_95%_HPD={0.0,2.999999992425728E-6}]:2.236579)[&length_median=5.79203,height_range={1.2760800000000074,3.6676180000000045},height_median=2.179577000000023,height=2.236579846846852,length_95%_HPD={4.787665,6.639451},length=5.77928906906907,posterior=1.0,length_range={4.564227,7.110435},height_95%_HPD={1.6646169999999927,3.0683380000000113}]:5.910606,(95[&length_median=3.956203,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2792792849258593E-6,length_95%_HPD={3.142411,4.70703},length=3.997027570570573,length_range={3.032821,5.612654},height_95%_HPD={0.0,2.999999992425728E-6}]:3.997028,96[&length_median=3.956203,height_range={0.0,4.0000000041118255E-6},height_median=1.0000000116860974E-6,height=1.2792792849258593E-6,length_95%_HPD={3.142411,4.70703},length=3.997027570570573,length_range={3.032821,5.612654},height_95%_HPD={0.0,2.999999992425728E-6}]:3.997028)[&length_median=4.095814,height_range={3.032821999999996,5.612656000000001},height_median=3.9562039999999854,height=3.997028849849855,length_95%_HPD={3.259403,5.228502},length=4.150157201201202,posterior=1.0,length_range={2.663308,5.637974},height_95%_HPD={3.1424120000000073,4.707031000000015}]:4.150157)[&length_median=0.796043,height_range={6.913169999999994,9.848318000000006},height_median=8.029284000000004,height=8.046157606779664,length_95%_HPD={0.421444,1.189049},length=0.8045417966101697,posterior=0.8858858858858859,length_range={0.255314,1.419379},height_95%_HPD={7.112973999999994,8.73286800000001}]:0.712732,76[&length_median=8.772204,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2552552601943358E-6,length_95%_HPD={7.384961,9.597316},length=8.728599882882884,length_range={6.999604,10.45683},height_95%_HPD={0.0,2.999999992425728E-6}]:8.859917)[&length_median=0.790887,height_range={7.491201000000004,10.488048000000006},height_median=8.837272000000013,height=8.859918273273278,length_95%_HPD={0.401143,1.216654},length=0.8071796276276278,posterior=1.0,length_range={0.360736,1.608138},height_95%_HPD={7.94042300000001,9.59731699999999}]:0.807180)[&length_median=0.471495,height_range={8.483479000000003,11.650794000000005},height_median=9.665729000000013,height=9.667097900900904,length_95%_HPD={0.21714,0.922469},length=0.5009849939939943,posterior=1.0,length_range={0.115658,1.045172},height_95%_HPD={8.877166000000003,10.397734000000014}]:0.500985,(57[&length_median=3.152169,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2792792843710813E-6,length_95%_HPD={1.631044,4.706616},length=3.187609564564564,length_range={1.570822,5.825346},height_95%_HPD={0.0,2.000000023372195E-6}]:3.187610,58[&length_median=3.152169,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2792792843710813E-6,length_95%_HPD={1.631044,4.706616},length=3.187609564564564,length_range={1.570822,5.825346},height_95%_HPD={0.0,2.000000023372195E-6}]:3.187610)[&length_median=7.024719,height_range={1.5708240000000018,5.82534600000001},height_median=3.1521690000000007,height=3.187610843843847,length_95%_HPD={5.007126,8.659133},length=6.980472051051046,posterior=1.0,length_range={4.362588,9.062453},height_95%_HPD={1.6310459999999978,4.7066170000000085}]:6.980472)[&length_median=2.705122,height_range={9.108443000000008,12.494961000000004},height_median=10.160729000000018,height=10.168082894894903,length_95%_HPD={2.015256,3.398441},length=2.7571245105105104,posterior=1.0,length_range={1.789352,5.605615},height_95%_HPD={9.39555700000001,10.956888000000006}]:2.757125)[&length_median=3.965536,height_range={11.749760000000009,16.800002000000006},height_median=12.864456000000004,height=12.925207405405418,length_95%_HPD={3.239648,4.74413},length=4.017842894894895,posterior=1.0,length_range={3.007423,9.765685},height_95%_HPD={12.05587100000001,13.560354000000004}]:4.017843,((((77[&length_median=8.860519,height_range={0.0,3.999999989900971E-6},height_median=1.0000000116860974E-6,height=1.2522522580097444E-6,length_95%_HPD={7.070076,10.222881},length=8.751972213213218,length_range={4.353127,11.012681},height_95%_HPD={0.0,2.9999999782148734E-6}]:8.776755,((97[&length_median=4.672305,height_range={0.0,3.0000000350582923E-6},height_median=1.0000000116860974E-6,height=1.3093093146666948E-6,length_95%_HPD={3.948621,5.522584},length=4.680733180180179,length_range={3.377256,5.737387},height_95%_HPD={0.0,2.999999992425728E-6}]:4.680733,98[&length_median=4.672305,height_range={0.0,3.0000000350582923E-6},height_median=1.0000000116860974E-6,height=1.3093093146666948E-6,length_95%_HPD={3.948621,5.522584},length=4.680733180180179,length_range={3.377256,5.737387},height_95%_HPD={0.0,2.999999992425728E-6}]:4.680733)[&length_median=1.852957,height_range={3.3772560000000027,5.73738800000001},height_median=4.67230600000002,height=4.680734489489498,length_95%_HPD={1.268999,2.543029},length=1.8954175975975978,posterior=1.0,length_range={1.027269,3.812855},height_95%_HPD={3.9486229999999978,5.5225850000000065}]:1.895861,87[&length_median=6.527922,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.243243248682079E-6,length_95%_HPD={5.49244,7.378176},length=6.552254432432432,length_range={4.353127,9.362512},height_95%_HPD={0.0,2.9999999782148734E-6}]:6.576594)[&length_median=2.3371579999999996,height_range={5.3404859999999985,8.284379000000001},height_median=6.527385000000002,height=6.537988819875782,length_95%_HPD={1.001808,3.65709},length=2.275321826086957,posterior=0.9669669669669669,length_range={0.009622,3.65709},height_95%_HPD={5.769121999999996,7.378177000000008}]:2.200161)[&length_median=2.595493,height_range={5.984994999999998,11.01268300000001},height_median=8.860518999999996,height=8.77675638738739,length_95%_HPD={1.246045,4.518222},length=2.7213923273273286,posterior=1.0,length_range={0.81556,6.054253},height_95%_HPD={7.070077000000012,10.111502000000002}]:2.721392,((88[&length_median=6.772271,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.228228233491597E-6,length_95%_HPD={5.888559,7.647164},length=6.7707116816816795,length_range={5.635144,9.045523},height_95%_HPD={0.0,2.000000023372195E-6}]:7.518037,89[&length_median=7.626547,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2282282334062465E-6,length_95%_HPD={6.079623,9.021344},length=7.518036510510505,length_range={5.635144,10.439591},height_95%_HPD={0.0,2.000000023372195E-6}]:7.518037)[&length_median=1.027854,height_range={5.6351450000000085,9.045524000000015},height_median=6.758786000000001,height=6.791895828947372,length_95%_HPD={0.367999,1.58779},length=1.038295032894737,posterior=0.45645645645645644,length_range={0.229585,2.170334},height_95%_HPD={5.7444400000000115,7.691615000000013}]:0.473936,78[&length_median=7.198362,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2312312363589924E-6,length_95%_HPD={6.036311,8.561752},length=7.244648150150149,length_range={5.697782,9.995727},height_95%_HPD={0.0,2.000000023372195E-6}]:7.991973)[&length_median=3.497151,height_range={6.796354000000008,10.439593000000002},height_median=7.948276000000021,height=7.991974210210219,length_95%_HPD={2.528842,4.447584},length=3.5061745045045067,posterior=1.0,length_range={2.082357,4.89802},height_95%_HPD={6.949826000000002,9.021346000000008}]:3.506175)[&length_median=1.196221,height_range={10.181126000000006,14.868074000000014},height_median=11.451786000000013,height=11.498148714714715,length_95%_HPD={0.626039,1.844317},length=1.2239973003003015,posterior=1.0,length_range={0.558651,2.388376},height_95%_HPD={10.395918000000009,12.37029600000001}]:1.223997,(59[&length_median=6.17369,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2462462513787734E-6,length_95%_HPD={4.378303,7.854342},length=6.171227399399397,length_range={4.161696,9.220929},height_95%_HPD={0.0,2.000000023372195E-6}]:6.171227,60[&length_median=6.17369,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2462462513787734E-6,length_95%_HPD={4.378303,7.854342},length=6.171227399399397,length_range={4.161696,9.220929},height_95%_HPD={0.0,2.000000023372195E-6}]:6.171227)[&length_median=6.597402,height_range={4.161697000000018,9.22093000000001},height_median=6.173693000000014,height=6.1712286456456535,length_95%_HPD={4.646831,8.091854},length=6.550917369369371,posterior=1.0,length_range={4.352351,10.810346},height_95%_HPD={4.378304000000014,7.854343000000014}]:6.550917)[&length_median=2.983824,height_range={11.368261000000018,16.58845300000001},height_median=12.633100999999996,height=12.722146015015015,length_95%_HPD={2.089104,3.752232},length=2.9991113303303334,posterior=1.0,length_range={1.804016,6.813266},height_95%_HPD={11.88461199999999,13.569503999999995}]:2.999541,(28[&length_median=13.431529,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2492492562092298E-6,length_95%_HPD={10.473106,15.462386},length=13.059641552552549,length_range={7.539276,20.734396},height_95%_HPD={0.0,2.00000000916134E-6}]:14.103368,(61[&length_median=12.442127,height_range={0.0,3.000000006636583E-6},height_median=1.0000000116860974E-6,height=1.2582582650674677E-6,length_95%_HPD={10.028855,15.077063},length=12.460660867867864,length_range={7.539276,19.138328},height_95%_HPD={0.0,2.000000023372195E-6}]:13.344232,(79[&length_median=11.28416,height_range={0.0,3.0000000208474376E-6},height_median=1.0000000116860974E-6,height=1.2462462530431078E-6,length_95%_HPD={9.7185,13.64386},length=11.4068708978979,length_range={7.087029,18.333756},height_95%_HPD={0.0,2.000000023372195E-6}]:11.424247,(90[&length_median=11.170343,height_range={0.0,3.999999975690116E-6},height_median=1.0000000116860974E-6,height=1.2762762821011392E-6,length_95%_HPD={9.215752,13.34312},length=11.229522774774768,length_range={7.0468,17.976566},height_95%_HPD={0.0,2.9999999640040187E-6}]:11.394549,(99[&length_median=11.240205,height_range={0.0,3.999999975690116E-6},height_median=1.0000000116860974E-6,height=1.2522522583938215E-6,length_95%_HPD={9.706848,13.747522},length=11.367487249249253,length_range={7.066881,17.976566},height_95%_HPD={0.0,2.999999992425728E-6}]:11.378817,100[&length_median=11.177128,height_range={0.0,3.999999975690116E-6},height_median=1.0000000116860974E-6,height=1.2702702762809978E-6,length_95%_HPD={9.186296,13.31255},length=11.223698126126123,length_range={7.0468,18.2033},height_95%_HPD={0.0,2.999999992425728E-6}]:11.378817)[&length_median=0.044536,height_range={9.352677,15.65573700000001},height_median=11.189538000000013,height=11.300271158878516,length_95%_HPD={0.015589,0.086895},length=0.048960093457943915,posterior=0.3213213213213213,length_range={0.010744,0.147733},height_95%_HPD={9.71069500000003,13.207143000000002}]:0.015732)[&length_median=0.0245965,height_range={7.066881000000009,18.20330100000001},height_median=11.261080500000006,height=11.361789277419374,length_95%_HPD={9.24E-4,0.083716},length=0.03190158064516129,posterior=0.9309309309309309,length_range={9.24E-4,0.235999},height_95%_HPD={9.706850000000017,13.312551999999997}]:0.029698)[&length_median=1.828997,height_range={7.087029000000015,18.333757000000013},height_median=11.284160999999983,height=11.408550902735572,length_95%_HPD={0.005191,3.081293},length=1.795019449848024,posterior=0.987987987987988,length_range={0.005191,3.646057},height_95%_HPD={9.892565999999988,13.863120000000023}]:1.919985)[&length_median=1.673999,height_range={9.086364000000003,18.60467900000002},height_median=12.811783999999989,height=12.619857408759133,length_95%_HPD={0.612684,3.297468},length=1.8451996204379573,posterior=0.4114114114114114,length_range={0.240513,4.993883},height_95%_HPD={10.822781000000006,13.824118000000006}]:0.759136)[&length_median=1.5951445,height_range={9.355614999999986,20.734397000000016},height_median=14.05427100000001,height=14.09957064759036,length_95%_HPD={0.176728,2.844487},length=1.6231919397590369,posterior=0.996996996996997,length_range={0.090946,5.016847},height_95%_HPD={12.612931000000003,15.546971000000006}]:1.618317)[&length_median=1.185742,height_range={14.372461999999992,22.874449000000013},height_median=15.634080000000012,height=15.721687096096105,length_95%_HPD={0.38758,1.918221},length=1.221363204204205,posterior=1.0,length_range={0.266542,4.297963},height_95%_HPD={14.86261900000001,16.41242100000001}]:1.221363)[&length_median=16.0,fossil_age={16.8,16.8},fossil_name=F_9,height_range={16.80000000000001,25.631966000000013},height_median=16.80000100000001,height=16.94305030030024,fossil_id=9,length_95%_HPD={16.0,16.0},length=15.898391957957958,posterior=1.0,length_range={8.887395,16.0},height_95%_HPD={16.800000999999995,16.800002000000006}]:15.898392)[&length_median=17.0,fossil_age={32.8,32.8},fossil_name=F_1,height_range={32.80000000000001,39.785077000000015},height_median=32.80000100000001,height=32.84144225825808,fossil_id=1,length_95%_HPD={17.0,17.0},length=16.958558978978978,posterior=1.0,length_range={10.014924,17.0},height_95%_HPD={32.800000999999995,32.800002000000006}]:16.958559)[&length_median=10.827755,fossil_age={49.8,49.8},fossil_name=F_6,height_range={49.80000000000001,49.80000200000002},height_median=49.80000100000001,height=49.80000123723702,fossil_id=6,length_95%_HPD={7.666797,12.962482},length=10.686524984984983,posterior=1.0,length_range={6.785077,13.71992},height_95%_HPD={49.800000999999995,49.800002000000006}]:10.686525)[&length_median=16.872245,height_range={56.585077999999996,63.51992100000001},height_median=60.62775600000001,height=60.486526222222246,length_95%_HPD={14.737518,20.033203},length=17.013475015015025,posterior=1.0,length_range={13.98008,20.914923},height_95%_HPD={57.46679800000001,62.762482999999996}]:17.013475)[&fossil_age={77.5,77.5},length=0.0,fossil_name=F_7,posterior=1.0,height_range={77.50000000000001,77.50000200000002},height_median=77.50000100000001,height=77.50000123723754,fossil_id=7,height_95%_HPD={77.500001,77.50000200000001}];
END;
BEGIN NOTES;
[! Notes block written by Bio::Phylo::Project 0.58 on Tue Aug 11 19:06:42 2015 ]
END;
