/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// local
#include "initialsurfaceplacementstrategy.h"
#include "logging.h"

// mir
#include "mir/shell/surface_creation_parameters.h"
#include "mir/geometry/rectangle.h"

namespace msh = mir::shell;
namespace msc = mir::scene;

InitialSurfacePlacementStrategy::InitialSurfacePlacementStrategy(
        const std::shared_ptr<msh::DisplayLayout> &displayLayout, QObject *parent)
  : QObject(parent)
  , m_displayLayout(displayLayout)
  , m_mainShellSurfaceFound(false)
{
    DLOG("InitialSurfacePlacementStrategy::InitialSurfacePlacementStrategy (%p)", this);
}

InitialSurfacePlacementStrategy::~InitialSurfacePlacementStrategy()
{
}

msh::SurfaceCreationParameters
InitialSurfacePlacementStrategy::place(const msh::Session &session,
                                       const msh::SurfaceCreationParameters &requestParameters)
{
    DLOG("InitialSurfacePlacementStrategy::place (%p, %s, %d)", this, session.name().c_str(), session.process_id());
    using namespace mir::geometry;
    auto placedParameters = requestParameters;

    // decide surface depth
    static const msc::DepthId shellSurfaceDepth{1};
    static const msc::DepthId oskSurfaceDepth{2};

    {   // as this method may be called by multiple threads, need to lock for m_mainShellSurfaceFound
        QMutexLocker lock(&m_mutex);

        if (session.process_id() == 0 && !m_mainShellSurfaceFound) {
            DLOG("Shell main surface found, placed at shell depth");
            m_mainShellSurfaceFound = true;
            placedParameters.depth = shellSurfaceDepth;
            placedParameters.input_mode = mir::input::InputReceptionMode::receives_all_input;
            return placedParameters;
        } else if (placedParameters.name == "MaliitOnScreenKeyboard") { // FIXME: string hardcoded into maliit
            DLOG("OSK depth");
            placedParameters.depth = oskSurfaceDepth;
        } else {
            DLOG("Default depth");
        }
    }

    // determine x,y position of the surface by asking shell, if it is up
    uint32_t x = 0;
    uint32_t y = 0;

    Q_EMIT requestPlacementForSession(&session, x, y);

    placedParameters.top_left = Point{ X{x}, Y{y} };

    return placedParameters;
}
