[comment {-*- tcl -*- doctools manpage}]
[manpage_begin des n 1.1]
[copyright {2005, Pat Thoyts <patthoyts@users.sourceforge.net>}]
[moddesc {Data Encryption Standard (DES)}]
[titledesc {Implementation of the DES and triple-DES ciphers}]
[category  {Hashes, checksums, and encryption}]
[require Tcl 8.2]
[require des 1.1]
[description]
[para]

This is an implementation in Tcl of the Data Encryption Standard (DES)
as published by the U.S. National Institute of Standards and
Technology (NIST) [lb]1[rb]. This implementation also supports triple
DES (3DES) extension to DES. DES is a 64-bit block cipher that uses a
56-bit key. 3DES uses a 168-bit key. DES has now officially been
superceeded by AES but is in common use in many protocols.

[para]

The tcllib implementation of DES and 3DES uses an implementation by
Mac Cody and is available as a separate download from [lb]2[rb]. For
anyone concerned about the details of exporting this code please see
the TclDES web pages. The tcllib specific code is a wrapper to the 
TclDES API that presents same API for the DES cipher as for other
ciphers in the library.

[section "COMMANDS"]

[list_begin definitions]
[call [cmd "::DES::des"] \
        [opt [arg "-mode [lb]ecb|cbc|cfb|ofb[rb]"]] \
        [opt [arg "-dir [lb]encrypt|decrypt[rb]"]] \
        [arg "-key keydata"] \
        [opt [arg "-iv vector"]] \
        [opt [arg "-hex"]] \
        [opt [arg "-weak"]] \
        [opt [arg "-out channel"]] \
        [opt [arg "-chunksize size"]] \
        [lb] [arg "-in channel"] | \
        [arg "data"] [rb]]

Perform the [package DES] algorithm on either the data provided
by the argument or on the data read from the [arg "-in"] channel. If
an [arg "-out"] channel is given then the result will be written to
this channel.

[para]

The [arg -key] option must be given. This parameter takes a binary
string of 8 bytes in length and is used to generate the key schedule.
In DES only 56 bits of key data are used. The highest bit from each
byte is discarded.

[para]

The [arg -mode] and [arg -dir] options are optional and default to cbc
mode and encrypt respectively. The initialization vector [arg -iv]
takes an 8 byte binary argument. This defaults to all zeros. See 
[sectref "MODES OF OPERATION"] for more about [arg -mode] and the use
of the initialization vector.

[para]

DES is a 64-bit block cipher. This means that the data must be
provided in units that are a multiple of 8 bytes.

[list_end]

[section "PROGRAMMING INTERFACE"]

Internal state is maintained in an opaque structure that is returned
from the [cmd Init] function. In ECB mode the state is not affected by
the input but for other modes some input dependent state is maintained
and may be reset by calling the [cmd Reset] function with a new
initialization vector value.

[list_begin definitions]

[call [cmd "::DES::Init"] [arg "mode"] [arg "keydata"] [arg "iv"] [opt [arg "weak"]]]

Construct a new DES key schedule using the specified key data and the
given initialization vector. The initialization vector is not used
with ECB mode but is important for other usage modes.
See [sectref "MODES OF OPERATION"].

[para]

There are a small number of keys that are known to be weak when used
with DES. By default if such a key is passed in then an error will be
raised. If there is a need to accept such keys then the [arg weak]
parameter can be set true to avoid the error being thrown.

[call [cmd "::DES::Encrypt"] [arg "Key"] [arg "data"]]

Use a prepared key acquired by calling [cmd Init] to encrypt the
provided data. The data argument should be a binary array that is a
multiple of the DES block size of 8 bytes. The result is a binary
array the same size as the input of encrypted data.

[call [cmd "::DES::Decrypt"] [arg "Key"] [arg "data"]]

Decipher data using the key. Note that the same key may be used to
encrypt and decrypt data provided that the initialization vector is
reset appropriately for CBC mode.

[call [cmd "::DES::Reset"] [arg "Key"] [arg "iv"]]

Reset the initialization vector. This permits the programmer to re-use
a key and avoid the cost of re-generating the key schedule where the
same key data is being used multiple times.

[call [cmd "::DES::Final"] [arg "Key"]]

This should be called to clean up resources associated with [arg Key].
Once this function has been called the key may not be used again.

[list_end]

[section "MODES OF OPERATION"]

[list_begin definitions]
[def "Electronic Code Book (ECB)"]
ECB is the basic mode of all block ciphers. Each block is encrypted
independently and so identical plain text will produce identical
output when encrypted with the same key. Any encryption errors will
only affect a single block however this is vulnerable to known
plaintext attacks.

[def "Cipher Block Chaining (CBC)"]

CBC mode uses the output of the last block encryption to affect the
current block. An initialization vector of the same size as the cipher
block size is used to handle the first block. The initialization
vector should be chosen randomly and transmitted as the first block of
the output. Errors in encryption affect the current block and the next
block after which the cipher will correct itself. CBC is the most
commonly used mode in software encryption.

[def "Cipher Feedback (CFB)"]

CFB mode can be used to convert block ciphers into stream ciphers. In
CFB mode the initialization vector is encrypted and the output is then
xor'd with the plaintext stream. The result is then used as the
initialization vector for the next round. Errors will affect the
current block and the next block.

[def "Output Feedback (OFB)"]
OFB is similar to CFB except that the output of the cipher is fed back
into the next round and not the xor'd plain text. This means that
errors only affect a single block but the cipher is more vulnerable to
attack.

[list_end]

[section EXAMPLES]

[example {
% set ciphertext [DES::des -mode cbc -dir encrypt -key $secret $plaintext]
% set plaintext [DES::des -mode cbc -dir decrypt -key $secret $ciphertext]
}]

[example {
set iv [string repeat \\0 8]
set Key [DES::Init cbc \\0\\1\\2\\3\\4\\5\\6\\7 $iv]
set ciphertext [DES::Encrypt $Key "somedata"]
append ciphertext [DES::Encrypt $Key "moredata"]
DES::Reset $Key $iv
set plaintext [DES::Decrypt $Key $ciphertext]
DES::Final $Key
}]

[section "REFERENCES"]

[list_begin enumerated]

[enum]
    "Data Encryption Standard",
    Federal Information Processing Standards Publication 46-3, 1999,
    ([uri http://csrc.nist.gov/publications/fips/fips46-3/fips46-3.pdf])

[enum]
    "TclDES: munitions-grade Tcl scripting"
    [uri http://tcldes.sourceforge.net/]

[list_end]

[see_also aes(n) blowfish(n) rc4(n) md5(n) sha1(n) ]

[section "AUTHORS"]
Jochen C Loewer, 
Mac Cody,
Pat Thoyts

[section {BUGS, IDEAS, FEEDBACK}]

This document, and the package it describes, will undoubtedly contain
bugs and other problems.

Please report such in the category [emph des] of the
[uri {http://sourceforge.net/tracker/?group_id=12883} {Tcllib SF Trackers}].

Please also report any ideas for enhancements you may have for either
package and/or documentation.


[keywords DES 3DES {block cipher} security encryption {data integrity}]
[manpage_end]
