/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/bernoulli/cdf' );
import Bernoulli = require( './../../../../../base/dists/bernoulli/ctor' );
import entropy = require( './../../../../../base/dists/bernoulli/entropy' );
import kurtosis = require( './../../../../../base/dists/bernoulli/kurtosis' );
import mean = require( './../../../../../base/dists/bernoulli/mean' );
import median = require( './../../../../../base/dists/bernoulli/median' );
import mgf = require( './../../../../../base/dists/bernoulli/mgf' );
import mode = require( './../../../../../base/dists/bernoulli/mode' );
import pmf = require( './../../../../../base/dists/bernoulli/pmf' );
import quantile = require( './../../../../../base/dists/bernoulli/quantile' );
import skewness = require( './../../../../../base/dists/bernoulli/skewness' );
import stdev = require( './../../../../../base/dists/bernoulli/stdev' );
import variance = require( './../../../../../base/dists/bernoulli/variance' );

/**
* Interface describing the `bernoulli` namespace.
*/
interface Namespace {
	/**
	* Bernoulli distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param p - success probability
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 1.0, 0.5 );
	* // returns 1.0
	*
	* y = ns.cdf( 0.5, 0.1 );
	* // returns 0.9
	*
	* var mycdf = ns.cdf.factory( 0.5 );
	* y = mycdf( 3.0 );
	* // returns 1.0
	*
	* y = mycdf( -1.0 );
	* // returns 0.0
	*/
	cdf: typeof cdf;

	/**
	* Bernoulli distribution.
	*/
	Bernoulli: typeof Bernoulli;

	/**
	* Returns the entropy of a Bernoulli distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 0.1 );
	* // returns ~0.325
	*
	* @example
	* var v = ns.entropy( 0.5 );
	* // returns ~0.693
	*
	* @example
	* var v = ns.entropy( 0.9 );
	* // returns ~0.325
	*
	* @example
	* var v = ns.entropy( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a Bernoulli distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns kurtosis
	*
	* @example
	* var v = ns.kurtosis( 0.1 );
	* // returns ~5.111
	*
	* @example
	* var v = ns.kurtosis( 0.5 );
	* // returns -2.0
	*
	* @example
	* var v = ns.kurtosis( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Returns the expected value of a Bernoulli distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 0.1 );
	* // returns 0.1
	*
	* @example
	* var v = ns.mean( 0.5 );
	* // returns 0.5
	*
	* @example
	* var v = ns.mean( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a Bernoulli distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns median
	*
	* @example
	* var v = ns.median( 0.1 );
	* // returns 0
	*
	* @example
	* var v = ns.median( 0.8 );
	* // returns 1
	*
	* @example
	* var v = ns.median( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Bernoulli distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param p - success probability
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.2, 0.5 );
	* // returns ~1.111
	*
	* y = ns.mgf( 0.4, 0.5 );
	* // returns ~1.246
	*
	* var mymgf = ns.mgf.factory( 0.8 );
	* y = mymgf( -0.2 );
	* // returns ~0.855
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of a Bernoulli distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	* -   For `p = 0.5`, the mode is either `0` or `1`. This function returns `0` for `p = 0.5`.
	*
	* @param p - success probability
	* @returns mode
	*
	* @example
	* var v = ns.mode( 0.1 );
	* // returns 0
	*
	* @example
	* var v = ns.mode( 0.8 );
	* // returns 1
	*
	* @example
	* var v = ns.mode( 0.5 );
	* // returns 0
	*
	* @example
	* var v = ns.mode( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Bernoulli distribution probability mass function (PMF).
	*
	* @param x - input value
	* @param p - success probability
	* @returns evaluated PMF
	*
	* @example
	* var y = ns.pmf( 0.0, 0.3 );
	* // returns 0.7
	*
	* y = ns.pmf( 1.0, 0.7 );
	* // returns 0.7
	*
	* y = ns.pmf( -1.0, 0.5 );
	* // returns 0.0
	*
	* var mypmf = ns.pmf.factory( 0.5 );
	* y = mypmf( 2.0 );
	* // returns 0.0
	*
	* y = mypmf( 1.0 );
	* // returns 0.5
	*/
	pmf: typeof pmf;

	/**
	* Bernoulli distribution quantile function.
	*
	* @param r - input value
	* @param p - success probability
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 0.4 );
	* // returns 1
	*
	* y = ns.quantile( 0.5, 0.4 );
	* // returns 0
	*
	* var myquantile = ns.quantile.factory( 0.4 );
	* y = myquantile( 0.4 );
	* // returns 0
	*
	* y = myquantile( 0.8 );
	* // returns 1
	*
	* y = myquantile( 1.0 );
	* // returns 1
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a Bernoulli distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 0.1 );
	* // returns ~2.667
	*
	* @example
	* var v = ns.skewness( 0.5 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a Bernoulli distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 0.1 );
	* // returns ~0.3
	*
	* @example
	* var v = ns.stdev( 0.5 );
	* // returns 0.5
	*
	* @example
	* var v = ns.stdev( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a Bernoulli distribution.
	*
	* ## Notes
	*
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param p - success probability
	* @returns variance
	*
	* @example
	* var v = ns.variance( 0.1 );
	* // returns ~0.09
	*
	* @example
	* var v = ns.variance( 0.5 );
	* // returns 0.25
	*
	* @example
	* var v = ns.variance( 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Bernoulli distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
