/**
* @license Apache-2.0
*
* Copyright (c) 2020 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isNonNegativeInteger = require( './../../is-nonnegative-integer' ).isPrimitive;
var MAX = require( '@stdlib/constants/array/max-typed-array-length' );


// VARIABLES //

var MAX_LENGTH = MAX / 2; // every complex array element has both a real and imaginary component stored as separate numbers, so the maximum length is half that of a normal typed array


// MAIN //

/**
* Tests if a value is complex-typed-array-like.
*
* @param {*} value - value to test
* @returns {boolean} boolean indicating if a value is complex-typed-array-like
*
* @example
* var arr = {
*	'BYTES_PER_ELEMENT': 8,
*	'length': 10,
*	'byteOffset': 0,
*	'byteLength': 10,
*   'get': function get() {},
*   'set': function set() {}
* };
* var val = isComplexTypedArrayLike( arr );
* // returns true
*
* @example
* var Complex64Array = require( '@stdlib/array/complex64' );
*
* var val = isComplexTypedArrayLike( new Complex64Array( 4 ) );
* // returns true
*
* @example
* var val = isComplexTypedArrayLike( [] );
* // returns false
*
* @example
* var val = isComplexTypedArrayLike( {} );
* // returns false
*
* @example
* var val = isComplexTypedArrayLike( null );
* // returns false
*
* @example
* var val = isComplexTypedArrayLike( 'beep' );
* // returns false
*/
function isComplexTypedArrayLike( value ) {
	return (
		value !== null &&
		typeof value === 'object' &&

		// Check for standard typed array properties:
		isNonNegativeInteger( value.length ) &&
		value.length <= MAX_LENGTH &&
		typeof value.BYTES_PER_ELEMENT === 'number' &&
		typeof value.byteOffset === 'number' &&
		typeof value.byteLength === 'number' &&

		// Check for properties necessary for complex typed arrays:
		typeof value.get === 'function' &&
		typeof value.set === 'function'
	);
}


// EXPORTS //

module.exports = isComplexTypedArrayLike;
