/*
 * @(#)XMLUtilUTest.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.util.xml.v1;

import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;


/**
 * Tests the XMLUtil class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2003/11/24 22:33:36 $
 * @since     06-Mar-2002
 */
public class XMLUtilUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    private static final Class THIS_CLASS = XMLUtilUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public XMLUtilUTest( String name )
    {
        super( name );
    }



    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testConstruction()
    {
        assertNotNull(
            "getInstance() must never return null.",
            XMLUtil.getInstance() );
    }
    
    
    public void testUTFConvert1()
    {
        // can work with nulls
        assertNull(
            "Null conversion didn't work right.",
            XMLUtil.getInstance().utf2xml( null ) );
    }
    
    
    public void testUTFConvert2()
    {
        // test non-escaped characters
        String noescapes =
            "abcdefghijklmnopqrstuvwxyz "+
            "ABCDEFGHIJKLMNOPQRSTUVWXYZ "+
            "()=+-_!@#$%^*{}[]\\|;:/?.,\n\t\r";
        String val = XMLUtil.getInstance().utf2xml( noescapes );
        DOC.getLog().debug( "expected '"+noescapes+"'" );
        DOC.getLog().debug( "received '"+val+"'" );
        assertEquals(
            noescapes,
            val );
    }
    
    
    public void testUTFConvert2a()
    {
        // we've changed the non-escaped character translation such that
        // it's compatible with UTF-8 encoding.  Probably shouldn't have,
        // but here we are.
        String noescapes = "" +
            ((char)0xD7FF) + ((char)0xE000) + ((char)0xFFFD);
        String real = "&#55295;&#57344;&#65533;";
        String val = XMLUtil.getInstance().utf2xml( noescapes );
        DOC.getLog().debug( "expected '"+noescapes+"'" );
        DOC.getLog().debug( "received '"+val+"'" );
        assertEquals(
            real,
            val );
    }
    
    
    public void testUTFConvert3()
    {
        assertEquals(
            "&lt;",
            XMLUtil.getInstance().utf2xml( "<" ) );
    }
    
    
    public void testUTFConvert4()
    {
        assertEquals(
            "&gt;",
            XMLUtil.getInstance().utf2xml( ">" ) );
    }
    
    
    public void testUTFConvert5()
    {
        assertEquals(
            "&amp;",
            XMLUtil.getInstance().utf2xml( "&" ) );
    }
    
    
    public void testUTFConvert6()
    {
        assertEquals(
            "&quot;",
            XMLUtil.getInstance().utf2xml( "\"" ) );
    }
    
    
    public void testUTFConvert7()
    {
        assertEquals(
            "&apos;",
            XMLUtil.getInstance().utf2xml( "'" ) );
    }
    
    
    public void testUTFConvert8()
    {
        assertEquals(
            "&#2;",
            XMLUtil.getInstance().utf2xml( ""+((char)2) ) );
    }
    
    
    public void testUTFConvert9()
    {
        assertEquals(
            "a&lt;b",
            XMLUtil.getInstance().utf2xml( "a<b" ) );
    }
    
    
    public void testUTFConvert10()
    {
        assertEquals(
            "a&gt;b",
            XMLUtil.getInstance().utf2xml( "a>b" ) );
    }
    
    
    public void testUTFConvert11()
    {
        assertEquals(
            "&amp;a&lt;",
            XMLUtil.getInstance().utf2xml( "&a<" ) );
    }
    
    
    public void testUTFConvert12()
    {
        assertEquals(
            "a&quot;b&gt;c",
            XMLUtil.getInstance().utf2xml( "a\"b>c" ) );
    }
    
    
    public void testUTFConvert13()
    {
        assertEquals(
            "&apos;abc&gt;d",
            XMLUtil.getInstance().utf2xml( "'abc>d" ) );
    }
    
    
    public void testUTFConvert14()
    {
        assertEquals(
            "aaa&#2;b&#3;cfed asdf;lkj qwepiouf &amp;amp;f3df",
            XMLUtil.getInstance().utf2xml( "aaa"+((char)2)+
                'b' + ((char)3) + "cfed asdf;lkj qwepiouf &amp;f3df") );
    }
    
    
    public void testUTFAppend1()
    {
        try
        {
            XMLUtil.getInstance().utf2xml( null, null );
            fail( "did not throw an IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // test text?
        }
    }
    
    
    public void testUTFAppend2()
    {
        try
        {
            XMLUtil.getInstance().utf2xml( "abc", null );
            fail( "did not throw an IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // test text?
        }
    }
    
    
    public void testUTFAppend3()
    {
        // can work with nulls
        StringBuffer sb = new StringBuffer();
        XMLUtil.getInstance().utf2xml( null, sb );
        assertEquals(
            "Null conversion did not work.",
            "null",
            sb.toString() );
    }
    
    
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

