"""Provider interface implementation for Openstack with S3 storage"""

import os
import logging

from txaws.service import AWSServiceRegion

from juju.providers.openstack import provider as os_provider
from juju.providers.openstack import credentials
from juju.providers.ec2 import files as s3_files


log = logging.getLogger("juju.openstack_s3")


class HybridCredentials(credentials.OpenStackCredentials):
    """Encapsulation of credentials with S3 required values included"""

    _config_vars = {
        'combined-key':  ("EC2_ACCESS_KEY", "AWS_ACCESS_KEY_ID"),
        's3-uri': ("S3_URL",),
    }
    _config_vars.update(credentials.OpenStackCredentials._config_vars)


class MachineProvider(os_provider.MachineProvider):
    """MachineProvider for use in Openstack environment but with S3 API"""

    Credentials = HybridCredentials

    def __init__(self, environment_name, config):
        super(MachineProvider, self).__init__(environment_name, config)

        del self.swift

        # If access or secret keys are still blank, inside txaws environment
        # a ValueError will be raised after rechecking the environment.
        self._aws_service = AWSServiceRegion(
            access_key=self.credentials.combined_key,
            secret_key=self.credentials.secret_key,
            ec2_uri="", # The EC2 client will not be used
            s3_uri=self.credentials.s3_uri)

        if self._check_certs:
            s3_endpoint = self._aws_service.s3_endpoint
            s3_endpoint.ssl_hostname_verification = True
            if s3_endpoint.scheme != "https":
                log.warn("S3 API calls not using secure transport")

        self.s3 = self._aws_service.get_s3_client()

    @property
    def provider_type(self):
        return "openstack_s3"

    def get_file_storage(self):
        """Retrieve a S3 API compatible backend FileStorage class"""
        return s3_files.FileStorage(self.s3, self.config["control-bucket"])
