package retry_test

import (
	"errors"
	"fmt"
	"log"
	"math/rand"
	"net/http"
	"os"
	"time"

	"github.com/Rican7/retry"
	"github.com/Rican7/retry/backoff"
	"github.com/Rican7/retry/jitter"
	"github.com/Rican7/retry/strategy"
)

func Example() {
	retry.Retry(func(attempt uint) error {
		return nil // Do something that may or may not cause an error
	})
}

func Example_fileOpen() {
	const logFilePath = "/var/log/myapp.log"

	var logFile *os.File

	err := retry.Retry(func(attempt uint) error {
		var err error

		logFile, err = os.Open(logFilePath)

		return err
	})

	if err != nil {
		log.Fatalf("Unable to open file %q with error %q", logFilePath, err)
	}

	logFile.Chdir() // Do something with the file
}

func Example_httpGetWithStrategies() {
	var response *http.Response

	action := func(attempt uint) error {
		var err error

		response, err = http.Get("https://api.github.com/repos/Rican7/retry")

		if err == nil && response != nil && response.StatusCode > 200 {
			err = fmt.Errorf("failed to fetch (attempt #%d) with status code: %d", attempt, response.StatusCode)
		}

		return err
	}

	err := retry.Retry(
		action,
		strategy.Limit(5),
		strategy.Backoff(backoff.Fibonacci(10*time.Millisecond)),
	)

	if err != nil {
		log.Fatalf("Failed to fetch repository with error %q", err)
	}
}

func Example_withBackoffJitter() {
	action := func(attempt uint) error {
		fmt.Println("attempt", attempt)

		return errors.New("something happened")
	}

	seed := time.Now().UnixNano()
	random := rand.New(rand.NewSource(seed))

	retry.Retry(
		action,
		strategy.Limit(5),
		strategy.BackoffWithJitter(
			backoff.BinaryExponential(10*time.Millisecond),
			jitter.Deviation(random, 0.5),
		),
	)

	// Output:
	// attempt 1
	// attempt 2
	// attempt 3
	// attempt 4
	// attempt 5
}
