/***************************************************************************

  nes.c

  Driver file to handle emulation of the Nintendo Entertainment System (Famicom).

  MESS driver by Brad Oliver (bradman@pobox.com), NES sound code by Matt Conte.
  Based in part on the old xNes code, by Nicolas Hamel, Chuck Mason, Brad Oliver,
  Richard Bannister and Jeff Mitchell.

***************************************************************************/

#include "emu.h"
#include "includes/nes.h"
#include "cpu/m6502/n2a03.h"
#include "imagedev/flopdrv.h"
#include "formats/nes_dsk.h"

#include "machine/nes_slot.h"


READ8_MEMBER(nes_state::psg_4015_r)
{
	return m_sound->read(space, 0x15);
}

WRITE8_MEMBER(nes_state::psg_4015_w)
{
	m_sound->write(space, 0x15, data);
}

WRITE8_MEMBER(nes_state::psg_4017_w)
{
	m_sound->write(space, 0x17, data);
}

WRITE8_MEMBER(nes_state::nes_vh_sprite_dma_w)
{
	m_ppu->spriteram_dma(space, data);
}

static ADDRESS_MAP_START( nes_map, AS_PROGRAM, 8, nes_state )
	AM_RANGE(0x0000, 0x07ff) AM_RAM AM_MIRROR(0x1800)                   /* RAM */
	AM_RANGE(0x2000, 0x3fff) AM_DEVREADWRITE("ppu", ppu2c0x_device, read, write)        /* PPU registers */
	AM_RANGE(0x4000, 0x4013) AM_DEVREADWRITE("nessound", nesapu_device, read, write)       /* PSG primary registers */
	AM_RANGE(0x4014, 0x4014) AM_WRITE(nes_vh_sprite_dma_w)              /* stupid address space hole */
	AM_RANGE(0x4015, 0x4015) AM_READWRITE(psg_4015_r, psg_4015_w)       /* PSG status / first control register */
	AM_RANGE(0x4016, 0x4016) AM_READWRITE(nes_in0_r, nes_in0_w)         /* IN0 - input port 1 */
	AM_RANGE(0x4017, 0x4017) AM_READ(nes_in1_r)                         /* IN1 - input port 2 */
	AM_RANGE(0x4017, 0x4017) AM_WRITE(psg_4017_w)       /* PSG second control register */
	// 0x4100-0x5fff -> LOW HANDLER defined on a pcb base
	// 0x6000-0x7fff -> MID HANDLER defined on a pcb base
	// 0x8000-0xffff -> HIGH HANDLER defined on a pcb base
ADDRESS_MAP_END


static INPUT_PORTS_START( nes_pads12 )
	PORT_START("PAD1")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON2 ) PORT_NAME("P1 A") PORT_PLAYER(1)  PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 ) PORT_NAME("P1 B") PORT_PLAYER(1)  PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SELECT ) PORT_PLAYER(1)                     PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START ) PORT_PLAYER(1)                      PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP ) PORT_PLAYER(1)                PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)              PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)              PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)             PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)

	PORT_START("PAD2")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON2 ) PORT_NAME("P2 A") PORT_PLAYER(2)  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 ) PORT_NAME("P2 B") PORT_PLAYER(2)  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SELECT ) PORT_PLAYER(2)                     PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START ) PORT_PLAYER(2)                      PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP ) PORT_PLAYER(2)                PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN ) PORT_PLAYER(2)              PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT ) PORT_PLAYER(2)              PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(2)             PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
INPUT_PORTS_END

static INPUT_PORTS_START( nes_pads34 )
	PORT_START("PAD3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON2 ) PORT_NAME("P3 A") PORT_PLAYER(3)  PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 ) PORT_NAME("P3 B") PORT_PLAYER(3)  PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SELECT ) PORT_PLAYER(3)                     PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START ) PORT_PLAYER(3)                      PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP ) PORT_PLAYER(3)                PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN ) PORT_PLAYER(3)              PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT ) PORT_PLAYER(3)              PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(3)             PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)

	PORT_START("PAD4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON2 ) PORT_NAME("P4 A") PORT_PLAYER(4)  PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 ) PORT_NAME("P4 B") PORT_PLAYER(4)  PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SELECT ) PORT_PLAYER(4)                     PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START ) PORT_PLAYER(4)                      PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP ) PORT_PLAYER(4)                PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN ) PORT_PLAYER(4)              PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT ) PORT_PLAYER(4)              PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(4)             PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
INPUT_PORTS_END


static INPUT_PORTS_START( nes_powerpad )
// difference between the two sides is that we mirror the key mapping to match the real pad layout!
	PORT_START("POWERPAD1")
// side A layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad Top1")  PORT_CODE(KEYCODE_Y) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_UNUSED )                                                  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad Mid1")  PORT_CODE(KEYCODE_J) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_UNUSED )                                                  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad Mid2")  PORT_CODE(KEYCODE_H) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad Low1")  PORT_CODE(KEYCODE_N) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad Low2")  PORT_CODE(KEYCODE_B) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad Mid3")  PORT_CODE(KEYCODE_G) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
// side B layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 2")     PORT_CODE(KEYCODE_T) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 1")     PORT_CODE(KEYCODE_R) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 5")     PORT_CODE(KEYCODE_F) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 9")     PORT_CODE(KEYCODE_V) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 6")     PORT_CODE(KEYCODE_G) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 10")    PORT_CODE(KEYCODE_B) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 11")    PORT_CODE(KEYCODE_N) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 7")     PORT_CODE(KEYCODE_H) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_START("POWERPAD2")
// side A layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_UNUSED )                                                  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad Top2")  PORT_CODE(KEYCODE_T) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNUSED )                                                  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad Mid4")  PORT_CODE(KEYCODE_F) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
	PORT_BIT( 0xf0, IP_ACTIVE_HIGH, IPT_UNUSED )                                                  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0050)
// side B layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 4")     PORT_CODE(KEYCODE_U) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 3")     PORT_CODE(KEYCODE_Y) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 12")    PORT_CODE(KEYCODE_M) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("PowerPad 8")     PORT_CODE(KEYCODE_J) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
	PORT_BIT( 0xf0, IP_ACTIVE_HIGH, IPT_UNUSED )                                                  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0060)
INPUT_PORTS_END

static INPUT_PORTS_START( nes_zapper1 )
	PORT_START("ZAPPER1_X")
	PORT_BIT( 0xff, 0x80, IPT_LIGHTGUN_X) PORT_CROSSHAIR(X, 1.0, 0.0, 0) PORT_SENSITIVITY(70) PORT_KEYDELTA(30) PORT_MINMAX(0,255) PORT_PLAYER(1) PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0002)
	PORT_START("ZAPPER1_Y")
	PORT_BIT( 0xff, 0x80, IPT_LIGHTGUN_Y) PORT_CROSSHAIR(Y, 1.0, 0.0, 0) PORT_SENSITIVITY(50) PORT_KEYDELTA(30) PORT_MINMAX(0,255) PORT_PLAYER(1) PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0002)
	PORT_START("ZAPPER1_T")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1) PORT_NAME("P1 Lightgun Trigger") PORT_PLAYER(1) PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0002)
INPUT_PORTS_END

static INPUT_PORTS_START( nes_zapper2 )
	PORT_START("ZAPPER2_X")
	PORT_BIT( 0xff, 0x80, IPT_LIGHTGUN_X) PORT_CROSSHAIR(X, 1.0, 0.0, 0) PORT_SENSITIVITY(70) PORT_KEYDELTA(30) PORT_MINMAX(0,255 ) PORT_PLAYER(2) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0020)
	PORT_START("ZAPPER2_Y")
	PORT_BIT( 0xff, 0x80, IPT_LIGHTGUN_Y) PORT_CROSSHAIR(Y, 1.0, 0.0, 0) PORT_SENSITIVITY(50) PORT_KEYDELTA(30) PORT_MINMAX(0,255 ) PORT_PLAYER(2) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0020)
	PORT_START("ZAPPER2_T")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1) PORT_NAME("P2 Lightgun Trigger") PORT_PLAYER(2) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0020)
INPUT_PORTS_END

static INPUT_PORTS_START( nes_paddle )
	PORT_START("PADDLE")
	PORT_BIT( 0xff, 0x7f, IPT_PADDLE) PORT_SENSITIVITY(25) PORT_KEYDELTA(25) PORT_CENTERDELTA(0) PORT_MINMAX(0x62,0xf2) PORT_PLAYER(2) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0040)
	PORT_START("PADDLE_BUTTON")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1) PORT_NAME("Paddle button") PORT_PLAYER(2) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0040)
INPUT_PORTS_END

static INPUT_PORTS_START( nes )
	PORT_INCLUDE( nes_pads12 )
	PORT_INCLUDE( nes_pads34 )
	PORT_INCLUDE( nes_powerpad )
	PORT_INCLUDE( nes_zapper1 )
	PORT_INCLUDE( nes_zapper2 )
	PORT_INCLUDE( nes_paddle )

	PORT_START("CTRLSEL")
	PORT_CONFNAME( 0x000f, 0x0001, "P1 Controller")
	PORT_CONFSETTING(  0x0000, "Unconnected" )
	PORT_CONFSETTING(  0x0001, "Gamepad" )
	PORT_CONFSETTING(  0x0002, "Zapper" )
	PORT_CONFNAME( 0x00f0, 0x0010, "P2 Controller")
	PORT_CONFSETTING(  0x0000, "Unconnected" )
	PORT_CONFSETTING(  0x0010, "Gamepad" )
	PORT_CONFSETTING(  0x0020, "Zapper" )
	PORT_CONFSETTING(  0x0040, "Arkanoid paddle" )
	PORT_CONFSETTING(  0x0050, "Power Pad (Side A layout)" )
	PORT_CONFSETTING(  0x0060, "Power Pad (Side B layout)" )
	PORT_CONFNAME( 0x0f00, 0x0000, "P3 Controller")
	PORT_CONFSETTING(  0x0000, "Unconnected" )
	PORT_CONFSETTING(  0x0100, "Gamepad" )
	PORT_CONFNAME( 0xf000, 0x0000, "P4 Controller")
	PORT_CONFSETTING(  0x0000, "Unconnected" )
	PORT_CONFSETTING(  0x1000, "Gamepad" )

	PORT_START("CONFIG")
	PORT_CONFNAME( 0x01, 0x00, "Draw Top/Bottom 8 Lines")
	PORT_CONFSETTING(    0x01, DEF_STR(No) )
	PORT_CONFSETTING(    0x00, DEF_STR(Yes) )
	PORT_CONFNAME( 0x02, 0x00, "Enforce 8 Sprites/line")
	PORT_CONFSETTING(    0x02, DEF_STR(No) )
	PORT_CONFSETTING(    0x00, DEF_STR(Yes) )
INPUT_PORTS_END


static INPUT_PORTS_START( fc_pads12 )
	PORT_START("PAD1")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON2 ) PORT_NAME("P1 A") PORT_PLAYER(1)  PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 ) PORT_NAME("P1 B") PORT_PLAYER(1)  PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SELECT ) PORT_PLAYER(1)                     PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START ) PORT_PLAYER(1)                      PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP ) PORT_PLAYER(1)                PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)              PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)              PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)             PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0001)

	PORT_START("PAD2")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON2 ) PORT_NAME("P2 A") PORT_PLAYER(2)  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 ) PORT_NAME("P2 B") PORT_PLAYER(2)  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SELECT ) PORT_PLAYER(2)                     PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START ) PORT_PLAYER(2)                      PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP ) PORT_PLAYER(2)                PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN ) PORT_PLAYER(2)              PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT ) PORT_PLAYER(2)              PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(2)             PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0010)
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON2 ) PORT_NAME("P2 A") PORT_PLAYER(2)  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x00f0)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 ) PORT_NAME("P2 B") PORT_PLAYER(2)  PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x00f0)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SELECT ) PORT_NAME("Microphone") PORT_PLAYER(2) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x00f0)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_UNUSED )                                    PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x00f0)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP ) PORT_PLAYER(2)                PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x00f0)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN ) PORT_PLAYER(2)              PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x00f0)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT ) PORT_PLAYER(2)              PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x00f0)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(2)             PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x00f0)
INPUT_PORTS_END

static INPUT_PORTS_START( fc_pads34 )
	PORT_START("PAD3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON2 ) PORT_NAME("P3 A") PORT_PLAYER(3)  PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 ) PORT_NAME("P3 B") PORT_PLAYER(3)  PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SELECT ) PORT_PLAYER(3)                     PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START ) PORT_PLAYER(3)                      PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP ) PORT_PLAYER(3)                PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN ) PORT_PLAYER(3)              PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT ) PORT_PLAYER(3)              PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(3)             PORT_CONDITION("CTRLSEL", 0x0f00, EQUALS, 0x0100)

	PORT_START("PAD4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON2 ) PORT_NAME("P4 A") PORT_PLAYER(4)  PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 ) PORT_NAME("P4 B") PORT_PLAYER(4)  PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SELECT ) PORT_PLAYER(4)                     PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START ) PORT_PLAYER(4)                      PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP ) PORT_PLAYER(4)                PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN ) PORT_PLAYER(4)              PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT ) PORT_PLAYER(4)              PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(4)             PORT_CONDITION("CTRLSEL", 0xf000, EQUALS, 0x1000)
INPUT_PORTS_END

static INPUT_PORTS_START( fc_lightgun )
	PORT_START("ZAPPER2_X")
	PORT_BIT( 0xff, 0x80, IPT_LIGHTGUN_X) PORT_CROSSHAIR(X, 1.0, 0.0, 0) PORT_SENSITIVITY(70) PORT_KEYDELTA(30) PORT_MINMAX(0,255) PORT_PLAYER(2) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x01)
	PORT_START("ZAPPER2_Y")
	PORT_BIT( 0xff, 0x80, IPT_LIGHTGUN_Y) PORT_CROSSHAIR(Y, 1.0, 0.0, 0) PORT_SENSITIVITY(50) PORT_KEYDELTA(30) PORT_MINMAX(0,255) PORT_PLAYER(2) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x01)
	PORT_START("ZAPPER2_T")
	PORT_BIT( 0x03, IP_ACTIVE_HIGH, IPT_BUTTON1) PORT_NAME("Lightgun Trigger") PORT_PLAYER(2) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x01)
INPUT_PORTS_END

static INPUT_PORTS_START( fc_paddle )
	PORT_START("PADDLE")
	PORT_BIT( 0xff, 0x7f, IPT_PADDLE) PORT_SENSITIVITY(25) PORT_KEYDELTA(25) PORT_CENTERDELTA(0) PORT_MINMAX(0x62,0xf2) PORT_PLAYER(2) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x04)
	PORT_START("PADDLE_BUTTON")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1) PORT_NAME("Paddle button") PORT_PLAYER(2) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x04)
INPUT_PORTS_END

static INPUT_PORTS_START( fc_cclimb )
	PORT_START("CC_LEFT")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_UNUSED ) PORT_PLAYER(1)             PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0002)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_UNUSED ) PORT_PLAYER(1)             PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0002)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_SELECT ) PORT_PLAYER(1)             PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0002)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START ) PORT_PLAYER(1)              PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0002)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_UP ) PORT_PLAYER(1)    PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0002)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_DOWN ) PORT_PLAYER(1)  PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0002)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_LEFT ) PORT_PLAYER(1)  PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0002)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_RIGHT ) PORT_PLAYER(1) PORT_CONDITION("CTRLSEL", 0x000f, EQUALS, 0x0002)

	PORT_START("CC_RIGHT")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_UNUSED ) PORT_PLAYER(1)             PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0020)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_UNUSED ) PORT_PLAYER(1)             PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0020)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNUSED ) PORT_PLAYER(1)             PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0020)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_UNUSED ) PORT_PLAYER(1)             PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0020)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_UP ) PORT_PLAYER(1)   PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0020)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_DOWN ) PORT_PLAYER(1) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0020)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_LEFT ) PORT_PLAYER(1) PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0020)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_RIGHT ) PORT_PLAYER(1)PORT_CONDITION("CTRLSEL", 0x00f0, EQUALS, 0x0020)
INPUT_PORTS_END

static INPUT_PORTS_START( fc_keyboard )
	PORT_START("FCKEY0")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F8) PORT_CHAR(UCHAR_MAMEKEY(F8))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_ENTER) PORT_CHAR(13)           PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_CLOSEBRACE)    PORT_CHAR('[')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_BACKSLASH) PORT_CHAR(']')      PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("Kana")                                PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_RSHIFT)                        PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_BACKSLASH2)    PORT_CHAR('\\') PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("Stop") PORT_CODE(KEYCODE_BACKSPACE)   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)

	PORT_START("FCKEY1")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F7) PORT_CHAR(UCHAR_MAMEKEY(F7))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_OPENBRACE) PORT_CHAR('@')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_COLON) PORT_CHAR(':')      PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_QUOTE) PORT_CHAR(';')      PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CHAR('_')                               PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CHAR('/')                               PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_MINUS) PORT_CHAR('-')      PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_EQUALS)    PORT_CHAR('^')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)

	PORT_START("FCKEY2")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F6) PORT_CHAR(UCHAR_MAMEKEY(F6))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_O) PORT_CHAR('O')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_L) PORT_CHAR('L')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_K) PORT_CHAR('K')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_STOP)  PORT_CHAR('.')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_COMMA) PORT_CHAR(',')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_P) PORT_CHAR('P')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_0) PORT_CHAR('0')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)

	PORT_START("FCKEY3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F5) PORT_CHAR(UCHAR_MAMEKEY(F5))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_I) PORT_CHAR('I')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_U) PORT_CHAR('U')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_J) PORT_CHAR('J')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_M) PORT_CHAR('M')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_N) PORT_CHAR('N')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_9) PORT_CHAR('9')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_8) PORT_CHAR('8')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)

	PORT_START("FCKEY4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F4) PORT_CHAR(UCHAR_MAMEKEY(F4))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_Y) PORT_CHAR('Y')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_G) PORT_CHAR('G')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_H) PORT_CHAR('H')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_B) PORT_CHAR('B')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_V) PORT_CHAR('V')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_7) PORT_CHAR('7')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_6) PORT_CHAR('6')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)

	PORT_START("FCKEY5")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F3) PORT_CHAR(UCHAR_MAMEKEY(F3))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_T) PORT_CHAR('T')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_R) PORT_CHAR('R')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_D) PORT_CHAR('D')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F) PORT_CHAR('F')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_C) PORT_CHAR('C')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_5) PORT_CHAR('5')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_4) PORT_CHAR('4')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)

	PORT_START("FCKEY6")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F2) PORT_CHAR(UCHAR_MAMEKEY(F2))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_W) PORT_CHAR('W')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_S) PORT_CHAR('S')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_A) PORT_CHAR('A')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_X) PORT_CHAR('X')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_Z) PORT_CHAR('Z')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_E) PORT_CHAR('E')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_3) PORT_CHAR('3')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)

	PORT_START("FCKEY7")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F1) PORT_CHAR(UCHAR_MAMEKEY(F1))           PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_TAB) PORT_CHAR(UCHAR_MAMEKEY(ESC))         PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_Q) PORT_CHAR('Q')                          PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_LCONTROL)      PORT_CHAR(UCHAR_SHIFT_2)    PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_LSHIFT)        PORT_CHAR(UCHAR_SHIFT_1)    PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("Grph") PORT_CODE(KEYCODE_LALT)        PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_1) PORT_CHAR('1')              PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_2) PORT_CHAR('2')              PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)

	PORT_START("FCKEY8")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("Clr")                                 PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_UP)    PORT_CHAR(UCHAR_MAMEKEY(UP))        PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_RIGHT) PORT_CHAR(UCHAR_MAMEKEY(RIGHT))     PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_LEFT) PORT_CHAR(UCHAR_MAMEKEY(LEFT))       PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_DOWN) PORT_CHAR(UCHAR_MAMEKEY(DOWN))       PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_SPACE)     PORT_CHAR(' ')      PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("Del") PORT_CODE(KEYCODE_DEL)          PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("Ins") PORT_CODE(KEYCODE_INSERT)       PORT_CONDITION("EXP", 0x0f, EQUALS, 0x02)

INPUT_PORTS_END

static INPUT_PORTS_START( subor_keyboard )
	PORT_START("SUBKEY0")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_4) PORT_CHAR('4')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_G) PORT_CHAR('G')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F) PORT_CHAR('F')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_C) PORT_CHAR('C')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F2) PORT_CHAR(UCHAR_MAMEKEY(F2))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_E) PORT_CHAR('E')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_5) PORT_CHAR('5')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_V) PORT_CHAR('V')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY1")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_2) PORT_CHAR('2')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_D) PORT_CHAR('D')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_S) PORT_CHAR('S')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_END) PORT_CHAR(UCHAR_MAMEKEY(END)) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F1) PORT_CHAR(UCHAR_MAMEKEY(F1))       PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_W) PORT_CHAR('W')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_3) PORT_CHAR('3')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_X) PORT_CHAR('X')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY2")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_INSERT) PORT_CHAR(UCHAR_MAMEKEY(INSERT))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_BACKSPACE) PORT_CHAR(8)                PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("NEXT")                                            PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_RIGHT) PORT_CHAR(UCHAR_MAMEKEY(RIGHT)) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F8) PORT_CHAR(UCHAR_MAMEKEY(F8))       PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("PRIOR")                                       PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_DEL) PORT_CHAR(UCHAR_MAMEKEY(DEL)) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_HOME) PORT_CHAR(UCHAR_MAMEKEY(HOME))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_9) PORT_CHAR('9')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_I) PORT_CHAR('I')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_L) PORT_CHAR('L')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_COMMA) PORT_CHAR(',')          PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F5) PORT_CHAR(UCHAR_MAMEKEY(F5))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_O) PORT_CHAR('O')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_0) PORT_CHAR('0')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_STOP)  PORT_CHAR('.')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_CLOSEBRACE)    PORT_CHAR(']')          PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_ENTER) PORT_CHAR(13)               PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_UP) PORT_CHAR(UCHAR_MAMEKEY(UP))       PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_LEFT) PORT_CHAR(UCHAR_MAMEKEY(LEFT))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F7) PORT_CHAR(UCHAR_MAMEKEY(F7))       PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_OPENBRACE) PORT_CHAR('[')          PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_BACKSLASH) PORT_CHAR('\\')         PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_DOWN) PORT_CHAR(UCHAR_MAMEKEY(DOWN))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY5")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_Q) PORT_CHAR('Q')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_CAPSLOCK) PORT_CHAR(UCHAR_MAMEKEY(CAPSLOCK))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_Z) PORT_CHAR('Z')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_TAB)   PORT_CHAR('\t') PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_ESC) PORT_CHAR(UCHAR_MAMEKEY(ESC))         PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_A) PORT_CHAR('A')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_1) PORT_CHAR('1')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_LCONTROL)      PORT_CHAR(UCHAR_SHIFT_2)        PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY6")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_7) PORT_CHAR('7')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_Y) PORT_CHAR('Y')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_K) PORT_CHAR('K')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_M) PORT_CHAR('M')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F4) PORT_CHAR(UCHAR_MAMEKEY(F4))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_U) PORT_CHAR('U')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_8) PORT_CHAR('8')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_J) PORT_CHAR('J')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY7")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_MINUS) PORT_CHAR('-')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_COLON) PORT_CHAR(':')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_QUOTE) PORT_CHAR('\'') PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_SLASH) PORT_CHAR('/')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F6) PORT_CHAR(UCHAR_MAMEKEY(F6))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_P)     PORT_CHAR('P')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_EQUALS)    PORT_CHAR('=')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_LSHIFT)    PORT_CHAR(UCHAR_SHIFT_1)    PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY8")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_T) PORT_CHAR('T')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_H) PORT_CHAR('H')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_N) PORT_CHAR('N')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_SPACE) PORT_CHAR(' ')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F3) PORT_CHAR(UCHAR_MAMEKEY(F3))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_R) PORT_CHAR('R')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_6) PORT_CHAR('6')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_B) PORT_CHAR('B')  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY9")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY10")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("LMENU")   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_4_PAD) PORT_CHAR(UCHAR_MAMEKEY(4_PAD)) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_7_PAD) PORT_CHAR(UCHAR_MAMEKEY(7_PAD)) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F11)       PORT_CHAR(UCHAR_MAMEKEY(F11))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F12)       PORT_CHAR(UCHAR_MAMEKEY(F12))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_1_PAD) PORT_CHAR(UCHAR_MAMEKEY(1_PAD)) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_2_PAD) PORT_CHAR(UCHAR_MAMEKEY(2_PAD)) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_8_PAD) PORT_CHAR(UCHAR_MAMEKEY(8_PAD)) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY11")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_MINUS_PAD)PORT_CHAR(UCHAR_MAMEKEY(MINUS_PAD))  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_PLUS_PAD)  PORT_CHAR(UCHAR_MAMEKEY(PLUS_PAD))  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_ASTERISK)  PORT_CHAR(UCHAR_MAMEKEY(ASTERISK))  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_9_PAD) PORT_CHAR(UCHAR_MAMEKEY(9_PAD))     PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F10) PORT_CHAR(UCHAR_MAMEKEY(F10))         PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_5_PAD) PORT_CHAR(UCHAR_MAMEKEY(5_PAD))     PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_SLASH_PAD)PORT_CHAR(UCHAR_MAMEKEY(SLASH_PAD))  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_NUMLOCK)   PORT_CHAR(UCHAR_MAMEKEY(NUMLOCK))   PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

	PORT_START("SUBKEY12")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_TILDE) PORT_CHAR('`')                  PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_6_PAD) PORT_CHAR(UCHAR_MAMEKEY(6_PAD)) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("PAUSE")                                           PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("SPACE2")                                          PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_F9) PORT_CHAR(UCHAR_MAMEKEY(F9))           PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_3_PAD) PORT_CHAR(UCHAR_MAMEKEY(3_PAD)) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_NAME("Keypad .") PORT_CODE(KEYCODE_DEL_PAD)         PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD ) PORT_CODE(KEYCODE_0_PAD) PORT_CHAR(UCHAR_MAMEKEY(0_PAD)) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x03)

INPUT_PORTS_END

static INPUT_PORTS_START( mahjong_panel )
	PORT_START("MAH0")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)

	PORT_START("MAH1")
	PORT_BIT( 0x03, IP_ACTIVE_HIGH, IPT_UNUSED ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_MAHJONG_N ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_MAHJONG_M ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_MAHJONG_L ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_MAHJONG_K ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_MAHJONG_J ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_MAHJONG_I ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)

	PORT_START("MAH2")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_MAHJONG_H ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_MAHJONG_G ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_MAHJONG_F ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_MAHJONG_E ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_MAHJONG_D ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_MAHJONG_C ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_MAHJONG_B ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_MAHJONG_A ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)

	PORT_START("MAH3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_UNUSED ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_MAHJONG_RON ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_MAHJONG_REACH ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_MAHJONG_CHI ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_MAHJONG_PON ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_MAHJONG_KAN ) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_SELECT ) PORT_NAME("P1 Mahjong Select") PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_START ) PORT_NAME("P1 Mahjong Start") PORT_CONDITION("EXP", 0x0f, EQUALS, 0x07)
INPUT_PORTS_END

// these are read differently than the powerpad inputs, but we share the tags, to reduce
static INPUT_PORTS_START( fc_ftrainer )
// difference between the two sides is that we mirror the key mapping to match the real pad layout!
	PORT_START("FT_COL0")
// side A layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_UNUSED )                                                        PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer Mid1")  PORT_CODE(KEYCODE_J) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNUSED )                                                        PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
// side B layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 12")    PORT_CODE(KEYCODE_M) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 8")     PORT_CODE(KEYCODE_J) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 4")     PORT_CODE(KEYCODE_U) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)

	PORT_START("FT_COL1")
// side A layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer Low1")  PORT_CODE(KEYCODE_N) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer Mid2")  PORT_CODE(KEYCODE_H) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer Top1")  PORT_CODE(KEYCODE_Y) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
// side B layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 11")    PORT_CODE(KEYCODE_N) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 7")     PORT_CODE(KEYCODE_H) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 3")     PORT_CODE(KEYCODE_Y) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)

	PORT_START("FT_COL2")
// side A layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer Low2")  PORT_CODE(KEYCODE_B) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer Mid3")  PORT_CODE(KEYCODE_G) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer Top2")  PORT_CODE(KEYCODE_T) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
// side B layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 10")    PORT_CODE(KEYCODE_B) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 6")     PORT_CODE(KEYCODE_G) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 2")     PORT_CODE(KEYCODE_T) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)

	PORT_START("FT_COL3")
// side A layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_UNUSED )                                                        PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer Mid4")  PORT_CODE(KEYCODE_F) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNUSED )                                                        PORT_CONDITION("EXP", 0x0f, EQUALS, 0x05)
// side B layout
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 9")     PORT_CODE(KEYCODE_V) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 5")     PORT_CODE(KEYCODE_F) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_NAME("Family Trainer 1")     PORT_CODE(KEYCODE_R) PORT_CONDITION("EXP", 0x0f, EQUALS, 0x06)
INPUT_PORTS_END



static INPUT_PORTS_START( famicom )
	PORT_INCLUDE( fc_pads12 )
	PORT_INCLUDE( fc_pads34 )
	PORT_INCLUDE( fc_lightgun )
	// FIXME: was it possible to attache two paddles in a Famicom (as the Arkanoid 2 box suggests)?!? investigate...
	PORT_INCLUDE( fc_paddle )
	// Crazy Climber is not really a separate controller, but a couple of small sticks to be
	// put on top of d-pads of the regular controllers. Users should then control the game
	// by using both controllers, turned 90 degrees, as a couple of dual sticks as in the arcade
	PORT_INCLUDE( fc_cclimb )
	PORT_INCLUDE( fc_keyboard )
	PORT_INCLUDE( subor_keyboard )
	PORT_INCLUDE( mahjong_panel )
	PORT_INCLUDE( fc_ftrainer )

	PORT_START("CTRLSEL")
	PORT_CONFNAME( 0x000f, 0x0001, "P1 Controller")
	PORT_CONFSETTING(  0x0000, "Unconnected" )
	PORT_CONFSETTING(  0x0001, "Gamepad" )
	PORT_CONFSETTING(  0x0002, "Crazy Climber pad (Left)" )
	PORT_CONFNAME( 0x00f0, 0x0010, "P2 Controller")
	PORT_CONFSETTING(  0x0000, "Unconnected" )
	PORT_CONFSETTING(  0x0010, "Gamepad" )
	PORT_CONFSETTING(  0x0020, "Crazy Climber pad (Right)" )
	PORT_CONFSETTING(  0x00f0, "Gamepad (Older Version)" )
	PORT_CONFNAME( 0x0f00, 0x0000, "P3 Controller") PORT_CONDITION("EXP", 0x0f, EQUALS, 0x08)
	PORT_CONFSETTING(  0x0000, "Unconnected" )
	PORT_CONFSETTING(  0x0100, "Gamepad" )
	PORT_CONFNAME( 0xf000, 0x0000, "P4 Controller") PORT_CONDITION("EXP", 0x0f, EQUALS, 0x08)
	PORT_CONFSETTING(  0x0000, "Unconnected" )
	PORT_CONFSETTING(  0x1000, "Gamepad" )

	PORT_START("EXP")
	PORT_CONFNAME( 0x0f, 0x00, "Expansion Port")
	PORT_CONFSETTING(  0x00, "(Empty)" )
	PORT_CONFSETTING(  0x01, "Light Gun" )
	PORT_CONFSETTING(  0x02, "FC Keyboard" )
	PORT_CONFSETTING(  0x03, "Subor Keyboard" )
	PORT_CONFSETTING(  0x04, "Arkanoid paddle" )
	PORT_CONFSETTING(  0x05, "Family Trainer (Side A)" )
	PORT_CONFSETTING(  0x06, "Family Trainer (Side B)" )
	PORT_CONFSETTING(  0x07, "Mahjong Panel" )
	PORT_CONFSETTING(  0x08, "Hori Twin Adapter" )

	PORT_START("CONFIG")
	PORT_CONFNAME( 0x01, 0x00, "Draw Top/Bottom 8 Lines")
	PORT_CONFSETTING(    0x01, DEF_STR(No) )
	PORT_CONFSETTING(    0x00, DEF_STR(Yes) )
	PORT_CONFNAME( 0x02, 0x00, "Enforce 8 Sprites/line")
	PORT_CONFSETTING(    0x02, DEF_STR(No) )
	PORT_CONFSETTING(    0x00, DEF_STR(Yes) )

	PORT_START("FLIPDISK") /* fake key */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON3) PORT_NAME("Change Disk Side")
INPUT_PORTS_END


static const nesapu_interface nes_apu_interface =
{
	"maincpu"
};


void nes_state::ppu_nmi(int *ppu_regs)
{
	m_maincpu->set_input_line(INPUT_LINE_NMI, PULSE_LINE);
}


static const ppu2c0x_interface nes_ppu_interface =
{
	"maincpu",
	0,
	0,
	PPU_MIRROR_NONE
};

static const floppy_interface nes_floppy_interface =
{
	DEVCB_NULL,
	DEVCB_NULL,
	DEVCB_NULL,
	DEVCB_NULL,
	DEVCB_NULL,
	FLOPPY_STANDARD_5_25_DSHD,
	LEGACY_FLOPPY_OPTIONS_NAME(nes_only),
	"floppy_5_25",
	NULL
};


static const nes_cart_interface nes_crt_interface =
{
};


static const cassette_interface fc_cassette_interface =
{
	cassette_default_formats,
	NULL,
	(cassette_state)(CASSETTE_STOPPED | CASSETTE_MOTOR_ENABLED | CASSETTE_SPEAKER_ENABLED),
	"fc_cass",
	NULL
};


SLOT_INTERFACE_START(nes_cart)
// HROM, NROM, RROM, SROM & STROM
	SLOT_INTERFACE_INTERNAL("nrom",             NES_NROM)
// Nintendo Family BASIC pcb (NROM + 2K or 4K WRAM)
	SLOT_INTERFACE_INTERNAL("hvc_basic",        NES_FCBASIC)
// Game Genie
	SLOT_INTERFACE_INTERNAL("nrom_gg",          NES_GGENIE)
// UNROM/UOROM
	SLOT_INTERFACE_INTERNAL("uxrom",            NES_UXROM)
	SLOT_INTERFACE_INTERNAL("unrom_cc",         NES_UXROM_CC)
// CNROM
	SLOT_INTERFACE_INTERNAL("cnrom",            NES_CNROM)
// Bandai PT-554 (CNROM boards + special audio chip, used by Aerobics Studio)
	SLOT_INTERFACE_INTERNAL("bandai_pt554",     NES_BANDAI_PT554)
// CPROM
	SLOT_INTERFACE_INTERNAL("cprom",            NES_CPROM)
// AMROM, ANROM, AOROM
	SLOT_INTERFACE_INTERNAL("axrom",            NES_AXROM)
// PxROM
	SLOT_INTERFACE_INTERNAL("pxrom",            NES_PXROM)
// FxROM
	SLOT_INTERFACE_INTERNAL("fxrom",            NES_FXROM)
// BNROM
	SLOT_INTERFACE_INTERNAL("bnrom",            NES_BXROM)
// GNROM & MHROM
	SLOT_INTERFACE_INTERNAL("gxrom",            NES_GXROM)
// UN1ROM
	SLOT_INTERFACE_INTERNAL("un1rom",           NES_UN1ROM)
// SxROM
	SLOT_INTERFACE_INTERNAL("sxrom",            NES_SXROM)
	SLOT_INTERFACE_INTERNAL("sorom",            NES_SOROM)
	SLOT_INTERFACE_INTERNAL("sxrom_a",          NES_SXROM_A)  // in MMC1-A PRG RAM is always enabled
	SLOT_INTERFACE_INTERNAL("sorom_a",          NES_SOROM_A)  // in MMC1-A PRG RAM is always enabled
// TxROM
	SLOT_INTERFACE_INTERNAL("txrom",            NES_TXROM)
// HKROM
	SLOT_INTERFACE_INTERNAL("hkrom",            NES_HKROM)
// TQROM
	SLOT_INTERFACE_INTERNAL("tqrom",            NES_TQROM)
// TxSROM
	SLOT_INTERFACE_INTERNAL("txsrom",           NES_TXSROM)
// ExROM
	SLOT_INTERFACE_INTERNAL("exrom",            NES_EXROM)
// Nintendo Custom boards
	SLOT_INTERFACE_INTERNAL("pal_zz",           NES_ZZ_PCB)
	SLOT_INTERFACE_INTERNAL("nes_qj",           NES_QJ_PCB)
	SLOT_INTERFACE_INTERNAL("nes_event",        NES_EVENT)
// Discrete Components boards
// IC_74x139x74
	SLOT_INTERFACE_INTERNAL("discrete_74x139",  NES_74X139X74)
// IC_74x377
	SLOT_INTERFACE_INTERNAL("discrete_74x377",  NES_74X377)
// Discrete board IC_74x161x161x32
	SLOT_INTERFACE_INTERNAL("discrete_74x161", NES_74X161X161X32)
// Discrete board IC_74x161x138
	SLOT_INTERFACE_INTERNAL("bitcorp_dis",      NES_74X161X138)
// Bandai boards
	SLOT_INTERFACE_INTERNAL("lz93d50",          NES_LZ93D50)
	SLOT_INTERFACE_INTERNAL("lz93d50_ep1",      NES_LZ93D50_24C01)
	SLOT_INTERFACE_INTERNAL("lz93d50_ep2",      NES_LZ93D50_24C02)
	SLOT_INTERFACE_INTERNAL("fcg",              NES_FCG)
	SLOT_INTERFACE_INTERNAL("fjump2",           NES_FJUMP2)
	SLOT_INTERFACE_INTERNAL("datach",           NES_DATACH)
	SLOT_INTERFACE_INTERNAL("karastudio",       NES_KARAOKESTUDIO)
	SLOT_INTERFACE_INTERNAL("oekakids",         NES_OEKAKIDS)
// Irem boards
	SLOT_INTERFACE_INTERNAL("g101",             NES_G101)
	SLOT_INTERFACE_INTERNAL("lrog017",          NES_LROG017)
	SLOT_INTERFACE_INTERNAL("h3001",            NES_H3001)
	SLOT_INTERFACE_INTERNAL("holydivr",         NES_HOLYDIVR)
	SLOT_INTERFACE_INTERNAL("tam_s1",           NES_TAM_S1)
// Jaleco boards
	SLOT_INTERFACE_INTERNAL("jf11",             NES_JF11)
	SLOT_INTERFACE_INTERNAL("jf13",             NES_JF13)
	SLOT_INTERFACE_INTERNAL("jf16",             NES_JF16)
	SLOT_INTERFACE_INTERNAL("jf17",             NES_JF17)
	SLOT_INTERFACE_INTERNAL("jf17pcm",          NES_JF17_ADPCM)
	SLOT_INTERFACE_INTERNAL("jf19",             NES_JF19)
	SLOT_INTERFACE_INTERNAL("jf19pcm",          NES_JF19_ADPCM)
	SLOT_INTERFACE_INTERNAL("ss88006",          NES_SS88006)
	SLOT_INTERFACE_INTERNAL("jf23",             NES_JF23)
	SLOT_INTERFACE_INTERNAL("jf24",             NES_JF24)
	SLOT_INTERFACE_INTERNAL("jf29",             NES_JF29)
	SLOT_INTERFACE_INTERNAL("jf33",             NES_JF33)
// Konami boards
	SLOT_INTERFACE_INTERNAL("vrc1",             NES_VRC1)
	SLOT_INTERFACE_INTERNAL("vrc2",             NES_VRC2)
	SLOT_INTERFACE_INTERNAL("vrc3",             NES_VRC3)
	SLOT_INTERFACE_INTERNAL("vrc4",             NES_VRC4)
	SLOT_INTERFACE_INTERNAL("vrc6",             NES_VRC6)
	SLOT_INTERFACE_INTERNAL("vrc7",             NES_VRC7)
// Namcot boards
	SLOT_INTERFACE_INTERNAL("namcot_163",       NES_NAMCOT163)
	SLOT_INTERFACE_INTERNAL("namcot_175",       NES_NAMCOT175)
	SLOT_INTERFACE_INTERNAL("namcot_340",       NES_NAMCOT340)
	SLOT_INTERFACE_INTERNAL("namcot_3433",      NES_NAMCOT3433)  // DxROM is a Nintendo board for US versions of the 3433/3443 games
	SLOT_INTERFACE_INTERNAL("namcot_3425",      NES_NAMCOT3425)
	SLOT_INTERFACE_INTERNAL("namcot_3446",      NES_NAMCOT3446)
// Sunsoft boards
	SLOT_INTERFACE_INTERNAL("sunsoft1",         NES_SUNSOFT_1)
	SLOT_INTERFACE_INTERNAL("sunsoft2",         NES_SUNSOFT_2)
	SLOT_INTERFACE_INTERNAL("sunsoft3",         NES_SUNSOFT_3)
	SLOT_INTERFACE_INTERNAL("sunsoft4",         NES_SUNSOFT_4)
	SLOT_INTERFACE_INTERNAL("sunsoft_dcs",      NES_SUNSOFT_DCS)
	SLOT_INTERFACE_INTERNAL("sunsoft_fme7",     NES_SUNSOFT_FME7) // JxROM is a Nintendo board for US versions of the Sunsoft FME7 games
	SLOT_INTERFACE_INTERNAL("sunsoft5a",        NES_SUNSOFT_5)
	SLOT_INTERFACE_INTERNAL("sunsoft5b",        NES_SUNSOFT_5)
// Taito boards
	SLOT_INTERFACE_INTERNAL("tc0190fmc",        NES_TC0190FMC)
	SLOT_INTERFACE_INTERNAL("tc0190fmcp",       NES_TC0190FMC_PAL16R4)
	SLOT_INTERFACE_INTERNAL("tc0350fmr",        NES_TC0190FMC)
	SLOT_INTERFACE_INTERNAL("x1_005",           NES_X1_005)   // two variants exist, depending on pin17 & pin31 connections
	SLOT_INTERFACE_INTERNAL("x1_017",           NES_X1_017)
// Misc pirate boards (by AVE, Camerica, C&E, Nanjing, NTDEC, JY Company, Sachen, Tengen, TXC, Waixing, Henggendianzi, etc.)
	SLOT_INTERFACE_INTERNAL("nina001",          NES_NINA001)
	SLOT_INTERFACE_INTERNAL("nina006",          NES_NINA006)
	SLOT_INTERFACE_INTERNAL("bf9093",           NES_BF9093)
	SLOT_INTERFACE_INTERNAL("bf9096",           NES_BF9096)
	SLOT_INTERFACE_INTERNAL("goldenfive",       NES_GOLDEN5)
	SLOT_INTERFACE_INTERNAL("cne_decathl",      NES_CNE_DECATHL)
	SLOT_INTERFACE_INTERNAL("cne_fsb",          NES_CNE_FSB)
	SLOT_INTERFACE_INTERNAL("cne_shlz",         NES_CNE_SHLZ)
	SLOT_INTERFACE_INTERNAL("nanjing",          NES_NANJING) // mapper 163
	SLOT_INTERFACE_INTERNAL("ntdec_asder",      NES_NTDEC_ASDER) // mapper 112
	SLOT_INTERFACE_INTERNAL("ntdec_fh",         NES_NTDEC_FH)    // mapper 193
	SLOT_INTERFACE_INTERNAL("jyc_a",            NES_JY_TYPEA)    // mapper 90
	SLOT_INTERFACE_INTERNAL("jyc_b",            NES_JY_TYPEB)    // mapper 211
	SLOT_INTERFACE_INTERNAL("jyc_c",            NES_JY_TYPEC)    // mapper 209
	SLOT_INTERFACE_INTERNAL("sa009",            NES_SACHEN_SA009)
	SLOT_INTERFACE_INTERNAL("sa0036",           NES_SACHEN_SA0036)
	SLOT_INTERFACE_INTERNAL("sa0037",           NES_SACHEN_SA0037)
	SLOT_INTERFACE_INTERNAL("sa72007",          NES_SACHEN_SA72007)
	SLOT_INTERFACE_INTERNAL("sa72008",          NES_SACHEN_SA72008)
	SLOT_INTERFACE_INTERNAL("tca01",            NES_SACHEN_TCA01)
	SLOT_INTERFACE_INTERNAL("s8259a",           NES_SACHEN_8259A)
	SLOT_INTERFACE_INTERNAL("s8259b",           NES_SACHEN_8259B)
	SLOT_INTERFACE_INTERNAL("s8259c",           NES_SACHEN_8259C)
	SLOT_INTERFACE_INTERNAL("s8259d",           NES_SACHEN_8259D)
	SLOT_INTERFACE_INTERNAL("s74x374",          NES_SACHEN_74X374)
	SLOT_INTERFACE_INTERNAL("s74x374a",         NES_SACHEN_74X374_ALT)  /* FIXME: Made up boards some different handling */
	SLOT_INTERFACE_INTERNAL("tcu01",            NES_SACHEN_TCU01)
	SLOT_INTERFACE_INTERNAL("tcu02",            NES_SACHEN_TCU02)
	SLOT_INTERFACE_INTERNAL("tengen_800008",    NES_TENGEN_800008)   /* FIXME: Is this the same as CNROM? */
	SLOT_INTERFACE_INTERNAL("tengen_800032",    NES_TENGEN_800032)
	SLOT_INTERFACE_INTERNAL("tengen_800037",    NES_TENGEN_800037)
	SLOT_INTERFACE_INTERNAL("txc_22211",        NES_TXC_22211)
	SLOT_INTERFACE_INTERNAL("txc_dumarc",       NES_TXC_DUMARACING)
	SLOT_INTERFACE_INTERNAL("txc_mjblock",      NES_TXC_MJBLOCK)
	SLOT_INTERFACE_INTERNAL("txc_strikew",      NES_TXC_STRIKEW)
	SLOT_INTERFACE_INTERNAL("txc_commandos",    NES_TXC_COMMANDOS)
	SLOT_INTERFACE_INTERNAL("waixing_a",        NES_WAIXING_A)
	SLOT_INTERFACE_INTERNAL("waixing_a1",       NES_WAIXING_A1)   /* FIXME: Made up boards the different CHRRAM banks (see Ji Jia Zhan Shi) */
	SLOT_INTERFACE_INTERNAL("waixing_b",        NES_WAIXING_B)
	SLOT_INTERFACE_INTERNAL("waixing_c",        NES_WAIXING_C)
	SLOT_INTERFACE_INTERNAL("waixing_d",        NES_WAIXING_D)
	SLOT_INTERFACE_INTERNAL("waixing_e",        NES_WAIXING_E)
	SLOT_INTERFACE_INTERNAL("waixing_f",        NES_WAIXING_F)
	SLOT_INTERFACE_INTERNAL("waixing_g",        NES_WAIXING_G)
	SLOT_INTERFACE_INTERNAL("waixing_h",        NES_WAIXING_H)
	SLOT_INTERFACE_INTERNAL("waixing_h1",       NES_WAIXING_H1)   /* FIXME: Made up boards the different WRAM protect banks (see Shen Mi Jin San Jiao) */
	SLOT_INTERFACE_INTERNAL("waixing_i",        NES_WAIXING_I)
	SLOT_INTERFACE_INTERNAL("waixing_j",        NES_WAIXING_J)
	SLOT_INTERFACE_INTERNAL("waixing_sgz",      NES_WAIXING_SGZ)
	SLOT_INTERFACE_INTERNAL("waixing_sgzlz",    NES_WAIXING_SGZLZ)
	SLOT_INTERFACE_INTERNAL("waixing_sec",      NES_WAIXING_SEC)
	SLOT_INTERFACE_INTERNAL("waixing_ffv",      NES_WAIXING_FFV)
	SLOT_INTERFACE_INTERNAL("waixing_wxzs",     NES_WAIXING_WXZS)
	SLOT_INTERFACE_INTERNAL("waixing_wxzs2",    NES_WAIXING_WXZS2)
	SLOT_INTERFACE_INTERNAL("waixing_dq8",      NES_WAIXING_DQ8)
	SLOT_INTERFACE_INTERNAL("waixing_sh2",      NES_WAIXING_SH2)
	SLOT_INTERFACE_INTERNAL("fs304",            NES_WAIXING_FS304)  // used in Zelda 3 by Waixing
	SLOT_INTERFACE_INTERNAL("cony",             NES_CONY)
	SLOT_INTERFACE_INTERNAL("yoko",             NES_YOKO)
	SLOT_INTERFACE_INTERNAL("hengg_srich",      NES_HENGG_SRICH)
	SLOT_INTERFACE_INTERNAL("hengg_xhzs",       NES_HENGG_XHZS)
	SLOT_INTERFACE_INTERNAL("hengg_shjy3",      NES_HENGG_SHJY3) // mapper 253
	SLOT_INTERFACE_INTERNAL("hes",              NES_HES)
	SLOT_INTERFACE_INTERNAL("hosenkan",         NES_HOSENKAN)
	SLOT_INTERFACE_INTERNAL("ks7058",           NES_KS7058)
	SLOT_INTERFACE_INTERNAL("ks202",            NES_KS202)   // mapper 56
	SLOT_INTERFACE_INTERNAL("ks7022",           NES_KS7022)// mapper 175
	SLOT_INTERFACE_INTERNAL("ks7017",           NES_KS7017)
	SLOT_INTERFACE_INTERNAL("ks7032",           NES_KS7032) //  mapper 142
	SLOT_INTERFACE_INTERNAL("ks7012",           NES_KS7012)  // used in Zanac (FDS Conversion)
	SLOT_INTERFACE_INTERNAL("ks7013b",          NES_KS7013B) // used in Highway Star (FDS Conversion)
	SLOT_INTERFACE_INTERNAL("ks7031",           NES_KS7031) //  used in Dracula II (FDS Conversion)
	SLOT_INTERFACE_INTERNAL("gs2015",           NES_GS2015)
	SLOT_INTERFACE_INTERNAL("gs2004",           NES_GS2004)
	SLOT_INTERFACE_INTERNAL("gs2013",           NES_GS2013)
	SLOT_INTERFACE_INTERNAL("tf9in1",           NES_TF9IN1)
	SLOT_INTERFACE_INTERNAL("3dblock",          NES_3DBLOCK)    // NROM + IRQ?
	SLOT_INTERFACE_INTERNAL("racermate",        NES_RACERMATE)   // mapper 168
	SLOT_INTERFACE_INTERNAL("agci_50282",       NES_AGCI_50282)
	SLOT_INTERFACE_INTERNAL("dreamtech01",      NES_DREAMTECH01)
	SLOT_INTERFACE_INTERNAL("fukutake",         NES_FUKUTAKE)
	SLOT_INTERFACE_INTERNAL("futuremedia",      NES_FUTUREMEDIA)
	SLOT_INTERFACE_INTERNAL("magicseries",      NES_MAGSERIES)
	SLOT_INTERFACE_INTERNAL("daou_306",         NES_DAOU306)
	SLOT_INTERFACE_INTERNAL("subor0",           NES_SUBOR0)
	SLOT_INTERFACE_INTERNAL("subor1",           NES_SUBOR1)
	SLOT_INTERFACE_INTERNAL("cc21",             NES_CC21)
	SLOT_INTERFACE_INTERNAL("xiaozy",           NES_XIAOZY)
	SLOT_INTERFACE_INTERNAL("edu2k",            NES_EDU2K)
	SLOT_INTERFACE_INTERNAL("t230",             NES_T230)
	SLOT_INTERFACE_INTERNAL("mk2",              NES_MK2)
	SLOT_INTERFACE_INTERNAL("unl_whero",        NES_WHERO)    // mapper 27
	SLOT_INTERFACE_INTERNAL("unl_43272",        NES_43272)    // used in Gaau Hok Gwong Cheung
	SLOT_INTERFACE_INTERNAL("tf1201",           NES_TF1201)
	SLOT_INTERFACE_INTERNAL("unl_cfight",       NES_CITYFIGHT) //  used by City Fighter IV
// misc bootleg boards
	SLOT_INTERFACE_INTERNAL("ax5705",           NES_AX5705)
	SLOT_INTERFACE_INTERNAL("sc127",            NES_SC127)
	SLOT_INTERFACE_INTERNAL("mariobaby",        NES_MARIOBABY)
	SLOT_INTERFACE_INTERNAL("asnicol",          NES_ASN)
	SLOT_INTERFACE_INTERNAL("smb3pirate",       NES_SMB3PIRATE)
	SLOT_INTERFACE_INTERNAL("btl_dninja",       NES_BTL_DNINJA)
	SLOT_INTERFACE_INTERNAL("whirl2706",        NES_WHIRLWIND_2706)
	SLOT_INTERFACE_INTERNAL("smb2j",            NES_SMB2J)
	SLOT_INTERFACE_INTERNAL("smb2ja",           NES_SMB2JA)
	SLOT_INTERFACE_INTERNAL("smb2jb",           NES_SMB2JB)
	SLOT_INTERFACE_INTERNAL("09034a",           NES_09034A)
	SLOT_INTERFACE_INTERNAL("tobidase",         NES_TOBIDASE) // mapper 120
	SLOT_INTERFACE_INTERNAL("mmalee2",          NES_MMALEE)    // mapper 55?
	SLOT_INTERFACE_INTERNAL("unl_2708",         NES_2708)    // mapper 103
	SLOT_INTERFACE_INTERNAL("unl_lh32",         NES_LH32)   // used by Monty no Doki Doki Daidassou FDS conversion
	SLOT_INTERFACE_INTERNAL("unl_lh10",         NES_LH10)    // used in Fuuun Shaolin Kyo (FDS Conversion)
	SLOT_INTERFACE_INTERNAL("unl_lh53",         NES_LH53)    // used in Nazo no Murasamejou (FDS Conversion)
	SLOT_INTERFACE_INTERNAL("unl_ac08",         NES_AC08) //  used by Green Beret FDS conversion
	SLOT_INTERFACE_INTERNAL("unl_bb",           NES_UNL_BB) //  used by a few FDS conversions
	SLOT_INTERFACE_INTERNAL("sgpipe",           NES_SHUIGUAN)    // mapper 183
// misc MMC3 clone boards
	SLOT_INTERFACE_INTERNAL("dbz5",             NES_REX_DBZ5)
	SLOT_INTERFACE_INTERNAL("sl1632",           NES_REX_SL1632)
	SLOT_INTERFACE_INTERNAL("somari",           NES_SOMARI) // mapper 116
	SLOT_INTERFACE_INTERNAL("nitra",            NES_NITRA)
	SLOT_INTERFACE_INTERNAL("ks7057",           NES_KS7057) // mapper 196 alt (for Street Fighter VI / Fight Street VI)
	SLOT_INTERFACE_INTERNAL("sbros11",          NES_SBROS11)
	SLOT_INTERFACE_INTERNAL("unl_malisb",       NES_MALISB) //  used by Super Mali Splash Bomb
	SLOT_INTERFACE_INTERNAL("family4646",       NES_FAMILY4646)
	SLOT_INTERFACE_INTERNAL("pikay2k",          NES_PIKAY2K) // mapper 254
	SLOT_INTERFACE_INTERNAL("8237",             NES_8237)
	SLOT_INTERFACE_INTERNAL("8237a",            NES_NROM)    // UNSUPPORTED
	SLOT_INTERFACE_INTERNAL("sg_lionk",         NES_SG_LIONK)
	SLOT_INTERFACE_INTERNAL("sg_boog",          NES_SG_BOOG)
	SLOT_INTERFACE_INTERNAL("kasing",           NES_KASING)
	SLOT_INTERFACE_INTERNAL("kay",              NES_KAY)
	SLOT_INTERFACE_INTERNAL("h2288",            NES_H2288)
	SLOT_INTERFACE_INTERNAL("unl_6035052",      NES_6035052) // mapper 238?
	SLOT_INTERFACE_INTERNAL("txc_tw",           NES_TXC_TW)
	SLOT_INTERFACE_INTERNAL("kof97",            NES_KOF97)
	SLOT_INTERFACE_INTERNAL("kof96",            NES_KOF96)
	SLOT_INTERFACE_INTERNAL("sfight3",          NES_SF3)
	SLOT_INTERFACE_INTERNAL("gouder",           NES_GOUDER)
	SLOT_INTERFACE_INTERNAL("sa9602b",          NES_SA9602B)
	SLOT_INTERFACE_INTERNAL("unl_shero",        NES_SACHEN_SHERO)
// misc multigame cart boards
	SLOT_INTERFACE_INTERNAL("benshieng",        NES_BENSHIENG)
	SLOT_INTERFACE_INTERNAL("action52",         NES_ACTION52)
	SLOT_INTERFACE_INTERNAL("caltron6in1",      NES_CALTRON6IN1)
	SLOT_INTERFACE_INTERNAL("maxi15",           NES_MAXI15)        //  mapper 234
	SLOT_INTERFACE_INTERNAL("rumblestation",    NES_RUMBLESTATION)    // mapper 46
	SLOT_INTERFACE_INTERNAL("svision16",        NES_SVISION16)  // mapper 53
	SLOT_INTERFACE_INTERNAL("n625092",          NES_N625092)
	SLOT_INTERFACE_INTERNAL("a65as",            NES_A65AS)
	SLOT_INTERFACE_INTERNAL("t262",             NES_T262)
	SLOT_INTERFACE_INTERNAL("novel1",           NES_NOVEL1)
	SLOT_INTERFACE_INTERNAL("novel2",           NES_NOVEL2) // mapper 213... same as BMC-NOVELDIAMOND9999999IN1 board?
	SLOT_INTERFACE_INTERNAL("studyngame",       NES_STUDYNGAME) // mapper 39
	SLOT_INTERFACE_INTERNAL("sgun20in1",        NES_SUPERGUN20IN1)
	SLOT_INTERFACE_INTERNAL("bmc_vt5201",       NES_VT5201) // mapper 60 otherwise
	SLOT_INTERFACE_INTERNAL("bmc_d1038",        NES_VT5201) // mapper 60?
	SLOT_INTERFACE_INTERNAL("810544c",          NES_810544C)
	SLOT_INTERFACE_INTERNAL("ntd03",            NES_NTD03)
	SLOT_INTERFACE_INTERNAL("bmc_gb63",         NES_BMC_GB63)
	SLOT_INTERFACE_INTERNAL("bmc_gka",          NES_BMC_GKA)
	SLOT_INTERFACE_INTERNAL("bmc_gkb",          NES_BMC_GKB)
	SLOT_INTERFACE_INTERNAL("bmc_ws",           NES_BMC_WS)
	SLOT_INTERFACE_INTERNAL("bmc_g146",         NES_BMC_G146)
	SLOT_INTERFACE_INTERNAL("bmc_11160",        NES_BMC_11160)
	SLOT_INTERFACE_INTERNAL("bmc_8157",         NES_BMC_8157)
	SLOT_INTERFACE_INTERNAL("bmc_hik300",       NES_BMC_HIK300)
	SLOT_INTERFACE_INTERNAL("bmc_s700",         NES_BMC_S700)
	SLOT_INTERFACE_INTERNAL("bmc_ball11",       NES_BMC_BALL11)
	SLOT_INTERFACE_INTERNAL("bmc_22games",      NES_BMC_22GAMES)
	SLOT_INTERFACE_INTERNAL("bmc_64y2k",        NES_BMC_64Y2K)
	SLOT_INTERFACE_INTERNAL("bmc_12in1",        NES_BMC_12IN1)
	SLOT_INTERFACE_INTERNAL("bmc_20in1",        NES_BMC_20IN1)
	SLOT_INTERFACE_INTERNAL("bmc_21in1",        NES_BMC_21IN1)
	SLOT_INTERFACE_INTERNAL("bmc_31in1",        NES_BMC_31IN1)
	SLOT_INTERFACE_INTERNAL("bmc_35in1",        NES_BMC_35IN1)
	SLOT_INTERFACE_INTERNAL("bmc_36in1",        NES_BMC_36IN1)
	SLOT_INTERFACE_INTERNAL("bmc_64in1",        NES_BMC_64IN1)
	SLOT_INTERFACE_INTERNAL("bmc_70in1",        NES_BMC_70IN1)   // mapper 236?
	SLOT_INTERFACE_INTERNAL("bmc_72in1",        NES_BMC_72IN1)
	SLOT_INTERFACE_INTERNAL("bmc_76in1",        NES_BMC_76IN1)
	SLOT_INTERFACE_INTERNAL("bmc_s42in1",       NES_BMC_76IN1)
	SLOT_INTERFACE_INTERNAL("bmc_110in1",       NES_BMC_110IN1)
	SLOT_INTERFACE_INTERNAL("bmc_150in1",       NES_BMC_150IN1)
	SLOT_INTERFACE_INTERNAL("bmc_190in1",       NES_BMC_190IN1)
	SLOT_INTERFACE_INTERNAL("bmc_800in1",       NES_BMC_800IN1)   // mapper 236?
	SLOT_INTERFACE_INTERNAL("bmc_1200in1",      NES_BMC_1200IN1)
	SLOT_INTERFACE_INTERNAL("bmc_gold150",      NES_BMC_GOLD150) // mapper 235 with 2M PRG
	SLOT_INTERFACE_INTERNAL("bmc_gold260",      NES_BMC_GOLD260) // mapper 235 with 4M PRG
	SLOT_INTERFACE_INTERNAL("bmc_power255",     NES_BMC_CH001)   // mapper 63?
	SLOT_INTERFACE_INTERNAL("bmc_s22games",     NES_BMC_SUPER22) // mapper 233
	SLOT_INTERFACE_INTERNAL("bmc_reset4",       NES_BMC_4IN1RESET) // mapper 60 with 64k prg and 32k chr
	SLOT_INTERFACE_INTERNAL("bmc_reset42",      NES_BMC_42IN1RESET)  // mapper 60? or 226? or 233?
// misc multigame cart MMC3 clone boards
	SLOT_INTERFACE_INTERNAL("fk23c",            NES_FK23C)
	SLOT_INTERFACE_INTERNAL("fk23ca",           NES_FK23CA)
	SLOT_INTERFACE_INTERNAL("s24in1c03",        NES_S24IN1SC03)
	SLOT_INTERFACE_INTERNAL("bmc_15in1",        NES_BMC_15IN1)
	SLOT_INTERFACE_INTERNAL("bmc_sbig7in1",     NES_BMC_SBIG7)
	SLOT_INTERFACE_INTERNAL("bmc_hik8in1",      NES_BMC_HIK8)
	SLOT_INTERFACE_INTERNAL("bmc_hik4in1",      NES_BMC_HIK4)
	SLOT_INTERFACE_INTERNAL("bmc_mario7in1",    NES_BMC_MARIO7IN1)
	SLOT_INTERFACE_INTERNAL("bmc_gold7in1",     NES_BMC_GOLD7IN1)
	SLOT_INTERFACE_INTERNAL("bmc_gc6in1",       NES_BMC_GC6IN1)
	SLOT_INTERFACE_INTERNAL("bmc_411120c",      NES_BMC_411120C)
	SLOT_INTERFACE_INTERNAL("bmc_830118c",      NES_BMC_830118C)
	SLOT_INTERFACE_INTERNAL("pjoy84",           NES_PJOY84)
// other unsupported...
	SLOT_INTERFACE_INTERNAL("ninjaryu",         NES_NROM)    // mapper 111 - UNSUPPORTED
	SLOT_INTERFACE_INTERNAL("unl_dance",        NES_NROM)    // UNSUPPORTED
	SLOT_INTERFACE_INTERNAL("onebus",           NES_NROM)    // UNSUPPORTED
// are there dumps of games with these boards?
	SLOT_INTERFACE_INTERNAL("bmc_hik_kof",      NES_NROM) // mapper 251 - UNSUPPORTED
	SLOT_INTERFACE_INTERNAL("bmc_13in1jy110",   NES_NROM) //  [mentioned in FCEUMM source - we need more info] - UNSUPPORTED
	SLOT_INTERFACE_INTERNAL("bmc_gk_192",       NES_NROM) //  [mentioned in FCEUMM source - we need more info] - UNSUPPORTED
	SLOT_INTERFACE_INTERNAL("konami_qtai",      NES_NROM) //  [mentioned in FCEUMM source - we need more info] - UNSUPPORTED
	SLOT_INTERFACE_INTERNAL("unl_3d_block",     NES_NROM) //  [mentioned in FCEUMM source - we need more info] - UNSUPPORTED
	SLOT_INTERFACE_INTERNAL("unl_c_n22m",       NES_NROM) //  [mentioned in FCEUMM source - we need more info] - UNSUPPORTED
	SLOT_INTERFACE_INTERNAL("unl_pec_586",      NES_NROM) //  [mentioned in FCEUMM source - we need more info] - UNSUPPORTED
	SLOT_INTERFACE_INTERNAL("a9746",            NES_NROM) // mapper 219 - UNSUPPORTED (no dump available)
// legacy boards for FFE copier mappers (are there images available to fix/improve emulation?)
	SLOT_INTERFACE_INTERNAL("ffe3",             NES_FFE3)
	SLOT_INTERFACE_INTERNAL("ffe4",             NES_FFE4)
	SLOT_INTERFACE_INTERNAL("ffe8",             NES_FFE8)
SLOT_INTERFACE_INTERNAL("test",             NES_NROM)
//
	SLOT_INTERFACE_INTERNAL("unknown",          NES_NROM)  //  a few pirate dumps uses the wrong mapper...
SLOT_INTERFACE_END


static MACHINE_CONFIG_START( nes, nes_state )
	/* basic machine hardware */
	MCFG_CPU_ADD("maincpu", N2A03, NTSC_CLOCK)
	MCFG_CPU_PROGRAM_MAP(nes_map)

	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_REFRESH_RATE(60.098)
	// This isn't used so much to calulate the vblank duration (the PPU code tracks that manually) but to determine
	// the number of cycles in each scanline for the PPU scanline timer. Since the PPU has 20 vblank scanlines + 2
	// non-rendering scanlines, we compensate. This ends up being 2500 cycles for the non-rendering portion, 2273
	// cycles for the actual vblank period.
	MCFG_SCREEN_VBLANK_TIME(ATTOSECONDS_IN_USEC((113.66/(NTSC_CLOCK/1000000)) * (PPU_VBLANK_LAST_SCANLINE_NTSC-PPU_VBLANK_FIRST_SCANLINE+1+2)))
	MCFG_SCREEN_SIZE(32*8, 262)
	MCFG_SCREEN_VISIBLE_AREA(0*8, 32*8-1, 0*8, 30*8-1)
	MCFG_SCREEN_UPDATE_DRIVER(nes_state, screen_update_nes)


	MCFG_PALETTE_LENGTH(4*16*8)

	MCFG_PPU2C02_ADD("ppu", nes_ppu_interface)
	MCFG_PPU2C0X_SET_NMI(nes_state, ppu_nmi)

	/* sound hardware */
	MCFG_SPEAKER_STANDARD_MONO("mono")
	MCFG_SOUND_ADD("nessound", NES_APU, NTSC_CLOCK)
	MCFG_SOUND_CONFIG(nes_apu_interface)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 0.90)

	MCFG_NES_CARTRIDGE_ADD("nes_slot", nes_crt_interface, nes_cart, NULL)
	MCFG_SOFTWARE_LIST_ADD("cart_list", "nes")
	MCFG_SOFTWARE_LIST_ADD("ntb_list", "nes_ntbrom") // Nantettate Baseball mini_carts
MACHINE_CONFIG_END

static MACHINE_CONFIG_DERIVED( nespal, nes )

	/* basic machine hardware */
	MCFG_CPU_MODIFY("maincpu")
	MCFG_CPU_CLOCK(PAL_CLOCK)

	MCFG_DEVICE_REMOVE("ppu")
	MCFG_PPU2C07_ADD("ppu", nes_ppu_interface)
	MCFG_PPU2C0X_SET_NMI(nes_state, ppu_nmi)

	/* video hardware */
	MCFG_SCREEN_MODIFY("screen")
	MCFG_SCREEN_REFRESH_RATE(53.355)
	MCFG_SCREEN_VBLANK_TIME(ATTOSECONDS_IN_USEC((106.53/(PAL_CLOCK/1000000)) * (PPU_VBLANK_LAST_SCANLINE_PAL-PPU_VBLANK_FIRST_SCANLINE+1+2)))
	MCFG_SCREEN_SIZE(32*8, 312)
	MCFG_SCREEN_VISIBLE_AREA(0*8, 32*8-1, 0*8, 30*8-1)

	/* sound hardware */
	MCFG_SOUND_REPLACE("nessound", NES_APU, PAL_CLOCK)
	MCFG_SOUND_CONFIG(nes_apu_interface)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 0.90)
MACHINE_CONFIG_END

static MACHINE_CONFIG_DERIVED( dendy, nes )

	/* basic machine hardware */
	MCFG_CPU_MODIFY( "maincpu" )
	MCFG_CPU_CLOCK( 26601712/15 ) /* 26.601712MHz / 15 == 1.77344746666... MHz */

	MCFG_DEVICE_REMOVE("ppu")
	MCFG_PPU2C07_ADD("ppu", nes_ppu_interface)
	MCFG_PPU2C0X_SET_NMI(nes_state, ppu_nmi)

	/* video hardware */
	MCFG_SCREEN_MODIFY("screen")
	MCFG_SCREEN_REFRESH_RATE(50.00697796827)
	MCFG_SCREEN_VBLANK_TIME(ATTOSECONDS_IN_USEC((106.53/(PAL_CLOCK/1000000)) * (PPU_VBLANK_LAST_SCANLINE_PAL-PPU_VBLANK_FIRST_SCANLINE+1+2)))

	/* sound hardware */
	MCFG_SOUND_REPLACE("nessound", NES_APU, 26601712/15) /* 26.601712MHz / 15 == 1.77344746666... MHz */
	MCFG_SOUND_CONFIG(nes_apu_interface)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 0.90)
MACHINE_CONFIG_END

static MACHINE_CONFIG_DERIVED( famicom, nes )
	MCFG_DEVICE_REMOVE("nes_slot")
	MCFG_NES_CARTRIDGE_ADD("nes_slot", nes_crt_interface, nes_cart, NULL)
	MCFG_NES_CARTRIDGE_NOT_MANDATORY

	MCFG_LEGACY_FLOPPY_DRIVE_ADD(FLOPPY_0, nes_floppy_interface)
	MCFG_SOFTWARE_LIST_ADD("flop_list", "famicom_flop")

	MCFG_CASSETTE_ADD( "tape", fc_cassette_interface )
	MCFG_SOFTWARE_LIST_ADD("cass_list", "famicom_cass")
MACHINE_CONFIG_END


/* rom regions are just place-holders: they get removed and re-allocated when a cart is loaded */
ROM_START( nes )
	ROM_REGION( 0x10000, "maincpu", ROMREGION_ERASE00 )  /* Main RAM */
ROM_END

ROM_START( nespal )
	ROM_REGION( 0x10000, "maincpu", ROMREGION_ERASE00 )  /* Main RAM */
ROM_END

ROM_START( famicom )
	ROM_REGION( 0x10000, "maincpu", 0 )  /* Main RAM */
	ROM_SYSTEM_BIOS( 0, "2c33a-01a", "Famicom Disk System Bios")
	ROMX_LOAD( "rp2c33a-01a.bin", 0xe000, 0x2000, CRC(5e607dcf) SHA1(57fe1bdee955bb48d357e463ccbf129496930b62), ROM_BIOS(1)) // newer, Nintendo logo has no shadow
	ROM_SYSTEM_BIOS( 1, "2c33-01", "Famicom Disk System Bios, older")
	ROMX_LOAD( "rp2c33-01.bin", 0xe000, 0x2000, CRC(1c7ae5d5) SHA1(af5af53f66982e749643fdf8b2acbb7d4d3ed229), ROM_BIOS(2)) // older, Nintendo logo has shadow
ROM_END

ROM_START( famitwin )
	ROM_REGION( 0x10000, "maincpu", 0 )  /* Main RAM */
	ROM_LOAD( "rp2c33a-02.bin", 0xe000, 0x2000, CRC(4df24a6c) SHA1(e4e41472c454f928e53eb10e0509bf7d1146ecc1) ) // "Famicom" logo instead of Nintendo logo
ROM_END

ROM_START( m82 )
	ROM_REGION( 0x14000, "maincpu", 0 )  /* Main RAM + program banks */
	/* Banks to be mapped at 0xe000? More investigations needed... */
	ROM_LOAD( "m82_v1_0.bin", 0x10000, 0x4000, CRC(7d56840a) SHA1(cbd2d14fa073273ba58367758f40d67fd8a9106d) )
ROM_END

// see http://www.disgruntleddesigner.com/chrisc/drpcjr/index.html
// and http://www.disgruntleddesigner.com/chrisc/drpcjr/DrPCJrMemMap.txt
ROM_START( drpcjr )
	ROM_REGION( 0x18000, "maincpu", 0 )  /* Main RAM + program banks */
	/* 4 banks to be mapped in 0xe000-0xffff (or 8 banks to be mapped in 0xe000-0xefff & 0xf000-0xffff).
	Banks selected by writing at 0x4180 */
	ROM_LOAD("drpcjr_bios.bin", 0x10000, 0x8000, CRC(c8fbef89) SHA1(2cb0a817b31400cdf27817d09bae7e69f41b062b) ) // bios vers. 1.0a
	// Not sure if we should support this: hacked version 1.5a by Chris Covell with bugfixes and GameGenie support
//  ROM_LOAD("drpcjr_v1_5_gg.bin", 0x10000, 0x8000, CRC(98f2033b) SHA1(93c114da787a19279d1a46667c2f69b49e25d4f1) )
ROM_END

ROM_START( dendy )
	ROM_REGION( 0x10000, "maincpu", ROMREGION_ERASE00 )  /* Main RAM */
ROM_END

ROM_START( gchinatv )
	ROM_REGION( 0x10000, "maincpu", ROMREGION_ERASE00 )  /* Main RAM */
	ROM_REGION( 0x800,   "ciram", ROMREGION_ERASE00 )  /* CI RAM */
ROM_END

/***************************************************************************

  Game driver(s)

***************************************************************************/

/*     YEAR  NAME      PARENT  COMPAT MACHINE   INPUT    INIT    COMPANY       FULLNAME */
CONS( 1985, nes,       0,      0,     nes,      nes, driver_device,     0,       "Nintendo",  "Nintendo Entertainment System / Famicom (NTSC)", GAME_IMPERFECT_GRAPHICS | GAME_SUPPORTS_SAVE )
CONS( 1987, nespal,    nes,    0,     nespal,   nes, driver_device,     0,       "Nintendo",  "Nintendo Entertainment System (PAL)", GAME_IMPERFECT_GRAPHICS | GAME_SUPPORTS_SAVE )
CONS( 1983, famicom,   nes,    0,     famicom,  famicom, nes_state, famicom, "Nintendo",  "Famicom (w/ Disk System add-on)", GAME_IMPERFECT_GRAPHICS | GAME_SUPPORTS_SAVE )
CONS( 1986, famitwin,  nes,    0,     famicom,  famicom, nes_state, famicom, "Sharp",     "Famicom Twin", GAME_IMPERFECT_GRAPHICS | GAME_SUPPORTS_SAVE )
CONS( 198?, m82,       nes,    0,     nes,      nes, driver_device,     0,       "Nintendo",  "M82 Display Unit", GAME_IMPERFECT_GRAPHICS | GAME_NOT_WORKING | GAME_SUPPORTS_SAVE )
CONS( 1996, drpcjr,    nes,    0,     famicom,  famicom, nes_state, famicom, "Bung",      "Doctor PC Jr", GAME_IMPERFECT_GRAPHICS | GAME_SUPPORTS_SAVE )
CONS( 1992, dendy,     nes,    0,     dendy,    nes, driver_device,     0,       "Steepler",  "Dendy Classic", GAME_IMPERFECT_GRAPHICS | GAME_SUPPORTS_SAVE )
CONS( 198?, gchinatv,  nes,    0,     nespal,   nes, driver_device,     0,       "Golden China",  "Golden China TV Game", GAME_IMPERFECT_GRAPHICS | GAME_SUPPORTS_SAVE )
