﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/apigatewayv2/model/DeploymentStatus.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>

using namespace Aws::Utils;

namespace Aws {
namespace ApiGatewayV2 {
namespace Model {
namespace DeploymentStatusMapper {

static const int PENDING_HASH = HashingUtils::HashString("PENDING");
static const int FAILED_HASH = HashingUtils::HashString("FAILED");
static const int DEPLOYED_HASH = HashingUtils::HashString("DEPLOYED");

DeploymentStatus GetDeploymentStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == PENDING_HASH) {
    return DeploymentStatus::PENDING;
  } else if (hashCode == FAILED_HASH) {
    return DeploymentStatus::FAILED;
  } else if (hashCode == DEPLOYED_HASH) {
    return DeploymentStatus::DEPLOYED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<DeploymentStatus>(hashCode);
  }

  return DeploymentStatus::NOT_SET;
}

Aws::String GetNameForDeploymentStatus(DeploymentStatus enumValue) {
  switch (enumValue) {
    case DeploymentStatus::NOT_SET:
      return {};
    case DeploymentStatus::PENDING:
      return "PENDING";
    case DeploymentStatus::FAILED:
      return "FAILED";
    case DeploymentStatus::DEPLOYED:
      return "DEPLOYED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace DeploymentStatusMapper
}  // namespace Model
}  // namespace ApiGatewayV2
}  // namespace Aws
