// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// The background script of the test app.
//
// The app exposes a certificate to Chrome via chrome.certificateProvider API.
// This allows to test that the app and the API work by initiating an SSL
// connection that requests the client certificate. The app also tests
// displaying of the PIN dialog. Note that the app does NOT implement the real
// signing operation, therefore it's expected that the SSL connections will fail
// to establish in the end.
//
// Note that the debug output from this app is printed to the JavaScript
// console, which on the login screen may only be accessed via remote debugging.
// For instructions, please refer to:
// * https://chromedevtools.github.io/devtools-protocol/#remote
// * https://www.chromium.org/developers/how-tos/run-chromium-with-flags#TOC-Setting-Flags-for-Chrome-OS


// Raw contents of the DER-encoded X.509 certificate.
// Generated using the following command (assuming that the certificate was
// bundled in the PKCS #12 archive <path_to_pkcs12_file>):
//   openssl pkcs12 -in <path_to_pkcs12_file> -nodes | \
//     openssl x509 -outform der | xxd -i
// Current certificate expires on Dec 15, 2017.
// In case an update is required, please reach out to the managed-devices@
// mailing list.
var TEST_CERTIFICATE =
    (new Uint8Array([
      0x30, 0x82, 0x06, 0x78, 0x30, 0x82, 0x05, 0x60, 0xa0, 0x03, 0x02, 0x01,
      0x02, 0x02, 0x0a, 0x17, 0xa5, 0x51, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x8a, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
      0x01, 0x0a, 0x30, 0x00, 0x30, 0x64, 0x31, 0x13, 0x30, 0x11, 0x06, 0x0a,
      0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x03,
      0x6e, 0x65, 0x74, 0x31, 0x1b, 0x30, 0x19, 0x06, 0x0a, 0x09, 0x92, 0x26,
      0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x0b, 0x74, 0x6f, 0x74,
      0x68, 0x65, 0x63, 0x6c, 0x6f, 0x75, 0x64, 0x73, 0x31, 0x12, 0x30, 0x10,
      0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19,
      0x16, 0x02, 0x61, 0x64, 0x31, 0x1c, 0x30, 0x1a, 0x06, 0x03, 0x55, 0x04,
      0x03, 0x13, 0x13, 0x54, 0x6f, 0x74, 0x68, 0x65, 0x63, 0x6c, 0x6f, 0x75,
      0x64, 0x73, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x30, 0x1e,
      0x17, 0x0d, 0x31, 0x36, 0x31, 0x32, 0x31, 0x35, 0x31, 0x34, 0x34, 0x34,
      0x31, 0x39, 0x5a, 0x17, 0x0d, 0x31, 0x37, 0x31, 0x32, 0x31, 0x35, 0x31,
      0x34, 0x34, 0x34, 0x31, 0x39, 0x5a, 0x30, 0x81, 0x9e, 0x31, 0x13, 0x30,
      0x11, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01,
      0x19, 0x16, 0x03, 0x6e, 0x65, 0x74, 0x31, 0x1b, 0x30, 0x19, 0x06, 0x0a,
      0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x0b,
      0x74, 0x6f, 0x74, 0x68, 0x65, 0x63, 0x6c, 0x6f, 0x75, 0x64, 0x73, 0x31,
      0x12, 0x30, 0x10, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c,
      0x64, 0x01, 0x19, 0x16, 0x02, 0x61, 0x64, 0x31, 0x0e, 0x30, 0x0c, 0x06,
      0x03, 0x55, 0x04, 0x03, 0x13, 0x05, 0x55, 0x73, 0x65, 0x72, 0x73, 0x31,
      0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0a, 0x61, 0x64,
      0x66, 0x73, 0x20, 0x74, 0x65, 0x73, 0x74, 0x31, 0x31, 0x31, 0x30, 0x2f,
      0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16,
      0x22, 0x61, 0x64, 0x66, 0x73, 0x2d, 0x74, 0x65, 0x73, 0x74, 0x31, 0x40,
      0x64, 0x65, 0x76, 0x69, 0x63, 0x65, 0x73, 0x2e, 0x6e, 0x6f, 0x6d, 0x61,
      0x64, 0x69, 0x63, 0x66, 0x65, 0x65, 0x74, 0x2e, 0x63, 0x6f, 0x6d, 0x30,
      0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
      0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30,
      0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xdf, 0xf4, 0x9a, 0x94,
      0xad, 0x0f, 0xf4, 0xaf, 0xd6, 0x0e, 0xc5, 0x15, 0xf9, 0x20, 0x73, 0xa3,
      0x73, 0x04, 0xb7, 0x6a, 0x32, 0xec, 0xa6, 0x1a, 0x91, 0x1f, 0x80, 0x1f,
      0xc6, 0xfa, 0x33, 0x17, 0x1d, 0x9b, 0xb3, 0xd5, 0xe1, 0x2f, 0xcf, 0x1e,
      0xfb, 0xd7, 0x43, 0xe9, 0xe8, 0x5f, 0x8e, 0x9d, 0x9e, 0x83, 0x9b, 0x95,
      0x98, 0x5d, 0x21, 0xd2, 0xc9, 0x8a, 0x45, 0x5f, 0x0f, 0x98, 0x84, 0xfa,
      0xe8, 0x0a, 0x7d, 0x9c, 0xab, 0xe6, 0xbe, 0x03, 0x18, 0x05, 0x2a, 0x46,
      0xf0, 0x03, 0x2b, 0xf1, 0x36, 0xf6, 0x1f, 0x0a, 0xac, 0x04, 0xd6, 0x22,
      0x0a, 0xce, 0x2c, 0x3f, 0x2f, 0x25, 0xd5, 0xad, 0x84, 0xf4, 0x6b, 0xd9,
      0xbe, 0x77, 0x78, 0xcb, 0xa3, 0xe4, 0x8c, 0x2c, 0x8d, 0xdc, 0x83, 0x61,
      0x18, 0x00, 0x0f, 0x2e, 0x2c, 0x57, 0x6b, 0xfa, 0xfb, 0x1d, 0x19, 0xf7,
      0x8b, 0x2f, 0xdc, 0x3c, 0xb5, 0x69, 0x44, 0xfa, 0x81, 0x7d, 0x3d, 0x26,
      0x84, 0xad, 0xe5, 0xe6, 0xce, 0x0f, 0x1b, 0x13, 0xf6, 0x60, 0xd4, 0x5d,
      0x3f, 0xce, 0x09, 0x2d, 0x2f, 0x71, 0xda, 0x3f, 0x88, 0x51, 0xe1, 0x75,
      0xcd, 0xf0, 0xbf, 0x84, 0xa8, 0xc5, 0xb5, 0x4e, 0x26, 0x59, 0x92, 0x0a,
      0x3f, 0x5f, 0x5f, 0xcd, 0xa8, 0x67, 0x4b, 0xed, 0x22, 0xa0, 0x0d, 0x41,
      0x31, 0x0a, 0x73, 0x10, 0x42, 0xa0, 0x08, 0xe1, 0x94, 0x1b, 0xd5, 0x3e,
      0x97, 0x2b, 0x53, 0x9c, 0x65, 0xd3, 0xc8, 0x1d, 0x56, 0x0e, 0x0b, 0x4d,
      0x32, 0x6d, 0x54, 0xfc, 0x5a, 0xa8, 0x3b, 0x61, 0xc1, 0xe2, 0xf7, 0x79,
      0x16, 0xec, 0x9e, 0x81, 0x58, 0xf6, 0x9b, 0x13, 0xd9, 0x2f, 0x9b, 0x3c,
      0x5c, 0x47, 0x87, 0xff, 0x5b, 0x8d, 0xc0, 0xbe, 0x34, 0xaf, 0xd3, 0x23,
      0xeb, 0x67, 0x5a, 0xa1, 0xff, 0x9a, 0xb2, 0xf3, 0xea, 0x74, 0xbc, 0x89,
      0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, 0x02, 0xef, 0x30, 0x82, 0x02,
      0xeb, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14,
      0xbe, 0x17, 0x87, 0x1f, 0xa6, 0x79, 0xe3, 0xa6, 0x50, 0x93, 0x97, 0xa8,
      0x11, 0x60, 0x09, 0x45, 0x85, 0xd4, 0x87, 0xad, 0x30, 0x1f, 0x06, 0x03,
      0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xd0, 0xd1, 0x3f,
      0x7c, 0xcd, 0xc8, 0xe5, 0xff, 0x3f, 0x17, 0xb2, 0xfb, 0xc7, 0x51, 0xf1,
      0xbd, 0x99, 0x5d, 0x27, 0xc0, 0x30, 0x82, 0x01, 0x72, 0x06, 0x03, 0x55,
      0x1d, 0x1f, 0x04, 0x82, 0x01, 0x69, 0x30, 0x82, 0x01, 0x65, 0x30, 0x82,
      0x01, 0x61, 0xa0, 0x82, 0x01, 0x5d, 0xa0, 0x82, 0x01, 0x59, 0x86, 0x81,
      0xd1, 0x6c, 0x64, 0x61, 0x70, 0x3a, 0x2f, 0x2f, 0x2f, 0x43, 0x4e, 0x3d,
      0x54, 0x6f, 0x74, 0x68, 0x65, 0x63, 0x6c, 0x6f, 0x75, 0x64, 0x73, 0x25,
      0x32, 0x30, 0x52, 0x6f, 0x6f, 0x74, 0x25, 0x32, 0x30, 0x43, 0x41, 0x2c,
      0x43, 0x4e, 0x3d, 0x77, 0x69, 0x6e, 0x2d, 0x70, 0x6b, 0x69, 0x2d, 0x73,
      0x65, 0x72, 0x76, 0x65, 0x72, 0x30, 0x31, 0x2c, 0x43, 0x4e, 0x3d, 0x43,
      0x44, 0x50, 0x2c, 0x43, 0x4e, 0x3d, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63,
      0x25, 0x32, 0x30, 0x4b, 0x65, 0x79, 0x25, 0x32, 0x30, 0x53, 0x65, 0x72,
      0x76, 0x69, 0x63, 0x65, 0x73, 0x2c, 0x43, 0x4e, 0x3d, 0x53, 0x65, 0x72,
      0x76, 0x69, 0x63, 0x65, 0x73, 0x2c, 0x43, 0x4e, 0x3d, 0x43, 0x6f, 0x6e,
      0x66, 0x69, 0x67, 0x75, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x44,
      0x43, 0x3d, 0x61, 0x64, 0x2c, 0x44, 0x43, 0x3d, 0x74, 0x6f, 0x74, 0x68,
      0x65, 0x63, 0x6c, 0x6f, 0x75, 0x64, 0x73, 0x2c, 0x44, 0x43, 0x3d, 0x6e,
      0x65, 0x74, 0x3f, 0x63, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61,
      0x74, 0x65, 0x52, 0x65, 0x76, 0x6f, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e,
      0x4c, 0x69, 0x73, 0x74, 0x3f, 0x62, 0x61, 0x73, 0x65, 0x3f, 0x6f, 0x62,
      0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x3d, 0x63, 0x52,
      0x4c, 0x44, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
      0x6e, 0x50, 0x6f, 0x69, 0x6e, 0x74, 0x86, 0x41, 0x68, 0x74, 0x74, 0x70,
      0x3a, 0x2f, 0x2f, 0x70, 0x6b, 0x69, 0x2e, 0x74, 0x6f, 0x74, 0x68, 0x65,
      0x63, 0x6c, 0x6f, 0x75, 0x64, 0x73, 0x2e, 0x6e, 0x65, 0x74, 0x2f, 0x43,
      0x65, 0x72, 0x74, 0x45, 0x6e, 0x72, 0x6f, 0x6c, 0x6c, 0x2f, 0x54, 0x6f,
      0x74, 0x68, 0x65, 0x63, 0x6c, 0x6f, 0x75, 0x64, 0x73, 0x25, 0x32, 0x30,
      0x52, 0x6f, 0x6f, 0x74, 0x25, 0x32, 0x30, 0x43, 0x41, 0x2e, 0x63, 0x72,
      0x6c, 0x86, 0x40, 0x66, 0x69, 0x6c, 0x65, 0x3a, 0x2f, 0x2f, 0x5c, 0x5c,
      0x77, 0x69, 0x6e, 0x2d, 0x70, 0x6b, 0x69, 0x2d, 0x73, 0x65, 0x72, 0x76,
      0x65, 0x72, 0x30, 0x31, 0x5c, 0x43, 0x65, 0x72, 0x74, 0x45, 0x6e, 0x72,
      0x6f, 0x6c, 0x6c, 0x5c, 0x54, 0x6f, 0x74, 0x68, 0x65, 0x63, 0x6c, 0x6f,
      0x75, 0x64, 0x73, 0x25, 0x32, 0x30, 0x52, 0x6f, 0x6f, 0x74, 0x25, 0x32,
      0x30, 0x43, 0x41, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x3b, 0x06, 0x08, 0x2b,
      0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x2f, 0x30, 0x2d, 0x30,
      0x2b, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x86,
      0x1f, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x70, 0x6b, 0x69, 0x2e,
      0x74, 0x6f, 0x74, 0x68, 0x65, 0x63, 0x6c, 0x6f, 0x75, 0x64, 0x73, 0x2e,
      0x6e, 0x65, 0x74, 0x2f, 0x6f, 0x63, 0x73, 0x70, 0x30, 0x17, 0x06, 0x09,
      0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x14, 0x02, 0x04, 0x0a, 0x1e,
      0x08, 0x00, 0x55, 0x00, 0x73, 0x00, 0x65, 0x00, 0x72, 0x30, 0x0e, 0x06,
      0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x05,
      0xa0, 0x30, 0x29, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x22, 0x30, 0x20,
      0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x0a, 0x03, 0x04,
      0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x04, 0x06, 0x08,
      0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x30, 0x5c, 0x06, 0x03,
      0x55, 0x1d, 0x11, 0x04, 0x55, 0x30, 0x53, 0xa0, 0x2d, 0x06, 0x0a, 0x2b,
      0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x14, 0x02, 0x03, 0xa0, 0x1f, 0x0c,
      0x1d, 0x61, 0x64, 0x66, 0x73, 0x2d, 0x74, 0x65, 0x73, 0x74, 0x31, 0x40,
      0x61, 0x64, 0x2e, 0x74, 0x6f, 0x74, 0x68, 0x65, 0x63, 0x6c, 0x6f, 0x75,
      0x64, 0x73, 0x2e, 0x6e, 0x65, 0x74, 0x81, 0x22, 0x61, 0x64, 0x66, 0x73,
      0x2d, 0x74, 0x65, 0x73, 0x74, 0x31, 0x40, 0x64, 0x65, 0x76, 0x69, 0x63,
      0x65, 0x73, 0x2e, 0x6e, 0x6f, 0x6d, 0x61, 0x64, 0x69, 0x63, 0x66, 0x65,
      0x65, 0x74, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x44, 0x06, 0x09, 0x2a, 0x86,
      0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x0f, 0x04, 0x37, 0x30, 0x35, 0x30,
      0x0e, 0x06, 0x08, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x03, 0x02, 0x02,
      0x02, 0x00, 0x80, 0x30, 0x0e, 0x06, 0x08, 0x2a, 0x86, 0x48, 0x86, 0xf7,
      0x0d, 0x03, 0x04, 0x02, 0x02, 0x00, 0x80, 0x30, 0x07, 0x06, 0x05, 0x2b,
      0x0e, 0x03, 0x02, 0x07, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0x86,
      0xf7, 0x0d, 0x03, 0x07, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
      0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00,
      0x2d, 0xb1, 0x92, 0xad, 0xc5, 0x3d, 0x4d, 0x41, 0x51, 0x7d, 0x38, 0xa0,
      0x7a, 0x9b, 0x67, 0x51, 0x7e, 0x5b, 0x8f, 0xd6, 0x2a, 0x6f, 0xb5, 0x67,
      0x58, 0xeb, 0x86, 0x1f, 0xf6, 0x9e, 0xfa, 0x06, 0x12, 0x5d, 0x3d, 0xce,
      0x61, 0xba, 0x8f, 0x23, 0x89, 0xba, 0xd4, 0x2a, 0x50, 0xee, 0x1d, 0xda,
      0x23, 0x7f, 0x19, 0x30, 0x10, 0xf6, 0xa4, 0xea, 0xbf, 0x0d, 0xbb, 0x4d,
      0x3d, 0x4b, 0x1c, 0x03, 0xd5, 0x85, 0xbc, 0x42, 0x69, 0x59, 0xed, 0xb2,
      0x0c, 0x68, 0x39, 0x7e, 0x8a, 0x00, 0xd0, 0x46, 0xe3, 0x59, 0xa4, 0xf5,
      0x99, 0x43, 0xb1, 0xdf, 0x36, 0xfa, 0x18, 0x3c, 0xef, 0xc2, 0x51, 0x96,
      0x55, 0x35, 0x79, 0xb6, 0x01, 0x1f, 0x54, 0xd1, 0xf0, 0xc5, 0x36, 0xe7,
      0x11, 0x6f, 0xe7, 0x33, 0x5c, 0x87, 0xb5, 0x23, 0x9b, 0xcd, 0x5c, 0x97,
      0x3e, 0x8c, 0x64, 0x43, 0xb3, 0xe0, 0x73, 0x1c, 0x89, 0xd5, 0x83, 0x60,
      0x8c, 0xa7, 0xc7, 0x57, 0x8f, 0xc1, 0x62, 0x0a, 0x65, 0x5e, 0x80, 0x66,
      0x76, 0xb8, 0x69, 0x71, 0x6a, 0x00, 0x3a, 0xda, 0x26, 0x07, 0x9e, 0x9e,
      0x9f, 0x39, 0x62, 0x98, 0x6f, 0x76, 0x84, 0xbc, 0x8b, 0xd0, 0x12, 0x41,
      0x2e, 0x1e, 0xca, 0x58, 0xae, 0x36, 0x9a, 0x1f, 0x10, 0x6b, 0x95, 0xda,
      0x93, 0x34, 0x41, 0xba, 0xc8, 0x02, 0x37, 0x91, 0xb7, 0x28, 0x2a, 0xc5,
      0x61, 0xc6, 0x2d, 0xd0, 0x27, 0xa2, 0xeb, 0xbc, 0x51, 0xe4, 0xf2, 0x79,
      0x14, 0x2f, 0x13, 0x0a, 0xb1, 0x77, 0x0f, 0x01, 0x9a, 0xf9, 0x92, 0x2a,
      0x98, 0xb7, 0x64, 0xf2, 0xef, 0x7d, 0xb8, 0x11, 0x25, 0x2d, 0xb9, 0xce,
      0x9b, 0xb2, 0x4c, 0xde, 0x5b, 0x1e, 0x33, 0x6e, 0xa9, 0x1b, 0xbf, 0x8b,
      0x1c, 0x9a, 0x00, 0x3b, 0x28, 0x13, 0x00, 0x5c, 0x3c, 0x3b, 0x69, 0x50,
      0x5e, 0xb6, 0xfd, 0x98
    ])).buffer;

// Pretend to support all types of hashes (as documented at
// https://developer.chrome.com/extensions/certificateProvider#type-Hash).
var TEST_CERTIFICATE_SUPPORTED_HASHES =
    ['MD5_SHA1', 'SHA1', 'SHA256', 'SHA384', 'SHA512'];

// The expected PIN code.
var VALID_PIN = '1234';

// The number of wrong attempts after which requesting the PIN will be stopped.
var MAXIMUM_PIN_FAILED_ATTEMPT_COUNT = 3;

// Data that is used as a response for the sign digest request. This data is
// random and invalid (as obtaining the real signature requires bundling the
// app with a private key and with a piece of cryptographic code).
var FAKE_SIGN_DIGEST_REQUEST_RESPONSE = (new Uint8Array([1, 2, 3])).buffer;


function log(message) {
  console.log('[Sign-in Screen Test App] ' + message);
}

function logError(message) {
  console.error('[Sign-in Screen Test App] ' + message);
}

function arrayBufferToByteArray(arrayBuffer) {
  return Array.from(new Uint8Array(arrayBuffer));
}

// Returns the text dump of the array of objects of CertificateInfo type from
// the chrome.certificateProvider API.
function dumpCertificateInfos(certificateInfos) {
  var transformedList = certificateInfos.map(function(certificateInfo) {
    var transformedItem = Object.assign({}, certificateInfo);
    transformedItem.certificate = arrayBufferToByteArray(
        transformedItem.certificate);
    return transformedItem;
  });
  return JSON.stringify(transformedList);
}

function dumpArrayBuffer(arrayBuffer) {
  return JSON.stringify(arrayBufferToByteArray(arrayBuffer));
}

function dumpArrayBuffers(arrayBuffers) {
  return JSON.stringify(arrayBuffers.map(function(arrayBuffer) {
    return arrayBufferToByteArray(arrayBuffer);
  }));
}

function arrayBufferEquals(firstArrayBuffer, secondArrayBuffer) {
  return dumpArrayBuffer(firstArrayBuffer) ==
      dumpArrayBuffer(secondArrayBuffer);
}

// Listener for the chrome.certificateProvider.onCertificatesRequested event.
function certificatesRequestedListener(reportCallback) {
  log('The onCertificatesRequested event received');
  var response = [{
    certificate: TEST_CERTIFICATE,
    supportedHashes: TEST_CERTIFICATE_SUPPORTED_HASHES
  }];
  log('Responding with ' + response.length +
      ' certificate(s): ' + dumpCertificateInfos(response));
  reportCallback(response, function(rejectedCertificates) {
    if (chrome.runtime.lastError) {
      logError(
          'Failed to report the certificates: ' +
          chrome.runtime.lastError.message);
      return;
    }
    if (rejectedCertificates.length) {
      logError(
          rejectedCertificates.length + ' certificate(s) were rejected: ' +
          dumpArrayBuffers(rejectedCertificates));
      return;
    }
    log('Successfully reported the certificates');
  });
}

// Listener for the chrome.certificateProvider.onSignDigestRequested event.
function signDigestRequestedListener(request, reportCallback) {
  log('The onSignDigestRequested event received: signRequestId=' +
      request.signRequestId + ', digest=' + dumpArrayBuffer(request.digest) +
      ', hash="' + request.hash +
      '", certificate=' + dumpArrayBuffer(request.certificate));
  if (!arrayBufferEquals(request.certificate, TEST_CERTIFICATE)) {
    logError('The signing request refers to an unknown certificate');
    abandonSignRequest(request, reportCallback);
    return;
  }
  inputPinAndProcessSignRequest(
      1 /* attemptNumber */, null /* errorType */, request, reportCallback);
}

// Requests the PIN from the user and responds to the sign digest request - with
// some data if the PIN was entered successfully, or with an error otherwise.
function inputPinAndProcessSignRequest(
    attemptNumber, errorType, signRequest, signRequestReportCallback) {
  requestPin(
      signRequest.signRequestId, attemptNumber, errorType, function(userInput) {
        if (chrome.runtime.lastError) {
          logError(
              'The PIN request failed: ' + chrome.runtime.lastError.message);
          abandonSignRequest(signRequest, signRequestReportCallback);
          return;
        }
        log('The PIN dialog returned "' + userInput + '"');
        if (userInput == VALID_PIN) {
          log('The entered PIN is correct');
          stopPinRequest(null /* errorType */, signRequest.signRequestId);
          processSignRequestWithValidPin(
              signRequest, signRequestReportCallback);
        } else if (attemptNumber < MAXIMUM_PIN_FAILED_ATTEMPT_COUNT) {
          log('The entered PIN is wrong, showing the dialog again');
          inputPinAndProcessSignRequest(
              attemptNumber + 1, 'INVALID_PIN', signRequest,
              signRequestReportCallback);
        } else {
          log('The entered PIN is wrong, and no attempts left');
          stopPinRequest('MAX_ATTEMPTS_EXCEEDED', signRequest.signRequestId);
          abandonSignRequest(signRequest, signRequestReportCallback);
        }
      });
}

// Requests the PIN from the user by displaying the PIN dialog. The passed
// callback will be called with the string entered by the user, or with null in
// case of an error.
function requestPin(signRequestId, attemptNumber, errorType, callback) {
  var parameters = {signRequestId: signRequestId};
  if (errorType) {
    parameters.errorType = errorType;
    parameters.attemptsLeft =
        MAXIMUM_PIN_FAILED_ATTEMPT_COUNT - attemptNumber + 1;
  }
  log('Requesting PIN with parameters ' + JSON.stringify(parameters));
  chrome.certificateProvider.requestPin(parameters, function(result) {
    callback(result ? result.userInput : null);
  });
}

// Stops the PIN request: if no error is passed (null), the PIN dialog is
// closed; if there is some, then the corresponding message will be displayed in
// the dialog that allows no further input.
function stopPinRequest(errorType, signRequestId) {
  var parameters = {signRequestId: signRequestId};
  if (errorType)
    parameters.errorType = errorType;
  log('Stopping the PIN request with parameters ' + JSON.stringify(parameters));
  chrome.certificateProvider.stopPinRequest(parameters, function() {
    if (chrome.runtime.lastError) {
      logError(
          'Failed to stop the PIN request: ' +
          chrome.runtime.lastError.message);
      return;
    }
    log('Successfully stopped the PIN request');
  });
}

// Responds to the signing request after the PIN was entered correctly.
function processSignRequestWithValidPin(
    signRequest, signRequestReportCallback) {
  log('Responding to the signing request with a fake data');
  signRequestReportCallback(FAKE_SIGN_DIGEST_REQUEST_RESPONSE);
}

// Responds to the signing request with an error, which unblocks Chrome from
// waiting for the response from this extension.
function abandonSignRequest(signRequest, signRequestReportCallback) {
  log('Responding to the signing request with an error');
  signRequestReportCallback();
}


log('Loaded');
if (chrome.certificateProvider) {
  chrome.certificateProvider.onCertificatesRequested.addListener(
      certificatesRequestedListener);
  chrome.certificateProvider.onSignDigestRequested.addListener(
      signDigestRequestedListener);
  log('Successfully set up listeners for the certificateProvider API');
} else {
  logError('The certificateProvider API is unavailable');
}
