/* SPDX-FileCopyrightText: 2025  NVIDIA Corporation. All rights reserved.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later */
#pragma once

#include <pxr/usd/usd/stage.h>

namespace blender::io::usd {

struct USDExportParams;

/**
 * This function processes the USD stage generated by the USD hierarchy iterator to
 * change scene graph instancing prototypes from defined USD prims to abstract prims.
 *
 * In the following example, instance /root/proto/Plane_0 references prototype prim
 * /root/proto_001/Plane_0:
 *
 *   def Xform "root"
 *   {
 *       def Xform "proto_001"
 *       {
 *           def Xform "Plane_0"
 *           {
 *               def Mesh "Plane"
 *               {
 *               }
 *           }
 *       }
 *
 *       def Xform "proto"
 *       {
 *           def Xform "Plane_0" (
 *               instanceable = true
 *               prepend references = </root/proto_001/Plane_0>
 *           )
 *           {
 *           }
 *       }
 *   }
 *
 * The function will copy prototype /root/proto_001/Plane_0 under a new class prim
 * named /root/prototypes and convert prim /root/proto_001/Plane_0 to be an instance referencing
 * the copied abstract prim /root/prototypes/Plane_0.
 *
 *   def Xform "root"
 *   {
 *       def Xform "proto"
 *       {
 *           def Xform "Plane_0" (
 *               instanceable = true
 *               prepend references = </root/prototypes/Plane_0>
 *           )
 *           {
 *           }
 *       }
 *
 *       def Xform "proto_001"
 *       {
 *           def Xform "Plane_0" (
 *               instanceable = true
 *               references = </root/prototypes/Plane_0>
 *           )
 *           {
 *           }
 *       }
 *
 *      class "prototypes"
 *      {
 *           def Xform "Plane_0"
 *           {
 *               def Mesh "Plane"
 *               {
 *               }
 *           }
 *      }
 *   }
 *
 * \param export_params: The export parameters
 *
 * \param stage: Pointer to the stage to process
 *
 */
void process_scene_graph_instances(const USDExportParams &export_params,
                                   pxr::UsdStageRefPtr stage);

}  // namespace blender::io::usd
