/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/exponential/cdf' );
import Exponential = require( './../../../../../base/dists/exponential/ctor' );
import entropy = require( './../../../../../base/dists/exponential/entropy' );
import kurtosis = require( './../../../../../base/dists/exponential/kurtosis' );
import logcdf = require( './../../../../../base/dists/exponential/logcdf' );
import logpdf = require( './../../../../../base/dists/exponential/logpdf' );
import mean = require( './../../../../../base/dists/exponential/mean' );
import median = require( './../../../../../base/dists/exponential/median' );
import mgf = require( './../../../../../base/dists/exponential/mgf' );
import mode = require( './../../../../../base/dists/exponential/mode' );
import pdf = require( './../../../../../base/dists/exponential/pdf' );
import quantile = require( './../../../../../base/dists/exponential/quantile' );
import skewness = require( './../../../../../base/dists/exponential/skewness' );
import stdev = require( './../../../../../base/dists/exponential/stdev' );
import variance = require( './../../../../../base/dists/exponential/variance' );

/**
* Interface describing the `exponential` namespace.
*/
interface Namespace {
	/**
	* Exponential distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param lambda - rate parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 0.1 );
	* // returns ~0.181
	*
	* var myCDF = ns.cdf.factory( 0.5 );
	* y = myCDF( 3.0 );
	* // returns ~0.777
	*/
	cdf: typeof cdf;

	/**
	* Exponential distribution.
	*/
	Exponential: typeof Exponential;

	/**
	* Returns the entropy of an exponential distribution.
	*
	* ## Notes
	*
	* -   If provided a negative value for `λ`, the function returns `NaN`.
	*
	* @param lambda - rate parameter
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 9.0 );
	* // returns ~-1.197
	*
	* @example
	* var v = ns.entropy( 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.entropy( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of an exponential distribution.
	*
	* ## Notes
	*
	* -   If provided a negative value for `λ`, the function returns `NaN`.
	*
	* @param lambda - rate parameter
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 9.0 );
	* // returns 6.0
	*
	* @example
	* var v = ns.kurtosis( 1.0 );
	* // returns 6.0
	*
	* @example
	* var v = ns.kurtosis( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Exponential distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param lambda - rate parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 2.0, 0.1 );
	* // returns ~-1.708
	*
	* var mylogcdf = ns.logcdf.factory( 0.5 );
	* y = mylogcdf( 3.0 );
	* // returns ~-0.252
	*/
	logcdf: typeof logcdf;

	/**
	* Exponential distribution natural logarithm of probability density function (logPDF).
	*
	* @param x - input value
	* @param lambda - rate parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 0.3, 4.0 );
	* // returns ~0.186
	*
	* var mylogpdf = ns.logpdf.factory( 0.5 );
	*
	* y = mylogpdf( 3.0 );
	* // returns ~-2.193
	*
	* y = mylogpdf( 1.0 );
	* // returns ~-1.193
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of an exponential distribution.
	*
	* ## Notes
	*
	* -   If provided a negative value for `λ`, the function returns `NaN`.
	*
	* @param lambda - rate parameter
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 9.0 );
	* // returns ~0.111
	*
	* @example
	* var v = ns.mean( 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.mean( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of an exponential distribution.
	*
	* ## Notes
	*
	* -   If provided a negative value for `λ`, the function returns `NaN`.
	*
	* @param lambda - rate parameter
	* @returns median
	*
	* @example
	* var v = ns.median( 9.0 );
	* // returns ~0.077
	*
	* @example
	* var v = ns.median( 1.0 );
	* // returns ~0.693
	*
	* @example
	* var v = ns.median( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Exponential distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param lambda - rate parameter
	* @returns evaluated MGF
	*
	* @example
	* var v = ns.mgf( 2.0, 3.0 );
	* // returns 3.0
	*
	* var mymgf = ns.mgf.factory( 4.0 );
	* v = mymgf( 3.0 );
	* // returns 4.0
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of an exponential distribution.
	*
	* ## Notes
	*
	* -   If provided a negative value for `λ`, the function returns `NaN`.
	*
	* @param lambda - rate parameter
	* @returns mode
	*
	* @example
	* var v = ns.mode( 9.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Exponential distribution probability density function (PDF).
	*
	* @param x - input value
	* @param lambda - rate parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 0.3, 4.0 );
	* // returns ~1.205
	*
	* var myPDF = ns.pdf.factory( 0.5 );
	*
	* y = myPDF( 3.0 );
	* // returns ~0.112
	*
	* y = myPDF( 1.0 );
	* // returns ~0.303
	*/
	pdf: typeof pdf;

	/**
	* Exponential distribution quantile function.
	*
	* @param p - input value
	* @param lambda - rate parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.5, 4.0 );
	* // returns ~0.173
	*
	* var myQuantile = ns.quantile.factory( 0.4 );
	*
	* y = myQuantile( 0.4 );
	* // returns ~1.277
	*
	* y = myQuantile( 1.0 );
	* // returns Infinity
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of an exponential distribution.
	*
	* ## Notes
	*
	* -   If provided a negative value for `λ`, the function returns `NaN`.
	*
	* @param lambda - rate parameter
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 9.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.skewness( 1.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.skewness( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of an exponential distribution.
	*
	* ## Notes
	*
	* -   If provided a negative value for `λ`, the function returns `NaN`.
	*
	* @param lambda - rate parameter
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 9.0 );
	* // returns ~0.11
	*
	* @example
	* var v = ns.stdev( 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.stdev( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of an exponential distribution.
	*
	* ## Notes
	*
	* -   If provided a negative value for `λ`, the function returns `NaN`.
	*
	* @param lambda - rate parameter
	* @returns variance
	*
	* @example
	* var v = ns.variance( 9.0 );
	* // returns ~0.012
	*
	* @example
	* var v = ns.variance( 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.variance( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Exponential distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
