package main

import (
	"pkg.deepin.io/daemon/sync/infrastructure/hardware"
	"pkg.deepin.io/daemon/sync/infrastructure/log"
	"pkg.deepin.io/daemon/sync/infrastructure/utils"
	"pkg.deepin.io/daemon/sync/modules"

	"math/rand"
	"time"

	"github.com/godbus/dbus"
)

const (
	regionCN = "CN"
)

type deepinidToken struct {
}

func (dt *deepinidToken) Token() string {
	conn, err := dbus.SessionBus()
	if err != nil {
		log.Error(err)
		return ""
	}

	caller := conn.Object("com.deepin.deepinid", "/com/deepin/deepinid")
	token := ""
	caller.Call("com.deepin.deepinid.GetToken", 0).Store(&token)

	return token
}

// SyncState indicate the state of synchronization
type SyncState struct {
	Code    modules.StateType
	Message string
}

// Daemon is sync main daemon
type Daemon struct {
	utils.DBusProxy

	conn  *dbus.Conn
	core  *modules.Manager
	hw    *hardware.Hardware
	state *SyncState

	userID       int64
	lastSyncTime int64  // last sync time
	region       string // only support 'CN'
	token        *deepinidToken

	machineName string
}

func newDaemon(hw *hardware.Hardware) (*Daemon, error) {
	conn, err := dbus.SessionBus()
	if err != nil {
		return nil, err
	}

	var d = Daemon{
		conn:  conn,
		hw:    hw,
		token: &deepinidToken{},
		state: &SyncState{},
	}

	return &d, nil
}

func (d *Daemon) initModules() error {
	core, err := modules.NewManager(&d.userID, d.hw.ID, d.token)
	if err != nil {
		return err
	}
	d.core = core
	return nil
}

// Monitor user info change
func (d *Daemon) Monitor(conn *dbus.Conn) {
	// Setup sync daemon
	// Receive pusher server message
	caller := conn.Object("com.deepin.pusher", "/com/deepin/pusher")
	caller.Call("com.deepin.pusher.Register", 0, "sync", dbusName, dbusPath, dbusIFC).Store()

	// Receive login signal
	caller = conn.Object("com.deepin.deepinid", "/com/deepin/deepinid")
	caller.AddMatchSignal("org.freedesktop.DBus.Properties", "PropertiesChanged")

	go func() {
		c := make(chan *dbus.Signal, 10)
		d.conn.Signal(c)
		for v := range c {
			if len(v.Body) < 2 {
				continue
			}
			values := v.Body[1].(map[string]dbus.Variant)
			v, ok := values["UserInfo"]
			if !ok {
				continue
			}
			userInfo := utils.UserInfo{}
			userInfo.FromDBus(v.Value())
			d.userID = userInfo.UserID
			d.region = userInfo.Region
			if userInfo.IsValid() {
				d.pushMachineData()
				d.lastSyncTime, _ = getLastSyncTime(d.userID)
				println("Last sync time from file:", d.lastSyncTime)
			}
		}
	}()
}

func (d *Daemon) startTimer(n int) {
	time.Sleep(time.Second * 5)
	if n == 0 {
		rand.Seed(time.Now().UnixNano())
		// 5~30 min after boot
		n = rand.Intn(1800)
		if n < 300 {
			n = 900 - n
		}
	}
	log.Debug("[Timer] duration seconds:", n)
	ticker := time.NewTicker(time.Second * time.Duration(n))
	syncing := false
	for {
		select {
		case <-ticker.C:
			if !syncing {
				syncing = true
				d.doSync()
				syncing = false
			}
		}
	}
	ticker.Stop()
}

func (d *Daemon) doSync() {
	// only sync region in 'CN'
	if d.userID < 1 || d.region != regionCN {
		return
	}

	d.state.Code = modules.CodeSynchronizing
	d.state.Message = modules.CodeSynchronizing.Message()
	d.Emit(dbusPropState, d.state)
	_, serr := d.core.Sync()
	if serr != nil {
		d.state.Code = serr.Code
		d.state.Message = serr.Error()
		d.Emit(dbusPropState, d.state)
		log.Warning("Failed to sync modules:", serr)
		return
	}

	d.state.Code = modules.CodeSuccess
	d.state.Message = modules.CodeSuccess.Message()
	d.Emit(dbusPropState, d.state)

	// TODO(jouyouyun): optimize
	if !d.core.SwitcherGet("enabled") {
		return
	}

	timestamp := time.Now().Unix()
	d.SetInt64Prop(&d.lastSyncTime, dbusPropLastSyncTime, timestamp)
	err := setLastSyncTime(d.userID, timestamp)
	if err != nil {
		log.Warning("Failed to set last sync time:", err)
	}
}

func (d *Daemon) pushMachineData() {
	preData, _ := loadMachineData()
	data := d.hw.Bytes()
	if string(preData) == string(data) {
		log.Debug("The machine data had uploaded, nothing to do...")
		return
	}
	err := putMachineData(data, d.hw.ID, d.token)
	if err != nil {
		log.Warning("Failed to upload machine data:", err)
		return
	}
	err = saveMachineData(data)
	if err != nil {
		log.Warning("Failed to save machine data:", err)
	}
}
