#!/usr/bin/python

try:
  import os, sys
  if os.path.exists('/usr/lib/woody'):
    sys.path.append('/usr/lib/woody')
  from woody_defs import *
  import woody_tree
  #import woody
  from woody_iface import *
except IOError, msg:
  print "%s: Fatal Error loading program modules: %s" % (__name__, msg)
  sys.exit(1)
except ImportError, msg:
  print "%s: Fatal Error loading program modules: %s" % (__name__, msg)
  sys.exit(1)


from slang import *


class slang_iface(iface):


  def __init__(self, cfg):
    global version

    self.scr_wid = 80
    self.scr_hgt = 24
    self.statustextl = program_name
    self.statustextr = "Version " + version
    self.tty_cursor_x = 0
    self.tty_cursor_y = 0

    # Color-pair definitions...
    self.first_color = 1
    self.num_colors = 19
    (self.POPUP_C, self.EDIT_BORDER_C, self.EDIT_TEXT_C, \
     self.TOPBAR_C, self.BOTBAR_C, \
     self.NODE_C, self.C_NODE_C, \
     self.PAST_DUE_C, self.C_PAST_DUE_C, \
     self.MENUBAR_C, self.H_MENUBAR_C, self.C_MENUBAR_C, \
     self.DIALOG_C, self.C_DIALOG_C,
     self.FILE_BKGD_C, self.FILE_TEXT_C, \
     self.FILE_ENTRY_C, self.FILE_PARTIAL_C, self.FILE_COMPLETE_C, ) = \
     range(self.first_color, self.num_colors + self.first_color)

    sl_init()

    self.scr_wid, self.scr_hgt = get_screen_size()

    # Doesn't seem to work:
    #import signal
    #
    #signal.signal(signal.SIGWINCH, self.sigwinch_handler)
    
    # Colors:
    self.colorstrings = [
      ("popup", self.POPUP_C),
      ("edit border", self.EDIT_BORDER_C),
      ("edit text", self.EDIT_TEXT_C),
      ("top bar", self.TOPBAR_C),
      ("bottom bar", self.BOTBAR_C),
      ("node", self.NODE_C),
      ("selected node", self.C_NODE_C),
      ("past due", self.PAST_DUE_C),
      ("selected past due", self.C_PAST_DUE_C),
      ("menu bar", self.MENUBAR_C),
      ("highlit menu bar", self.H_MENUBAR_C),
      ("menu bar cursor", self.C_MENUBAR_C),
      ("dialog box", self.DIALOG_C),
      ("dialog button", self.C_DIALOG_C),
      ("file background", self.FILE_BKGD_C),
      ("file text", self.FILE_TEXT_C),
      ("file entry", self.FILE_ENTRY_C),
      ("file partial", self.FILE_PARTIAL_C),
      ("file complete", self.FILE_COMPLETE_C),
      #("", self._C),
      ]

    self.default_colors = {
      "popup" : ("yellow", "gray"),
      "edit border" : ("yellow", "gray"),
      "edit text" : ("white", "gray"),
      "top bar" : ("white", "blue"),
      "bottom bar" : ("white", "blue"),
      "node" : ("white", "black"),
      "selected node" : ("white", "magenta"),
      "past due" : ("red", "black"),
      "selected past due" : ("red", "magenta"),
      "menu bar" : ("yellow", "red"),
      "highlit menu bar" : ("white", "red"),
      "menu bar cursor" : ("red", "white"),
      "dialog box" : ("white", "blue"),
      "dialog button" : ("black", "white"),
      "file background" : ("yellow", "gray"),
      "file text" : ("white", "gray"),
      "file entry" : ("white", "red"),
      "file partial" : ("white", "magenta"),
      "file complete" : ("white", "blue"),
      #"" : ("", ""),
      }

    SLtt_set_color(0, '', 'gray', 'default');

    for col in self.colorstrings:
      str, num = col
      if cfg.colors.has_key(str):
        fore, back = cfg.colors[str]
      else:
        fore, back = self.default_colors[str]
      SLtt_set_color(num, '', fore, back);
        
        


  ############################################################
  #
  #
  def sigwinch_handler(self, signum, frame):
    self.scr_wid, self.scr_hgt = get_screen_size()
    print "Changed screen size to %sx%s" % (self.scr_wid, self.scr_hgt)
    

  ############################################################
  # Controls the statusbar at the bottom...
  #
  def set_status(self, left=None, right=None):

    if left != None:
      self.statustextl = left
    if right != None:
      self.statustextr = right

    SLsmg_gotorc(self.scr_hgt-1, 0)
    cprint(self.BOTBAR_C, self.lrpad(self.statustextl, self.statustextr + " "))

    # position cursor for braille tty users
    SLsmg_gotorc(self.tty_cursor_y, self.tty_cursor_x)

    SLsmg_refresh()

  ############################################################
  # 
  #
  def suspend(self):
    SLtt_set_cursor_visibility(1)
    SLsmg_suspend_smg()

  ############################################################
  # 
  #
  def resume(self):
    # FIXME: This is a nasty hack!
    # (slang doesn't seem to have these basic tty control functions)
    import curses
    curses.initscr()
    curses.cbreak()
    curses.raw()
    curses.noecho()

    SLsmg_resume_smg()
    SLtt_set_cursor_visibility(0)

  ############################################################
  # Returns text with space between left and right halves...
  #
  def lrpad(self, left, right):
    pad = " " * (self.scr_wid - len(left) - len(right))
    text = left + pad + right
    return text

  ############################################################
  # - introduction screen function
  #
  def intro(self, text):
    wid, hgt = self.scr_wid, self.scr_hgt
    cx, cy = wid/2, hgt/2
    SLsmg_set_color(self.POPUP_C)
    SLsmg_fill_region(cy - hgt/2, cx - wid/2, hgt, wid, ord(" "))
    SLsmg_draw_box(cy - hgt/2, cx - wid/2, hgt, wid)
    SLsmg_write_wrapped_string(text, cy-hgt/2+1, cx-wid/2+1,
                               hgt-3, wid-3, 0)
    SLsmg_refresh()
    SLkp_getkey()
    SLsmg_cls()
    SLsmg_refresh()
    

  ############################################################
  # - Generic message to user...
  #
  def message(self, text):
    wid, hgt = (self.scr_wid * 3 / 4, self.scr_hgt)
    cx, cy = (self.scr_wid / 2,  self.scr_hgt / 2)
    SLsmg_set_color(self.POPUP_C)
    SLsmg_fill_region(cy - hgt/2, cx - wid/2, hgt, wid, ord(" "))
    SLsmg_draw_box(cy - hgt/2, cx - wid/2, hgt, wid)
    SLsmg_write_wrapped_string(text, cy-hgt/2+1, cx-wid/2+1,
                               hgt-3, wid-3, 0)
    SLsmg_refresh()
    SLkp_getkey()
    #SLsmg_cls()
    #SLsmg_refresh()

  ############################################################
  # - Generic message to user...
  #
  def note(self, text):
    import string
    lines = string.split(text, "\n")
    wid = self.scr_wid * 3 / 4
    hgt = len(lines) + 2
    if hgt > self.scr_hgt:  hgt = self.scr_hgt
    cx, cy = (self.scr_wid / 2,  self.scr_hgt / 2)
    SLsmg_set_color(self.POPUP_C)
    SLsmg_fill_region(cy - hgt/2, cx - wid/2, hgt, wid, ord(" "))
    SLsmg_draw_box(cy - hgt/2, cx - wid/2, hgt, wid)
    SLsmg_write_wrapped_string(text, cy-hgt/2+1, cx-wid/2+1,
                               hgt-2, wid-3, 0)
    SLsmg_refresh()

  ############################################################
  #
  #
  def main_loop(self, cfg, wood):
  
    state = screen_state()

    state.quit = 0
    state.screen_top = 0
    state.cursor_pos = 0
    state.window_hgt = state.items_shown = self.scr_hgt - 3
    state.title_left = "%s (%s items)" % (wood.children.text, wood.len())
    state.title_right = "%s %s" % (program_name, version)
    state.empty = 1
    state.foocount = 0
    state.options = wood.options
    state.helptext = \
"""Help is supposed to go here...

But it's not here yet.

For now, look at the included help tree:
  woody help.tree

This contains all the current documentation.
"""
    

    ## Get the screen size
    self.scr_wid, self.scr_hgt = get_screen_size()

    wood.vis_count()
    wlist = wood.vis_list()


    ## Main UI loop
    while 0 == state.quit:

      # Make sure cursor is non-negative
      if state.cursor_pos < 0: state.cursor_pos = 0
      # Make sure cursor doesn't go off screen top
      if state.cursor_pos < state.screen_top:
        state.screen_top = state.cursor_pos
      # Make sure cursor doesn't fall off the list
      if state.cursor_pos >= len(wlist): state.cursor_pos = len(wlist)-1
      # Make sure screen top stays on list
      if state.screen_top >= len(wlist):
        state.screen_top = len(wlist) - 1
      # Make sure cursor doesn't drop off bottom of screen
      if state.cursor_pos >= state.screen_top + state.items_shown:
        state.screen_top = state.cursor_pos - state.items_shown + 1
      # Check again to make sure everything is non-negative
      if state.screen_top < 0: state.screen_top = 0
      if state.cursor_pos < 0: state.cursor_pos = 0

      if wlist == []:
        state.empty = 1
      else:
        state.empty = 0


      if wood.modified:
        state.title_left = "%s (%s items)" % (wood.children.text, wood.len())

      self.draw_screen(state, wlist, wood)

      
      if wood.modified:
        self.set_status(right="(modified) %s" % wood.filename)
      else:
        self.set_status(right=wood.filename)
      key = SLkp_getkey()
      self.set_status("")

      #branch = wood.get_node(state.cursor_pos)
      if state.empty:
        branch, depth = None, 0
      else:
        branch, depth = wlist[state.cursor_pos]
    
      state.help = 0
      # TODO: Use a key table and functions, instead of if/else (?)
      # TODO: Figure out proper keycodes for arrows
      # Perhaps..
      # - get a key, then run...
      # - keyfuncs[keytable[key]](state, wlist)
      #   (where keytable translates numbers to keycodes, and
      #    keyfuncs translates keycodes to function pointers)



      # List of keys and what they should do...
      # - q: quit
      # - up/down: move cursor
      # - pgup/pgdown: move cursor faster
      # - left/right: (de)indent branches
      # - enter: edit node text
      # - N: edit note
      # - e: edit node (pop up node info screen)
      # - n/p: move node down/up (next/prev)
      # - space: collapse/expand branch
      # - d: delete node
      # - c: create node
      # - t: type (todo, progress, or normal)
      # - x: check todo, or pop up percentage done selector
      # - !: choose priority
      # - o: tree options
      # - h/?: help
      # - s: save

      #if key < 256:
      #  bar = "Key: %s (%s)" % (key, chr(key))
      #else:
      #  bar = "Key: %s" % key
      #print bar
      #self.set_status("", bar)
      
      # q
      if key == ord('q'):
        state.quit = 1
        if wood.modified:
          ans = self.dialog_box("Unsaved work...",
                                "You have made changes since you last " \
                                "saved.  Are you sure you want to quit?",
                                [("Yes", "quit"),
                                 "No",
                                 ("Save & Quit", "save")], 1)
          if ans == "quit":
            state.quit = 1
          elif ans == "save":
            self.set_status("Saving file %s.." % wood.filename)
            wood.save()
            self.set_status("Saved file %s.." % wood.filename)
            state.quit = 1
          else:
            state.quit = 0
      # s
      elif key == ord('s'):
        self.set_status("Saving file %s.." % wood.filename)
        wood.save()
        self.set_status("Saved file %s.." % wood.filename)
        
      # i
      elif key == ord('i'):
        self.set_status("Importing tree...")
        import woody
        filename = self.fileselector("Import", "Choose a tree to import")
        if filename:
          twig = branch
          if not branch:
            twig = wood.children
          twig.import_tree(filename, cfg, wood.callback)
          wlist = wood.vis_list()
          self.set_status("Imported tree: %s" % filename)
        else:
          self.set_status("Import aborted.")
        
      # f
      elif key == ord('f'):
        self.fileselector("Title",
                          "You want to do something with a file here...")
        
      # T
      elif key == ord('T'):
        self.set_status("Exporting file %s.txt.." % wood.filename,
                        wood.filename)
        wood.save_text()
        self.set_status("Exported file %s.txt.." % wood.filename,
                        wood.filename)
        
      # z
      elif key == ord('z'):
        self.set_status("Starting shell...")
        self.run_shell()
        self.set_status("Shell done..")
        
      # enter
      elif key == 13  and  not state.empty:
        wood.modified = 1
        if wood.options.editor:
          self.suspend()
          foop = run_external_editor(wood, branch.text)
          self.resume()
          if type(foop) == type(1)  and  foop < 0:
            self.edit_text(branch, depth, state)
          else:
            branch.text = foop
        else:
          self.edit_text(branch, depth, state)
      # N
      elif key == ord('N')  and  not state.empty:
        wood.modified = 1
        if wood.options.note_editor:
          self.suspend()
          foop = run_external_editor(wood, branch.note)
          self.resume()
          if type(foop) == type(1)  and  foop < 0:
            print "External editor failed!"
            self.edit_text(branch, depth, state)
          else:
            branch.note = foop
        else:
          # TODO: implement notes in internal editor
          pass
      # c
      elif key == ord('c'):
        wood.modified = 1
        self.set_status("Creating node...")
        #n = node(text="Blubber %s" % state.foocount)
        n = woody_tree.node()
        state.foocount = state.foocount + 1
        if state.empty:
          wood.add(n)
        else:
          branch.add(n)
        state.cursor_pos = wood.get_num_node(n) - 1
        #print "Cursor_pos = %s" % state.cursor_pos
        wlist = wood.vis_list()
        if state.cursor_pos >= state.screen_top + state.window_hgt:
          state.screen_top = state.cursor_pos - state.window_hgt + 1
        self.draw_screen(state, wlist, wood)
        branch, depth = wlist[state.cursor_pos]
        self.edit_text(branch, depth, state)
        self.set_status("Creating node...  Done.")
      # d
      elif key == ord('d')  and  not state.empty:
        wood.modified = 1
        self.set_status("Deleting node...")
        # FIXME: confirm deletion...
        ans = self.dialog_box("Delete branch",
                              "Are you sure you want to delete this branch?",
                              [("Yes", "delete"), "No"], 0)
        if ans == "delete":
          branch.destroy()
          wlist = wood.vis_list()
          self.set_status("Deleting node...  Done.")
        else:
          self.set_status("Deleting node...  Aborted.")
##         if branch.todo  and  branch.parent:
##           branch.todo = None
##           branch.parent.recalc_progress()
      # space: expand/collapse branch
      elif key == ord(' ')  and  not state.empty:
        wood.modified = 1
        self.set_status("Expanding/collapsing branch...")
        branch.expand_collapse(wood)
        state.cursor_pos = wood.get_num_node(branch) - 1
        #if len(branch.children) > 0:
        #  if branch.expanded: branch.expanded = 0
        #  else: branch.expanded = 1
        wlist = wood.vis_list()
        self.set_status("Expanding/collapsing branch...  Done.")
      # t
      elif key == ord('t')  and  not state.empty:
        wood.modified = 1
        #print "Todo? %s %s" % (branch.todo, type(branch.todo))
        if not branch.todo  or  branch.todo == 0:
          branch.todo = "todo"
        elif branch.todo == "todo":
          branch.todo = "progress"
        elif branch.todo == "progress":
          branch.todo = None
        if branch:
          branch.recalc_progress()
        if branch.parent:
          branch.parent.recalc_progress()
      # x
      elif key == ord('x')  and  not state.empty:
        wood.modified = 1
        if branch.todo:
          if branch.todo == "todo":
            if branch.done:
              branch.done = 0
            else:
              branch.done = 1
          elif branch.todo == "progress":
            # TODO: pop up a percentage window...
            self.choose_percent(branch, depth, state)
          if branch.parent:
            branch.parent.recalc_progress()
          if not wood.options.show_done_items:
            wlist = wood.vis_list()
      # S
      elif key == ord('S')  and  not state.empty:
        wood.modified = 1
        self.set_status("Sorting branch...")
        if branch.children:
          branch.sort("name")
        elif branch.parent:
          branch.parent.sort("name")
        wlist = wood.vis_list()
        self.set_status("Sorting branch...  Done.")
      # h
      elif key == ord('h'):
        # TODO: write a *real* help screen...
        self.message(state.helptext)
      # Up
      elif key == 257:
        state.cursor_pos = state.cursor_pos - 1
      # Down
      elif key == 258:
        state.cursor_pos = state.cursor_pos + 1
        self.check_bottom(state)
      # PgUp
      elif key == 261:
        if state.cursor_pos > 0:
          state.cursor_pos = state.cursor_pos - state.items_shown + 1
          if state.cursor_pos < 0: state.cursor_pos = 0
      # PgDn
      elif key == 262:
        if state.cursor_pos < len(wlist):
          state.cursor_pos = state.cursor_pos + state.items_shown - 1
          self.check_bottom(state)
          if state.cursor_pos >= len(wlist): state.cursor_pos = len(wlist)
      # Left
      elif key == 259  and  not state.empty:
        wood.modified = 1
        branch.move_left()
        wlist = wood.vis_list()
      # Right
      elif key == 260  and  not state.empty:
        wood.modified = 1
        branch.move_right()
        wlist = wood.vis_list()
        self.check_bottom(state)
      # n
      elif key == ord('n'):
        wood.modified = 1
        self.set_status("Moving node down...")
        branch.move_down()
        state.cursor_pos = wood.get_num_node(branch) - 1
        wlist = wood.vis_list()
        self.set_status("Moving node down...  Done.")
      # p
      elif key == ord('p'):
        wood.modified = 1
        self.set_status("Moving node up...")
        branch.move_up()
        state.cursor_pos = wood.get_num_node(branch) - 1
        wlist = wood.vis_list()
        self.set_status("Moving node up...  Done.")
      # m, F10
      elif key == ord('m')  or  key == 522:
        self.set_status("Invoking menu...")
        self.handle_menus()
        self.set_status("Menu done.")
      # Ctrl-L
      elif key == 12:
        SLsmg_touch_lines(0, self.scr_hgt)
      else:
        pass
        foop = "Unrecognized key! (%s)" % key
        if key >= 32  and  key < 256:
          foop = foop + ' "%s"' % chr(key)
        print foop


      
    SLsmg_cls()
    SLsmg_refresh()

    return None

  ############################################################
  #
  #
  def draw_screen(self, state, wlist, wood, refresh=1):
    import string

    ### Menu bar at the top...
    SLsmg_gotorc(0, 0)
    self.draw_menubar()

    ### Screen header and title...
    SLsmg_gotorc(1, 0)
    cprint(self.TOPBAR_C, self.lrpad(state.title_left, state.title_right))

    ### Node list...
    self.tty_cursor_y = 0
    self.tty_cursor_x = 0
    depth = 0
    i = 0
    y = 0
    state.item_lengths = []
    while y < state.window_hgt:
      # Handle rows beyond the end of the list
      if i + state.screen_top >= len(wlist)  or  i < 0:
        line = ""
      # Draw a node
      else:
        line = ""
        #print "screen_top: %s  i: %s" % (state.screen_top, i)
        n, depth = wlist[state.screen_top + i]
        line = woody_tree.node_to_str(wood, n, depth, wid=self.scr_wid)

      #print "line: %s" % line
      lines = string.split(line, "\n")
      state.item_lengths.append(len(lines))

      if i + state.screen_top == state.cursor_pos:
        col = self.C_NODE_C
        self.tty_cursor_y = y+2
        self.tty_cursor_x = 1 + depth * 4
      else:
        col = self.NODE_C

      for line in lines:
        SLsmg_gotorc(y+2, 0)
        cprint(col, self.lrpad(line, ""))
        y = y + 1

      
      i = i + 1

    # Figure out how long the next item is, for better scrolling...
    if i - 1 + state.screen_top == state.cursor_pos  and \
       i + state.screen_top < len(wlist):
      n, depth = wlist[state.screen_top + i]
      line = woody_tree.node_to_str(wood, n, depth, wid=self.scr_wid)
      lines = string.split(line, "\n")
      state.next_item_len = len(lines)
    else:
      state.next_item_len = 0
    
    state.leftover = y - state.window_hgt
    state.items_shown = i

    ### Screen footer and status / summary
    SLsmg_gotorc(self.scr_hgt-1, 0)
    cprint(self.BOTBAR_C, self.lrpad(self.statustextl, self.statustextr + " "))


    # position cursor for braille tty users
    SLsmg_gotorc(self.tty_cursor_y, self.tty_cursor_x)


    if refresh:
      SLsmg_refresh()


  ############################################################
  #
  #
  def draw_menubar(self, top=None):
    import woody_menu

    if not top:
      top = woody_menu.woody_menus()

    SLsmg_gotorc(0,0)
    cprint(self.MENUBAR_C, " " * self.scr_wid)
    SLsmg_gotorc(0,0)
    
    for entry in range(len(top.menubar.items)):
      item = top.menubar.items[entry]
      if entry + 1 == top.menubar.cursor:
        color = self.C_MENUBAR_C
      else:
        color = self.MENUBAR_C
      for i in range(len(item.text)):
        letter = item.text[i]
        if letter == "&":
          if entry + 1 == top.menubar.cursor:
            color = self.C_MENUBAR_C
          else:
            color = self.H_MENUBAR_C
        else:
          cprint(color, letter)
          if entry + 1 == top.menubar.cursor:
            color = self.C_MENUBAR_C
          else:
            color = self.MENUBAR_C
      cprint(self.MENUBAR_C, "  ")


  ############################################################
  #
  #
  def handle_menus(self, m=None):
    import woody_menu

    #SLsmg_gotorc(0,0)
    #saveunder = None
    #saveunder = SLsmg_read_raw(self.scr_wid * self.scr_hgt)

    self.set_status("Menus don't work yet.  Press any key to continue.", "")

    #cprint(self.POPUP_C, "Booger butt!")

    if m:
      pass
    else:
      top = woody_menu.woody_menus()
      top.menubar.cursor = 1
      self.draw_menubar(top)

      #SLsmg_gotorc(0,0)
      #SLsmg_write_raw(saveunder)
      #SLsmg_write_raw(self.scr_wid * self.scr_hgt)

      SLsmg_refresh()
      key = SLkp_getkey()
    

  ############################################################
  #
  def check_bottom(self, state):
    # If we're at the bottom of the screen...
    # ... make sure the last item is entirely visible!
    if state.cursor_pos >= state.screen_top + state.items_shown - 1:
      #self.set_status("Scrolling to fit last item..", "")
      if state.leftover <= 0:
        state.leftover = state.next_item_len
      if state.leftover > 0:
        #self.set_status("LeftOver: %s" % state.leftover, "")
        if state.item_lengths:
          state.screen_top = state.screen_top + 1
          state.leftover = state.leftover - state.item_lengths[0]
          #self.set_status("LeftOver: %s" % state.leftover, "")
          del state.item_lengths[0]
          while state.leftover > 0  and \
                state.cursor_pos > state.screen_top:
            state.screen_top = state.screen_top + 1
            state.leftover = state.leftover - state.item_lengths[0]
            #self.set_status("LeftOver: %s" % state.leftover, "")
            del state.item_lengths[0]
        else:
          #self.set_status("No item length info!", "")
          pass
          
  ############################################################
  #
  def edit_text(self, branch, depth, state):
    #top = state.cursor_pos - state.screen_top
    top = 1
    for i in range(state.cursor_pos - state.screen_top):
      top = top + state.item_lengths[i]
    indent_unit = 3
    if state.options.show_todo:
      indent_unit = indent_unit + 1
    left = depth * indent_unit + 3
    if state.options.show_todo:
      left = left + 4
    if state.options.show_priorities:
      left = left + 4
    hgt = 6
    wid = -20

    branch.text = self.entry_box(branch.text, top, left, hgt, wid)

  ############################################################
  #
  def entry_box(self, text, top, left, hgt=1, wid=-1):

    import string

    SLtt_set_cursor_visibility(1) # doesn't work on win32

    # do automatic sizing...
    if hgt <= 0:
      foo = self.scr_hgt - top
      if foo < -hgt:
        foo = -hgt
      hgt = foo
    if wid <= 0:
      foo = self.scr_wid - left
      if foo < -wid:
        foo = -wid
      wid = foo

    # and be sure to check screen edges to avoid flowing over..
    if top + hgt >= self.scr_hgt:
      top = self.scr_hgt - hgt
    if left + wid >= self.scr_wid:
      left = self.scr_wid - wid

    if top < 0:
      top = 0
    if wid < 0:
      wid = 0

    if top + hgt >= self.scr_hgt:
      hgt = self.scr_hgt - top
    if left + wid >= self.scr_wid:
      wid = self.scr_wid - left


    if not text:
      text = ""


    # TODO: proper line calculation
    # TODO: word wrap!
    # TODO: A good text editor for these entries!
    lines = []
    done = 0
    while not done:
      SLsmg_set_color(self.EDIT_BORDER_C)
      SLsmg_fill_region(top, left, hgt, wid, ord(" "))
      SLsmg_draw_box(top, left, hgt, wid)
      SLsmg_set_color(self.EDIT_TEXT_C)
      if text:
        offset = 0
        lines = woody_tree.wordwrap(text, wid-3)
        offset = len(lines) - hgt + 2
        if offset < 0:  offset = 0
        display_text = string.join(lines[offset:], "\n")
        SLsmg_write_wrapped_string(display_text, top+1, left+1,
                                   hgt-2, wid-3, 0)
      else:
        SLsmg_write_wrapped_string(text, top+1, left+1,
                                   hgt-2, wid-3, 0)
        
      SLsmg_refresh()
      key = SLkp_getkey()

      # Enter: done
      if key == 13:
        done = 1
      # Tab (inserts a return)
      elif key == ord("\t"):
        text = text + "\n"
      # Ctrl-u or Ctrl-k deletes current line
      elif key == 21  or  key == 11:
        if lines != []:
          offset = string.rfind(lines[-1], "\n")
          if offset > 0:
            offset = offset - len(lines[-1])
            text = text[:offset]
          else:
            offset = len(lines[-1])
            if offset > 0:
              text = text[:-offset]
            elif len(text) > 0:
              text = text[:-1]
      # Regular character
      elif key >= 32  and  key < 256:
        text = text + "%s" % chr(key)
      # Backspace
      elif key == 272  or  key == 8:
        if len(text) > 0:
          text = text[:-1]

    SLtt_set_cursor_visibility(0) # doesn't work on win32
    return text
  
    
  ############################################################
  #
  def choose_percent(self, branch, depth, state):
    import string
    
    # This code copied from above, in edit_text()
    top = 1
    for i in range(state.cursor_pos - state.screen_top):
      top = top + state.item_lengths[i]
    indent_unit = 3
    if state.options.show_todo:
      indent_unit = indent_unit + 1
    left = depth * indent_unit # + 3
    if state.options.show_todo:
      left = left + 4
    if state.options.show_priorities:
      left = left + 4
    if branch.done >= 0.1:
      left = left - 1
    if branch.done >= 1.0:
      left = left - 1
    # end copied code

    hgt = 3
    wid = 10
    SLsmg_set_color(self.POPUP_C)

    if branch.done:
      text = "%3.2f" % (float(branch.done) * 100.0)
    else:
      text = "0"

    text = self.entry_box(text, top, left, hgt, wid)

    #for i in range(len(text)):
    #  if text[i] not in "0123456789.-+":
    #while text[-1] not in "%":
    #  text = text[:-1]

    try:
      percent = float(text) / 100.0
      if percent < 0.0:
        percent = 0.0
      if percent > 1.0:
        percent = 1.0

      branch.done = percent
    except ValueError, msg:
      pass
    


  ############################################################
  #
  def dialog_box(self, title, text, buttons, default):
    # first calculate width and height of box
    # width
    wid = 2
    for button in buttons:
      if type(button) == type((1,2)):
        label, value = button
      else:
        label = button
        value = button
      wid = wid + 6 + len(label)

    wid = max(wid, len(title))
    wid = max(wid, self.scr_wid / 3)

    # height
    import string
    lines = woody_tree.wordwrap(text, wid-4)
    text = string.join(lines, "\n")
    hgt = 4 + len(lines)

    # Draw the box
    cx = self.scr_wid / 2
    cy = self.scr_hgt / 2
    left = cx - (wid/2)
    top =  cy - (hgt/2)

    SLsmg_set_color(self.DIALOG_C)
    SLsmg_fill_region(top, left, hgt, wid, ord(" "))
    SLsmg_draw_box(top, left, hgt, wid)
        
    # draw the title
    if title:
      SLsmg_write_wrapped_string(title, top, cx-(len(title)/2),
                                 1, len(title), 0)
    # draw the text
    SLsmg_write_wrapped_string(text, top+1, left+2,
                               hgt-3, wid-4, 0)
    # loop:
    done = 0
    if default >= 0:
      current = default

    value = None
    
    while not done:
      #   draw the buttons
      column = left + 2
      for i in range(len(buttons)):
        SLsmg_set_color(self.DIALOG_C)

        button = buttons[i]
        if type(button) == type((1,2)):
          label, tmp = button
        else:
          label = button
          tmp = button

        if i == current:
          self.tty_cursor_x = column + 1
          self.tty_cursor_y = top + hgt - 2
          value = tmp
          SLsmg_set_color(self.C_DIALOG_C)

        SLsmg_gotorc(top + hgt - 2, column)
        SLsmg_write_string("[ %s ]" % label)
        column = column + len(label) + 6
          
      #   get a keypress
      SLsmg_gotorc(self.tty_cursor_y, self.tty_cursor_x)
      
      SLsmg_refresh()
      key = SLkp_getkey()

      #   check the key value...
      # return
      if key == 10  or  key == 13:
        return value
      # left / up
      elif key == 257  or  key == 259:
        current = current - 1
        if current < 0:
          current = 0
      # right / down
      elif key == 258  or  key == 260:
        current = current + 1
        if current >= len(buttons):
          current = len(buttons) - 1
      # tab
      elif key == 9:
        current = current + 1
        if current >= len(buttons):
          current = 0
      
    # return button (code should never get here, though)
    return value


  ############################################################
  # This lets the user choose a file.
  #
  # Arrows move the cursor up and down in the list
  # enter selects a file or opens a directory
  # right opens a directory
  # left opens the ".." directory
  # ascii keystrokes go into an entry box
  # Tab completes the current filename, with the currently selected
  #     incomplete match.
  #
  # When entering a filename, cursor colors will change.
  # There are colors to distinguish between complete filenames
  # (matches one in the list), incomplete filenames (matches part of a
  # file in the list), and new files (doesn't match at all).
  #
  def fileselector(self, title, text="Choose a file", default=None, dir="."):
    #self.set_status("File selector!")
    import glob, string

    # Draw once: title, text
    # Draw many: border, dir, files, cursor, entry, scrollbar

    done = 0
    result = default
    entrytext = default
    entrystatus = check_filename(entrytext)
    
    top = 3
    numlines = self.scr_hgt - 4 - top
    top_offset = 0
    reread_dir = 1
    cursor_pos = 0

    if dir and  dir != ".":
      try:
        os.chdir(dir)
        dir = "."
      except:
        pass

    # Draw the background
    cx = self.scr_wid / 2
    cy = self.scr_hgt / 2

    SLsmg_set_color(self.FILE_BKGD_C)
    SLsmg_fill_region(0, 0, self.scr_hgt, self.scr_wid, ord(" "))


    # draw the title
    if not title:
      title = " "
    SLsmg_write_wrapped_string(title, 1, cx - (len(title)/2),
                               1, len(title), 0)

    # draw the text message
    if text:
      SLsmg_write_wrapped_string(text, 2, cx-(len(text)/2),
                                 1, len(text), 0)
      top = 4
      numlines = self.scr_hgt - 4 - top

    # main loop...
    while not done:
      # draw the border
      SLsmg_set_color(self.FILE_BKGD_C)
      SLsmg_draw_box(0, 0, self.scr_hgt, self.scr_wid)

      # draw the directory
      dirtext = "[ Directory: %s ]" % (os.getcwd())
      SLsmg_gotorc(0, cx - len(dirtext)/2)
      cprint(self.FILE_TEXT_C, dirtext)

      # Get a list of filenames...
      if reread_dir:
        reread_dir = 0
        filelist = glob.glob("%s/*" % dir)
        filenamelist = []
        filenamelist.append(("../", "d"))
        for file in filelist:
          slash = string.rfind(file, "/") + 1
          if os.path.isdir(file):
            filetype = "d"
            file = file + "/"
          else:
            filetype = "f"
          filenamelist.append((file[slash:], filetype))
        filenamelist.sort(filesorter)

      # Draw the file list
      if cursor_pos < top_offset:
        top_offset = cursor_pos
      if cursor_pos >= top_offset + numlines:
        top_offset = cursor_pos - numlines + 1
      for i in range(numlines):
        if (i+top_offset) >= len(filenamelist):
          SLsmg_gotorc(i+top, 3)
          cprint(self.FILE_TEXT_C, " " * (self.scr_wid - 6))

        else:
          filename, filetype = filenamelist[i + top_offset]
          if i + top_offset == cursor_pos:
            color = self.FILE_COMPLETE_C
          else:
            color = self.FILE_TEXT_C
          SLsmg_gotorc(i+top, 3)
          cprint(color, "%s%s" % \
                 (filename, " " * (self.scr_wid - 6 - len(filename))))

      # Draw the entry box
      #entrytext = result
      if entrystatus == "complete":
        color = self.FILE_COMPLETE_C
      elif entrystatus == "partial":
        color = self.FILE_PARTIAL_C
      elif entrystatus == "new":
        color = self.FILE_ENTRY_C

      filename = entrytext
      if not filename:
        filename, filetype = filenamelist[cursor_pos]

      entrytextdisplay = filename + \
                         " " * (self.scr_wid - len(filename) - 6)
      SLsmg_gotorc(self.scr_hgt-3, 3)
      cprint(color, entrytextdisplay)

      SLsmg_refresh()
      try:
        key = SLkp_getkey()
      except KeyboardInterrupt:
        done = 1
        return None

      # Enter: select file
      if key == 10  or  key == 13:
        filename, filetype = filenamelist[cursor_pos]
        if entrystatus == "complete"  and  filetype == 'd':
          done = 0
          reread_dir = 1
          os.chdir(filename)
          dir = "."
          cursor_pos = 0
          entrytext = ""
          entrystatus = "complete"
        else:
          if entrytext:
            result = entrytext
          else:
            result = filename
          done = 1


      # Escape (doesn't work): cancel
      elif key == 27:
        done = 1
        result = None

      # up
      elif key == 257:
        cursor_pos = cursor_pos - 1
        if cursor_pos < 0:  cursor_pos = 0
        result, foo = filenamelist[cursor_pos]
        entrytext = None
        entrystatus = "complete"
        
      # PgUp
      elif key == 261:
        cursor_pos = cursor_pos - numlines
        if cursor_pos < 0:  cursor_pos = 0
        result, foo = filenamelist[cursor_pos]
        entrytext = None
        entrystatus = "complete"
        
      # down
      elif key == 258:
        cursor_pos = cursor_pos + 1
        if cursor_pos >= len(filenamelist):  cursor_pos = len(filenamelist) - 1
        result, foo = filenamelist[cursor_pos]
        entrytext = None
        entrystatus = "complete"

      # PgDown
      elif key == 262:
        cursor_pos = cursor_pos + numlines
        if cursor_pos >= len(filenamelist):  cursor_pos = len(filenamelist) - 1
        result, foo = filenamelist[cursor_pos]
        entrytext = None
        entrystatus = "complete"


      # tab
      elif key == 9:
        filename, filetype = filenamelist[cursor_pos]
        entrytext = ""
        entrystatus = "complete"

      elif key >= 32  and  key <= 127:
        if not entrytext:
          entrytext = chr(key)
        else:
          entrytext = entrytext + chr(key)
        entrystatus = check_filename(entrytext)
        cursor_pos = findfileindex(filenamelist, entrytext)

      # backspace
      elif key == 272  or  key == 8:
        if len(entrytext) > 0:
          entrytext = entrytext[:-1]
          entrystatus = check_filename(entrytext)
        if entrytext:
          cursor_pos = findfileindex(filenamelist, entrytext)

    result = "%s/%s" % (os.getcwd(), result)
    #self.set_status("File selector done! (%s)" % result)
    return result


  ############################################################
  #
  def run_shell(self):
    self.suspend()
    shell = os.environ["SHELL"]
    os.system(shell)
    self.resume()
    
  
  ############################################################
  # - shutdown: should restore screen state, etc..
  def shutdown(self):
    SLsmg_cls()
    SLsmg_refresh()
    sl_exit()
    if not debug and os.path.exists("pyslang.err"):
      os.system("/bin/rm -f pyslang.err")


  ############################################################
  #
  def callback(self, what, data):

    SLsmg_set_color(self.POPUP_C)

    if what == "file load":
      name, current, total = data
      wid = self.scr_wid * 3 / 4
      progress_meter("Loading %s" % name, self.scr_hgt/2 - 3,
                     self.scr_wid/2 - (wid / 2),
                     self.scr_wid * 3 / 4,
                     current, total)
      SLsmg_refresh()

    elif what == "file save":
      name, current, total = data
      wid = self.scr_wid * 3 / 4
      progress_meter("Saving %s" % name, self.scr_hgt/2 - 3,
                     self.scr_wid/2 - (wid / 2),
                     self.scr_wid * 3 / 4,
                     current, total)
      SLsmg_refresh()
  
############################################################
#
#
def cprint(color, text):
  SLsmg_set_color(color)
  SLsmg_write_string(text)



def greybar(y, x, len, current, total):
  if total > 0:
    donelen = current * len / total
    # TODO: Make progress bars look better...
    SLsmg_fill_region(y, x, 1, donelen, ord("#"))
    SLsmg_fill_region(y, x+donelen, 1, len-donelen, ord("-"))


def progress_meter(text, y, x, wid, current, total):
  hgt = 7
  cx = x + (wid/2)
  SLsmg_fill_region(y, x, hgt, wid, ord(" "))
  SLsmg_draw_box(y, x, hgt, wid)
  SLsmg_gotorc(y + 2, cx - len(text)/2)
  SLsmg_write_string(text)
  greybar(y+hgt-3, x+2, wid-4, current, total)


def filesorter(a, b):
  n1, t1 = a
  n2, t2 = b

  return cmp(n1, n2)
#  if t1 == t2:
#    return cmp(n1,n2)
#
#  return cmp(t1,t2)

def check_filename(filename):
  import os.path, glob
  if not filename:
    return "complete"
  if os.path.exists(filename):
    return "complete"

  if glob.glob("%s*" % filename):
    return "partial"

  return "new"

def findfileindex(filelist, pattern):
  for i in range(len(filelist)):
    filename, filetype = filelist[i]
    #print "Comparing %s to %s..." % (filename, pattern)
    check = cmp(filename, pattern)
    if check >= 0:
      return i

  return 0

############################################################
#
#
# TODO: Support "center" gravity and "ellipsis" style (?)
def padcrop(text, size, gravity="left", style="crop"):
  if len(text) <= size:
    if gravity == "left":
      str = text + " " * (size - len(text))
      return str
    elif gravity == "right":
      str = " " * (size - len(text)) + text
      return str
    elif gravity == "center":
      pass
    return None

  # Text is too big; must crop.
  if style == "crop":
    if gravity == "left":
      str = text[:size]
      return str
    elif gravity == "right":
      str = text[-size:]
      return str
    elif gravity == "center":
      pass
    return None


############################################################
#
#
# TODO: Figure out the terminal size USING SLANG, instead of stty
def get_screen_size():
  #SLtt_get_screen_size()
  #return (SLtt_Screen_Cols, SLtt_Screen_Rows)
  
  import os
  import string
  tty = os.popen("/bin/stty size")
  text = tty.readline()
  tty.close()
  list = string.split(string.strip(text))
  hgt = string.atoi(list[0])
  wid = string.atoi(list[1])

  return (wid, hgt)


def sl_init():
  import sys
  SLtt_get_terminfo()
  SLang_init_tty(-1, 0, 0)
  SLsmg_init_smg()
  SLkp_init()
  SLtt_set_cursor_visibility(0) # doesn't work on win32
  #SLtt_set_mouse_mode(1,1)
  sys.stdout = sys.stderr

def sl_exit():
  SLtt_set_cursor_visibility(1) # doesn't work on win32
  SLsmg_reset_smg();
  SLang_reset_tty();


############################################################
#
#
def is_screen_state(obj):
  return hasattr(obj,'is_screen_state') and obj.is_screen_state

class screen_state:
  is_screen_state = 1
  
  def __init__(self):
    self.cursor_pos = 0
    self.screen_top = 0
    self.window_hgt = 0
    self.items_shown = 0
    self.item_lengths = []
    self.next_item_len = 0
    self.leftover = 0
    self.description = 1
    self.help = 0
    self.helptext = ""
    self.size = ""
    self.title_left = ""
    self.title_right = ""
    self.options = None




############################################################
#
#
def run():
  print "This module is not intended to be executable."


############################################################
#
#
if __name__ == '__main__':
  run()

