#!/usr/bin/python

"""
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
"""

__version__ = "$Revision: 1.16 $"
# $Source: /cvsroot/quantlib/QuantLib-Python/QuantLib/test/barrier_option.py,v $

import QuantLib
import unittest
from math import fabs

class BarrierOptionTest(unittest.TestCase):
    def runTest(self):
        "Testing barrier option pricer"
        maxErrorAllowed = 5e-5
        maxStraddleErrorAllowed = 5e-4
        underPrice = 100
        rebate = 3
        resTime = 0.5
        rRate = 0.08
        divRate = 0.04
        # this table is from:
        # "Option pricing formulas", E.G. Haug, McGraw-Hill 1998
        # pag 72
        values = []
#                       barrType,  vol,strike, barrier, (   Call,    Put)
        values.append(("DownOut", 0.25,    90,      95, ( 9.0246,  2.2798)))
        values.append(("DownOut", 0.25,   100,      95, ( 6.7924,  2.2947)))
        values.append(("DownOut", 0.25,   110,      95, ( 4.8759,  2.6252)))
        values.append(("DownOut", 0.25,    90,     100, ( 3.0000,  3.0000)))
        values.append(("DownOut", 0.25,   100,     100, ( 3.0000,  3.0000)))
        values.append(("DownOut", 0.25,   110,     100, ( 3.0000,  3.0000)))
        values.append(("UpOut",   0.25,    90,     105, ( 2.6789,  3.7760)))
        values.append(("UpOut",   0.25,   100,     105, ( 2.3580,  5.4932)))
        values.append(("UpOut",   0.25,   110,     105, ( 2.3453,  7.5187)))

        values.append(("DownIn",  0.25,    90,      95, ( 7.7627,  2.9586)))
        values.append(("DownIn",  0.25,   100,      95, ( 4.0109,  6.5677)))
        values.append(("DownIn",  0.25,   110,      95, ( 2.0576, 11.9752)))
        values.append(("DownIn",  0.25,    90,     100, (13.8333,  2.2845)))
        values.append(("DownIn",  0.25,   100,     100, ( 7.8494,  5.9085)))
        values.append(("DownIn",  0.25,   110,     100, ( 3.9795, 11.6465)))
        values.append(("UpIn",    0.25,    90,     105, (14.1112,  1.4653)))
        values.append(("UpIn",    0.25,   100,     105, ( 8.4482,  3.3721)))
        values.append(("UpIn",    0.25,   110,     105, ( 4.5910,  7.0846)))

        values.append(("DownOut", 0.30,    90,      95, ( 8.8334,  2.4170)))
        values.append(("DownOut", 0.30,   100,      95, ( 7.0285,  2.4258)))
        values.append(("DownOut", 0.30,   110,      95, ( 5.4137,  2.6246)))
        values.append(("DownOut", 0.30,    90,     100, ( 3.0000,  3.0000)))
        values.append(("DownOut", 0.30,   100,     100, ( 3.0000,  3.0000)))
        values.append(("DownOut", 0.30,   110,     100, ( 3.0000,  3.0000)))
        values.append(("UpOut",   0.30,    90,     105, ( 2.6341,  4.2293)))
        values.append(("UpOut",   0.30,   100,     105, ( 2.4389,  5.8032)))
        values.append(("UpOut",   0.30,   110,     105, ( 2.4315,  7.5649)))

        values.append(("DownIn",  0.30,    90,      95, ( 9.0093,  3.8769)))
        values.append(("DownIn",  0.30,   100,      95, ( 5.1370,  7.7989)))
        values.append(("DownIn",  0.30,   110,      95, ( 2.8517, 13.3078)))
        values.append(("DownIn",  0.30,    90,     100, (14.8816,  3.3328)))
        values.append(("DownIn",  0.30,   100,     100, ( 9.2045,  7.2636)))
        values.append(("DownIn",  0.30,   110,     100, ( 5.3043, 12.9713)))
        values.append(("UpIn",    0.30,    90,     105, (15.2098,  2.0658)))
        values.append(("UpIn",    0.30,   100,     105, ( 9.7278,  4.4226)))
        values.append(("UpIn",    0.30,   110,     105, ( 5.8350,  8.3686)))

        for option in values:
            barrType, vol, strike, barrier, results = option
            opCall = QuantLib.BarrierOption(barrType, "Call", underPrice,
                strike, divRate, rRate, resTime, vol, barrier, rebate)
            calculated = opCall.value()
            expected   = results[0]
            error = fabs(calculated - expected)
            if not (error <= maxErrorAllowed):
                self.fail("""
%(barrType)7s Call %4(strike)i %4(barrier)i
    value:    %(calculated)8.4f
    expected: %(expected)8.4f
    error:    %(error)12.2e
                """ % locals())

            opPut = QuantLib.BarrierOption(barrType, "Put", underPrice,
                strike, divRate, rRate, resTime, vol, barrier, rebate)
            calculated = opPut.value()
            expected   = results[1]
            error = fabs(calculated - expected)
            if not (error <= maxErrorAllowed):
                self.fail("""
%(barrType)7s Put %4(strike)i %4(barrier)i
    value:    %(calculated)8.4f
    expected: %(expected)8.4f
    error:    %(error)12.2e
                """ % locals())

            opStraddle = QuantLib.BarrierOption(barrType, "Straddle",
                underPrice, strike, divRate, rRate, resTime, vol, barrier,
                rebate)
            calculated = opStraddle.value()
            expected   = results[0] + results[1]
            error = fabs(calculated - expected)
            if not (error <= maxStraddleErrorAllowed):
                self.fail("""
%(barrType)7s Straddle %(strike)4i %(barrier)4i
    value:    %(calculated)8.4f
    expected: %(expected)8.4f
    error:    %(error)12.2e
                """ % locals())

if __name__ == '__main__':
    print 'testing QuantLib', QuantLib.__version__, QuantLib.QuantLibc.__file__, QuantLib.__file__
    import sys
    suite = unittest.TestSuite()
    suite.addTest(BarrierOptionTest())
    if sys.hexversion >= 0x020100f0:
        unittest.TextTestRunner(verbosity=2).run(suite)
    else:
        unittest.TextTestRunner().run(suite)
    raw_input('press any key to continue')

