
/*
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

// $Id: Vectors.i,v 1.16 2002/01/16 14:50:51 nando Exp $

#ifndef quantlib_vectors_i
#define quantlib_vectors_i

%include Types.i
%include String.i

%{
#include <vector>
typedef std::vector<int> IntVector;
typedef std::vector<double> DoubleVector;
using QuantLib::Null;
using QuantLib::IntegerFormatter;
using QuantLib::DoubleFormatter;
%}

// typemap Python sequence of ints to std::vector<int>

%typemap(python,in) IntVector (IntVector temp), IntVector * (IntVector temp),
  const IntVector & (IntVector temp), IntVector & (IntVector temp) {
    IntVector* v;
    if (PyTuple_Check($source) || PyList_Check($source)) {
        /* Size */ int size = (PyTuple_Check($source) ?
            PyTuple_Size($source) :
            PyList_Size($source));
        temp = IntVector(size);
        $target = &temp;
        for (/* Size */ int i=0; i<size; i++) {
            PyObject* o = PySequence_GetItem($source,i);
            if (o == Py_None) {
                (*$target)[i] = Null<int>();
                Py_DECREF(o);
            } else if (PyInt_Check(o)) {
                (*$target)[i] = int(PyInt_AsLong(o));
                Py_DECREF(o);
            } else {
                PyErr_SetString(PyExc_TypeError,"ints expected");
                Py_DECREF(o);
                return NULL;
            }
        }
    } else if ((SWIG_ConvertPtr($source,(void **) &v,
                                SWIGTYPE_p_IntVector,0)) != -1) {
        $target = v;
    } else {
        PyErr_SetString(PyExc_TypeError,"IntVector expected");
        return NULL;
    }
};

class IntVector {
  public:
    ~IntVector();
};

%addmethods IntVector {
    IntVector(const IntVector& v) {
        return new IntVector(v);
    }
    String __str__() {
        String s = "(";
        /* Size */ int size_ = self->size();
        if (size_>0) {
            for (/* Size */ int i=0; i<size_-1; i++) {
                s += IntegerFormatter::toString((*self)[i]);
                s += ", ";
            }
            s += IntegerFormatter::toString((*self)[self->size()-1]);
        }
        s += ")";
        return s;
    }
    int __len__() {
        return self->size();
    }
    int __getitem__(int i) {
        int size_ = static_cast<int>(self->size());
        if (i>=0 && i<size_) {
            return (*self)[i];
        } else if (i<0 && -i<=size_) {
            return (*self)[size_+i];
        } else {
            throw IndexError("IntVector index out of range");
        }
        QL_DUMMY_RETURN(0)
    }
    void __setitem__(int i, int x) {
        int size_ = static_cast<int>(self->size());
        if (i>=0 && i<size_) {
            (*self)[i] = x;
        } else if (i<0 && -i<=size_) {
            (*self)[size_+i] = x;
        } else {
            throw IndexError("IntVector index out of range");
        }
    }
    IntVector __getslice__(int i, int j) {
        int size_ = static_cast<int>(self->size());
        if (i<0)
            i = size_+i;
        if (j<0)
            j = size_+j;
        i = QL_MAX(0,i);
        j = QL_MIN(size_,j);
        IntVector tmp(j-i);
        std::copy(self->begin()+i,self->begin()+j,tmp.begin());
        return tmp;
    }
    void __setslice__(int i, int j, const IntVector& rhs) {
        int size_ = static_cast<int>(self->size());
        if (i<0)
            i = size_+i;
        if (j<0)
            j = size_+j;
        i = QL_MAX(0,i);
        j = QL_MIN(size_,j);
        QL_ENSURE(static_cast<int>(rhs.size()) == j-i,
			"IntVectors are not resizable");
        std::copy(rhs.begin(),rhs.end(),self->begin()+i);
    }
    bool __nonzero__() {
        return (self->size() != 0);
    }
};


// typemap Python sequence of doubles to std::vector<double>

%typemap(python,in) DoubleVector (DoubleVector temp),
  DoubleVector * (DoubleVector temp), const DoubleVector & (DoubleVector temp),
  DoubleVector & (DoubleVector temp) {
    DoubleVector* v;
    if (PyTuple_Check($source) || PyList_Check($source)) {
        /* Size */ int size = (PyTuple_Check($source) ?
            PyTuple_Size($source) :
            PyList_Size($source));
        temp = DoubleVector(size);
        $target = &temp;
        for (/* Size */ int i=0; i<size; i++) {
            PyObject* o = PySequence_GetItem($source,i);
            if (o == Py_None) {
                (*$target)[i] = Null<double>();
                Py_DECREF(o);
            } else if (PyFloat_Check(o)) {
                (*$target)[i] = PyFloat_AsDouble(o);
                Py_DECREF(o);
            } else if (PyInt_Check(o)) {
                (*$target)[i] = double(PyInt_AsLong(o));
                Py_DECREF(o);
            } else {
                PyErr_SetString(PyExc_TypeError,"doubles expected");
                Py_DECREF(o);
                return NULL;
            }
        }
    } else if ((SWIG_ConvertPtr($source,(void **) &v,
                                SWIGTYPE_p_DoubleVector,0)) != -1) {
        $target = v;
    } else {
        PyErr_SetString(PyExc_TypeError,"DoubleVector expected");
        return NULL;
    }
};

class DoubleVector {
  public:
    ~DoubleVector();
};

%addmethods DoubleVector {
    DoubleVector(const DoubleVector& v) {
        return new DoubleVector(v);
    }
    String __str__() {
        String s = "(";
        /* Size */ int size_ = self->size();
        if (size_>0) {
            for (/* Size */ int i=0; i<size_-1; i++) {
                s += DoubleFormatter::toString((*self)[i]);
                s += ", ";
            }
            s += DoubleFormatter::toString((*self)[size_-1]);
        }
        s += ")";
        return s;
    }
    int __len__() {
        return self->size();
    }
    double __getitem__(int i) {
        int size_ = static_cast<int>(self->size());
        if (i>=0 && i<size_) {
            return (*self)[i];
        } else if (i<0 && -i<=size_) {
            return (*self)[size_+i];
        } else {
            throw IndexError("DoubleVector index out of range");
        }
        QL_DUMMY_RETURN(0.0)
    }
    void __setitem__(int i, double x) {
        int size_ = static_cast<int>(self->size());
        if (i>=0 && i<size_) {
            (*self)[i] = x;
        } else if (i<0 && -i<=size_) {
            (*self)[size_+i] = x;
        } else {
            throw IndexError("DoubleVector index out of range");
        }
    }
    DoubleVector __getslice__(int i, int j) {
        int size_ = static_cast<int>(self->size());
        if (i<0)
            i = size_+i;
        if (j<0)
            j = size_+j;
        i = QL_MAX(0,i);
        j = QL_MIN(size_,j);
        DoubleVector tmp(j-i);
        std::copy(self->begin()+i,self->begin()+j,tmp.begin());
        return tmp;
    }
    void __setslice__(int i, int j, const DoubleVector& rhs) {
        int size_ = static_cast<int>(self->size());
        if (i<0)
            i = size_+i;
        if (j<0)
            j = size_+j;
        i = QL_MAX(0,i);
        j = QL_MIN(size_,j);
        QL_ENSURE(static_cast<int>(rhs.size()) == j-i,
			"DoubleVectors are not resizable");
        std::copy(rhs.begin(),rhs.end(),self->begin()+i);
    }
    bool __nonzero__() {
        return (self->size() != 0);
    }
};


// vector of C++ classes
%define ExportVector(Type,Vector,VectorAlias)
%{
typedef std::vector<Type> Vector;
%}

%name(VectorAlias) class Vector {
  public:
    ~Vector();
};

%addmethods Vector {
    Vector(const Vector& v) {
        return new Vector(v);
    }
    int __len__() {
        return self->size();
    }
    Type __getitem__(int i) {
        int size_ = static_cast<int>(self->size());
        if (i>=0 && i<size_) {
            return (*self)[i];
        } else if (i<0 && -i<=size_) {
            return (*self)[size_+i];
        } else {
            throw IndexError("VectorAlias" " index out of range");
        }
        QL_DUMMY_RETURN(Type())
    }
    void __setitem__(int i, const Type& x) {
        int size_ = static_cast<int>(self->size());
        if (i>=0 && i<size_) {
            (*self)[i] = x;
        } else if (i<0 && -i<=size_) {
            (*self)[size_+i] = x;
        } else {
            throw IndexError("VectorAlias" " index out of range");
        }
    }
    Vector __getslice__(int i, int j) {
        int size_ = static_cast<int>(self->size());
        if (i<0)
            i = size_+i;
        if (j<0)
            j = size_+j;
        i = QL_MAX(0,i);
        j = QL_MIN(size_,j);
        Vector tmp(j-i);
        std::copy(self->begin()+i,self->begin()+j,tmp.begin());
        return tmp;
    }
    void __setslice__(int i, int j, const Vector& rhs) {
        int size_ = static_cast<int>(self->size());
        if (i<0)
            i = size_+i;
        if (j<0)
            j = size_+j;
        i = QL_MAX(0,i);
        j = QL_MIN(size_,j);
        QL_ENSURE(static_cast<int>(rhs.size()) == j-i,
			"VectorAlias" "s are not resizable");
        std::copy(rhs.begin(),rhs.end(),self->begin()+i);
    }
    bool __nonzero__() {
        return (self->size() != 0);
    }
};
%enddef

%define TypemapVector(Type,TypeAlias,Vector,VectorAlias)
%{
typedef std::vector<Type> Vector;
%}
%types(Vector *);
%typemap(python,in) Vector (Vector temp), Vector * (Vector temp),
  const Vector & (Vector temp), Vector & (Vector temp) {
    Vector* v;
    if (PyTuple_Check($source) || PyList_Check($source)) {
        /* Size */ int size = (PyTuple_Check($source) ?
            PyTuple_Size($source) :
            PyList_Size($source));
        temp = Vector(size);
        $target = &temp;
        for (/* Size */ int i=0; i<size; i++) {
            Type* x;
            PyObject* o = PySequence_GetItem($source,i);
            if ((SWIG_ConvertPtr(o,(void **) &x,
                                 SWIGTYPE_p_##Type,0)) != -1) {
                (*$target)[i] = *x;
                Py_DECREF(o);
            } else {
                PyErr_SetString(PyExc_TypeError,"TypeAlias" "s expected");
                Py_DECREF(o);
                return NULL;
            }
        }
    } else if ((SWIG_ConvertPtr($source,(void **) &v,
                                SWIGTYPE_p_##Vector,0)) != -1) {
        $target = v;
    } else {
        PyErr_SetString(PyExc_TypeError,"VectorAlias" " expected");
        return NULL;
    }
};
%enddef

%define TypemapVectorAllowingNone(Type,TypeAlias,Vector,VectorAlias)
%{
typedef std::vector<Type> Vector;
%}
%types(Vector *);
%typemap(python,in) Vector (Vector temp), Vector * (Vector temp),
  const Vector & (Vector temp), Vector & (Vector temp) {
    Vector* v;
    if (PyTuple_Check($source) || PyList_Check($source)) {
        /* Size */ int size = (PyTuple_Check($source) ?
            PyTuple_Size($source) :
            PyList_Size($source));
        temp = Vector(size);
        $target = &temp;
        for (/* Size */ int i=0; i<size; i++) {
            Type* x;
            PyObject* o = PySequence_GetItem($source,i);
            if (o == Py_None) {
                (*$target)[i] = Null<Type>();
                Py_DECREF(o);
            } else if ((SWIG_ConvertPtr(o,(void **) &x,
                                        SWIGTYPE_p_##Type,0)) != -1) {
                (*$target)[i] = *x;
                Py_DECREF(o);
            } else {
                PyErr_SetString(PyExc_TypeError,"TypeAlias" "s expected");
                Py_DECREF(o);
                return NULL;
            }
        }
    } else if ((SWIG_ConvertPtr($source,(void **) &v,
                                SWIGTYPE_p_##Vector,0)) != -1) {
        $target = v;
    } else {
        PyErr_SetString(PyExc_TypeError,"VectorAlias" " expected");
        return NULL;
    }
};
%enddef


#endif
