
/*
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

// $Id: CashFlows.i,v 1.24 2002/03/05 16:58:17 lballabio Exp $

#ifndef quantlib_cash_flows_i
#define quantlib_cash_flows_i

%include Date.i
%include Types.i
%include Calendars.i
%include DayCounters.i
%include Indexes.i
%include TermStructures.i
%include Vectors.i

%{
    using QuantLib::Handle;
    using QuantLib::CashFlow;
    using QuantLib::CashFlows::SimpleCashFlow;
    using QuantLib::CashFlows::FixedRateCoupon;
    using QuantLib::CashFlows::FloatingRateCoupon;
    typedef Handle<CashFlow> CashFlowHandle;
    typedef Handle<SimpleCashFlow> SimpleCashFlowHandle;
    typedef Handle<FixedRateCoupon> FixedRateCouponHandle;
    typedef Handle<FloatingRateCoupon> FloatingRateCouponHandle;
%}

// Python cash flow class
%typemap(python,in) PyObject* pyCashFlow {
	$target = $source;
}

%{
// its C++ wrapper
class PyCashFlow : public CashFlow {
  public:
	PyCashFlow(PyObject* pyCashFlow) : pyCashFlow_(pyCashFlow) {
	    /* make sure the Python object stays alive
	       as long as we need it */
	    Py_XINCREF(pyCashFlow_);
    }
    PyCashFlow(const PyCashFlow& cf)
    : pyCashFlow_(cf.pyCashFlow_) {
	    /* make sure the Python object stays alive
	       as long as we need it */
	    Py_XINCREF(pyCashFlow_);
    }
    PyCashFlow& operator=(const PyCashFlow& cf) {
        if ((this != &cf) && (pyCashFlow_ != cf.pyCashFlow_)) {
            Py_XDECREF(pyCashFlow_);
            pyCashFlow_ = cf.pyCashFlow_;
    	    Py_XINCREF(pyCashFlow_);
        }
        return *this;
    }
    ~PyCashFlow() {
        // now it can go as far as we are concerned
        Py_XDECREF(pyCashFlow_);
    }
	double amount() const {
		PyObject* pyResult =
		  PyObject_CallMethod(pyCashFlow_,"amount",NULL);
		QL_ENSURE(pyResult != NULL,
		  "failed to call amount() on Python cash flow");
		double result = PyFloat_AsDouble(pyResult);
		Py_XDECREF(pyResult);
		return result;
	}
	Date date() const {
		PyObject* pyResult =
		  PyObject_CallMethod(pyCashFlow_,"date",NULL);
		QL_ENSURE(pyResult != NULL,
		  "failed to call date() on Python cash flow");
		Date result, *x;
        if ((SWIG_ConvertPtr(pyResult,(void **)&x,SWIGTYPE_p_Date,0)) != -1) {
                result = *x;
                Py_XDECREF(pyResult);
        } else {
            Py_XDECREF(pyResult);
            throw Error("calling date() on Python cash flow "
                        "did not return a Date");
        }
        return result;
	}
  private:
	PyObject* pyCashFlow_;
};
%}

%name(CashFlow) class CashFlowHandle : public ObservableHandle {
  public:
    // constructor redefined below
    ~CashFlowHandle();
};

%addmethods CashFlowHandle {
    CashFlowHandle(PyObject* pyCashFlow) {
        return new CashFlowHandle(new PyCashFlow(pyCashFlow));
    }
    double amount() {
        return (*self)->amount();
    }
    Date date() {
        return (*self)->date();
    }
    void notifyObservers() {
        (*self)->notifyObservers();
    }
}


// implementations

// Fake inheritance to allow passing where Handle<CashFlow> is wanted
%name(SimpleCashFlow) class SimpleCashFlowHandle : public CashFlowHandle {
  public:
    // constructor redefined below
    ~SimpleCashFlowHandle();
};

%addmethods SimpleCashFlowHandle {
    SimpleCashFlowHandle(double amount, Date date) {
        return new SimpleCashFlowHandle(
            new SimpleCashFlow(amount,date));
    }
}


// Fake inheritance to allow passing where Handle<CashFlow> is wanted
%name(FixedRateCoupon) class FixedRateCouponHandle : public CashFlowHandle {
  public:
    // constructor redefined below
    ~FixedRateCouponHandle();
};

%addmethods FixedRateCouponHandle {
    FixedRateCouponHandle(double nominal, Rate rate,
      Calendar calendar, RollingConvention convention,
      DayCounter dayCounter, Date startDate, Date endDate,
      Date refPeriodStart, Date refPeriodEnd) {
        return new FixedRateCouponHandle(
            new FixedRateCoupon(nominal, rate, calendar, convention,
                dayCounter, startDate, endDate, refPeriodStart,
                refPeriodEnd));
    }
}


// Fake inheritance to allow passing where Handle<CashFlow> is wanted
%name(FloatingRateCoupon)
class FloatingRateCouponHandle : public CashFlowHandle {
  public:
    // constructor redefined below
    ~FloatingRateCouponHandle();
};

%addmethods FloatingRateCouponHandle {
    FloatingRateCouponHandle(double nominal, XiborHandle index,
      TermStructureRelinkableHandle termStructure,
      Date startDate, Date endDate, int fixingDays, Spread spread,
      Date refPeriodStart, Date refPeriodEnd) {
        return new FloatingRateCouponHandle(
            new FloatingRateCoupon(nominal, index,
                termStructure, startDate, endDate, fixingDays, spread,
                refPeriodStart, refPeriodEnd));
    }
}


// typedef Python list of cash flows to std::vector<Handle<CashFlow> >

TypemapVector(CashFlowHandle,CashFlow,CashFlowHandleVector,CashFlowVector);
ExportVector(CashFlowHandle,CashFlowHandleVector,CashFlowVector);


// cash flow vector builders

%{
using QuantLib::CashFlows::FixedRateCouponVector;
using QuantLib::CashFlows::FloatingRateCouponVector;
%}

class FixedRateCouponVector : public CashFlowHandleVector {
  public:
    FixedRateCouponVector(
        DoubleVector nominals, DoubleVector couponRates,
        Date startDate, Date endDate, int frequency,
        Calendar calendar, RollingConvention convention,
        bool isAdjusted, DayCounter dayCount,
        DayCounter firstPeriodDayCount, Date stubDate);
    ~FixedRateCouponVector();
};

class FloatingRateCouponVector : public CashFlowHandleVector {
  public:
    // constructor redefined below
    ~FloatingRateCouponVector();
};

%addmethods FloatingRateCouponVector {
    FloatingRateCouponVector(DoubleVector nominals,
      Date startDate, Date endDate,
      int frequency, Calendar calendar,
      RollingConvention convention,
      TermStructureRelinkableHandle termStructure,
      XiborHandle index, int indexFixingDays, DoubleVector spreads,
      Date stubDate) {
        return new FloatingRateCouponVector(nominals,
            startDate, endDate, frequency, calendar,
            convention, termStructure, index,
            indexFixingDays, spreads, stubDate);
    }
}


#endif
