/*  tilp - link program for TI calculators
 *  Copyright (C) 1999, 2000  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __TIFFEP_TIFILES__
#define __TIFFEP_TIFILES__

#include "types.h"
#include "tiffep.h"

/***************/
/* Definitions */
/***************/

/*
  This structure is used for getting informations about a variable stored
  in a file.
*/
struct var_info_
{
  gchar *var_path;         // Path of the variable
  gchar *var_name_b;       // Name of the variable (binary name)
  gchar *var_name_t;       // Name of the variable (translated name)
  gint var_type;           // Type of the variable (calc indep)
  gint var_atrb;           // Attribute of the var (none, protected, ...)
  gint var_size;           // Real size of the var (length of buffer 'data')
  guchar *data;            // Data content
  gint calc_type;          // Calculator type
  glong offset;            // Offset of the var entry in the file
};
//typedef struct var_info_ VarInfo;
typedef struct var_info_ TiffepVarInfo;

/*
  This structure is used for storing informations about a folder entry.
*/
struct fld_info_
{
  gchar *name;   // Folder name (8 chars max)
  gint num_vars; // Number of vars in the given folder
  glong offset;  // Offset on the first var in the folder entry
  gint type;     // Variable type
};
//typedef struct fld_info_ DirInfo;
typedef struct fld_info_ TiffepFolderInfo;

/* 
   This structure is used for storing some informations about the file 
   but without loading its data content.
*/
struct file_info_
{
  // General informations
  gchar *filename;	// Name of the file (fully qualified: path and name)
  gchar *name;	        // Base-name of the file without extension
  gchar *extension;	// Extension of the file
  gint file_type;	// File type: single, group, backup or FLASH file
  gint calc_type;	// Calculator type: 82, 83, ...
  gint num_of_vars;	// Numbers of variables into if group file, 1 else
  gchar *comment;       // Comment placed within the file
  gint data_size;       // Size of data in the file (sum of all data vars)
  gint file_size;       // Size of the file itself
  
  // These fields are always defined
  GList *var_list;      // List of variables
  GList *fld_list;      // List of folders: unused
};
//typedef struct file_info_ FileInfo;
typedef struct file_info_ TiffepFileInfo;


/*********************/
/* Convenient macros */
/*********************/

#define FILEINFO_FILENAME(x)  ((x)->filename)
#define FILEINFO_NAME(x)      ((x)->name)
#define FILEINFO_EXTENSION(x) ((x)->extension)
#define FILEINFO_FILETYPE(x)  ((x)->file_type)
#define FILEINFO_CALCTYPE(x)  ((x)->calc_type)
#define FILEINFO_NUMVARS(x)   ((x)->num_of_vars)
#define FILEINFO_COMMENT(x)   ((x)->comment)
#define FILEINFO_VARLIST(x)   ((x)->var_list)
#define FILEINFO_DIRLIST(x)   ((x)->fld_list)
#define FILEINFO_DATASIZE(x)  ((x)->data_size)
#define FILEINFO_FILESIZE(x)  ((x)->file_size)

#define VARINFO_PATH(x)       ((x)->var_path)
#define VARINFO_CALCNAME(x)   ((x)->var_name_b) //preferred
#define VARINFO_BINARYNAME(x) ((x)->var_name_b)
#define VARINFO_REALNAME(x)   ((x)->var_name_t) //preferred
#define VARINFO_TRANSNAME(x)  ((x)->var_name_t)
#define VARINFO_TYPE(x)       ((x)->var_type)
#define VARINFO_ATTRIBUTE(x)  ((x)->var_atrb)
#define VARINFO_SIZE(x)       ((x)->var_size) //preferred
#define VARINFO_VARSIZE(x)    ((x)->var_size)
#define VARINFO_CONTENT(x)    ((x)->data)
#define VARINFO_CALCTYPE(x)   ((x)->calc_type)
#define VARINFO_OFFSET(x)     ((x)->offset)

#define DIRINFO_NAME(x)       ((x)->name)
#define DIRINFO_NUMVARS(x)    ((x)->num_vars)
#define DIRINFO_SIZE(x)       ((x)->num_vars)  //preferred
#define DIRINFO_OFFSET(x)     ((x)->offset)
#define DIRINFO_TYPE(x)       ((x)->type)

// some shortcuts for internal use
#define FI_FILENAME(x)  FILEINFO_FILENAME(x)
#define FI_NAME(x)      FILEINFO_NAME(x)
#define FI_EXTENSION(x) FILEINFO_EXTENSION(x)
#define FI_FILETYPE(x)  FILEINFO_FILETYPE(x)
#define FI_CALCTYPE(x)  FILEINFO_CALCTYPE(x)
#define FI_NUMVARS(x)   FILEINFO_NUMVARS(x)
#define FI_COMMENT(x)   FILEINFO_COMMENT(x)
#define FI_VARLIST(x)   FILEINFO_VARLIST(x)
#define FI_DIRLIST(x)   FILEINFO_DIRLIST(x)
#define FI_DATASIZE(x)  FILEINFO_DATASIZE(x)
#define FI_FILESIZE(x)  FILEINFO_FILESIZE(x)

#define VI_PATH(x)      VARINFO_PATH(x)
#define VI_CALCNAME(x)  VARINFO_CALCNAME(x)
#define VI_REALNAME(x)  VARINFO_REALNAME(x)
#define VI_TYPE(x)      VARINFO_TYPE(x)
#define VI_ATTRIBUTE(x) VARINFO_ATTRIBUTE(x)
#define VI_SIZE(x)      VARINFO_VARSIZE(x)
#define VI_CONTENT(x)   VARINFO_CONTENT(x)
#define VI_CALCTYPE(x)  VARINFO_CALCTYPE(x)
#define VI_OFFSET(x)    VARINFO_OFFSET(x)

#define DI_NAME(x)      DIRINFO_NAME(x)
#define DI_NUMVARS(x)   DIRINFO_NUMVARS(x)
#define DI_SIZE(x)      DIRINFO_SIZE(x)    //preferred
#define DI_OFFSET(x)    DIRINFO_OFFSET(x)
#define DI_TYPE(x)      DIRINFO_TYPE(x)

/******************/
/* Cast operators */
/******************/

#define FILEINFO(x)           ((TiffepFileInfo *)(x))
#define VARINFO(x)            ((TiffepVarInfo *)(x))
#define DIRINFO(x)            ((TiffepFolderInfo *)(x))

#define VARLIST_DATA(glist)   ((TiffepVarInfo *)(glist->data))
#define DIRLIST_DATA(glist)   ((TiffepFolderInfo *)(glist->data))
#define VARLIST_NEXT(glist)   (g_list_next(glist))
#define DIRLIST_NEXT(glist)   (g_list_next(glist))

/***********************/
/* Available functions */
/***********************/

TIEXPORT
int tiffep_load_file_informations(gchar *filename, TiffepFileInfo **fileinfo);
TIEXPORT
int tiffep_load_file_content(gchar *filename, TiffepFileInfo **fileinfo);
TIEXPORT
int tiffep_save_file_content(gchar *filename, TiffepFileInfo *fileinfo);


TIEXPORT
TiffepFileInfo* tiffep_alloc_fi_struct(void);
TIEXPORT
int tiffep_free_fi_struct(TiffepFileInfo **fi);
TIEXPORT
TiffepFileInfo* tiffep_dup_fi_struct(TiffepFileInfo *fi);
TIEXPORT
TiffepFileInfo* tiffep_clear_fi_struct(TiffepFileInfo *fi);
TIEXPORT
int tiffep_print_fi_struct(TiffepFileInfo *fi);

TIEXPORT
int tiffep_create_single_file(gchar *filename, gint calc_type, gint var_type,
			      char *comment, TiffepFileInfo *fileinfo);
TIEXPORT
int tiffep_create_group_file(gchar *filename, gint calc_type,
			     char *comment, TiffepFileInfo *fileinfo);




#endif

