/*  tilp - link program for TI calculators
 *  Copyright (C) 1999, 2000  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "hfiles.h"
#include "tiffep.h"


/*
  Allocate a TiffepFileInfo structure and fill it with null values.
  - TiffepFileInfo [out]: the newly allocated structure, NULL otherwise.
*/
TIEXPORT
TiffepFileInfo* tiffep_alloc_fi_struct(void)
{ 
  TiffepFileInfo *fileinfo = (TiffepFileInfo *)g_malloc0(sizeof(TiffepFileInfo));
  return fileinfo;
}

/* Free allocated memory in TiffepVarInfo structure */
static void free_var_content_1(gpointer data, gpointer user_data)
{
  //TiffepVarInfo *vi = (TiffepVarInfo *)data;
  //tiffep_print_vi_struct(vi);
  //tiffep_free_vi_struct(&vi); //commented else segfaults !
}

/* Free allocated memory in TiffepVarInfo structure */
static void free_fld_content_1(gpointer data, gpointer user_data)
{
  //TiffepFolderInfo *di = (TiffepFolderInfo *)data;
  //tiffep_print_vi_struct(di);
  //tiffep_free_vi_struct(&di); //commented else segfaults !
}

/*
  Free all allocated memory
  - fi [in]: the informations stored in an allocated structure
  for being freed
  - int [out]: 0 if success, an error code otherwise.
*/
TIEXPORT
int tiffep_free_fi_struct(TiffepFileInfo **fileinfo)
{
  GList *varlist;
  GList *fldlist;
  TiffepFileInfo *fi = *fileinfo;

  if(fi == NULL) 
    {
      g_warning(_("tiffep_free_fi_struct: NULL pointer.\n"));
      return -1;
    }
  varlist = FI_VARLIST(fi);
  fldlist = FI_DIRLIST(fi);

  // free the fields first
  g_free(FI_FILENAME(fi));
  g_free(FI_NAME(fi));
  g_free(FI_EXTENSION(fi));
  g_free(FI_COMMENT(fi));

  // free the linked list elements
  if(FI_VARLIST(fi) != NULL)
    g_list_foreach(varlist, free_var_content_1, NULL);
  if(FI_DIRLIST(fi) != NULL)
    g_list_foreach(fldlist, free_fld_content_1, NULL);
  // and next, free the linked list
  g_list_free(FI_VARLIST(fi));
  g_list_free(FI_DIRLIST(fi));

  // at last, free the structure itself
  g_free(fi);
  *fileinfo = NULL;

  return 0;
}

/*
  Duplicate a TiffepFileInfo structure and the TiffepVarInfo linked list ('wide' copy)
  - fi [in]: the TiffepFileInfo structure to duplicate
  - TiffepFileInfo [out]: the newly allocated structure if success, 
  NULL otherwise.
*/
TIEXPORT
TiffepFileInfo* tiffep_dup_fi_struct(TiffepFileInfo *fi)
{
  TiffepFileInfo *src = fi;
  TiffepFileInfo *dst = NULL;
  GList *p_dst;
  GList *p_src;

  dst = tiffep_alloc_fi_struct();

  FI_FILENAME(dst)   = g_strdup(FI_FILENAME(src));
  FI_NAME(dst)       = g_strdup(FI_NAME(src));
  FI_EXTENSION(dst)  = g_strdup(FI_EXTENSION(src));
  FI_FILETYPE(dst)   = FI_FILETYPE(src);
  FI_CALCTYPE(dst)   = FI_CALCTYPE(src);
  FI_NUMVARS(dst)    = FI_NUMVARS(src);
  FI_COMMENT(dst)    = g_strdup(FI_COMMENT(src));
  FI_DATASIZE(dst)   = FI_DATASIZE(src);
  FI_FILESIZE(dst)   = FI_FILESIZE(src);
  
  //var list
  FI_VARLIST(dst) = g_list_copy(FI_VARLIST(src));
  p_dst           = FI_VARLIST(dst);
  p_src           = FI_VARLIST(src);
  while(p_dst != NULL)
    {
      TiffepVarInfo *v_src = VARLIST_DATA(p_src);
      TiffepVarInfo *v_dst = VARLIST_DATA(p_dst);
      v_dst = tiffep_dup_vi_struct(v_src);

      p_dst = VARLIST_NEXT(p_dst);
    }
  // folder list
  dst->fld_list    = g_list_copy(src->fld_list);
  p_dst            = FI_DIRLIST(dst);
  p_src            = FI_DIRLIST(src);
  while(p_dst != NULL)
    {
      TiffepFolderInfo *f_src = DIRLIST_DATA(p_src);
      TiffepFolderInfo *f_dst = DIRLIST_DATA(p_dst);
      f_dst = tiffep_dup_di_struct(f_src);

      p_dst = DIRLIST_NEXT(p_dst);
    }

  return dst;
}

/*
  Clear a TiffepFileInfo structure
  - fi [in/out]: the TiffepFileInfo structure to clear/cleared
  - TiffepFileInfo [out]: the cleared structure
  NULL otherwise.
*/
TIEXPORT
TiffepFileInfo* tiffep_clear_fi_struct(TiffepFileInfo *fi)
{
  memset((void *)fi, 0, sizeof(TiffepFileInfo));
  return fi;
}

/*
  DISPLAY2 a TiffepFileInfo structure
  - fi [in]: the TiffepVarInfo structure to print
  - int [out]: none
*/
TIEXPORT
int tiffep_print_fi_struct(TiffepFileInfo *fi)
{
  if(fi == NULL)
    {
      g_warning("tiffep_print_fi_struct: NULL pointer.\n");
      return -1;
    }
  DISPLAY2("filename    = <%s>\n", fi->filename);
  DISPLAY2("name        = <%s>\n", fi->name);
  DISPLAY2("extension   = <%s>\n", fi->extension);
  DISPLAY2("file_type   = %i\n", fi->file_type);
  DISPLAY2("calc_type   = %i <%s>\n", fi->calc_type,
	  tiffep_calc_type_to_string(fi->calc_type));
  DISPLAY2("num_of_vars = %i\n", fi->num_of_vars);
  DISPLAY2("comment     = <%s>\n", fi->comment);
  DISPLAY2("data_size   = %i\n", fi->data_size);
  DISPLAY2("file_size   = %i\n", fi->file_size);
  DISPLAY2("var_list    = %p\n", fi->var_list);
  DISPLAY2("fld_list    = %p\n", fi->fld_list);

  return 0;
}
