/* Subwindow.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.subwindows;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.MouseEvent;
import javax.swing.AbstractAction;
import javax.swing.JComponent;
import javax.swing.KeyStroke;
import javax.swing.UIManager;
import javax.swing.event.MouseInputListener;

/**
 * Wraps a child component into a small `window' with a title bar. Clicking on
 * this title bar <i>collapses</i> the component: the child component is hidden
 * and only the title bar is still visible. Clicking again, resets the subwindow
 * to its full size.
 * <p>For best effect, subwindows of this kind should be stacked in a container
 * which uses a vertical box layout, as in the folllowing example
 * <pre>
 *      Subwindow subwindow1 = new Subwindow(component1, true);
 *      Subwindow subwindow2 = new Subwindow(component2, false);
 *
 *      JPanel container = new JPanel ();
 *      container.setLayout(new BoxLayout (container, BoxLayout.Y_AXIS));
 *      container.add (subwindow1);
 *      container.add (subwindow2);
 *      container.add (Box.createVerticalGlue());
 *      container.setBackground(Color.WHITE);
 * </pre>
 */
@SuppressWarnings("PMD.TooManyFields")
public class Subwindow extends JComponent {
    
    // TODO: make some windows incollapsible
    
    //
    private final JComponent child;
    
    /** 
     * Create a new subwindow which displays the given child component.The 
     * name of the child conmoponent is used as the caption of the title bar
     * of this window.
     */
    @SuppressWarnings("PMD.ConstructorCallsOverridableMethod")
    public Subwindow(JComponent child, boolean collapsed) {
        this.child = child;
        this.collapsed = collapsed;
        
        // set colors
        this.titleBarColorNoFocusSetExplicitely = false;
        this.titleBarTextColorNoFocusSetExplicitely = false;
        this.titleBarBorderColorSetExplicitely = false;
        setTitleBarColor(Color.GRAY);
        setTitleBarTextColor(Color.WHITE);
        
        
        this.titleBarFont = UIManager.getFont("Label.font");
        this.titleBarFontNoFocus = UIManager.getFont("TextField.font");
        this.titleBarHeight = 22;
        this.horizontalWindowPadding = 14;
        this.verticalWindowPadding = 10;
        
        this.titleBarHighlighted = false;
        
        add(child);
        child.setVisible(! collapsed);
        
        setFocusable(true);
        requestFocusInWindow();
        
        addFocusListener(new FocusListener() {
            public void focusGained(FocusEvent ev) {
                repaintTitleBar();
            }
            
            public void focusLost(FocusEvent ev) {
                repaintTitleBar();
            }
        });
        
        addMouseListener(new MouseHandler());
        addMouseMotionListener(new MouseHandler());
        
        // keyboard actions
        getActionMap().put("toggleCollapsed", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                toggleCollapsed();
            }
        });
        getInputMap().put(KeyStroke.getKeyStroke(' '), "toggleCollapsed");
    }
    
    
    /**
     * Return the child component which is managed by this subwindow.
     * The name of the child component is used as the caption of the
     * title bar.
     */
    public JComponent getChildComponent () {
        return this.child;
    }
    
    //
    private boolean collapsed;
    
    /**
     * Is this window collapsed?
     */
    public boolean isCollapsed() {
        return this.collapsed;
    }
    
    /**
     * Change the 'collapsed' state of this window.
     */
    public void setCollapsed(boolean collapsed) {
        if (this.collapsed != collapsed) {
            this.collapsed = collapsed;
            child.setVisible(! collapsed);
            child.revalidate();
            revalidate();
            repaint();
            fireCollapsedStateChanged ();
        }
    }
    
    /**
     * Toggle the 'collapsed' state of this window.
     */
    public void toggleCollapsed() {
        setCollapsed(! collapsed);
    }
    
    //
    private boolean titleBarColorNoFocusSetExplicitely;
    
    //
    private Color titleBarColor;
    
    //
    private Color titleBarColorNoFocus;
    
    //
    private void createTitleBarColorNoFocus() {
        int r = titleBarColor.getRed() + 40;
        int g = titleBarColor.getGreen() + 40;
        int b = titleBarColor.getBlue() + 40;
        this.titleBarColorNoFocus
                = new Color(r < 255 ? r : 255, g < 255 ? g : 255, b < 255 ? b : 255);
    }
    
    /**
     * The color of the title bar.
     */
    public Color getTitleBarColor() {
        return titleBarColor;
    }
    
    /**
     * The color of the title bar when  the subwindow
     * does not own the focus, as set by the user.
     * @return null when the user did not explicitely set this color
     */
    public Color getTitleBarColorNoFocus() {
        if (titleBarColorNoFocusSetExplicitely)
            return titleBarColorNoFocus;
        else
            return null;
    }
    
    /**
     * Set the color to be used for the title bar. Defaults to gray.
     */
    public void setTitleBarColor(Color titleBarColor) {
        if (titleBarColor != this.titleBarColor) {
            this.titleBarColor = titleBarColor;
            if (! titleBarColorNoFocusSetExplicitely)
                createTitleBarColorNoFocus();
            if (titleBarBorderColorSetExplicitely) {
                repaintTitleBar();
            } else {
                this.titleBarBorderColor = titleBarColor;
                repaint();
            }
        }
    }
    
    /**
     * Set the color to be used for the title bar when the subwindow
     * does not own the focus. Defaults to a paler version of the title
     * bar color.
     */
    public void setTitleBarColorNoFocus(Color titleBarColorNoFocus) {
        if (titleBarColorNoFocus != this.titleBarColorNoFocus) {
            if (titleBarColorNoFocus == null) {
                titleBarColorNoFocusSetExplicitely = false;
                createTitleBarColorNoFocus();
            } else {
                titleBarColorNoFocusSetExplicitely = true;
                this.titleBarColorNoFocus = titleBarColorNoFocus;
            }
            repaintTitleBar();
        }
    }
    
    //
    private Color titleBarTextColor;
    
    //
    private Color titleBarTextColorNoFocus;
    
    //
    private boolean titleBarTextColorNoFocusSetExplicitely;
    
    /**
     * Return the color for the text of the title bar.
     */
    public Color getTitleBarTextColor() {
        return titleBarTextColor;
    }
    
    /**
     * Set the color to be used for the text in the title bar. Defaults to white
     */
    public void setTitleBarTextColor(Color titleBarTextColor) {
        if (titleBarTextColor != this.titleBarTextColor) {
            this.titleBarTextColor = titleBarTextColor;
            if (! titleBarTextColorNoFocusSetExplicitely)
                this.titleBarTextColorNoFocus = titleBarTextColor;
            repaintTitleBar();
        }
    }
    
    /**
     * Set the color to be used for the text in the title bar when the subwindow
     * does not own the focus. Defaults to the title bar text color.
     */
    public void setTitleBarTextColorNoFocus(Color titleBarTextColorNoFocus) {
        if (titleBarTextColorNoFocus != this.titleBarTextColorNoFocus) {
            if (titleBarTextColorNoFocus == null) {
                titleBarTextColorNoFocusSetExplicitely = false;
                this.titleBarTextColorNoFocus = titleBarTextColor;
            } else {
                titleBarTextColorNoFocusSetExplicitely = true;
                this.titleBarTextColorNoFocus = titleBarTextColorNoFocus;
            }
            repaintTitleBar();
        }
    }
    
    /**
     * The color of the text in the title bar when  the subwindow
     * does not own the focus, as set by the user.
     * @return null when the user did not explicitely set this color
     */
    public Color getTitleBarTextColorNoFocus() {
        if (titleBarTextColorNoFocusSetExplicitely)
            return titleBarTextColorNoFocus;
        else
            return null;
    }
    
    //
    private boolean titleBarBorderColorSetExplicitely;
    
    //
    private Color titleBarBorderColor;
    
    /**
     * Set the color to be used for the subwindow border.
     * Defaults to the title bar color.
     */
    public void setTitleBarBorderColor(Color titleBarBorderColor) {
        if (titleBarBorderColor != this.titleBarBorderColor) {
            if (titleBarBorderColor == null) {
                titleBarBorderColorSetExplicitely = false;
                this.titleBarBorderColor = titleBarColor;
            } else {
                titleBarBorderColorSetExplicitely = true;
                this.titleBarBorderColor = titleBarBorderColor;
            }
            repaint();
        }
    }
    
    /**
     * The color of the subwindow border, as set by the user.
     * @return null when the user did not explicitely set this color
     */
    public Color getTitleBarBorderColor() {
        if (titleBarBorderColorSetExplicitely)
            return titleBarBorderColor;
        else
            return null;
    }
    
    //
    private final Font titleBarFont;
    
    //
    private final Font titleBarFontNoFocus;
    
    // TODO: make title fonts setable ?
    
    
    //
    private int titleBarHeight;
    
    /**
     * The height of the title bar in pixels.
     */
    public int getTitleBarHeight() {
        return titleBarHeight;
    }
    
    /**
     * Set the height of the title bar.
     */
    public void setTitleBarHeight(int titleBarHeight) {
        if (titleBarHeight != this.titleBarHeight) {
            this.titleBarHeight = titleBarHeight;
            revalidate();
            repaint();
        }
    }
    
    //
    private int horizontalWindowPadding;
    
    /**
     * Horizontal padding for this subwindow.
     */
    public int getHorizontalWindowPadding() {
        return horizontalWindowPadding;
    }
    
    /**
     * Sets the horizontal  for this subwindow.
     */
    public void setHorizontalWindowPadding(int horizontalWindowPadding) {
        if (this.horizontalWindowPadding != horizontalWindowPadding) {
            this.horizontalWindowPadding = horizontalWindowPadding;
            revalidate();
            repaint();
        }
    }
    
    //
    private int verticalWindowPadding;
    
    /**
     * Vertical padding for this subwindow.
     */
    public int getVerticalWindowPadding() {
        return verticalWindowPadding;
    }
    
    /**
     * Sets the vertical  for this subwindow.
     */
    public void setVerticalWindowPadding(int verticalWindowPadding) {
        if (this.verticalWindowPadding != verticalWindowPadding) {
            this.verticalWindowPadding = verticalWindowPadding;
            revalidate();
            repaint();
        }
    }
    
    public void paintComponent(Graphics g) {
        
        boolean highlighted = titleBarHighlighted || isFocusOwner();
        int internalWidth = getWidth() - 2 * horizontalWindowPadding;
        g.setColor(highlighted ? titleBarColor : titleBarColorNoFocus);
        g.fillRect(horizontalWindowPadding, verticalWindowPadding,
                internalWidth, titleBarHeight);
        
        g.setColor(titleBarBorderColor);
        g.drawRect(horizontalWindowPadding-1, verticalWindowPadding-1,
                internalWidth + 1, getHeight() - 2*verticalWindowPadding + 1);
        g.drawLine(horizontalWindowPadding-1,
                verticalWindowPadding + titleBarHeight - 1,
                horizontalWindowPadding + internalWidth,
                verticalWindowPadding + titleBarHeight - 1);
        String caption = child.getName();
        if (caption != null) {
            g.setFont(highlighted ? titleBarFont : titleBarFontNoFocus);
            g.setColor(highlighted ? titleBarTextColor : titleBarTextColorNoFocus);
            int baseLine = (titleBarHeight + g.getFontMetrics().getAscent() -
                    g.getFontMetrics().getDescent()) / 2;
            g.drawString(caption, 2*horizontalWindowPadding,
                    verticalWindowPadding + baseLine);
        }
        
    }
    
    //
    private Dimension adaptedDimension(Dimension p) {
        int newWidth = p.width + 2*horizontalWindowPadding;
        int newHeight = p.height + 2*verticalWindowPadding + titleBarHeight;
        
        // detect overflow, and normalize 'infinity' to Short.MAX_VALUE in all
        // cases (some components use Integer.MAX_VALUE)
        if (newWidth < 0 || newWidth > Short.MAX_VALUE)
            newWidth = Short.MAX_VALUE;
        if (newHeight < 0|| newHeight > Short.MAX_VALUE)
            newHeight = Short.MAX_VALUE;
        
        if (collapsed)
            return new Dimension(newWidth,
                    2*verticalWindowPadding + titleBarHeight);
        else
            return new Dimension(newWidth, newHeight);
    }
    
    /**
     * The preferred size is derived from the preferred size of the child
     * unless it has been explicitely set. For a collapsed window, only the
     * preferred width of the child component is taken into account.
     */
    public Dimension getPreferredSize() {
        if (isPreferredSizeSet())
            return super.getPreferredSize();
        else
            return adaptedDimension(child.getPreferredSize());
    }
    
    
    /**
     * The minimum size is derived from the minimum size of the child
     * unless it has been explicitely set. For a collapsed window, only the
     * preferred width of the child component is taken into account.
     */
    public Dimension getMinimumSize() {
        if (isMinimumSizeSet())
            return super.getMinimumSize();
        else
            return adaptedDimension(child.getMinimumSize());
    }
    
    /**
     * The maximum size is derived from the maximum size of the child
     * unless it has been explicitely set. For a collapsed window, only the
     * preferred width of the child component is taken into account.
     */
    public Dimension getMaximumSize() {
        if (isMaximumSizeSet())
            return super.getMaximumSize();
        else
            return adaptedDimension(child.getMaximumSize());
    }
    
    
    public void doLayout() {
        int width = getWidth() - 2*horizontalWindowPadding;
        int height = getHeight() - 2*verticalWindowPadding - titleBarHeight;
        child.setBounds(horizontalWindowPadding, verticalWindowPadding + titleBarHeight,
                width, height);
    }
    
    //
    private boolean inTitleBar(Point point) {
        return  point.y >= verticalWindowPadding &&
                point.y < verticalWindowPadding + titleBarHeight &&
                point.x >= horizontalWindowPadding &&
                point.x < getWidth() - horizontalWindowPadding;
    }
    
    //
    private void repaintTitleBar() {
        repaint(horizontalWindowPadding, verticalWindowPadding,
                getWidth()-2*horizontalWindowPadding, titleBarHeight);
    }
    
    //
    private boolean titleBarHighlighted;
    
    /**
     * Change the highlight status of the title bar.
     */
    private void setTitleBarHighlighted(boolean highlighted) {
        if (highlighted != titleBarHighlighted) {
            titleBarHighlighted = highlighted;
            repaintTitleBar();
        }
    }
    
    //
    private static final Class<SubwindowListener> SUBWINDOW_LISTENER_CLASS
            = SubwindowListener.class;
    
    /**
     * Notify all listeners of a change in the collapsed state of this subwindow.
     */
    private void fireCollapsedStateChanged() {
        
        Object[] listeners = listenerList.getListenerList();
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==SUBWINDOW_LISTENER_CLASS) {
                ((SubwindowListener)listeners[i+1]).changedCollapsedState(this, collapsed);
            }
        }
    }
    
    /**
     * Register a listener which will be notified of all changes in the collapsed
     * state of the subwindow.
     */
    public void addSubwindowListener (SubwindowListener listener) {
        listenerList.add(SUBWINDOW_LISTENER_CLASS, listener);
    }
    
    /**
     * Undo the registration of the given listener as a subwindow listener.
     */
    public void removeSubwindowListener(SubwindowListener listener) {
        listenerList.remove(SUBWINDOW_LISTENER_CLASS, listener);
    }
    
    private class MouseHandler implements MouseInputListener {
        
        MouseHandler() {
            // avoid creation of accessor type
        }
        
        public void mouseClicked(MouseEvent e) {
            if (inTitleBar(e.getPoint()))
                toggleCollapsed();
        }
        
        public void mousePressed(MouseEvent e) {
            // intentionally empty
        }
        
        public void mouseReleased(MouseEvent e) {
            // intentionally empty
        }
        
        public void mouseEntered(MouseEvent e) {
            // intentionally empty
        }
        
        public void mouseExited(MouseEvent e) {
            setCursor(null);
            setTitleBarHighlighted(false);
        }
        
        public void mouseDragged(MouseEvent e) {
            // intentionally empty
        }
        
        public void mouseMoved(MouseEvent e) {
            if (inTitleBar(e.getPoint())) {
                setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
                setTitleBarHighlighted(true);
            } else {
                setCursor(null);
                setTitleBarHighlighted(false);
            }
            
        }
        
    }
}
