/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: FWObject.cc,v 1.8 2001/12/28 08:49:18 lord Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <fwbuilder/Tools.hh>
#include <fwbuilder/XMLTools.hh>
#include <fwbuilder/FWObject.hh>
#include <fwbuilder/FWObjectDatabase.hh>
#include <fwbuilder/FWObjectReference.hh>

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>

#include <iostream>
#include <algorithm>
#include <functional>
#include <stack>

using namespace std;
using namespace libfwbuilder;

const char *FWObject::TYPENAME={"UNDEF"};
string FWObject::NOT_FOUND="";


void FWObject::fromXML(xmlNodePtr root)
{
    assert(root!=NULL);

    const char *n;

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("name")));
    if(n)  setName(n);

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("id")));
    if(n)  setId(n);

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("comment")));
    if(n)  setComment(XMLTools::unquote_linefeeds(n));

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("library")));
    if(n)  setLibrary(n);


    ref_counter=0;

    for(xmlNodePtr cur=root->xmlChildrenNode; cur; cur=cur->next) 
    {
        if(cur && !xmlIsBlankNode(cur))  
        {
            FWObject *o=FWObjectDatabase::db->createFromXML(cur);
            if(o!=NULL) 
            {
		add(o);
		o->fromXML(cur);
	    }
        }
    }
    
    setDirty(false);
}

xmlNodePtr FWObject::toXML(xmlNodePtr xml_parent_node)
{
    return toXML(xml_parent_node, true);
}

xmlNodePtr FWObject::toXML(xmlNodePtr parent, bool process_children)
{
    xmlNodePtr me = xmlNewChild(parent, NULL, xml_name.empty()?STRTOXMLCAST(getTypeName()):STRTOXMLCAST(xml_name), NULL);

    for(map<string, string>::iterator i=data.begin(); i!=data.end(); ++i) 
    {
        string name  = (*i).first;
        string value = (*i).second;

        xmlAttrPtr pr = xmlNewProp(me, 
                                   STRTOXMLCAST(name) , 
                                   STRTOXMLCAST((name=="comment"?XMLTools::quote_linefeeds(value):value)));
        
        if(name=="id")
        {
            xmlAddID(NULL, parent->doc, STRTOXMLCAST(value), pr);
        } else if(name=="ref") 
        {
            xmlAddRef(NULL, parent->doc, STRTOXMLCAST(value), pr);
        }
    }

    if(process_children)
    {
        vector<FWObject*>::iterator j;
        for(j=begin(); j!=end(); ++j) 
            (*j)->toXML(me);
    }
    
    return me;
}


FWObject::FWObject()
{
    ref_counter = 0;
    parent      = NULL;

    // When object created we assign it unique Id
    setId(FWObjectDatabase::db->generateUniqueId());

    setDirty(false);
}

FWObject::FWObject(const FWObject &c)
{
    *this=c;
}


FWObject::FWObject(FWObject *par)
{
    setName("New object");
    
    ref_counter = 0    ;
    parent      = par  ;
    
    // When object created we assign it unique Id
    setId(FWObjectDatabase::db->generateUniqueId());

    setDirty(false);
}


FWObject::~FWObject() 
{
    clearChildren();
    data.clear();
}

FWObject* FWObject::getParent()
{
    return parent;
}

void FWObject::setParent(FWObject *p)
{
    parent=p;
}

void FWObject::setXMLName(const string &n)
{
    xml_name = n;
}

FWObject* FWObject::_find(const string& name)   // find child by name
{
    iterator i=find_if(begin(),end(), FWObjectNameEQPredicate(name));
    return i==end()?NULL:(*i);
}

FWObject& FWObject::operator=(const FWObject &x)
{
    clearChildren();

    ref_counter = 0          ;
    parent      = x.parent   ;
    xml_name    = x.xml_name ;
    data        = x.data     ;
    
    for(vector<FWObject*>::const_iterator m=x.begin(); m!=x.end(); ++m) 
    {
        FWObject *o=*m;
        FWObject *o1 = FWObjectDatabase::db->create(o->getTypeName());
        assert(o1!=NULL);
        *o1=*o;
        add(o1);
    }
    setDirty(true);

    return *this;
}

FWObject& FWObject::duplicate(const FWObject *x)
{
    string id=getId();
    
    clearChildren();

    data=x->data;

    if(id!="")   // some objects do not have ID per DTD (e.g. Src, Dst, etc.)
	setId(id);

    for(vector<FWObject*>::const_iterator m=x->begin(); m!=x->end(); ++m) 
    {
        FWObject *o=*m;
        FWObject *o1=FWObjectDatabase::db->create( o->getTypeName());
        assert(o1!=NULL);
        o1->duplicate(o);
        add(o1);
    }
    
    setDirty(true);

    return *this;
}

const string &FWObject::getName() const 
{ 
    return getStr("name"); 
}

void FWObject::setName(const string &n)   
{
    setStr("name",n);
    setDirty(true);
}

const string &FWObject::getLibrary() const
{
    return getStr("library"); 
}

void FWObject::setLibrary(const string& c)
{
    setStr("library",c);
    setDirty(true);
}

const string &FWObject::getComment() const
{ 
    return getStr("comment"); 
}

void FWObject::setComment(const string &c)
{
    setStr("comment",c);
    setDirty(true);
}

const string &FWObject::getId() const
{ 
    return getStr("id");
}

void FWObject::setId(const string &c)
{
    setStr("id",c);
    setDirty(true);
}

bool FWObject::exists(const string &name) const 
{
    return data.count(name)!=0; 
}

const string &FWObject::getStr(const string &name) const
{
    if(exists(name)) 
    {
        map<string,string>::const_iterator i=data.find(name);
        return (*i).second;
    } else
    {
        return NOT_FOUND;
    }
}

void FWObject::remStr(const string &name)
{
    if(exists(name)) 
    {
	map<string, string>::iterator m=data.find(name);
	if(m != data.end()) 
        {
	    data.erase(m);
	    setDirty(true);
	}
    }
}

void FWObject::setStr(const string &name, const string &val)
{
    data[name]=val;
    setDirty(true);
}

int FWObject::getInt(const string &name) const
{
    string s=getStr(name);
    if(s!="") 
        return( atol(s.c_str()) );
    else   
        return(-1);
}

void FWObject::setInt(const string &name, int val)
{
    gchar str[128];
    sprintf(str,"%d",val);
    setStr(name, str);
    setDirty(true);
}

bool FWObject::getBool(const string &name) const
{
    if(exists(name)) 
    {
	return(getStr(name)=="1" || 
               strcasecmp(getStr(name).c_str() , "true")==0);
    } else
        return false;
}

void FWObject::setBool(const string &name, bool val)
{
    setStr(name, (val)?"True":"False");
    setDirty(true);
}

void FWObject::setBool(const string &name, const string &val)
{
    if(!name.empty())
	setBool(name,
		(val=="1" || strcasecmp(val.c_str(),"true")==0)); 
}

void FWObject::Show()
{
    setBool("read",true);
}

void FWObject::Hide()
{
    setBool("read",false);
}



void FWObject::dump(bool recursive,bool brief,int offset)
{
    dump(cerr,recursive,brief,offset);
}

void   FWObject::dump(std::ostream &f,bool recursive,bool brief,int offset)
{
    FWObject *o;
    string    n;

    if (brief) {
	f << string(offset,' ');
	f << " Obj=" << this;
	f << " ID="  << getId();
	f << " Name=" << getName();
	f << " Type=" << getTypeName();
	f << " Library=" << getLibrary();

	if (FWReference::cast(this)!=0) {
	    f << " Ref=" << FWReference::cast(this)->getPointer();
	}

	f << endl;

	if (recursive) {
	    vector<FWObject*>::iterator m;
	    for (m=begin(); m!=end(); ++m) {
		if (  (o=(*m))!=NULL)  o->dump(f,recursive,brief,offset+2);
	    }
	}
    } else {

	f << string(offset,' ') << string(16,'-') << endl;
	f << string(offset,' ') << "Obj:    " << this << endl;
	f << string(offset,' ') << "ID:     " << getId() << endl;
	f << string(offset,' ') << "Name:   " << getName() << endl;
	f << string(offset,' ') << "Ref.ctr:" << ref_counter << endl;
	f << string(offset,' ') << "Type:   " << getTypeName() << endl;
	f << string(offset,' ') << "Library:" << getLibrary() << endl;
//    f << string(offset,' ') << "Path:   " << getPath() << endl;
	n=(getParent()!=NULL)?getParent()->getName():"";
	f << string(offset,' ') << "Parent: " << getParent()
	  << "  name=" << n << endl;
	f << string(offset,' ') << "Root:   " << getRoot() << endl;

	map<string, string>::iterator d;
	for (d=data.begin(); d!=data.end(); ++d) {
	    if((*d).first=="name") 
                continue;
	    f << string(offset,' ');
	    f << (*d).first << ": " << (*d).second << endl;
	}
	if (recursive) {
	    vector<FWObject*>::iterator m;
	    for (m=begin(); m!=end(); ++m) {
		if (  (o=(*m))!=NULL)  o->dump(f,recursive,brief,offset+2);
	    }
	}
    }

}

void FWObject::_adopt(FWObject *obj)
{
    obj->ref();
    obj->setParent(this);
}

void FWObject::addAt(const string& where_id, FWObject *obj)
{
    FWObject *p=getById( where_id , true );
    assert (p!=NULL);
    p->add(obj);
}

void FWObject::add(FWObject *obj)
{
    if(validateChild(obj)) 
    {
	push_back(obj);
	_adopt(obj);
	setDirty(true);
    }
}

FWReference* FWObject::createRef( FWObject* obj )
{
    return new FWObjectReference(obj);
}


void FWObject::addRef(FWObject *obj)
{
    if(validateChild(obj)) 
    {
	FWReference *oref = createRef(obj);
	obj->ref();

	push_back(oref);
	_adopt(oref);
	setDirty(true);
    }
}

void FWObject::insert_before(FWObject *o1, FWObject *obj)
{
    if(!obj) 
        return;
    
    vector<FWObject*>::iterator m=find(begin(),end(),o1);
    if(m!=end())
    {
        insert(m, obj);
        _adopt(obj);
        setDirty(true);
    }
}

void FWObject::insert_after(FWObject *o1, FWObject *obj)
{
    if(!obj) 
        return;

    vector<FWObject*>::iterator m=find(begin(),end(),o1);
    if(m!=end())
    {
        insert(++m,obj);
        _adopt(obj);
        setDirty(true);
    }
}

void FWObject::swapObjects(FWObject *o1, FWObject *o2)
{
    for(vector<FWObject*>::iterator m=begin(); m!=end(); ++m) 
    {
        if(*m==o1) 
        {
            *m=o2;
        } else if(*m==o2)
        {
            *m=o1;
        }
    }
    setDirty(true);
}

void FWObject::remove(FWObject *obj, bool delete_if_last=true)
{
    FWObject::iterator fi=std::find(begin(), end(), obj);
    if(fi!=end())
    {
        erase(fi);
        setDirty(true);
        obj->unref();
        if(delete_if_last && !obj->ref_counter)
            delete obj;
    }
}

class RemoveChild 
{
    FWObject *r;

    public:

    RemoveChild(FWObject *_o) { r=_o; }
    
    void operator()(FWObject *obj) 
    {
	std::for_each(obj->begin(), obj->end(), RemoveChild(r));
        obj->remove(r, false);
    }
};

void FWObject::removeAllInstances(FWObject *rm)
{
    removeAllReferences(rm);
    (RemoveChild(rm))(this);
    
    if(!rm->ref_counter) 
        delete rm;
}

void FWObject::findAllReferences(const FWObject *obj, std::set<FWReference*> &res)
{
    for(vector<FWObject*>::iterator m=begin(); m!=end(); ++m) 
    {
        FWObject *o=*m;
        FWReference *oref=FWReference::cast(o);
        if(oref)
        {
	    if(oref->getPointer()==obj) 
                res.insert(oref);
	} else
        {
            o->findAllReferences(obj, res);
        }
    }
}

set<FWReference*> FWObject::findAllReferences(const FWObject *obj)
{
    set<FWReference*> res;
    findAllReferences(obj, res);
    return res;
}

void FWObject::removeRef(FWObject *obj)
{
    for(vector<FWObject*>::iterator m=begin(); m!=end(); ++m) 
    {
        FWObject *o=*m;
        if(FWReference::cast(o)!=NULL)
        {
	    FWReference *oref=FWReference::cast(o);
	    if(oref && oref->getPointer()==obj)
            {
                // do not delete object even if this reference was the last one (?)
		obj->unref();  
                // remove will delete o because reference counter goes to zero
		FWObject::remove(o); 
		return;
	    }
	}
    }
}

class RemoveReference {
    FWObject *r;

    public:
    RemoveReference(FWObject *_o) { r=_o; }
    void operator()(FWObject *obj) {
	std::for_each(obj->begin(), obj->end(), RemoveReference(r) );
	obj->removeRef(r);
    }
};

void FWObject::removeAllReferences(FWObject *rm)
{
    (RemoveReference(rm))(this);
//    std::for_each(begin(), end(), RemoveReference(obj) );
}

bool FWObject::validateChild(FWObject *obj)
{ 
    return true;

    /*
     *  Check if object "this" is a descendant of object "obj" to avoid loops
     *
     *  check disabled for now since we need to be able to add firewall to its
     *  own policy
     */
    FWObject *p;
    p=this;
    do {
	if (p==obj) return false;
    } while ((p=p->getParent())!=NULL);
    return true;
}

void FWObject::clearChildren()
{
    for(vector<FWObject*>::iterator m=begin(); m!=end(); ++m) 
    {
        FWObject *o=*m;
        o->unref();
        if(o->ref_counter==0) 
            delete o;
    }
    clear();
    setDirty(true);
}

void FWObject::clearChildren(const string &type_name)
{
 loop:
    for(vector<FWObject*>::iterator m=begin(); m!=end(); ++m) 
    {
        FWObject *o=(*m);
        if(o->getTypeName()==type_name)
        {
	    remove(o); 
            goto loop;
        }
    }
}

int FWObject::getChildrenCount()
{
    return(size());
}

FWObject* FWObject::getRoot()
{
    FWObject *p=this;

    while(p->getParent()!=NULL) p=p->getParent();
    return p;
}


FWObject* FWObject::getById  (const string &id, bool recursive,bool dereference)
{
    if(id==getId())
        return this;
    
    vector<FWObject*>::iterator j;
    for(j=begin(); j!=end(); ++j)     
    {
        FWObject *o=*j;
        string oid=o->getId();
        if(id==oid) 
            return o;
        /*
         *  if we follow references, do it only for ID comparison. Do not
         *  follow references recursively as it poses danger of loops
         */
        if (dereference && FWReference::cast(o)!=NULL) {
            FWObject *o1=FWReference::cast(o)->getPointer();
            if (o1 && id==o1->getId() )
                return o1;
        }

        if(recursive)  
        {
            o=o->getById(id, true);
            if(o) 
                return o;
        }
    }
    return NULL; // not found
}


FWObject* FWObject::getFirstByType(const string &type_name)
{
    iterator i=find_if(begin(),end(), FWObjectTypeNameEQPredicate(type_name));
    return i==end()?NULL:(*i);
}

vector<FWObject*> FWObject::getByType(const string &type_name)
{
    vector<FWObject*> res;
    for (iterator i=begin(); i!=end(); ++i) {
	i=find_if( i, end(), FWObjectTypeNameEQPredicate(type_name));
	if (i==end()) break;
	res.push_back(*i);
    } 
    return res; 
}

void  FWObject::setDirty(bool f, bool recursive)
{
    dirty=f;
    if(recursive) 
	for(vector<FWObject*>::iterator m=begin(); m!=end(); ++m) 
            (*m)->setDirty(f, TRUE);
}

bool FWObject::isDirty(bool recursive)
{
    if(dirty)
        return TRUE;

    if(recursive) 
    {
	for(vector<FWObject*>::iterator m=begin(); m!=end(); ++m)
            if((*m)->isDirty(recursive))
                return TRUE;
    }
    return FALSE;
}

bool FWObject::GUISortOrder(const FWObject *a, const FWObject *b)
{
    return a->getName() < b->getName();
}

class sort_order_func_adaptor 
{
    private:
    
    FWObject *o;
    
    public:
    
    explicit sort_order_func_adaptor(FWObject *p) { o=p; }
    
    bool operator()(const FWObject *a, const FWObject *b) 
    { 
        return o->GUISortOrder(a,b);
    }
};

void FWObject::sortChildren()
{
    std::sort(begin(), end(), sort_order_func_adaptor(this));
    setDirty(true);
}
