/** -*- C++ -*-
    @file cache/version.h
    @author Peter Rockai <me@mornfall.net>
*/

#include <wibble/mixin.h>
#include <wibble/fallback.h>
#include <wibble/shared.h>
#include <ept/forward.h>
#include <ept/cache/entity.h>

#ifndef EPT_CACHE_VERSION
#define EPT_CACHE_VERSION

namespace ept {
namespace t {
namespace cache {

template< typename C >
struct Version : EntityMixin< Version< C > >
{
    typedef typename C::VersionPointer Pointer;
    typedef typename C::Package Package;
    typedef typename C::Relation Relation;

    Version( Pointer p ) : m_pointer( p ) {}
    Version() {}

    bool valid() const { return pointer().valid(); }
    Pointer &pointer() { return m_pointer; }
    const Pointer &pointer() const { return m_pointer; }
    const typename C::Aggregator &aggregator() const { return pointer().aggregator(); }

    Package package() const {
        return valid() ? aggregator().index().packageForVersion( *this ) : Package();
    }

    long id() const { return pointer().id(); }

    std::string versionString( wibble::Fallback< std::string > d = wibble::Error() ) const {
        return valid() ? aggregator().index().versionString( *this ) : d.get();
    }

    typedef wibble::Fallback< std::string > FallbackString;

    std::string shortDescription( FallbackString d = wibble::Error() ) const {
        return valid() ? record().shortDescription : d.get();
    }

    std::string longDescription( FallbackString d = wibble::Error() ) const {
        return valid() ? record().longDescription : d.get();
    }

    std::string name( FallbackString d = wibble::Error() ) const {
        return valid() ? package().name() : d.get();
    }

    std::string maintainer( FallbackString d = wibble::Error() ) const {
        return valid() ? record().maintainer : d.get();
    }

    std::string architecture( FallbackString d = wibble::Error() ) const {
        return valid() ? aggregator().index().architecture( *this ) : d.get();
    }

    std::string section( FallbackString d = wibble::Error() ) const {
        return valid() ? aggregator().index().section( *this ) : d.get();
    }

    std::string source( FallbackString d = wibble::Error() ) const {
        return valid() ? record().source : d.get();
    }

    std::string fileName( FallbackString d = wibble::Error() ) const {
        return valid() ? record().fileName : d.get();
    }

    std::string md5sum( FallbackString d = wibble::Error() ) const {
        return valid() ? record().md5sum : d.get();
    }

    int installedSize( wibble::Fallback< int > d = wibble::Error() ) const {
        return valid() ? aggregator().index().installedSize( *this ) : d.get();
    }

    std::string installedSizeString( wibble::Fallback< std::string > d = wibble::Error() ) const {
        return valid() ? C::State::sizeString( installedSize() ) : d.get();
    }

    wibble::Range< Relation > relations() const {
        wibble::SharedVector< Relation > &r = wibble::shared< wibble::SharedVector< Relation > >();
        Relation rel = aggregator().index().firstRelationForVersion( *this );
        while( rel.valid() ) {
            consumer( r ).consume( rel );
            rel = rel.nextInCache();
        }
        return range( r );
    }
    // wibble::Range< Relation > depends() const;

    // XXX what here?
    std::string gettext( std::string x ) { return x; }
    std::string completeRecord() {
        std::string na = /*getstring*/( "not available" );
        return std::string("Name: ") + package().name( na ) +
            gettext( "\nPriority: not available" ) +
            gettext( "\nSection: " ) + section( na ) +
            gettext( "\nInstalled-Size: " ) + installedSizeString( na ) +
            gettext( "\nMaintainer: " ) + maintainer( na ) +
            gettext( "\nArchitecture: " ) + architecture( na ) +
            gettext( "\nVersion: "  ) + versionString( na ) +
            gettext( "\nDepends: not available" ) +
            gettext( "\nFilename: " ) + fileName( na ) +
            gettext( "\nSize: not available" ) +
            gettext( "\nMD5Sum: " ) + md5sum( na ) +
            gettext( "\nDescription: " ) + shortDescription( na ) + "\n" + longDescription( na );
    }


protected:
    const typename C::Records::Record &record() const {
        return aggregator().records().record( *this );
    }
    Pointer m_pointer;

};

}
}
}

#endif
